/* PSPP - computes sample statistics.
   Copyright (C) 1997-9, 2000 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <stdlib.h>
#include "avl.h"
#include "command.h"
#include "error.h"
#include "file-handle.h"
#include "lexer.h"
#include "sfm.h"
#include "str.h"
#include "var.h"

#undef DEBUGGING
/*#define DEBUGGING 1*/
#include "debug-print.h"

/* Parses and executes APPLY DICTIONARY. */
int
cmd_apply_dictionary (void)
{
  struct file_handle *handle;
  struct dictionary *dict;

  int n_matched = 0;

  int i;
  
  lex_match_id ("APPLY");
  lex_match_id ("DICTIONARY");
  
  lex_match_id ("FROM");
  lex_match ('=');
  handle = fh_parse_file_handle ();
  if (!handle)
    return CMD_FAILURE;

  dict = sfm_read_dictionary (handle, NULL);
  if (dict == NULL)
    return CMD_FAILURE;

  for (i = 0; i < dict->nvar; i++)
    {
      struct variable *s = dict->var[i];
      struct variable *t = find_variable (s->name);
      if (t == NULL)
	continue;

      n_matched++;
      if (s->type != t->type)
	{
	  msg (SW, _("Variable %s is %s in target file, but %s in "
		     "source file."),
	       s->name,
	       t->type == ALPHA ? _("string") : _("numeric"),
	       s->type == ALPHA ? _("string") : _("numeric"));
	  continue;
	}

      if (s->label && strcspn (s->label, " ") != strlen (s->label))
	{
	  free (t->label);
	  t->label = s->label;
	  s->label = NULL;
	}

      if (s->val_lab && t->width > MAX_SHORT_STRING)
	msg (SW, _("Cannot add value labels from source file to "
		   "long string variable %s."),
	     s->name);
      else if (s->val_lab)
	{
	  if (t->width < s->width)
	    {
	      avl_traverser iter;
	      struct value_label *lab;

	      avl_traverser_init (iter);
	      while ((lab = avl_traverse (s->val_lab, &iter)) != NULL)
		{
		  int j;

		  /* If the truncated characters aren't all blanks
		     anyway, then don't apply the value labels. */
		  for (j = t->width; j < s->width; j++)
		    if (lab->v.s[j] != ' ')
		      goto skip_value_labels;
		}
	    }
	  else
	    {
	      /* Fortunately, we follow the convention that all value
		 label values are right-padded with spaces, so it is
		 unnecessary to bother padding values here. */
	    }
	  
	  avl_destroy (t->val_lab, free_val_lab);
	  t->val_lab = s->val_lab;
	  s->val_lab = NULL;
	}
    skip_value_labels: ;

      if (s->miss_type != MISSING_NONE && t->width > MAX_SHORT_STRING)
	msg (SW, _("Cannot apply missing values from source file to "
		   "long string variable %s."),
	     s->name);
      else if (s->miss_type != MISSING_NONE)
	{
	  if (t->width < s->width)
	    {
	      static const int miss_count[MISSING_COUNT] = 
		{
		  0, 1, 2, 3, 2, 1, 1, 3, 2, 2,
		};

	      int j, k;
	      
	      for (j = 0; j < miss_count[s->miss_type]; j++)
		for (k = t->width; k < s->width; k++)
		  if (s->missing[j].s[k] != ' ')
		    goto skip_missing_values;
	    }

	  t->miss_type = s->miss_type;
	  memcpy (t->missing, s->missing, sizeof s->missing);
	}

      if (s->type == NUMERIC)
	{
	  t->print = s->print;
	  t->write = s->write;
	}
    }

  if (!n_matched)
    msg (SW, _("No matching variables found between the source "
	       "and target files."));
      
  /* Weighting. */
  {
    const int tfw = find_variable (default_dict.weight_var) != 0;
    const int sfw = dict->weight_var[0] != 0;
    struct variable *w;

    switch (10 * tfw + sfw)
      {
      case 10:
	/* The working file retains its weighting variable. */
	break;

      case 00:
      case 01:
	/* Fall through to case 11. */

      case 11:
	w = find_variable (dict->weight_var);
	if (w)
	  strcpy (default_dict.weight_var, dict->weight_var);
	break;
      }
  }
 skip_missing_values: ;
  
  sfm_maybe_close (handle);

  return lex_end_of_command ();
}
