/****************************************************************************
** profiles.cpp - deal with profiles
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include"profiles.h"
#include"common.h"
#include<qdir.h>
#include<qfileinfo.h>
#include<qdom.h>

#include<qapplication.h>
#include"eventdlg.h"
#include"chatdlg.h"


static QDomElement textTag(QDomDocument &doc, const QString &name, const QString &content)
{
	QDomElement tag = doc.createElement(name);
	QDomText text = doc.createTextNode(content);
	tag.appendChild(text);

	return tag;
}

static QDomElement textTag(QDomDocument &doc, const QString &name, int content)
{
	QDomElement tag = doc.createElement(name);
	QDomText text = doc.createTextNode(QString::number(content));
	tag.appendChild(text);

	return tag;
}

static QDomElement textTag(QDomDocument &doc, const QString &name, bool content)
{
	QDomElement tag = doc.createElement(name);
	QDomText text = doc.createTextNode(content ? "true" : "false");
	tag.appendChild(text);

	return tag;
}

static QDomElement textTag(QDomDocument &doc, const QString &name, QSize s)
{
	QString str;
	str.sprintf("%d,%d", s.width(), s.height());

	QDomElement tag = doc.createElement(name);
	QDomText text = doc.createTextNode(str);
	tag.appendChild(text);

	return tag;
}

static QDomElement textTag(QDomDocument &doc, const QString &name, QRect r)
{
	QString str;
	str.sprintf("%d,%d,%d,%d", r.x(), r.y(), r.width(), r.height());

	QDomElement tag = doc.createElement(name);
	QDomText text = doc.createTextNode(str);
	tag.appendChild(text);

	return tag;
}

static QString tagContent(const QDomElement &e)
{
	// look for some tag content
	for(QDomNode n = e.firstChild(); !n.isNull(); n = n.nextSibling()) {
		QDomText i = n.toText();
		if(i.isNull())
			continue;
		return i.data();
	}

	return "";
}

static QDomElement findSubTag(const QDomElement &e, const QString &name, bool *found)
{
	if(found)
		*found = FALSE;

	for(QDomNode n = e.firstChild(); !n.isNull(); n = n.nextSibling()) {
		QDomElement i = n.toElement();
		if(i.isNull())
			continue;
		if(i.tagName() == name) {
			if(found)
				*found = TRUE;
			return i;
		}
	}

	QDomElement tmp;
	return tmp;
}

static void readEntry(const QDomElement &e, const QString &name, QString *v)
{
	bool found = FALSE;
	QDomElement tag = findSubTag(e, name, &found);
	if(!found)
		return;
	*v = tagContent(tag);
}

static void readNumEntry(const QDomElement &e, const QString &name, int *v)
{
	bool found = FALSE;
	QDomElement tag = findSubTag(e, name, &found);
	if(!found)
		return;
	*v = tagContent(tag).toInt();
}

static void readBoolEntry(const QDomElement &e, const QString &name, bool *v)
{
	bool found = FALSE;
	QDomElement tag = findSubTag(e, name, &found);
	if(!found)
		return;
	*v = (tagContent(tag) == "true") ? TRUE: FALSE;
}

static void readSizeEntry(const QDomElement &e, const QString &name, QSize *v)
{
	bool found = FALSE;
	QDomElement tag = findSubTag(e, name, &found);
	if(!found)
		return;
	QStringList list = QStringList::split(',', tagContent(tag));
	if(list.count() != 2)
		return;
	QSize s;
	s.setWidth(list[0].toInt());
	s.setHeight(list[1].toInt());
	*v = s;
}

static void readRectEntry(const QDomElement &e, const QString &name, QRect *v)
{
	bool found = FALSE;
	QDomElement tag = findSubTag(e, name, &found);
	if(!found)
		return;
	QStringList list = QStringList::split(',', tagContent(tag));
	if(list.count() != 4)
		return;
	QRect r;
	r.setX(list[0].toInt());
	r.setY(list[1].toInt());
	r.setWidth(list[2].toInt());
	r.setHeight(list[3].toInt());
	*v = r;
}

static void readColorEntry(const QDomElement &e, const QString &name, QColor *v)
{
	bool found = FALSE;
	QDomElement tag = findSubTag(e, name, &found);
	if(!found)
		return;
	QColor c;
	c.setNamedColor(tagContent(tag));
	if(c.isValid())
		*v = c;
}

static void setBoolAttribute(QDomElement e, const QString &name, bool b)
{
	e.setAttribute(name, b ? "true" : "false");
}

static void readBoolAttribute(QDomElement e, const QString &name, bool *v)
{
	if(e.hasAttribute(name)) {
		QString s = e.attribute(name);
		*v = (s == "true") ? TRUE: FALSE;
	}
}


UserAccount::UserAccount()
{
	reset();
}

void UserAccount::reset()
{
	name = "Default";
	opt_auto = FALSE;
	opt_ssl = FALSE;
	tog_offline = TRUE;
	tog_away = TRUE;
	tog_agents = TRUE;
	user = "";
	pass = "";
	host = "";
	opt_pass = FALSE;
	port = 5222;
	resource = "Psi";
	priority = 5;
	olr_string = "";
	opt_keepAlive = TRUE;
	opt_plain = FALSE;
	opt_log = TRUE;
	opt_reconn = FALSE;

	roster.clear();
}

QString UserAccount::jid() const
{
	return QString("%1@%2").arg(user).arg(host);
}

// FIXME: this should be a const function (as should other ::toXml functions)
QDomElement UserAccount::toXml(QDomDocument &doc, const QString &tagName)
{
	QDomElement a = doc.createElement(tagName);

	setBoolAttribute(a, "auto", opt_auto);
	setBoolAttribute(a, "ssl", opt_ssl);
	setBoolAttribute(a, "showOffline", tog_offline);
	setBoolAttribute(a, "showAway", tog_away);
	setBoolAttribute(a, "showAgents", tog_agents);
	setBoolAttribute(a, "keepAlive", opt_keepAlive);
	setBoolAttribute(a, "plain", opt_plain);
	setBoolAttribute(a, "log", opt_log);
	setBoolAttribute(a, "reconn", opt_reconn);

	a.appendChild(textTag(doc, "name", name));
	a.appendChild(textTag(doc, "username", user));
	if(opt_pass)
		a.appendChild(textTag(doc, "password", encodePassword(pass, jid()) ));
	a.appendChild(textTag(doc, "host", host));
	a.appendChild(textTag(doc, "port", QString::number(port)));
	a.appendChild(textTag(doc, "resource", resource));
	a.appendChild(textTag(doc, "priority", QString::number(priority)));
	a.appendChild(textTag(doc, "OLR", olr_string));

	QDomElement r = doc.createElement("roster");
	a.appendChild(r);
	for(JabRosterEntry *i = roster.first(); i; i = roster.next()) {
		QDomElement tag = doc.createElement("item");
		tag.setAttribute("jid", i->jid);
		tag.setAttribute("name", i->nick);
		tag.setAttribute("subscription", i->sub);
		for(QStringList::ConstIterator it = i->groups.begin(); it != i->groups.end(); ++it)
			tag.appendChild(textTag(doc, "group", *it));
		r.appendChild(tag);
	}

	return a;
}

void UserAccount::fromXml(const QDomElement &a)
{
	reset();

	readEntry(a, "name", &name);
	readBoolAttribute(a, "auto", &opt_auto);
	readBoolAttribute(a, "ssl", &opt_ssl);
	readBoolAttribute(a, "showOffline", &tog_offline);
	readBoolAttribute(a, "showAway", &tog_away);
	readBoolAttribute(a, "showAgents", &tog_agents);
	readBoolAttribute(a, "keepAlive", &opt_keepAlive);
	readBoolAttribute(a, "plain", &opt_plain);
	readBoolAttribute(a, "log", &opt_log);
	readBoolAttribute(a, "reconn", &opt_reconn);

	readEntry(a, "username", &user);
	readEntry(a, "host", &host);
	readEntry(a, "password", &pass);
	if(pass.isEmpty())
		opt_pass = FALSE;
	else {
		pass = decodePassword(pass, jid());
		opt_pass = TRUE;
	}
	readNumEntry(a, "port", &port);
	readEntry(a, "resource", &resource);
	readNumEntry(a, "priority", &priority);
	readEntry(a, "OLR", &olr_string);

	bool found;
	QDomElement r = findSubTag(a, "roster", &found);
	if(found) {
		for(QDomNode n = r.firstChild(); !n.isNull(); n = n.nextSibling()) {
			QDomElement i = n.toElement();
			if(i.isNull())
				continue;

			if(i.tagName() == "item") {
				JabRosterEntry *entry = new JabRosterEntry;

				entry->jid = i.attribute("jid");
				entry->sub = i.attribute("subscription");
				entry->nick = i.attribute("name");

				// grab the groups
				for(QDomNode n = i.firstChild(); !n.isNull(); n = n.nextSibling()) {
					QDomElement tag = n.toElement();
					if(tag.isNull())
						continue;

					if(tag.tagName() == "group")
						entry->groups.append(tagContent(tag));
				}
				roster.add(entry);
			}
		}
	}
}

UserProfile::UserProfile()
{
	reset();
}

void UserProfile::reset()
{
	bool nix, win, mac;
	nix = win = mac = FALSE;

#ifdef Q_WS_X11
	nix = TRUE;
#endif
#ifdef Q_WS_WIN
	win = TRUE;
#endif
#ifdef Q_WS_MAC
	mac = TRUE;
#endif

	progver = PROG_VERSION;

	// global
	mwgeom.setRect(64, 64, 150, 360);
	lastStatusString = "";
	useSound = TRUE;

	acc.clear();

	// prefs
	prefs.useleft = FALSE;
	prefs.singleclick = FALSE;
	prefs.defaultAction = 0;
	prefs.delChats = dcHour;
	prefs.browser = 0;
	prefs.customBrowser = "";
	prefs.customMailer = "";
	prefs.alwaysOnTop = win ? TRUE : FALSE;
	prefs.keepSizes = FALSE;
	prefs.ignoreHeadline = FALSE;
	prefs.useDock = win ? TRUE: FALSE;
	prefs.dockDCstyle = win ? TRUE: FALSE;
	prefs.alertStyle = 2;
	prefs.popupMsgs = FALSE;
	prefs.popupChats = FALSE;
	prefs.raise = FALSE;
	prefs.incomingAs = 0;
	prefs.outgoingAs = 0;
	prefs.askOnline = FALSE;
	prefs.rosterAnim = TRUE;
	prefs.asAway = 10;
	prefs.asXa = 30;
	prefs.asOffline = 0;
	prefs.scrollTo = TRUE;

	prefs.sp.clear();
	prefs.sp.set(QObject::tr("Away from desk"), QObject::tr("I am away from my desk.  Leave a message."));
	prefs.sp.set(QObject::tr("Showering"), QObject::tr("I'm in the shower.  You'll have to wait for me to get out."));
	prefs.sp.set(QObject::tr("Eating"), QObject::tr("Out eating.  Mmmm.. food."));
	prefs.sp.set(QObject::tr("Sleep"), QObject::tr("Sleep is good.  Zzzzz"));
	prefs.sp.set(QObject::tr("Work"), QObject::tr("Can't chat.  Gotta work."));
	prefs.sp.set(QObject::tr("Air"), QObject::tr("Stepping out to get some fresh air."));
	prefs.sp.set(QObject::tr("Movie"), QObject::tr("Out to a movie.  Is that OK with you?"));
	prefs.sp.set(QObject::tr("Secret"), QObject::tr("I'm not available right now and that's all you need to know."));
	prefs.sp.set(QObject::tr("Out for the night"), QObject::tr("Out for the night."));
	prefs.sp.set(QObject::tr("Greece"), QObject::tr("I have gone to a far away place.  I will be back someday!"));

	prefs.iconset = "stellar";

	prefs.color[cOnline]    = QColor("#0060C0");
	prefs.color[cListBack]  = QColor("#C0C0C0");
	prefs.color[cAway]      = QColor("#008080");
	prefs.color[cDND]       = QColor("#800000");
	prefs.color[cOffline]   = QColor("#000000");
	prefs.color[cGroupFore] = QColor("#000000");
	prefs.color[cGroupBack] = QColor("#FFFFFF");

	prefs.font[fRoster] = QApplication::font().toString();
	prefs.font[fMessage] = QApplication::font().toString();
	prefs.font[fChat] = QApplication::font().toString();

	prefs.player = "play";
	prefs.noAwaySound = FALSE;

	prefs.onevent[eMessage] = g.pathBase + "/sound/chat2.wav";
	prefs.onevent[eChat1]   = g.pathBase + "/sound/chat1.wav";
	prefs.onevent[eChat2]   = g.pathBase + "/sound/chat2.wav";
	prefs.onevent[eSystem]  = g.pathBase + "/sound/chat2.wav";
	prefs.onevent[eOnline]  = g.pathBase + "/sound/online.wav";
	prefs.onevent[eOffline] = g.pathBase + "/sound/offline.wav";
	prefs.onevent[eSend]    = g.pathBase + "/sound/send.wav";

	prefs.sizeEventDlg = EventDlg::defaultSize();
	prefs.sizeChatDlg = ChatDlg::defaultSize();
}

bool UserProfile::toFile(const QString &fname)
{
//	FILE *f = fopen("c:\\psi.txt", "w");
//	fprintf(f, "would have opened [%s]\n", fname.latin1());
//	fclose(f);
//	return;

	QDomDocument doc;

	QDomElement base = doc.createElement("psiconf");
	base.setAttribute("version", "1.0");
	doc.appendChild(base);

	base.appendChild(textTag(doc, "progver", PROG_VERSION));
	base.appendChild(textTag(doc, "geom", mwgeom));
	base.appendChild(textTag(doc, "lastStatusString", lastStatusString));
	base.appendChild(textTag(doc, "useSound", useSound));

	QDomElement accs = doc.createElement("accounts");
	base.appendChild(accs);
	for(UserAccountList::Iterator ai = acc.begin(); ai != acc.end(); ++ai)
		accs.appendChild((*ai).toXml(doc, "account"));

	QDomElement p = doc.createElement("preferences");
	base.appendChild(p);

	{
		QDomElement p_general = doc.createElement("general");
		p.appendChild(p_general);

		{
			QDomElement p_roster = doc.createElement("roster");
			p_general.appendChild(p_roster);

			p_roster.appendChild(textTag(doc, "useleft", prefs.useleft));
			p_roster.appendChild(textTag(doc, "singleclick", prefs.singleclick));
			p_roster.appendChild(textTag(doc, "defaultAction", prefs.defaultAction));
		}
		{
			QDomElement p_misc = doc.createElement("misc");
			p_general.appendChild(p_misc);

			p_misc.appendChild(textTag(doc, "delChats", prefs.delChats));
			p_misc.appendChild(textTag(doc, "browser", prefs.browser));
			p_misc.appendChild(textTag(doc, "customBrowser", prefs.customBrowser));
			p_misc.appendChild(textTag(doc, "customMailer", prefs.customMailer));
			p_misc.appendChild(textTag(doc, "alwaysOnTop", prefs.alwaysOnTop));
			p_misc.appendChild(textTag(doc, "keepSizes", prefs.keepSizes));
			p_misc.appendChild(textTag(doc, "ignoreHeadline", prefs.ignoreHeadline));
			p_misc.appendChild(textTag(doc, "scrollTo", prefs.scrollTo));
		}
		{
			QDomElement p_dock = doc.createElement("dock");
			p_general.appendChild(p_dock);

			p_dock.appendChild(textTag(doc, "useDock", prefs.useDock));
			p_dock.appendChild(textTag(doc, "dockDCstyle", prefs.dockDCstyle));
		}
	}

	{
		QDomElement p_events = doc.createElement("events");
		p.appendChild(p_events);

		p_events.appendChild(textTag(doc, "alertstyle", prefs.alertStyle));

		{
			QDomElement p_receive = doc.createElement("receive");
			p_events.appendChild(p_receive);

			p_receive.appendChild(textTag(doc, "popupMsgs", prefs.popupMsgs));
			p_receive.appendChild(textTag(doc, "popupChats", prefs.popupChats));
			p_receive.appendChild(textTag(doc, "raise", prefs.raise));
			p_receive.appendChild(textTag(doc, "incomingAs", prefs.incomingAs));
		}
		{
			QDomElement p_send = doc.createElement("send");
			p_events.appendChild(p_send);

			p_send.appendChild(textTag(doc, "outgoingAs", prefs.outgoingAs));
		}
	}

	{
		QDomElement p_pres = doc.createElement("presence");
		p.appendChild(p_pres);

		{
			QDomElement tag = doc.createElement("misc");
			p_pres.appendChild(tag);

			tag.appendChild(textTag(doc, "askOnline", prefs.askOnline));
			tag.appendChild(textTag(doc, "rosterAnim", prefs.rosterAnim));
		}
		{
			QDomElement tag = doc.createElement("autostatus");
			p_pres.appendChild(tag);

			tag.appendChild(textTag(doc, "away", prefs.asAway));
			tag.appendChild(textTag(doc, "xa", prefs.asXa));
			tag.appendChild(textTag(doc, "offline", prefs.asOffline));
		}
		{
			p_pres.appendChild(prefs.sp.toXml(doc, "statuspresets"));
		}
	}

	{
		QDomElement p_lnf = doc.createElement("lookandfeel");
		p.appendChild(p_lnf);

		p_lnf.appendChild(textTag(doc, "iconset", prefs.iconset));

		{
			QDomElement tag = doc.createElement("colors");
			p_lnf.appendChild(tag);

			tag.appendChild(textTag(doc, "online", prefs.color[cOnline].name() ));
			tag.appendChild(textTag(doc, "listback", prefs.color[cListBack].name() ));
			tag.appendChild(textTag(doc, "away", prefs.color[cAway].name() ));
			tag.appendChild(textTag(doc, "dnd", prefs.color[cDND].name() ));
			tag.appendChild(textTag(doc, "offline", prefs.color[cOffline].name() ));
			tag.appendChild(textTag(doc, "groupfore", prefs.color[cGroupFore].name() ));
			tag.appendChild(textTag(doc, "groupback", prefs.color[cGroupBack].name() ));
		}

		{
			QDomElement tag = doc.createElement("fonts");
			p_lnf.appendChild(tag);

			tag.appendChild(textTag(doc, "roster", prefs.font[fRoster] ));
			tag.appendChild(textTag(doc, "message", prefs.font[fMessage] ));
			tag.appendChild(textTag(doc, "chat", prefs.font[fChat] ));
		}
	}

	{
		QDomElement p_sound = doc.createElement("sound");
		p.appendChild(p_sound);

		p_sound.appendChild(textTag(doc, "player", prefs.player));
		p_sound.appendChild(textTag(doc, "noawaysound", prefs.noAwaySound));

		{
			QDomElement p_onevent = doc.createElement("onevent");
			p_sound.appendChild(p_onevent);

			p_onevent.appendChild(textTag(doc, "message", prefs.onevent[eMessage]));
			p_onevent.appendChild(textTag(doc, "chat1", prefs.onevent[eChat1]));
			p_onevent.appendChild(textTag(doc, "chat2", prefs.onevent[eChat2]));
			p_onevent.appendChild(textTag(doc, "system", prefs.onevent[eSystem]));
			p_onevent.appendChild(textTag(doc, "online", prefs.onevent[eOnline]));
			p_onevent.appendChild(textTag(doc, "offline", prefs.onevent[eOffline]));
			p_onevent.appendChild(textTag(doc, "send", prefs.onevent[eSend]));
		}
	}

	{
		QDomElement p_sizes = doc.createElement("sizes");
		p.appendChild(p_sizes);

		p_sizes.appendChild(textTag(doc, "eventdlg", prefs.sizeEventDlg));
		p_sizes.appendChild(textTag(doc, "chatdlg", prefs.sizeChatDlg));
	}

	QFile f(fname);
	if(!f.open(IO_WriteOnly))
		return FALSE;
	QTextStream t;
	t.setDevice(&f);
	t.setEncoding(QTextStream::UnicodeUTF8);
	t << doc.toString();
	t.unsetDevice();
	f.close();

	return TRUE;
}

bool UserProfile::fromFile(const QString &fname)
{
	QString confver;
	QDomDocument doc;

	QFile f(fname);
	if(!f.open(IO_ReadOnly))
		return FALSE;
	if(!doc.setContent(&f))
		return FALSE;
	f.close();

	QDomElement base = doc.documentElement();
	if(base.tagName() != "psiconf")
		return FALSE;
	confver = base.attribute("version");
	if(confver != "1.0")
		return FALSE;

	readEntry(base, "progver", &progver);

	readRectEntry(base, "geom", &mwgeom);
	readEntry(base, "lastStatusString", &lastStatusString);
	readBoolEntry(base, "usesound", &useSound);

	bool found;
	QDomElement accs = findSubTag(base, "accounts", &found);
	if(found) {
		for(QDomNode n = accs.firstChild(); !n.isNull(); n = n.nextSibling()) {
			QDomElement a = n.toElement();
			if(a.isNull())
				continue;

			if(a.tagName() == "account") {
				UserAccount ua;
				ua.fromXml(a);
				acc.append(ua);
			}
		}
	}

	QDomElement p = findSubTag(base, "preferences", &found);
	if(found) {
		bool found;

		QDomElement p_general = findSubTag(p, "general", &found);
		if(found) {
			bool found;

			QDomElement p_roster = findSubTag(p_general, "roster", &found);
			if(found) {
				readBoolEntry(p_roster, "useleft", &prefs.useleft);
				readBoolEntry(p_roster, "singleclick", &prefs.singleclick);
				readNumEntry(p_roster, "defaultAction", &prefs.defaultAction);
			}

			QDomElement tag = findSubTag(p_general, "misc", &found);
			if(found) {
				readNumEntry(tag, "delChats", &prefs.delChats);
				readNumEntry(tag, "browser", &prefs.browser);
				readEntry(tag, "customBrowser", &prefs.customBrowser);
				readEntry(tag, "customMailer", &prefs.customMailer);
				readBoolEntry(tag, "alwaysOnTop", &prefs.alwaysOnTop);
				readBoolEntry(tag, "keepSizes", &prefs.keepSizes);
				readBoolEntry(tag, "ignoreHeadline", &prefs.ignoreHeadline);
				readBoolEntry(tag, "scrollTo", &prefs.scrollTo);
			}

			tag = findSubTag(p_general, "dock", &found);
			if(found) {
				readBoolEntry(tag, "useDock", &prefs.useDock);
				readBoolEntry(tag, "dockDCstyle", &prefs.dockDCstyle);
			}
		}

		QDomElement p_events = findSubTag(p, "events", &found);
		if(found) {
			bool found;

			readNumEntry(p_events, "alertstyle", &prefs.alertStyle);

			QDomElement tag = findSubTag(p_events, "receive", &found);
			if(found) {
				readBoolEntry(tag, "popupMsgs", &prefs.popupMsgs);
				readBoolEntry(tag, "popupChats", &prefs.popupChats);
				readBoolEntry(tag, "raise", &prefs.raise);
				readNumEntry(tag, "incomingAs", &prefs.incomingAs);
			}

			tag = findSubTag(p_events, "send", &found);
			if(found) {
				readNumEntry(tag, "outgoingAs", &prefs.outgoingAs);
			}
		}

		QDomElement p_pres = findSubTag(p, "presence", &found);
		if(found) {
			bool found;

			QDomElement tag = findSubTag(p_pres, "misc", &found);
			if(found) {
				readBoolEntry(tag, "askOnline", &prefs.askOnline);
				readBoolEntry(tag, "rosterAnim", &prefs.rosterAnim);
			}

			tag = findSubTag(p_pres, "autostatus", &found);
			if(found) {
				readNumEntry(tag, "away", &prefs.asAway);
				readNumEntry(tag, "xa", &prefs.asXa);
				readNumEntry(tag, "offline", &prefs.asOffline);
			}

			tag = findSubTag(p_pres, "statuspresets", &found);
			if(found)
				prefs.sp.fromXml(tag);
		}

		QDomElement p_lnf = findSubTag(p, "lookandfeel", &found);
		if(found) {
			bool found;

			readEntry(p_lnf, "iconset", &prefs.iconset);

			QDomElement tag = findSubTag(p_lnf, "colors", &found);
			if(found) {
				readColorEntry(tag, "online", &prefs.color[cOnline]);
				readColorEntry(tag, "listback", &prefs.color[cListBack]);
				readColorEntry(tag, "away", &prefs.color[cAway]);
				readColorEntry(tag, "dnd", &prefs.color[cDND]);
				readColorEntry(tag, "offline", &prefs.color[cOffline]);
				readColorEntry(tag, "groupfore", &prefs.color[cGroupFore]);
				readColorEntry(tag, "groupback", &prefs.color[cGroupBack]);
			}

			tag = findSubTag(p_lnf, "fonts", &found);
			if(found) {
				readEntry(tag, "roster", &prefs.font[fRoster]);
				readEntry(tag, "message", &prefs.font[fMessage]);
				readEntry(tag, "chat", &prefs.font[fChat]);
			}
		}

		QDomElement p_sound = findSubTag(p, "sound", &found);
		if(found) {
			bool found;

			readEntry(p_sound, "player", &prefs.player);
			readBoolEntry(p_sound, "noawaysound", &prefs.noAwaySound);

			QDomElement tag = findSubTag(p_sound, "onevent", &found);
			if(found) {
				readEntry(tag, "message", &prefs.onevent[eMessage]);
				readEntry(tag, "chat1", &prefs.onevent[eChat1]);
				readEntry(tag, "chat2", &prefs.onevent[eChat2]);
				readEntry(tag, "system", &prefs.onevent[eSystem]);
				readEntry(tag, "online", &prefs.onevent[eOnline]);
				readEntry(tag, "offline", &prefs.onevent[eOffline]);
				readEntry(tag, "send", &prefs.onevent[eSend]);
			}
		}

		QDomElement p_sizes = findSubTag(p, "sizes", &found);
		if(found) {
			readSizeEntry(p_sizes, "eventdlg", &prefs.sizeEventDlg);
			readSizeEntry(p_sizes, "chatdlg", &prefs.sizeChatDlg);
		}
	}

	return TRUE;
}


QString pathToProfile(const QString &name)
{
	return g.pathProfiles + "/" + name;
}

QString pathToProfileConfig(const QString &name)
{
	return pathToProfile(name) + "/config.xml";
}

QStringList getProfilesList()
{
	QStringList list;

	QDir d(g.pathProfiles);
	if(!d.exists())
		return list;

	QStringList entries = d.entryList();
	for(QStringList::Iterator it = entries.begin(); it != entries.end(); ++it) {
		if(*it == "." || *it == "..")
			continue;
		QFileInfo info(d, *it);
		if(!info.isDir())
			continue;

		list.append(*it);
	}

	list.sort();

	return list;
}

bool profileExists(const QString &_name)
{
	QString name = _name.lower();

	QStringList list = getProfilesList();
	for(QStringList::ConstIterator it = list.begin(); it != list.end(); ++it) {
		if((*it).lower() == name)
			return TRUE;
	}
	return FALSE;
}

bool profileNew(const QString &name)
{
	if(name.isEmpty())
		return FALSE;

	// verify the string is sane
	for(int n = 0; n < (int)name.length(); ++n) {
		if(!name.at(n).isLetterOrNumber())
			return FALSE;
	}

	// make it
	QDir d(g.pathProfiles);
	if(!d.exists())
		return FALSE;
	QDir p(g.pathProfiles + "/" + name);
	if(!p.exists()) {
		d.mkdir(name);
		if(!p.exists())
			return FALSE;
	}

	p.mkdir("history");
	p.mkdir("info");

	return TRUE;
}

bool profileRename(const QString &oldname, const QString &name)
{
	// verify the string is sane
	for(int n = 0; n < (int)name.length(); ++n) {
		if(!name.at(n).isLetterOrNumber())
			return FALSE;
	}

	// locate the folder
	QDir d(g.pathProfiles);
	if(!d.exists())
		return FALSE;
	if(!d.rename(oldname, name))
		return FALSE;

	return TRUE;
}

static bool folderRemove(const QDir &_d)
{
	QDir d = _d;

	QStringList entries = d.entryList();
	for(QStringList::Iterator it = entries.begin(); it != entries.end(); ++it) {
		if(*it == "." || *it == "..")
			continue;
		QFileInfo info(d, *it);
		if(info.isDir()) {
			if(!folderRemove(QDir(info.filePath())))
				return FALSE;
		}
		else {
			//printf("deleting [%s]\n", info.filePath().latin1());
			d.remove(info.filePath());
		}
	}
	QString name = d.dirName();
	if(!d.cdUp())
		return FALSE;
	//printf("removing folder [%s]\n", d.filePath(name).latin1());
	d.rmdir(name);

	return TRUE;
}

bool profileDelete(const QString &path)
{
	QDir d(path);
	if(!d.exists())
		return TRUE;

	return folderRemove(QDir(path));
}
