/****************************************************************************
** eventdlg.cpp - multi-window chat dialog for sending/receiving events
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include<qtimer.h>
#include<qiconset.h>
#include<qdatetime.h>
#include<qmessagebox.h>
#include<qgroupbox.h>
#include<qtooltip.h>
#include<qstringlist.h>
#include<qdragobject.h>
#include<qhbox.h>

#include"eventdlg.h"
#include"common.h"


/****************************************************************************
  AnimatedButtom - an animated button!
****************************************************************************/
AnimatedButton::AnimatedButton(QWidget *parent, const char *name)
:QPushButton(parent, name)
{
	setWFlags(getWFlags() | WRepaintNoErase);

	animStep = 0;
	v_isActive = FALSE;
}

void AnimatedButton::setAnim(Anim *_anim)
{
	if(_anim) {
		anim = _anim;
		animStep = 0;
		if(option.alertStyle == 0)
			changeIcon(anim->base());
		if(!v_isActive) {
			v_isActive = TRUE;

			t = new QTimer(this);
			connect(t, SIGNAL(timeout()), SLOT(animate()));
			t->start(120);
		}
	}
	else {
		// no icon
	        QIconSet icon;
		setIconSet(icon);

		anim = 0;
		if(v_isActive) {
			v_isActive = FALSE;
			delete t;
		}
	}
}

void AnimatedButton::animate()
{
	if(anim->isAnim() && option.alertStyle == 2) {
		changeIcon(anim->frame(animStep));

		++animStep;
		if(animStep >= anim->numFrames()-1)
			animStep = 0;
	}
	else if(option.alertStyle != 0) {
		if(animStep == 0)
			changeIcon(anim->base());
		else if(animStep == 4)
			changeIcon(*pix_blank);

		++animStep;
		if(animStep == 8)
			animStep = 0;
	}
}

void AnimatedButton::changeIcon(const QPixmap &pix)
{
	//setUpdatesEnabled(FALSE);
	setIconSet(pix);
	//setUpdatesEnabled(TRUE);
	//update();
}


/****************************************************************************
  ELineEdit - a line edit that handles dropping of Jids better
****************************************************************************/
ELineEdit::ELineEdit(QWidget *parent, const char *name)
:QLineEdit(parent, name)
{
	setAcceptDrops(TRUE);
}

void ELineEdit::dragEnterEvent(QDragEnterEvent *e)
{
	e->accept(QTextDrag::canDecode(e));
}

void ELineEdit::dropEvent(QDropEvent *e)
{
	QString str;

	if(QTextDrag::decode(e, str)) {
		if(text().isEmpty())
			setText(str);
		else {
			if(text().at(text().length()-1) == ',')
				setText(text() + str);
			else
				setText(text() + ',' + str);
		}
		setCursorPosition(text().length());

		repaint();
	}
}

/****************************************************************************
  EventDlg - a window to read and write events
****************************************************************************/
EventDlg::EventDlg(UserListItem *item, int x_localStatus, bool isUnique, bool compose, bool urlMode)
:QWidget(0, compose ? "compose" : "readmessage"), UniqueWindow(isUnique, "EventDlg", this, cleanJid(item->jid))
{
	// setup vars
	isCompose = compose;
	isMulti = FALSE;

	// save data internally. no external dependance!
	if(item) {
		v_jid = item->jid;
		v_nick = item->nick;
		v_status = item->state;
	}
	else
		v_jid = "";

	localStatus = x_localStatus;

	init();

	if(item) {
		le_jid->setText(item->jid);
		le_jid->setCursorPosition(0);
		QToolTip::add(le_jid, item->jid);
		updateContact(item);
	}

	if(urlMode) {
		tb_url->setOn(TRUE);
		le_url->setFocus();
	}
}

EventDlg::EventDlg(const QStringList &_to, int x_localStatus)
:QWidget(0, "compose"), UniqueWindow(FALSE, "EventDlg", this, "")
{
	isCompose = TRUE;
	isMulti = TRUE;

	localStatus = x_localStatus;

	init();

	QStringList to = _to;
	QString str;
	bool first = TRUE;
	for(QStringList::Iterator it = to.begin(); it != to.end(); ++it) {
		if(first == FALSE)
			str += ",";
		else
			first = FALSE;

		str += *it;
	}
	le_to->setText(str);
	le_to->setCursorPosition(0);

	if(!le_to->text().isEmpty())
		mle->setFocus();
}

void EventDlg::init()
{
	// set window flags
	setWFlags(getWFlags() | WDestructiveClose);

	// if multi (isCompose is implied here)
	if(isMulti)
		setCaption(tr("Multi"));

	msg = 0;

	// ---- setup the layout
	layout1 = new QVBoxLayout(this, 4);


	// ---- single
	lo_single = new QWidget(this);
	QHBoxLayout *lo_single_hb = new QHBoxLayout(lo_single);
	lb_statusIcon = new QLabel(lo_single);
	lo_single_hb->addWidget(lb_statusIcon);
	lo_single_hb->addSpacing(2);

	le_nick = new QLineEdit(lo_single);
	le_nick->setReadOnly(TRUE);
	le_nick->setFixedWidth(120);
	lo_single_hb->addWidget(le_nick);

	lo_single_hb->addSpacing(16);

	QLabel *lo_single_jid = new QLabel(tr("Jabber ID"), lo_single);
	lo_single_hb->addWidget(lo_single_jid);
	le_jid = new QLineEdit(lo_single);
	le_jid->setReadOnly(TRUE);
	lo_single_hb->addWidget(le_jid);

	// only put the multi button if composing
	if(isCompose) {
		pb_toMulti = new QPushButton(QString("<< ") + tr("Multi"), lo_single);
		QToolTip::add(pb_toMulti, tr("Switch to Multi mode.  List each recipient in the \"To\" field."));
		connect(pb_toMulti, SIGNAL(clicked()), SLOT(showMulti()));
		lo_single_hb->addWidget(pb_toMulti);
	}

	layout1->addWidget(lo_single);

	if(isMulti)
		lo_single->hide();


	// ---- multi
	lo_multi = new QWidget(this);
	QHBoxLayout *lo_multi_hb = new QHBoxLayout(lo_multi);
	QLabel *lb_multi_to = new QLabel(tr("To:"), lo_multi);
	lo_multi_hb->addWidget(lb_multi_to);

	le_to = new ELineEdit(lo_multi);
	lo_multi_hb->addWidget(le_to);

	// no swiching to single-mode from forced multi-mode
	if(!isMulti) {
		pb_toSingle = new QPushButton(QString("<< ") + tr("Single"), lo_multi);
		QToolTip::add(pb_toSingle, tr("Switch to Single mode"));
		connect(pb_toSingle, SIGNAL(clicked()), SLOT(showSingle()));
		lo_multi_hb->addWidget(pb_toSingle);
	}

	layout1->addWidget(lo_multi);

	if(!isMulti)
		lo_multi->hide();



	QHBoxLayout *hb;
	QLabel *l;

	// ---- info / toolbar
	hb = new QHBoxLayout;
	layout1->addLayout(hb);

	//lb_statusIcon = new QLabel(this);
	//hb->addWidget(lb_statusIcon);
	lb_statusStr = new QLabel(this);
	lb_statusStr->setFrameStyle( QFrame::Panel | QFrame::Sunken );
	lb_statusStr->hide();
	//hb->addWidget(lb_statusStr);


	l = new QLabel(tr("Time"), this);
	hb->addWidget(l);
	lb_time = new QLabel(this);
	lb_time->setFrameStyle( QFrame::Panel | QFrame::Sunken );
	hb->addWidget(lb_time);

	//hb->addStretch(1);

	l = new QLabel(tr("Type"), this);
	hb->addWidget(l);

	// in compose-mode, use a combo-box
	if(isCompose) {
		cb_eventType = new QComboBox(this);
		cb_eventType->insertItem(tr("Normal"));
		cb_eventType->insertItem(tr("Chat"));
		if(option.outgoingAs == 2)
			cb_eventType->setCurrentItem(1);

		hb->addWidget(cb_eventType);
	}
	else {
		lb_eventType = new QLabel(this);
		lb_eventType->setFrameStyle( QFrame::Panel | QFrame::Sunken );
		hb->addWidget(lb_eventType);
	}

	hb->addStretch(1);

	tb_subj = new QToolButton(this);
	tb_subj->setText(tr("+Subj"));
	tb_subj->setToggleButton(TRUE);
	connect(tb_subj, SIGNAL(toggled(bool)), SLOT(showSubject(bool)));
	hb->addWidget(tb_subj);
	if(!isCompose)
		tb_subj->setEnabled(FALSE);

	tb_url = new QToolButton(this);
	tb_url->setText(tr("+URL"));
	tb_url->setToggleButton(TRUE);
	connect(tb_url, SIGNAL(toggled(bool)), SLOT(showUrl(bool)));
	hb->addWidget(tb_url);
	if(!isCompose)
		tb_url->setEnabled(FALSE);

	/*QToolButton *file = new QToolButton(this);
	file->setText("File");
	file->setToggleButton(TRUE);
	connect(file, SIGNAL(toggled(bool)), SLOT(showFile(bool)));
	hb->addWidget(file);*/

	tb_info = new QToolButton(this);
	tb_info->setIconSet(QIconSet(*pix_info));
	connect(tb_info, SIGNAL(clicked()), SLOT(doInfo()));
	hb->addWidget(tb_info);
	if(isMulti)
		tb_info->setEnabled(FALSE);

	tb_history = new QToolButton(this);
	tb_history->setIconSet(QIconSet(*pix_history));
	connect(tb_history, SIGNAL(clicked()), SLOT(doHistory()));
	hb->addWidget(tb_history);
	if(isMulti)
		tb_history->setEnabled(FALSE);

	gb_subj = new QGroupBox(1, QGroupBox::Horizontal, tr("Subject"), this);
	gb_subj->hide();
	layout1->addWidget(gb_subj);
	le_subj = new QLineEdit(gb_subj);

	gb_url = new QGroupBox(2, QGroupBox::Horizontal, tr("Attached URL"), this);
	gb_url->hide();
	layout1->addWidget(gb_url);
	lb_url = new QLabel(tr("URL:"), gb_url);
	QHBox *w_url = new QHBox(gb_url);
	le_url = new QLineEdit(w_url);
	QPushButton *pb_urlgo = new QPushButton(tr("Go!"), w_url);
	connect(pb_urlgo, SIGNAL(clicked()), SLOT(goToURL()));
	lb_desc = new QLabel(tr("Desc:"), gb_url);
	le_desc = new QLineEdit(gb_url);


	mle = new MsgMle(this);
	mle->setText("");
	connect(mle, SIGNAL(linkClicked(const QString &)), SIGNAL(aOpenURL(const QString &)));

	layout1->addWidget(mle);

	hb = new QHBoxLayout;
	layout1->addLayout(hb);


	// bottom buttons
	QPushButton *pb_close = new QPushButton(tr("&Close"), this);
	pb_close->setMinimumWidth(80);
	connect(pb_close, SIGNAL(clicked()), SLOT(close()));
	hb->addWidget(pb_close);

	hb->addStretch(1);

	pb_next = 0;
	if(!isCompose) {
		pb_next = new AnimatedButton(this);
		pb_next->setText(tr("&Next"));
		pb_next->setMinimumWidth(96);
		pb_next->setEnabled(FALSE);
		connect(pb_next, SIGNAL(clicked()), SLOT(doReadNext()));
		hb->addWidget(pb_next);
	}

	pb_button1 = new QPushButton(this);
	pb_button1->setMinimumWidth(96);
	connect(pb_button1, SIGNAL(clicked()), SLOT(doButton1()));
	hb->addWidget(pb_button1);


	setTabOrder(mle, pb_button1);

	if(isCompose) {
		if(isMulti)
			le_to->setFocus();
		else
			mle->setFocus();
	}
	else {
		mle->setReadOnly(TRUE);
		mle->setUndoRedoEnabled(FALSE);
		le_subj->setReadOnly(TRUE);
		le_url->setReadOnly(TRUE);
		le_desc->setReadOnly(TRUE);
		pb_close->setFocus();
	}

	setTime(QDateTime::currentDateTime());

	eventType = -1;
	numReadNext = 0;

	flag_button1 = TRUE;
	setButton1();

	if(isCompose) {
		setIcon(*pix_send);
		pb_button1->setText(tr("&Send"));
		QIconSet icon;
		pb_button1->setIconSet(icon);

		QTimer *t = new QTimer(this);
		connect(t, SIGNAL(timeout()), SLOT(setCurrentTime()));
		t->start(5000);

		mle->setTextFormat(PlainText);
	}

	resize(option.sizeEventDlg);

	optionsUpdate();
}

EventDlg::~EventDlg()
{
        if(msg) {
                delete msg;
                msg = 0;
        }
}

QSize EventDlg::defaultSize()
{
	return QSize(420, 280);
}

void EventDlg::resizeEvent(QResizeEvent *e)
{
	if(option.keepSizes)
		option.sizeEventDlg = e->size();
}

void EventDlg::keyPressEvent(QKeyEvent *e)
{
        if(e->key() == Key_Escape)
                close();
        else if(isCompose && e->key() == Key_Return && ((e->state() & ControlButton) || (e->state() & AltButton)) )
                doButton1();
        else if(e->key() == Key_H && (e->state() & ControlButton))
                doHistory();
        else
                e->ignore();
}

void EventDlg::doButton1()
{
        // don't operate if disabled
        if(!pb_button1->isEnabled())
                return;

        if(isCompose) {
                if(mle->text().isEmpty() && le_url->text().isEmpty()) {
                        QMessageBox::information(this, CAP(tr("Warning")), tr("Please type in a message first."));
                        return;
                }

		Message msg;
		if(cb_eventType->currentItem() == 0)
			msg.type = MESSAGE_NORM;
		else
			msg.type = MESSAGE_CHAT;
		msg.text = mle->text();
		msg.timeStamp = QDateTime::currentDateTime();
		msg.originLocal = TRUE;
		msg.late = FALSE;
		if(!gb_subj->isHidden())
			msg.subject = le_subj->text();
		if(!gb_url->isHidden()) {
			msg.url = le_url->text();
			msg.url_desc = le_desc->text();
		}

		if(isMulti) {
			int numSent = 0;
			QStringList list = QStringList::split(',', le_to->text());
			for(QStringList::Iterator it = list.begin(); it != list.end(); ++it) {
				QString to = (*it).stripWhiteSpace();
				if(to == ",")
					continue;

				msg.to = to;
				aSend(msg);
				++numSent;
			}

			if(numSent == 0) {
				QMessageBox::warning(this, CAP("Error"), "No recipients have been specified!");
				return;
			}
		}
		else {
			msg.to = v_jid;
			aSend(msg);
		}

                close();
		return;
        }

        if(eventType == MESSAGE_AUTHREQ) {
                aAuth(le_jid->text());
                flag_button1 = FALSE;
                setButton1();
                closeAfterReply();
        }
        else
                aReply(v_jid);
}

void EventDlg::doReadNext()
{
        aReadNext(v_jid);
}

void EventDlg::closeAfterReply()
{
        if(numReadNext == 0)
                close();
}

void EventDlg::doHistory()
{
        aHistory(v_jid);
}

void EventDlg::doInfo()
{
	aInfo(v_jid);
}

void EventDlg::updateEvent(const Message &_msg)
{
	if(msg) {
		delete msg;
		msg = 0;
	}

	msg = new Message(_msg);

	eventAnim = messagetype2anim(msg->type);

	// update the icon and button only if the event type is different
	if(msg->type != eventType) {
		setIcon(eventAnim->base());

		if(msg->type == MESSAGE_AUTHREQ) {
			pb_button1->setText(tr("&Add/Auth"));
			pb_button1->setIconSet(QIconSet(*pix_add));
		}
		else {
			pb_button1->setText(tr("&Reply"));
			QIconSet icon;
			pb_button1->setIconSet(icon);
		}
	}

	flag_button1 = TRUE;
	setButton1();


	le_jid->setText(msg->from);
	le_jid->setCursorPosition(0);
	QToolTip::add(le_jid, msg->from);

	lb_eventType->setText(messagetype2str(msg->type));

	eventType = msg->type;
	mle->moveCursor(QTextEdit::MoveHome, FALSE);

	mle->setTextFormat(RichText);

	QString txt;
	if(eventType == MESSAGE_NORM || eventType == MESSAGE_CHAT || eventType == MESSAGE_HEADLINE)
		txt = plain2rich(msg->text);
	else
		txt = msg->text;

	mle->setText("<qt>" + linkify(txt) + "</qt>");

	if(!msg->subject.isEmpty()) {
		le_subj->setText(msg->subject);
		tb_subj->setOn(TRUE);
	}
	else
		tb_subj->setOn(FALSE);

	if(!msg->url.isEmpty()) {
		le_url->setText(msg->url);
		le_desc->setText(msg->url_desc);
		tb_url->setOn(TRUE);
	}
	else
		tb_url->setOn(FALSE);

	setTime(msg->timeStamp);
}

void EventDlg::updateReadNext(int x_nextType, int amount)
{
	int oldamount = numReadNext;

	nextType = x_nextType;
	numReadNext = amount;

	if(amount == 0) {
		pb_next->setAnim(0);
		pb_next->setEnabled(FALSE);
		QString str(tr("&Next"));
		pb_next->setText(str);

		if(pb_button1->isEnabled())
			pb_button1->setFocus();
	}
	else {
		pb_next->setEnabled(TRUE);
		QString str(tr("&Next"));
		str += QString(" - %1").arg(amount);
		pb_next->setText(str);

		pb_next->setAnim(messagetype2anim(nextType));

		if(numReadNext > oldamount)
			pb_next->setFocus();
	}
}

void EventDlg::setButton1()
{
        if(localStatus == STATUS_OFFLINE) {
                pb_button1->setEnabled(FALSE);
                return;
        }

        pb_button1->setEnabled(flag_button1);
}

void EventDlg::optionsUpdate()
{
	// update the font
	QFont f;
	f.fromString(option.font[fMessage]);
	mle->setFont(f);

	// update the status icon
	lb_statusIcon->setPixmap(jidIsAgent(v_jid) ? transport2icon(v_jid, v_status) : status2pix(v_status));

	// tool buttons
	tb_info->setIconSet(QIconSet(*pix_info));

	// update the readnext icon
	if(numReadNext > 0)
		pb_next->setAnim(messagetype2anim(nextType));

	// update the widget icon
	if(isCompose) {
		setIcon(*pix_send);
	}
	else {
		if(eventType != -1) {
			eventAnim = messagetype2anim(eventType);
			setIcon(eventAnim->base());
		}
	}
}

void EventDlg::localUpdate(const JabRosterEntry &e)
{
        localStatus = e.localStatus();

        setButton1();
}

void EventDlg::updateContact(UserListItem *item)
{
        if(jidcmp(item->jid, v_jid)) {
                v_nick = item->nick;
                v_status = item->state;

                QString nick = jidnick(item->jid, item->nick);

                if(nick != le_nick->text()) {
                        le_nick->setText(nick);
			le_nick->setCursorPosition(0);
			QToolTip::add(le_nick, nick);
		}
		if(!isMulti) {
                	if(nick != caption())
                        	setCaption(nick);
		}

                lb_statusStr->setText(status2txt(item->state));
		lb_statusIcon->setPixmap(jidIsAgent(v_jid) ? transport2icon(v_jid, v_status) : status2pix(v_status));

		QString head = tr("Status Message");
		QString str("<qt>");
                str += QString("<nobr><u>%1</u></nobr><br>%2").arg(head).arg(plain2rich(clipStatus(item->statusString, 50, 12)));
		str += "</qt>";
		QToolTip::add(lb_statusIcon, str);
        }
}

void EventDlg::setTime(const QDateTime &time)
{
        QString str;
        if(msg && msg->late)
                str.sprintf("<font color=\"red\"><nobr>%02d/%02d %02d:%02d:%02d</nobr></font>", time.date().month(), time.date().day(), time.time().hour(), time.time().minute(), time.time().second());
        else
                str.sprintf("<nobr>%02d/%02d %02d:%02d:%02d</nobr>", time.date().month(), time.date().day(), time.time().hour(), time.time().minute(), time.time().second());
        lb_time->setText(str);
}

void EventDlg::setCurrentTime()
{
        setTime(QDateTime::currentDateTime());
}

void EventDlg::showSubject(bool x)
{
	if(x) {
		gb_subj->show();
		le_subj->setFocus();
	}
	else {
		gb_subj->hide();
		mle->setFocus();
	}
}

void EventDlg::showUrl(bool x)
{
	if(x) {
		gb_url->show();
		le_url->setFocus();
	}
	else {
		gb_url->hide();
		mle->setFocus();
	}
}

void EventDlg::showMulti()
{
	isMulti = TRUE;

	lo_single->hide();

	le_to->setText(v_jid);
	lo_multi->show();

	tb_history->setEnabled(FALSE);
	tb_info->setEnabled(FALSE);

	setCaption("Multi");
}

void EventDlg::showSingle()
{
	isMulti = FALSE;

	lo_multi->hide();
	lo_single->show();

	tb_history->setEnabled(TRUE);
	tb_info->setEnabled(TRUE);

	setCaption(jidnick(v_jid, v_nick));
}

void EventDlg::goToURL()
{
	QString url = le_url->text();
	if(url.isEmpty())
		return;

	if(url.left(7) != "http://")
		url.insert(0, "http://");

	aOpenURL(url);
}

/*static*/ EventDlg * EventDlg::find(const QString &xjid)
{
        return (EventDlg *)UniqueWindowBank::find("EventDlg", cleanJid(xjid));
}
