/****************************************************************************
** contactview.cpp - an Licq-like contact list
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include<qpopupmenu.h>
#include<qmessagebox.h>
#include<qinputdialog.h>
#include<qiconset.h>
#include<qheader.h>
#include<qpalette.h>
#include<qdragobject.h>
#include<qapplication.h>
#include<qdropsite.h>

#include"contactview.h"
#include"common.h"
#include"userlist.h"


//*******************************************************
//  ContactView
//*******************************************************
ContactView::ContactView(QWidget *parent, const char *name)
:QListView(parent, name), QToolTip(viewport())
{
	// setup the QListView
	setAllColumnsShowFocus(TRUE);
	setHScrollBarMode(AlwaysOff);
	setMinimumSize(96,64);
	//setRootIsDecorated(TRUE);
	setTreeStepSize(4);
	setSorting(-1, TRUE);
	setDefaultRenameAction(QListView::Accept);
	//setSelectionMode(QListView::Extended);

	// create columns and hide the header
	addColumn("Contact");
	header()->hide();
	setResizeMode(QListView::LastColumn);
	setShowToolTips(FALSE);

	// initialize the alerting stuff
	QTimer *t = new QTimer(this);
	connect(t, SIGNAL(timeout()), SLOT(animAlert()));
	t->start(120);

	// catch signals
	connect(this, SIGNAL(itemRenamed(QListViewItem *, int, const QString &)), SLOT(qlv_itemRenamed(QListViewItem *, int, const QString &)));
	connect(this, SIGNAL(mouseButtonPressed(int, QListViewItem *, const QPoint &, int)),SLOT(qlv_singleclick(int, QListViewItem *, const QPoint &, int)) );
	connect(this, SIGNAL(doubleClicked(QListViewItem *)),SLOT(qlv_doubleclick(QListViewItem *)) );

	v_showOffline = TRUE;
	v_showAway = TRUE;
	v_showAgents = TRUE;

	lcto_active = FALSE;

	// sorting
	sortTimer = new QTimer(this);
	connect(sortTimer, SIGNAL(timeout()), SLOT(actualContactSort()));

	// actions
	qa_send = new QAction("", tr("Send &message"), QListView::CTRL+QListView::Key_M, this);
	connect(qa_send, SIGNAL(activated()), SLOT(doSendMessage()));
	qa_ren = new QAction("", tr("Re&name"), QListView::Key_F2, this);
	connect(qa_ren, SIGNAL(activated()), SLOT(doRename()));
	qa_chat = new QAction("", tr("Open &chat window"), QListView::CTRL+QListView::Key_C, this);
	connect(qa_chat, SIGNAL(activated()), SLOT(doOpenChat()));
	qa_hist = new QAction("", tr("&History"), QListView::CTRL+QListView::Key_H, this);
	connect(qa_hist, SIGNAL(activated()), SLOT(doHistory()));
	qa_logon = new QAction("", tr("&Log on"), QListView::CTRL+QListView::Key_L, this);
	connect(qa_logon, SIGNAL(activated()), SLOT(doLogon()));
	qa_recv = new QAction("", tr("&Receive incoming event"), QListView::CTRL+QListView::Key_R, this);
	connect(qa_recv, SIGNAL(activated()), SLOT(doRecvEvent()));
	qa_rem = new QAction("", tr("Rem&ove"), QListView::Key_Delete, this);
	connect(qa_rem, SIGNAL(activated()), SLOT(doRemove()));

	optionsUpdate();

	setAcceptDrops(TRUE);
}

ContactView::~ContactView()
{
	pdb(DEBUG_CV, "ContactView::~ContactView()\n");

	// delete the profiles
	profiles.setAutoDelete(TRUE);
	profiles.clear();
}

void ContactView::resizeEvent(QResizeEvent *e)
{
	QListView::resizeEvent(e);
}

void ContactView::keyPressEvent(QKeyEvent *e)
{
	if(e->key() == QListView::Key_Enter || e->key() == QListView::Key_Return)
		doEnter();
	else if(e->key() == QListView::Key_Space)
		doContext();
	else
		QListView::keyPressEvent(e);
}

void ContactView::viewportMousePressEvent(QMouseEvent *e)
{
	QListView::viewportMousePressEvent(e);
	if(e->button() == QListView::LeftButton)
		mousePressPos = e->pos();
}

void ContactView::viewportMouseMoveEvent(QMouseEvent *e)
{
	ContactViewItem *i = LVI2CVI(selectedItem());
	if(!i)
		return;
	if(i->type() != CV_CONTACT)
		return;

	if((e->state() & QListView::LeftButton) && !mousePressPos.isNull() && (QPoint(e->pos() - mousePressPos).manhattanLength() > 8)) {
		QDragObject *d = new QTextDrag(i->jid(), this);
		d->setPixmap(*pix_online, QPoint(8,8));
		d->dragCopy();
	}
}

void ContactView::dragEnterEvent(QDragEnterEvent *e)
{
	QString text;

	// only accept text jids that are in the contact list
	if(QTextDrag::decode(e, text)) {
		if(defPro()->findCVIByJid(text)) {
			e->accept(TRUE);
			return;
		}
	}

	e->accept(FALSE);
}

void ContactView::dropEvent(QDropEvent *e)
{
	ContactViewItem *i = LVI2CVI(itemAt(e->pos()));
	if(!i)
		return;

	ContactViewItem *dest;
	if(i->type() == CV_GROUP)
		dest = i;
	else
		dest = LVI2CVI(i->parent());

	QString text;

	if(QTextDrag::decode(e, text)) {
		ContactViewItem *src = defPro()->findCVIByJid(text);
		if(!src)
			return;
		if(src == i)
			return;
		if(src->isAgent() || !src->inList())
			return;

		QString newgroup;
		if(dest->groupType() == CVG_USER)
			newgroup = dest->groupName();
		else
			newgroup = "";

		if(src->group() == newgroup.mid(0))
			return;

		pdb(DEBUG_CV, QString("dragdrop: moving [%1] to group [%2] (oldgroup=[%3])\n").arg(text).arg(newgroup).arg(src->group()));
		if(newgroup != src->group()) {
			defPro()->scGroupChange(src->jid(), newgroup);
			//actionGroupChange(src->jid(), newgroup);
		}
	}
}

/*QDragObject *ContactView::dragObject()
{
	ContactViewItem *i = LVI2CVI(selectedItem());
	if(!i)
		return 0;

	QDragObject *d = new QTextDrag(i->jid(), this);
	d->setPixmap(*pix_online, QPoint(8,8));
	//d->dragCopy();
	return d;
}*/

void ContactView::sort()
{
	setSorting(0, TRUE);
	QListView::sort();
	setSorting(-1, TRUE);
}

void ContactView::maybeTip(const QPoint &pos)
{
	ContactViewItem *i = LVI2CVI(itemAt(pos));
	if(!i)
		return;
	if(i->type() != CV_CONTACT)
		return;

	QRect r(itemRect(i));

	QString str = QString("<qt>");
	str += QString("<nobr>%1: %2</nobr>").arg(i->nick()).arg(status2txt(i->status()));
	if(i->jid() != i->nick())
		str += QString("<br>[%1]").arg(i->jid());

	// subscription
	str += QString("<br><nobr>") + QString(tr("Subscription") + ": %1").arg(i->sub()) + "</nobr>";

	// resources
	if(!i->resList().isEmpty()) {
		QString resources;
		int n = 0;
		JabResource *pr = i->resList().priority();
		QPtrListIterator<JabResource> it(i->resList());
		JabResource *r;
		for(; (r = it.current()); ++it) {
			// skip 'blank' resources (non-jabber contacts)
			if(r->name.isEmpty())
				continue;

			if(n > 0)
				resources += ", ";

			// priority?
			if(r == pr)
				resources += "*";
			resources += r->name;
			++n;
		}

		if(!resources.isEmpty())
			str += QString("<br><nobr>") + QString(tr("Resource(s)") + " : %1").arg(resources) + "</nobr>";

		if(pr->timeStamp.isValid()) {
			QString d = pr->timeStamp.toString();
			str += QString("<br><nobr>") + tr("Last Status") + " @ " + d + "</nobr>";
		}

		if(!pr->songTitle.isEmpty())
			str += QString("<br><nobr>") + QString(tr("Listening to") + ": %1").arg(clipStatus(pr->songTitle, 200, 1)) + "</nobr>";
	}

	if(!i->statusString().isEmpty()) {
		QString head = tr("Status Message");
		str += QString("<br><nobr><u>%1</u></nobr><br>%2").arg(head).arg(plain2rich(clipStatus(i->statusString(), 200, 12)));
	}
	str += "</qt>";

	tip(r, str);
}

void ContactView::qlv_singleclick(int button, QListViewItem *i, const QPoint &pos, int c)
{
	lcto_active = FALSE;
	QToolTip::hide();

	if(!i)
		return;

	ContactViewItem *item = LVI2CVI(i);
	setSelected(item, TRUE);

	if(option.useleft) {
		if(button == QListView::LeftButton) {
			if(option.singleclick) {
				qlv_contextPopup(i, pos, c);
			}
			else {
				lcto_active = TRUE;
				lcto_pos = pos;
				lcto_item = i;
				QTimer::singleShot(QApplication::doubleClickInterval()/2, this, SLOT(leftClickTimeOut()));
			}
		}
		else if(option.singleclick && button == QListView::RightButton) {
			if(item->type() == CV_CONTACT)
				owner(item)->scActionDefault(item);
		}
	}
	else {
		if(button == QListView::RightButton) {
			qlv_contextPopup(i, pos, c);
		}
		else if(button == QListView::LeftButton && option.singleclick) {
			if(item->type() == CV_CONTACT)
				owner(item)->scActionDefault(item);
		}
	}
}

void ContactView::qlv_doubleclick(QListViewItem *i)
{
	lcto_active = FALSE;

	if(!i)
		return;

	if(option.singleclick)
		return;

	ContactViewItem *item = LVI2CVI(i);

	if(item->type() == CV_CONTACT)
		owner(item)->scActionDefault(item);
}

void ContactView::leftClickTimeOut()
{
	if(lcto_active) {
		QToolTip::hide();
		qlv_contextPopup(lcto_item, lcto_pos, 0);
		lcto_active = FALSE;
	}
}


// Profile related functions
ContactProfile *ContactView::createProfile(const QString &jid, const QString &name, bool unique)
{
	ContactProfile *pro = findProfile(jid);
	if(pro)
		removeProfile(pro);

	pro = new ContactProfile(jid, name, STATUS_OFFLINE, unique, this);
	profiles.append(pro);

	return pro;
}

ContactProfile *ContactView::findProfile(const QString &jid)
{
	QPtrListIterator<ContactProfile> it(profiles);
	for(ContactProfile *p; (p = it.current()); ++it) {
		if(jidcmp(p->jid, jid))
			return p;
	}
	return 0;
}

void ContactView::removeProfile(const QString &jid)
{
	ContactProfile *pro = findProfile(jid);
	if(pro)
		removeProfile(pro);
}

void ContactView::removeProfile(ContactProfile *p)
{
	profiles.remove(p);
}

ContactProfile *ContactView::defPro()
{
	return profiles.first();
}

// which profile likely deals with this item?
ContactProfile *ContactView::owner(ContactViewItem *i)
{
	QPtrListIterator<ContactProfile> it(profiles);
	for(ContactProfile *p; (p = it.current()); ++it) {
		if(p->isMine(i))
			return p;
	}
	return 0;
}

void ContactView::ensureVisible(ContactViewItem *item)
{
	// ensure the item is visible
	ContactViewItem *group_item = LVI2CVI(item->parent());

	// if the group is a user group, go up a step
	if(group_item && group_item->groupType() == CVG_USER)
		group_item = LVI2CVI(group_item->parent());

	if(group_item && !group_item->isVisible()) {
		if(group_item->groupType() == CVG_OFFLINE)
			setShowOffline(TRUE);
		if(group_item->groupType() == CVG_AGENTS)
			setShowAgents(TRUE);
	}
	if(!isShowAway() && item->isAway()) {
		setShowAway(TRUE);
	}

	ensureItemVisible(item);
}

void ContactView::animAlert()
{
	// are any of the items animating?
	QListViewItemIterator it(this);
	ContactViewItem *item;

	for(; it.current() ; ++it) {
		item = LVI2CVI(it.current());
		if(item->isAlert())
			item->animate();
		if(item->isAnimateNick())
			item->animateNick();
	}
}

void ContactView::clear()
{
	profiles.setAutoDelete(TRUE);
	profiles.clear();
	profiles.setAutoDelete(FALSE);
}

// right-click context menu
void ContactView::qlv_contextPopup(QListViewItem *xi, const QPoint &pos, int)
{
	ContactViewItem *i = LVI2CVI(xi);
	if(!i)
		return;

	owner(i)->doContextMenu(i, pos);
}

void ContactView::doRecvEvent()
{
	ContactViewItem *i = LVI2CVI(selectedItem());
	if(!i)
		return;

	if(i->type() == CV_CONTACT)
		owner(i)->scRecvEvent(i);
}

void ContactView::doRename()
{
	ContactViewItem *i = LVI2CVI(selectedItem());
	if(!i)
		return;

	owner(i)->scRename(i);
}

void ContactView::doEnter()
{
	ContactViewItem *i = LVI2CVI(selectedItem());
	if(!i)
		return;

	if(i->type() == CV_CONTACT)
		owner(i)->scActionDefault(i);
}

void ContactView::doContext()
{
	ContactViewItem *i = LVI2CVI(selectedItem());
	if(!i)
		return;
	ensureItemVisible(i);

	if(i->type() == CV_GROUP)
		setOpen(i, !i->isOpen());
	else
		qlv_contextPopup(i, viewport()->mapToGlobal(QPoint(32, itemPos(i))), 0);
}

void ContactView::doSendMessage()
{
	ContactViewItem *i = LVI2CVI(selectedItem());
	if(!i)
		return;

	owner(i)->scSendMessage(i);
}

void ContactView::doOpenChat()
{
	ContactViewItem *i = LVI2CVI(selectedItem());
	if(!i)
		return;

	owner(i)->scOpenChat(i);
}

void ContactView::doHistory()
{
	ContactViewItem *i = LVI2CVI(selectedItem());
	if(!i)
		return;

	owner(i)->scHistory(i);
}

void ContactView::doLogon()
{
	ContactViewItem *i = LVI2CVI(selectedItem());
	if(!i)
		return;
	if(i->type() != CV_CONTACT)
		return;

	if(i->isAgent())
		owner(i)->scAgentSetStatus(i, STATUS_ONLINE);
}

void ContactView::doRemove()
{
	ContactViewItem *i = LVI2CVI(selectedItem());
	if(!i)
		return;

	owner(i)->scRemove(i);
}

void ContactView::qlv_itemRenamed(QListViewItem *xi, int, const QString &text)
{
	ContactViewItem *i = LVI2CVI(xi);

	owner(i)->doItemRenamed(i, text);
}

void ContactView::setShowOffline(bool x)
{
	bool oldstate = v_showOffline;
	v_showOffline = x;

	if(v_showOffline != oldstate) {
		showOffline(v_showOffline);

		// set the visibility of all the offline groups
		QPtrListIterator<ContactProfile> it(profiles);
		for(ContactProfile *p; (p = it.current()); ++it) {
			ContactViewItem *group_item = p->checkGroup(CVG_OFFLINE);
			if(group_item)
				group_item->setVisible(v_showOffline);
		}
	}
}

void ContactView::setShowAway(bool x)
{
	bool oldstate = v_showAway;
	v_showAway = x;

	if(v_showAway != oldstate) {
		showAway(v_showAway);

		// we want to modify _all_ profiles, so just burn down the list
		QListViewItemIterator it(this);
		ContactViewItem *item;
		for(; it.current() ; ++it) {
			item = LVI2CVI(it.current());
			if(item->type() != CV_CONTACT)
				continue;

			if(!item->isAgent())
				setVisibility(item);
		}
	}
}

void ContactView::setShowAgents(bool x)
{
	bool oldstate = v_showAgents;
	v_showAgents = x;

	if(v_showAgents != oldstate) {
		showAgents(v_showAgents);

		// set the visibility of all the offline groups
		QPtrListIterator<ContactProfile> it(profiles);
		for(ContactProfile *p; (p = it.current()); ++it) {
			ContactViewItem *group_item = p->checkGroup(CVG_AGENTS);
			if(group_item)
				group_item->setVisible(v_showAgents);
		}
	}
}

void ContactView::optionsUpdate()
{
	// set the font
	QFont f;
	f.fromString(option.font[fRoster]);
	QListView::setFont(f);

	// set the text and background colors
	QPalette mypal = QListView::palette();
	mypal.setColor(QColorGroup::Text, option.color[cOnline]);
	mypal.setColor(QColorGroup::Base, option.color[cListBack]);
	QListView::setPalette(mypal);

	// reload the icons
	QListViewItemIterator it(this);
	ContactViewItem *item;
	for(; it.current() ; ++it) {
		item = LVI2CVI(it.current());
		item->optionsUpdate();
	}

	update();
}

void ContactView::setVisibility(ContactViewItem *i)
{
	ContactViewItem *par = LVI2CVI(i->parent());
	if(par && !par->isVisible())
		return;

	if(!isShowAway() && i->isAway())
		i->setVisible(FALSE);
	else
		i->setVisible(TRUE);
}

void ContactView::deferredContactSort()
{
	sortTimer->start(500, TRUE);
}

void ContactView::actualContactSort()
{
	QListViewItemIterator it(this);
	ContactViewItem *item;

	for(; it.current() ; ++it) {
		item = LVI2CVI(it.current());
		if(item->type() != CV_GROUP)
			continue;

		if(item->needSorting)
			item->sort();

		owner(item)->updateGroupInfo(item);
	}

	sort();
}


//*******************************************************
//  ContactViewItem
//*******************************************************
ContactViewItem::ContactViewItem(const QString &xgroupname, int xgrouptype, QListView *parent)
:QListViewItem(parent)
{
	v_type = CV_GROUP;
	v_groupName = xgroupname;
	v_groupType = xgrouptype;
	v_isAlert = FALSE;
	v_isAnimateNick = FALSE;

	resetGroupName();
	setOpen(TRUE);

	// sorting stuff
	needSorting = FALSE;

	//if(xgrouptype == CVG_USER)
		//setRenameEnabled(1, TRUE);

	drawGroupIcon();

	//setDropEnabled(TRUE);
}

ContactViewItem::ContactViewItem(const QString &xgroupname, int xgrouptype, QListViewItem *parent)
:QListViewItem(parent)
{
	if(!parent->isVisible())
		setVisible(FALSE);

	v_type = CV_GROUP;
	v_groupName = xgroupname;
	v_groupType = xgrouptype;
	v_isAlert = FALSE;
	v_isAnimateNick = FALSE;

	resetGroupName();
	setOpen(TRUE);

	// sorting stuff
	needSorting = FALSE;

	//if(xgrouptype == CVG_USER)
		//setRenameEnabled(1, TRUE);

	drawGroupIcon();

	//setDropEnabled(TRUE);
}

ContactViewItem::ContactViewItem(const QString &xjid, const QString &xnick, bool xisAgent, QListViewItem *parent)
:QListViewItem(parent)
{
	if(!parent->isVisible())
		setVisible(FALSE);

	v_type = CV_CONTACT;
	v_jid = xjid;
	v_nick = xnick;
	v_group = "";
	v_isAgent = xisAgent;
	v_isAlert = FALSE;
	v_isAnimateNick = FALSE;
	v_status = STATUS_OFFLINE;

	resetStatus();
	resetNick();

	//setDragEnabled(TRUE);
	//setDropEnabled(TRUE);
}

ContactViewItem::~ContactViewItem()
{
	// delete the alert queue
	while(!alertQueue.isEmpty()) {
		int *x = alertQueue.last();
		alertQueue.remove(x);
		delete x;
	}
}

void ContactViewItem::optionsUpdate()
{
	if(v_type == CV_GROUP) {
		drawGroupIcon();
	}
	else {
		if(!v_isAlert)
			resetStatus();
		else
			resetAnim();
	}
}

void ContactViewItem::resetAnim()
{
	animStep = 0;
	if(v_isAlert) {
		if(option.alertStyle == 0)
			setPixmap(0, messagetype2anim(alertType)->base());
	}
}

void ContactViewItem::paintFocus(QPainter *, const QColorGroup &, const QRect &)
{
	// re-implimented to do nothing.  selection is enough of a focus
}

void ContactViewItem::paintBranches(QPainter *p, const QColorGroup &cg, int w, int, int h)
{
	// paint a square of nothing
	p->fillRect(0, 0, w, h, cg.base());
}

void ContactViewItem::paintCell(QPainter *p, const QColorGroup & cg, int column, int width, int alignment)
{
	if(v_type == CV_CONTACT) {
		QColorGroup xcg = cg;
		if(v_status == STATUS_AWAY || v_status == STATUS_XA)
			xcg.setColor(QColorGroup::Text, option.color[cAway]);
		else if(v_status == STATUS_DND)
			xcg.setColor(QColorGroup::Text, option.color[cDND]);
		else if(v_status == STATUS_OFFLINE)
			xcg.setColor(QColorGroup::Text, option.color[cOffline]);

		if(v_isAnimateNick) {
			xcg.setColor(QColorGroup::Text, animateNickColor ? Qt::red : Qt::black);
			xcg.setColor(QColorGroup::HighlightedText, animateNickColor ? Qt::red : Qt::black);
		}

		QListViewItem::paintCell(p, xcg, column, width, alignment);
	}
	else if(v_type == CV_GROUP) {
		QColorGroup xcg = cg;
		xcg.setColor(QColorGroup::Text, option.color[cGroupFore]);
		xcg.setColor(QColorGroup::Base, option.color[cGroupBack]);

		//QFont f = p->font();
		//f.setPixelSize(8);
		//p->setFont(f);

		QListViewItem::paintCell(p, xcg, column, width, alignment);

		//QFontMetrics fm(p->font());
		//int x1 = fm.width(text(0)) + pixmap().width() + 4;

		//if(column == 0) {
			int x1, x2, y1, y2;

			x1 = 0;
			x2 = width - 1;
			y1 = 0;
			y2 = height() - 1;

			p->setPen(QPen(option.color[cGroupFore]));
			p->drawLine(x1, y1, x2, y1);
			p->drawLine(x1, y2, x2, y2);
			p->drawLine(x1, y1, x1, y2);
			p->drawLine(x2, y1, x2, y2);
		//}
		/*else {
			int x1, x2, y1, y2;

			x1 = 0;
			x2 = width - 1;
			y1 = 0;
			y2 = height() - 1;

			p->setPen(QPen(option.color[cGroupFore]));
			p->drawLine(x1, y1, x2, y1);
			p->drawLine(x1, y2, x2, y2);
			p->drawLine(x2, y1, x2, y2);
		}*/
	}
}

void ContactViewItem::setOpen(bool o)
{
	QListViewItem::setOpen(o);

	drawGroupIcon();
}

int ContactViewItem::parentGroupType()
{
	ContactViewItem *item = LVI2CVI(parent());
	return item->groupType();
}

void ContactViewItem::setStatus(int xstatus)
{
	if(v_status == xstatus)
		return;

	v_status = xstatus;
	resetStatus();

	ContactViewItem *i = LVI2CVI(parent());
	if(i)
		i->needSorting = TRUE;
}

void ContactViewItem::resetStatus()
{
	if(v_isAlert)
		return;

	setPixmap(0, v_isAgent ? transport2icon(v_jid, v_status) : status2pix(v_status));
}

void ContactViewItem::setNick(const QString &xnick)
{
	if(v_nick == xnick)
		return;

	v_nick = xnick;
	resetNick();

	ContactViewItem *i = LVI2CVI(parent());
	if(i)
		i->needSorting = TRUE;
}

void ContactViewItem::resetNick()
{
	setPixmap(0, v_isAgent ? transport2icon(v_jid, v_status) : status2pix(v_status));
	setText(0, v_nick);
}

void ContactViewItem::setGroup(const QString &xgroup)
{
	v_group = xgroup;
}

void ContactViewItem::setSub(const QString &xsub)
{
	v_sub = xsub;
}

void ContactViewItem::setStatusString(const QString &xstatusString)
{
	v_statusString = xstatusString;
}

void ContactViewItem::setInList(bool xinList)
{
	v_inList = xinList;
}

void ContactViewItem::setIsAgent(bool xisAgent)
{
	bool oldstate = v_isAgent;

	v_isAgent = xisAgent;

	// if different, then update the status
	if(v_isAgent != oldstate)
		resetStatus();
}

void ContactViewItem::setAlert(int xtype)
{
	if(!v_isAlert) {
		v_isAlert = TRUE;
		alertType = xtype;
		animStep = 0;
		if(option.alertStyle == 0)
			setPixmap(0, messagetype2anim(alertType)->base());
	}

	int *x = new int(xtype);
	alertQueue.insert(0, x);
}

void ContactViewItem::clearAlert(int at)
{
	if(alertQueue.isEmpty())
		return;

	// find and destroy
	int *x = alertQueue.last();
	for(int n = 0; n < at; ++n) {
		x = alertQueue.prev();
	}
	if(x) {
		alertQueue.remove(x);
		delete x;
	}

	// set the next alert (or stop alerting if empty)
	if(alertQueue.isEmpty()) {
		v_isAlert = FALSE;
		resetStatus();
	}
	else {
		x = alertQueue.last();
		alertType = *x;
		animStep = 0;
	}
}

void ContactViewItem::setAttr(const QString &xnick, const QString xgroup, const QString &sub, int status)
{
	setNick(xnick);
	setGroup(xgroup);
	setSub(sub);
	setStatus(status);
}

void ContactViewItem::setGroupName(const QString &name)
{
	v_groupName = name;
	resetGroupName();
}

void ContactViewItem::setGroupInfo(const QString &info)
{
	v_groupInfo = info;
	resetGroupName();
}

void ContactViewItem::resetGroupName(bool showInfo)
{
	QString str = v_groupName;
	if(showInfo && !v_groupInfo.isEmpty())
		str += QString(" ") + v_groupInfo;

	setText(0, str);
}

void ContactViewItem::animate()
{
	Anim *anim = messagetype2anim(alertType);

	if(anim->isAnim() && option.alertStyle == 2) {
		setPixmap(0, anim->frame(animStep));
		++animStep;
		if(animStep >= anim->numFrames()-1)
			animStep = 0;
	}
	else if(option.alertStyle != 0) {
		if(animStep == 0)
			setPixmap(0, anim->base());
		else if(animStep == 4)
			setPixmap(0, *pix_blank);

		++animStep;
		if(animStep == 8)
			animStep = 0;
	}
}

int ContactViewItem::rankGroup(int groupType) const
{
	static int rankgroups[7] = {
		CVG_SELF,
		CVG_ONLINE,
		CVG_USER,
		CVG_OFFLINE,
		CVG_AUTHWAIT,
		CVG_AGENTS,
		CVG_NOTINLIST,
	};

	int n;
	for(n = 0; n < 7; ++n) {
		if(rankgroups[n] == groupType)
			break;
	}
	if(n == 7)
		return 6;

	return n;
}

int ContactViewItem::rankStatus(int status) const
{
	static int rankstatuses[5] = {
		STATUS_ONLINE,
		STATUS_AWAY,
		STATUS_XA,
		STATUS_DND,
		STATUS_OFFLINE,
	};

	int n;
	for(n = 0; n < 5; ++n) {
		if(rankstatuses[n] == status)
			break;
	}
	if(n == 5)
		return 4;

	return n;
}

int ContactViewItem::compare(QListViewItem *xi, int, bool) const
{
	ContactViewItem *i = LVI2CVI(xi);
	int ret = 0;

	if(v_type == CV_GROUP) {
		// contacts always go before groups
		if(i->type() == CV_CONTACT)
			ret = 1;
		else {
			ret = rankGroup(v_groupType) - rankGroup(i->groupType());
			if(ret == 0)
				ret = text(0).lower().localeAwareCompare(i->text(0).lower());
		}
	}
	else if(v_type == CV_CONTACT) {
		// contacts always go before groups
		if(i->type() == CV_GROUP)
			ret = -1;
		else {
			//int x1 = v_isAlert ? 0 : (rankStatus(v_status) + 1);
			//int x2 = i->isAlert() ? 0 : (rankStatus(i->status()) + 1);

			ret = rankStatus(v_status) - rankStatus(i->status());
			//ret = x1 - x2;
			if(ret == 0)
				ret = text(0).lower().localeAwareCompare(i->text(0).lower());
		}
	}

	return ret;
}

void ContactViewItem::drawGroupIcon()
{
	if(childCount() == 0)
		setPixmap(0, *pix_arrow[2]);
	else if(isOpen())
		setPixmap(0, *pix_arrow[1]);
	else
		setPixmap(0, *pix_arrow[0]);
}

void ContactViewItem::insertItem(QListViewItem * newChild)
{
	QListViewItem::insertItem(newChild);

	drawGroupIcon();
}

void ContactViewItem::takeItem(QListViewItem * item)
{
	QListViewItem::takeItem(item);

	drawGroupIcon();
}

void ContactViewItem::setAnimateNick()
{
	stopAnimateNick();

	v_isAnimateNick = TRUE;
	animateNickX = 0;
	animateNick();
}

void ContactViewItem::stopAnimateNick()
{
	if(!v_isAnimateNick)
		return;

	v_isAnimateNick = FALSE;
	repaint();
}

void ContactViewItem::animateNick()
{
	animateNickColor = (animateNickX % 10) >= 5 ? 1: 0;
	if((animateNickX % 5) == 0)
		repaint();

	++animateNickX;
	if(animateNickX == 80)
		stopAnimateNick();
}

bool ContactViewItem::isAway()
{
	if(v_status == STATUS_AWAY || v_status == STATUS_XA || v_status == STATUS_DND)
		return TRUE;

	return FALSE;
}

void ContactViewItem::sort()
{
	QListView *i = listView();
	if(!i)
		return;

	i->setSorting(0, TRUE);
	QListViewItem::sort();
	i->setSorting(-1, TRUE);

	needSorting = FALSE;
}

/*bool ContactViewItem::acceptDrop(const QMimeSource *mime) const
{
	printf("accept?\n");
	return TRUE;// QTextDrag::canDecode(mime);
}

void ContactViewItem::dropped(QDropEvent *e)
{
	QString text;

	if(QTextDrag::decode(e, text)) {
		printf("dropped on %s, this text: %s\n", v_jid.latin1(), text.latin1());
	}
}
*/


//*******************************************************
//  ContactProfile
//*******************************************************
ContactProfile::ContactProfile(const QString &_jid, const QString &_name, int _localStatus, bool unique, ContactView *_cv)
{
	jid = _jid;
	name = _name;
	localStatus = _localStatus;
	cv = _cv;

	roster = new QDict<ContactProfileItem>(50, TRUE);
	roster->setAutoDelete(TRUE);

	if(!unique)
		cvi = new ContactViewItem(name, CVG_USER, cv);
	else
		cvi = 0;

	cvi_online = addGroup(CVG_ONLINE);
	cvi_offline = addGroup(CVG_OFFLINE);
}

ContactProfile::~ContactProfile()
{
	// delete the roster
	clear();

	// clean up
	delete cvi_online;
	delete cvi_offline;
	if(cvi)
		delete cvi;

	delete roster;
}

ContactProfileItem *ContactProfile::cvi2cpi(ContactViewItem *i)
{
	return roster->find(i->jid().lower());
}

ContactProfileItem *ContactProfile::findByJid(const QString &jid)
{
	return roster->find(jid.lower());
}

bool ContactProfile::isMine(ContactViewItem *i)
{
	// unique?  we'll take it
	if(cvi == 0)
		return TRUE;

	while(i) {
		if(i == cvi)
			return TRUE;

		// reduce the depth until we hit the base
		i = LVI2CVI(i->parent());
	}

	return FALSE;
}

ContactViewItem *ContactProfile::findCVIByJid(const QString &jid)
{
	ContactProfileItem *i = findByJid(jid);
	if(!i)
		return 0;
	return i->cvi;
}

ContactViewItem *ContactProfile::addGroup(int type)
{
	ContactViewItem *item;

	QString gname;
	if(type == CVG_ONLINE)
		gname = tr("Online");
	else if(type == CVG_OFFLINE)
		gname = tr("Offline");
	else if(type == CVG_NOTINLIST)
		gname = tr("Not in list");
	else if(type == CVG_AUTHWAIT)
		gname = tr("Need Authorization");
	else if(type == CVG_AGENTS)
		gname = tr("Agents/Transports");

	if(cvi)
		item = new ContactViewItem(gname, type, cvi);
	else
		item = new ContactViewItem(gname, type, cv);

	if((type == CVG_OFFLINE && !cv->isShowOffline()) || (type == CVG_AGENTS && !cv->isShowAgents()))
		item->setVisible(FALSE);

	cv->deferredContactSort();

	// debug
	QString dstr; dstr.sprintf("ContactProfile::addGroup: [%s]\n", gname.latin1());
	pdb(DEBUG_CV, dstr);

	return item;
}

ContactViewItem *ContactProfile::addGroup(const QString &name, ContactViewItem *par)
{
	ContactViewItem *item = new ContactViewItem(name, CVG_USER, par);
	cv->deferredContactSort();
	return item;
}

ContactViewItem *ContactProfile::checkGroup(int group)
{
	ContactViewItem *item;
	if(cvi)
		item = LVI2CVI(cvi->firstChild());
	else
		item = LVI2CVI(cv->firstChild());

	// do we have it?
	for(; item; item = LVI2CVI(item->nextSibling()) ) {
		if(item->type() == CV_GROUP && item->groupType() == group)
			return item;
	}

	return 0;
}

ContactViewItem *ContactProfile::checkGroup(const QString &group, ContactViewItem *par)
{
	// do we have it?
	ContactViewItem *item = LVI2CVI(par->firstChild());
	for(; item; item = LVI2CVI(item->nextSibling()) ) {
		if(item->type() == CV_GROUP && item->groupType() == CVG_USER) {
			if(item->groupName() == group)
				return item;
		}
	}

	return 0;
}

ContactViewItem *ContactProfile::ensureGroup(int group)
{
	ContactViewItem *group_item = checkGroup(group);
	if(group_item)
		return group_item;

	return addGroup(group);
}

ContactViewItem *ContactProfile::ensureGroup(const QString &group, int status)
{
	// empty string == default group (online/offline)
	if(group.isEmpty()) {
		int gtype = (status == STATUS_OFFLINE) ? CVG_OFFLINE : CVG_ONLINE;
		return ensureGroup(gtype);
	}

	ContactViewItem *group_item;
	ContactViewItem *par = (status == STATUS_OFFLINE) ? offline(): online();

	group_item = checkGroup(group, par);

	// doesn't exist?  make it!
	if(!group_item)
		group_item = addGroup(group, par);

	return group_item;
}

ContactProfileItem *ContactProfile::addEntry(const UserListItem &i)
{
	ContactViewItem *group_item;
	if(i.inList == FALSE)
		group_item = ensureGroup(CVG_NOTINLIST);
	else if(i.isTransport == TRUE)
		group_item = ensureGroup(CVG_AGENTS);
	else if(i.sub == "from" || i.sub == "none")
		group_item = ensureGroup(CVG_AUTHWAIT);
	else
		group_item = ensureGroup(i.group, i.state);

	QString nick = jidnick(i.jid, i.nick);
	ContactViewItem *item = new ContactViewItem(i.jid, nick, i.isTransport, group_item);
	item->setAttr(nick, i.group, i.sub, i.state);
	item->setInList(i.inList);
	item->setStatusString(i.statusString);

	item->resList() = i.res;

	// insert into the profile roster
	ContactProfileItem *pi = new ContactProfileItem;
	pi->uli = i;
	pi->cvi = item;
	roster->insert(i.jid.lower(), pi);

	// debug
	QString dstr; dstr.sprintf("ContactProfile::addEntry: adding [%s] to group [%s]\n", i.jid.latin1(), group_item->groupName().latin1());
	pdb(DEBUG_CV, dstr);

	return pi;
}

void ContactProfile::updateEntry(const UserListItem &i)
{
	ContactProfileItem *pi = findByJid(i.jid);
	ContactViewItem *item;
	if(!pi) {
		item = addEntry(i)->cvi;
	}
	else {
		item = pi->cvi;

		// if the group is different, we may need to remove a floating user group
		QString oldgroup = item->group();

		item->setAttr(jidnick(i.jid, i.nick), i.group, i.sub, i.state);
		item->setInList(i.inList);
		item->setStatusString(i.statusString);

		item->resList() = i.res;

		// it is possible that a contact could be recognized later as an agent i guess
		item->setIsAgent(i.isTransport); // ok 99, go-go-gadget shoe phone

		// update the group
		changeGroup(item, i.group);

		// destroy original group?
		checkDestroyGroup(oldgroup);

		// stop animating the nick if going offline
		if(i.state == STATUS_OFFLINE)
			item->stopAnimateNick();
	}

	cv->setVisibility(item);
	cv->deferredContactSort();
}


void ContactProfile::changeGroup(ContactViewItem *item, ContactViewItem *group_new)
{
	// contacts are always children of groups, so this should be safe:
	ContactViewItem *group_in = LVI2CVI(item->parent());

	if(group_in != group_new) {
		// take from the rich, give to the poor
		group_in->takeItem(item);
		group_new->insertItem(item);

		// visibility of item should match that of group_new
		item->setVisible(group_new->isVisible());
	}

	// we may need to destroy the original group(s):
	checkDestroyGroup(group_in);

	if(item->isAlert()) {
		if(option.scrollTo)
			cv->ensureVisible(item);
	}
}

void ContactProfile::changeGroup(ContactViewItem *item, const QString &group)
{
	ContactViewItem *group_item;

	if(item->inList() == FALSE) {
		item->setStatus(STATUS_OFFLINE);
		group_item = ensureGroup(CVG_NOTINLIST);
	}
	else if(item->isAgent() == TRUE)
		group_item = ensureGroup(CVG_AGENTS);
	else if(item->sub() == "from" || item->sub() == "none") {
		item->setStatus(STATUS_OFFLINE);
		group_item = ensureGroup(CVG_AUTHWAIT);
	}
	else
		group_item = ensureGroup(group, item->status());

	changeGroup(item, group_item);
}

void ContactProfile::checkDestroyGroup(const QString &group)
{
	if(group.isEmpty())
		return;

	ContactViewItem *group_item;

	group_item = checkGroup(group, offline());
	if(group_item)
		checkDestroyGroup(group_item);
	group_item = checkGroup(group, online());
	if(group_item)
		checkDestroyGroup(group_item);
}

void ContactProfile::checkDestroyGroup(ContactViewItem *group)
{
	// don't destroy online/offline psuedogroups
	if(group->groupType() == CVG_ONLINE || group->groupType() == CVG_OFFLINE)
		return;

	if(group->childCount() == 0) {
		// debug
		QString dstr; dstr.sprintf("ContactProfile::checkDestroyGroup: removing group [%s]\n", group->groupName().latin1());
		pdb(DEBUG_CV, dstr);

		delete group;
	}
}

void ContactProfile::removeEntry(const QString &jid)
{
	ContactProfileItem *pi = findByJid(jid);
	if(!pi)
		return;
	removeEntry(pi);
}

void ContactProfile::removeEntry(ContactProfileItem *pi)
{
	ContactViewItem *item = pi->cvi;

	// prepare to delete the parent group
	ContactViewItem *group_in = LVI2CVI(item->parent());

	// debug
	//QString dstr; dstr.sprintf("ContactProfile::removeEntry: removing contact [%s]\n", item->jid().latin1());
	//pdb(DEBUG_CV, dstr);

	clearAllAlerts(item);
	delete item;

	QString key = pi->uli.jid.lower();
	roster->remove(key);

	updateGroupInfo(group_in);

	// remove the group?
	checkDestroyGroup(group_in);
}

void ContactProfile::showAlert(const QString &jid, int type)
{
	ContactProfileItem *pi = findByJid(jid);
	if(!pi)
		return;
	ContactViewItem *item = pi->cvi;

	// reset all animations (so they are in sync with this new one)
	for(QListViewItemIterator it(cv); it.current() ; ++it) {
		ContactViewItem *item = LVI2CVI(it.current());
		if(item->type() == CV_CONTACT)
			item->resetAnim();
	}

	// flag the alert, and account the total
	item->setAlert(type);

	if(option.scrollTo)
		cv->ensureVisible(item);
}

void ContactProfile::clearAlert(const QString &jid, int at)
{
	ContactProfileItem *pi = findByJid(jid);
	if(!pi)
		return;
	ContactViewItem *item = pi->cvi;
	clearAlert(item, at);
}

void ContactProfile::clearAllAlerts(const QString &jid)
{
	ContactProfileItem *pi = findByJid(jid);
	if(!pi)
		return;
	ContactViewItem *item = pi->cvi;
	clearAllAlerts(item);
}

void ContactProfile::clearAlert(ContactViewItem *item, int at)
{
	if(item->isAlert())
		item->clearAlert(at);
}

void ContactProfile::clearAllAlerts(ContactViewItem *item)
{
	while(item->isAlert())
		item->clearAlert();
}

void ContactProfile::animateNick(const QString &jid)
{
	ContactProfileItem *pi = findByJid(jid);
	if(!pi)
		return;
	ContactViewItem *item = pi->cvi;
	item->setAnimateNick();
}

void ContactProfile::clear()
{
	QDictIterator<ContactProfileItem> it(*roster);

	for(ContactProfileItem *pi; (pi = it.current());)
		removeEntry(pi);
}

QStringList ContactProfile::buildGroupList()
{
	QStringList groupList;

	QListViewItemIterator it(cv);
	ContactViewItem *item;

	for(; it.current() ; ++it) {
		item = LVI2CVI(it.current());
		if(item->type() != CV_CONTACT)
			continue;

		if(item->group().isEmpty())
			continue;

		// weed out duplicates
		if(qstringlistmatch(groupList, item->group()) == -1)
			groupList.append(item->group());
	}

	groupList.sort();
	return groupList;
}

void ContactProfile::doContextMenu(ContactViewItem *i, const QPoint &pos)
{
	//ContactProfileItem *pi = cvi2cpi(i);
	QStringList groupList = buildGroupList();

	if(i->type() == CV_CONTACT) {
		QPopupMenu *popup = new QPopupMenu;

		if(i->parentGroupType() == CVG_NOTINLIST) {
			popup->insertItem(QIconSet(*pix_add), tr("Add/Authorize to contact list"), 10);
			if(localStatus == STATUS_OFFLINE)
				popup->setItemEnabled(10, FALSE);

			popup->insertSeparator();
		}
		if(i->isAlert()) {
			cv->qa_recv->addTo(popup);
			popup->insertSeparator();
		}

		cv->qa_send->setIconSet(*pix_send);
		cv->qa_send->addTo(popup);

		popup->insertItem(QIconSet(*pix_url), tr("Send &URL"), 2);

		cv->qa_chat->setIconSet(anim_chat->base());
		cv->qa_chat->addTo(popup);

		popup->insertItem("Offer &file", 3);
		popup->setItemEnabled(3, FALSE);
		popup->insertSeparator();

		if(i->parentGroupType() != CVG_NOTINLIST)
			cv->qa_ren->addTo(popup);

		if(!i->isAgent()) {
			if(i->parentGroupType() != CVG_NOTINLIST) {

				// --- groups ---
				QPopupMenu *groupmenu = new QPopupMenu(popup);

				groupmenu->setCheckable(TRUE);
				groupmenu->insertItem(tr("&None"), 8);
				groupmenu->insertSeparator();

				if(i->group().isEmpty())
					groupmenu->setItemChecked(8, TRUE);

				int n = 0;
				for(QStringList::Iterator it = groupList.begin(); it != groupList.end(); ++it, ++n) {
					QString str;
					if(n < 9)
						str = "&";
					str += QString("%1. %2").arg(n+1).arg(*it);
					groupmenu->insertItem(str, n+32);

					if(i->group() == *it)
						groupmenu->setItemChecked(n+32, TRUE);
				}
				if(n > 0)
					groupmenu->insertSeparator();

				groupmenu->insertItem(tr("&Create new..."), 9);

				popup->insertItem(tr("&Group"), groupmenu, 5);
				// --- groups ---
			}

			popup->insertItem(tr("(Re)send &authorization to"), 6);
			if(localStatus == STATUS_OFFLINE)
				popup->setItemEnabled(6, FALSE);

			//if(i->parentGroupType() == CVG_AUTHWAIT)
				popup->insertItem(tr("Rerequest authorization from"), 11);
			if(localStatus == STATUS_OFFLINE)
				popup->setItemEnabled(11, FALSE);
		}
		else {
			popup->insertSeparator();

			if(i->status() != STATUS_OFFLINE || localStatus == STATUS_OFFLINE)
				cv->qa_logon->setEnabled(FALSE);
			else
				cv->qa_logon->setEnabled(TRUE);

			cv->qa_logon->setIconSet(transport2icon(i->jid()));
			cv->qa_logon->addTo(popup);

			popup->insertItem(transport2icon(i->jid(), STATUS_OFFLINE), tr("Log off"), 16);
			if(i->status() == STATUS_OFFLINE || localStatus == STATUS_OFFLINE)
				popup->setItemEnabled(16, FALSE);
		}
		popup->insertSeparator();

		if(localStatus == STATUS_OFFLINE && i->inList())
			cv->qa_rem->setEnabled(FALSE);
		else
			cv->qa_rem->setEnabled(TRUE);
		cv->qa_rem->setIconSet(*pix_remove);
		cv->qa_rem->addTo(popup);

		//popup->insertItem(QIconSet(*pix_remove), tr("Rem&ove"), 7);
		//if(localStatus == STATUS_OFFLINE && i->inList())
		//	popup->setItemEnabled(7, FALSE);
		popup->insertSeparator();

		popup->insertItem(tr("Check &Status"), 4);
		popup->insertItem(QIconSet(*pix_info), tr("User &Info"), 12);

		cv->qa_hist->setIconSet(*pix_history);
		cv->qa_hist->addTo(popup);

		int x = popup->exec(pos);
		delete popup;

		if(x == -1)
			return;
		if(x == 0)
			actionRecvEvent(i);
		else if(x == 1)
			actionSendMessage(i);
		else if(x == 2) {
			actionSendUrl(i->jid());
		}
		else if(x == 3) {
			actionOfferFile(i->jid());
		}
		else if(x == 4) {
			actionStatusShow(i->jid());
		}
		else if(x == 6) {
			if(localStatus != STATUS_OFFLINE) {
				QString name = i->jid();
				if(i->nick() != i->jid())
					name += QString(" (%1)").arg(i->nick());

				actionAuthorize(i);
				QMessageBox::information(cv, CAP(tr("Authorize")),
				QString(tr("Authorization sent to <b>%1</b>.")).arg(name));
			}
		}
		else if(x == 8) {
			QString dstr; dstr.sprintf("contextPopup: groupchange: None\n");
			pdb(DEBUG_CV, dstr);

			// only send signal if we are actually changing group
			if(!i->group().isEmpty()) {
				actionGroupChange(i->jid(), "");

				//if(i->status() == STATUS_OFFLINE)
				//	warnOfflineGroupChange();
			}
		}
		else if(x == 9) {
			while(1) {
				bool ok = FALSE;
				QString newgroup = QInputDialog::getText(CAP(tr("Create New Group")), tr("Enter the new Group name:"), QLineEdit::Normal, QString::null, &ok, cv);
				if(!ok)
					break;

				if(newgroup.isEmpty()) {
					continue;
				}

				// make sure we don't have it already
				if(qstringlistmatch(groupList, newgroup) == -1) {
					QString dstr; dstr.sprintf("contextPopup: groupchange: [%s]\n", newgroup.latin1());
					pdb(DEBUG_CV, dstr);

					// only send signal if we are actually changing group
					if(i->group() != newgroup) {
						actionGroupChange(i->jid(), newgroup);

						//if(i->status() == STATUS_OFFLINE)
						//	warnOfflineGroupChange();
					}
					break;
				}
			}
		}
		else if(x == 10) {
			if(localStatus != STATUS_OFFLINE) {
				QString name = i->jid();
				if(i->nick() != i->jid())
					name += QString(" (%1)").arg(i->nick());

				actionAuthorize(i);

				// only add if it's _not_ a transport.  see jabcon.
				if(!i->isAgent())
					actionAdd(i);

				QMessageBox::information(cv, CAP(tr("Add")),
				QString(tr("Added/Authorized <b>%1</b> to the contact list.")).arg(name));
			}
		}
		else if(x == 11) {
			if(localStatus != STATUS_OFFLINE) {
				QString name = i->jid();
				if(i->nick() != i->jid())
					name += QString(" (%1)").arg(i->nick());

				actionAdd(i);
				QMessageBox::information(cv, CAP(tr("Authorize")),
				QString(tr("Rerequested authorization from <b>%1</b>.")).arg(name));
			}
		}
		else if(x == 12) {
			actionInfo(i->jid());
		}
		else if(x == 13) {
			actionHistory(i->jid());
		}
		else if(x == 14) {
			if(!i->isAlert())
				removeEntry(i->jid());
		}
		else if(x == 16) {
			actionAgentSetStatus(i->jid(), STATUS_OFFLINE);
		}
		else if(x >= 32) {
			int n = 0;
			int n2 = x - 32;

			QString newgroup;
			for(QStringList::Iterator it = groupList.begin(); it != groupList.end(); ++it, ++n) {
				if(n == n2) {
					newgroup = *it;
					break;
				}
			}
			if(n == n2) {
				if(i->group() != newgroup) {
					QString dstr; dstr.sprintf("contextPopup: group change [%s].\n", newgroup.latin1());
					pdb(DEBUG_CV, dstr);

					// only send signal if we are actually changing group
					if(i->group() != newgroup) {
						actionGroupChange(i->jid(), newgroup);

						//if(i->status() == STATUS_OFFLINE)
						//	warnOfflineGroupChange();
					}
				}
			}
		}
	}
	else if(i->type() == CV_GROUP) {
		QPopupMenu *popup = new QPopupMenu(cv);
		popup->insertItem(QIconSet(*pix_send), tr("Send message to group"), 0);
		popup->insertItem(tr("Re&name"), cv, SLOT(doRename()), QListView::Key_F2, 1);
		popup->insertSeparator();
		popup->insertItem(tr("Remove group"), 2);
		popup->insertItem(tr("Remove group and contacts"), 3);

		// disable if it's not a user group
		if(i->groupType() != CVG_USER) {
			popup->setItemEnabled(1,FALSE);
			popup->setItemEnabled(2,FALSE);
			popup->setItemEnabled(3,FALSE);
		}

		if(i->groupType() == CVG_OFFLINE || i->groupType() == CVG_AGENTS) {
			popup->insertSeparator();
			popup->insertItem(tr("Hide"), 4);
		}

		if(localStatus == STATUS_OFFLINE) {
			popup->setItemEnabled(2,FALSE);
			popup->setItemEnabled(3,FALSE);
		}

		int x = popup->exec(pos);
		delete popup;

		if(x == -1)
			return;
		if(x == 0) {
			QStringList list;

			list = contactListFromCVGroup(i);

			// send multi
			actionSendMessage(list);
		}
		else if(x == 2 && localStatus != STATUS_OFFLINE) {
			int n = QMessageBox::information(cv, CAP(tr("Remove Group")),tr(
			"Removing this group will cause all contacts associated with this group\n"
			"to be associated with no group (affects both online and offline contacts).\n"
			"\n"
			"Proceed?"), tr("&Yes"), tr("&No"));

			if(n == 0) {
				QStringList list = contactListFromGroup(i->groupName());
				for(QStringList::Iterator it = list.begin(); it != list.end(); ++it)
					actionGroupChange(*it, "");
			}
		}
		else if(x == 3 && localStatus != STATUS_OFFLINE) {
			int n = QMessageBox::information(cv, CAP(tr("Remove Group and Contacts")),tr(
			"WARNING!  This will remove all contacts associated with this group (affects\n"
			"both online and offline contacts)!\n"
			"\n"
			"Proceed?"), tr("&Yes"), tr("&No"));

			if(n == 0) {
				QStringList list = contactListFromGroup(i->groupName());
				for(QStringList::Iterator it = list.begin(); it != list.end(); ++it) {
					removeEntry(*it);
					actionRemove(*it);
				}
			}
		}
		else if(x == 4) {
			if(i->groupType() == CVG_OFFLINE)
				cv->setShowOffline(FALSE);
			if(i->groupType() == CVG_AGENTS)
				cv->setShowAgents(FALSE);
		}
	}
}

void ContactProfile::localUpdate(const JabRosterEntry &e)
{
	if(e.isAvailable())
		localStatus = e.local()->status;
	else
		localStatus = STATUS_OFFLINE;
}

void ContactProfile::scActionDefault(ContactViewItem *i)
{
	if(i->type() == CV_CONTACT)
		actionDefault(i);
}

void ContactProfile::scRecvEvent(ContactViewItem *i)
{
	if(i->type() == CV_CONTACT)
		actionRecvEvent(i);
}

void ContactProfile::scSendMessage(ContactViewItem *i)
{
	if(i->type() == CV_CONTACT)
		actionSendMessage(i);
}

void ContactProfile::scRename(ContactViewItem *i)
{
	if((i->type() == CV_CONTACT && i->inList()) || (i->type() == CV_GROUP && i->groupType() == CVG_USER)) {
		if(i->type() == CV_GROUP)
			i->resetGroupName(FALSE);

		i->setRenameEnabled(0, TRUE);
		i->startRename(0);
		i->setRenameEnabled(0, FALSE);
	}
}

void ContactProfile::scGroupChange(const QString &jid, const QString &newgroup)
{
	actionGroupChange(jid, newgroup);
}

void ContactProfile::scHistory(ContactViewItem *i)
{
	if(i->type() == CV_CONTACT)
		actionHistory(i->jid());
}

void ContactProfile::scOpenChat(ContactViewItem *i)
{
	if(i->type() == CV_CONTACT)
		actionOpenChat(i->jid());
}

void ContactProfile::scAgentSetStatus(ContactViewItem *i, int s)
{
	if(i->status() != STATUS_OFFLINE || localStatus == STATUS_OFFLINE)
		return;

	actionAgentSetStatus(i->jid(), s);
}

void ContactProfile::scRemove(ContactViewItem *i)
{
	ContactProfileItem *pi = cvi2cpi(i);

	if(!(localStatus == STATUS_OFFLINE && i->inList())) {
		QString name = i->jid();
		if(i->nick() != i->jid())
			name += QString(" (%1)").arg(i->nick());

		int n = 0;
		if(i->parentGroupType() != CVG_NOTINLIST) {
			n = QMessageBox::information(cv, CAP(tr("Remove")),
			QString(tr("Are you sure you want to remove <b>%1</b> from your contact list?")).arg(name),
			tr("&Yes"), tr("&No"));
		}

		if(n == 0) {
			QString jid = i->jid();
			removeEntry(pi);
			actionRemove(jid);
		}
	}
}

void ContactProfile::doItemRenamed(ContactViewItem *i, const QString &text)
{
	if(i->type() == CV_CONTACT) {
		// no change?
		if(text == i->nick())
			return;
		if(text.isEmpty()) {
			i->resetNick();
			QMessageBox::information(cv, CAP(tr("Error")), tr("You can't set a blank nickname."));
			return;
		}

		i->setNick(text);
		actionRename(i, text);
	}
	else {
		// no change?
		if(text == i->groupName()) {
			i->resetGroupName();
			return;
		}
		if(text.isEmpty()) {
			i->resetGroupName();
			QMessageBox::information(cv, CAP(tr("Error")), tr("You can't set a blank nickname."));
			return;
		}

		// make sure we don't have it already
		QStringList groupList = buildGroupList();
		if(qstringlistmatch(groupList, text) != -1) {
			i->resetGroupName();
			QMessageBox::information(cv, CAP(tr("Error")), tr("You already have a group with that name."));
			return;
		}

		QString oldName = i->groupName();

		// set group name
		i->setGroupName(text);
		cv->deferredContactSort();

		// send signal
		actionGroupRename(oldName, text);
	}
}

// return a list of contacts from a CVI group
QStringList ContactProfile::contactListFromCVGroup(ContactViewItem *group)
{
	QStringList list;

	for(ContactViewItem *item = LVI2CVI(group->firstChild()); item ; item = LVI2CVI(item->nextSibling())) {
		if(item->type() != CV_CONTACT)
			continue;

		list.append(item->jid());
	}

	return list;
}

// return a list of contacts associated with "groupName"
QStringList ContactProfile::contactListFromGroup(const QString &groupName)
{
	QStringList list;

	QDictIterator<ContactProfileItem> it(*roster);
	ContactViewItem *item;
	for(ContactProfileItem *pi; (pi = it.current()); ++it) {
		item = pi->cvi;
		if(!item->isAgent() && item->group() == groupName)
			list.append(item->jid());
	}

	return list;
}

int ContactProfile::contactSizeFromCVGroup(ContactViewItem *group)
{
	int total = 0;

	for(ContactViewItem *item = LVI2CVI(group->firstChild()); item ; item = LVI2CVI(item->nextSibling())) {
		if(item->type() != CV_CONTACT)
			continue;

		++total;
	}

	return total;
}

// return the number of contacts associated with "groupName"
int ContactProfile::contactSizeFromGroup(const QString &groupName)
{
	int total = 0;

	QDictIterator<ContactProfileItem> it(*roster);
	ContactViewItem *item;
	for(ContactProfileItem *pi; (pi = it.current()); ++it) {
		item = pi->cvi;
		if(!item)
			//printf("null item!!!\n");
		if(!item->parent()) {
			//printf("cvi: [%s]\n", item->jid().latin1());
			//printf("null parent!!!\n");
			continue;
		}
		int type = item->parentGroupType();
		if((type == CVG_ONLINE || type == CVG_OFFLINE || type == CVG_USER) && item->group() == groupName)
				++total;
	}

	return total;
}

void ContactProfile::updateGroupInfo(ContactViewItem *group)
{
	int inGroup;

	inGroup = contactSizeFromCVGroup(group);

	int type = group->groupType();
	if(type == CVG_ONLINE || type == CVG_OFFLINE || type == CVG_USER) {
		int total;
		QString gname;
		if(type == CVG_USER)
			gname = group->groupName();
		else
			gname = "";
		total = contactSizeFromGroup(gname);

		group->setGroupInfo(QString("(%1/%2)").arg(inGroup).arg(total));
	}
	else
		group->setGroupInfo(QString("(%1)").arg(inGroup));
}
