#include "mixxxclient.h"
#include "datadispatcher.h"
#include "gui.h"

#include <unistd.h>
#include <qtextstream.h>

// *** constructor ***
MixxxClient::MixxxClient() {
    connected  = false;
    connecting = false;
    sending    = false;
    socket     = new QSocket(this, "mixxxConection");
    timer      = new QTimer( this );
    listener   = NULL;

    connect( socket, SIGNAL(connected()),
             SLOT(slot_socketConnected()) );
    connect( socket, SIGNAL(connectionClosed()),
             SLOT(slot_socketConnectionClosed()) );
    connect( socket, SIGNAL(readyRead()),
             SLOT(slot_socketReadyRead()) );
    connect( socket, SIGNAL(error(int)),
             SLOT(slot_socketError(int)) );
    connect( timer, SIGNAL(timeout()), this,
             SLOT(slot_handleCommandQueue()) );
}

// ##############################################
// # establishes connection to mixxx
// ##############################################
void MixxxClient::establishConnection(QString host, int port) {
    if (verbose == 10)
        qWarning("connecting ...");
    if(!connected) {
        connecting = true;
        socket->connectToHost(host, port);
    }

    while(connecting) {
        sleep(1);
    }
}

void MixxxClient::disconnect() {
    if (verbose == 10)
        qWarning("disconnecting ...");
    socket->close();
    if ( socket->state() == QSocket::Closing ) {
        // We have a delayed close.
        connect( socket, SIGNAL(delayedCloseFinished()),
                 SLOT(slot_socketClosed()) );
    } else {
        // The socket is closed.
        slot_socketClosed();
    }
}

void MixxxClient::loadAtChannel(QString file, int channel) {
    QString player("player" + QString::number(channel));
    enqueueCommand( player + " load " + file );
}

void MixxxClient::loadAndPlayAtChannel(QString file, int channel) {
    QString player("player" + QString::number(channel));
    enqueueCommand( player + " load " + file );
    enqueueCommand( player + " start");
}

void MixxxClient::stopAtChannel(int channel) {
    QString player("player" + QString::number(channel));
    enqueueCommand( player + " stop");
}

void MixxxClient::pauseAtChannel(int channel) {
    QString player("player" + QString::number(channel));
    enqueueCommand( player + " stop");
}

void MixxxClient::continueAtChannel(int channel) {
    QString player("player" + QString::number(channel));
    enqueueCommand( player + " start");
}

void MixxxClient::enqueueCommand(QString command) {
    commandQueue.append(command);
    if( !timer->isActive() )
        timer->start( 500 ); // 500 ms
}

QString MixxxClient::dequeueCommand() {
    QString command = commandQueue.first();
    commandQueue.pop_front();
    if( commandQueue.empty() && timer->isActive() )
        timer->stop();
    return command;
}

void MixxxClient::sendToServer(QString command) {
    if(connected) {
        sending = true;
        if (verbose == 10)
            qWarning(command.local8Bit().data());
        // write to the server
        QTextStream os(socket);
        os << command << "\n";
    } else {
        MessageEvent* e = new MessageEvent(_("Lost connection to mixxx!"));
        QApplication::postEvent( gui, e );
    }
}

void MixxxClient::registerListener(GUI_MixxxQueue *l) {
    listener = l;
}

void MixxxClient::slot_handleCommandQueue() {
    if(sending || commandQueue.empty())
        return;

    sendToServer(dequeueCommand());
}

void MixxxClient::slot_socketReadyRead() {
    // read from the server
    while ( socket->canReadLine() ) {
        QString result = socket->readLine();
        if (verbose == 10)
            qWarning(result.local8Bit().data());
        if ( result.startsWith("OK") ) {
            sending = false;
            continue;
        } else if ( result.startsWith("ERR") ) {
            sending = false;
            MessageEvent* e = new MessageEvent(_("Latest command failed!"));
            QApplication::postEvent( gui, e );
        } else if ( result.startsWith("Mixxx")) {
            MessageEvent* e = new MessageEvent(/* TRANSLATORS: %1 = mixxx version string */ QString(_("Connection established to %1")).arg(result));
            QApplication::postEvent( gui, e );
        } else if ( listener && result.startsWith("player") ) {
            if(result.startsWith("player1 finished")) {
                listener->notifyFinished(1);
            } else if(result.startsWith("player2 finished")) {
                listener->notifyFinished(2);
            }
        } else {
            MessageEvent* e = new MessageEvent(QString(_("Unknown response")) + ":\n" + result);
            QApplication::postEvent( gui, e );
        }
    }
}

void MixxxClient::slot_socketConnected() {
    commandQueue.clear();
    connecting = false;
    connected  = true;

}

void MixxxClient::slot_socketConnectionClosed() {
    disconnect();
}

void MixxxClient::slot_socketClosed() {
    if( timer->isActive() )
        timer->stop();
    connected = false;
    app->lock()
    ;
    datadispatcher->eventLostMixxxConnection();
    MessageEvent* e = new MessageEvent(_("Connection to mixxx closed."));
    QApplication::postEvent( gui, e );
    app->unlock();
}

void MixxxClient::slot_socketError( int err ) {
    connecting = false;

    QString error;

    switch (err) {
    case QSocket::ErrConnectionRefused:
        error = _("The connection was refused.");
        break;
    case QSocket::ErrHostNotFound:
        error = _("The host was not found.");
        break;
    case QSocket::ErrSocketRead:
        error = _("A read from the socket failed");
        break;
    default:
        error = _("Unknown error");
    }

    MessageEvent* e = new MessageEvent(QString(_("Connection to mixxx could not be established.")) + "\n" + QString(_("Error code")) + ": " + QString::number(err) + " - " + error);
    QApplication::postEvent( gui, e );
}
