/*
 *  Copyright (C) 2001, 2002, 2003 Yoann Vandoorselaere.
 *
 *  This program is free software; you can redistribute it and/or modify 
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors: Yoann Vandoorselaere <yoann@prelude-ids.org>
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>
#include <unistd.h>
#include <inttypes.h>
#include <fcntl.h>
#include <sys/utsname.h>
#include <pthread.h>

#include "packet.h"

#include <libprelude/list.h>
#include <libprelude/prelude-log.h>
#include <libprelude/plugin-common.h>
#include <libprelude/idmef-message-id.h>
#include <libprelude/prelude-message-id.h>
#include <libprelude/idmef-tree.h>
#include <pcap.h>

#include "config.h"
#include "nids-alert.h"
#include "nids-alert-id.h"
#include "packet-decode.h"


#include <libprelude/prelude-message-buffered.h>
#include <libprelude/prelude-client.h>
#include <libprelude/idmef-msg-send.h>
#include <libprelude/sensor.h>



#define NIDS_CORE_NAME "Prelude NIDS core detection engine"
#define NIDS_CORE_AUTHOR "The Prelude team"
#define NIDS_CORE_CONTACT "prelude-list@prelude-ids.org"
#define NIDS_CORE_DESC "Internal Prelude NIDS detection routines"

#define NIDS_CLASS "NIDS"
#define NIDS_MODEL "Prelude NIDS"
#define NIDS_MANUFACTURER "The Prelude Team http://www.prelude-ids.org"


static plugin_generic_t prelude_core_plugin = {
        NIDS_CORE_NAME, sizeof(NIDS_CORE_NAME),
        NIDS_CORE_AUTHOR, sizeof(NIDS_CORE_AUTHOR),
        NIDS_CORE_CONTACT, sizeof(NIDS_CORE_CONTACT),
        NIDS_CORE_DESC, sizeof(NIDS_CORE_DESC),
};


static pthread_mutex_t mutex;
static prelude_msgbuf_t *msgbuf, *hb_msgbuf;
static idmef_analyzer_t analyzer;
static idmef_heartbeat_t heartbeat;


static void send_heartbeat(void *data) 
{
        struct timeval tv;
        
        gettimeofday(&tv, NULL);
        heartbeat.create_time.sec = tv.tv_sec;
        heartbeat.create_time.usec = tv.tv_usec;

        /*
         * we could use additional data to send stats.
         */
        prelude_msgbuf_set_header(hb_msgbuf, PRELUDE_MSG_IDMEF, 0);
        idmef_send_heartbeat(hb_msgbuf, &heartbeat);
        prelude_msgbuf_mark_end(hb_msgbuf);
}




static void send_additional_data(prelude_msgbuf_t *msgbuf, idmef_additional_data_type_t type,
                                 const char *meaning, size_t mlen, const char *data, size_t dlen) 
{        
        type = htonl(type);
        
        prelude_msgbuf_set(msgbuf, MSG_ADDITIONALDATA_TAG, 0, NULL);
        prelude_msgbuf_set(msgbuf, MSG_ADDITIONALDATA_TYPE, sizeof(type), &type);
        prelude_msgbuf_set(msgbuf, MSG_ADDITIONALDATA_MEANING, mlen, meaning);
        prelude_msgbuf_set(msgbuf, MSG_ADDITIONALDATA_DATA, dlen, data);
        prelude_msgbuf_set(msgbuf, MSG_END_OF_TAG, 0, NULL);
}




static void packet_to_msg(prelude_msgbuf_t *msgbuf, packet_container_t *pkt) 
{
        int i = 0;        
        uint8_t tag;
        
        if ( ! pkt )
                return;

        tag = MSG_FORMAT_PRELUDE_NIDS;
        
        prelude_msgbuf_set(msgbuf, MSG_OWN_FORMAT, sizeof(uint8_t), &tag);
        prelude_msgbuf_set(msgbuf, ID_PRELUDE_NIDS_PACKET, 0, NULL);
        
        do {
                if ( pkt->packet[i].proto == p_end ) 
                        prelude_msgbuf_set(msgbuf, p_end, 0, NULL);
                else
                        prelude_msgbuf_set(msgbuf, pkt->packet[i].proto, pkt->packet[i].len, pkt->packet[i].p.ip);
        } while ( pkt->packet[i++].proto != p_end );
        
        prelude_msgbuf_set(msgbuf, MSG_END_OF_TAG, 0, NULL);
}




/**
 * nids_alert_new:
 * @p: Pointer to a plugin.
 * @pkt: Pointer on a packet container.
 * @item: Number of item the caller want to put in the message.
 * @len: Length of all the item the caller want to put in the message.
 *
 * Setup a basic IDMEF alert containing NIDS based alert information.
 * The caller can still add it's own item when this call return, but you'll have
 * to call prelude_msg_set(msg, MSG_END_OF_TAG, 0, NULL); to set the end of the
 * alert.
 *
 * You don't have to count the alert ending tag in the item.
 *
 * Returns: a new #prelude_msg_t object.
 */
static int nids_alert_new(plugin_generic_t *p, packet_container_t *pkt, nids_alert_t *alert) 
{
        uint8_t priority;
        idmef_time_t time;
        struct timeval tv;
        const char pname[] = "Detection Plugin Name",
                pauthor[] = "Detection Plugin Author",
                pcontact[] = "Detection Plugin Contact",
                pdesc[] = "Detection Plugin Description";
        
        if ( ! p )
                p = &prelude_core_plugin;
                
        priority = PRELUDE_MSG_PRIORITY_HIGH;

        if ( alert->impact ) {
                switch (alert->impact->severity) {
                case impact_low:
                        priority = PRELUDE_MSG_PRIORITY_LOW;
                        break;
                        
                case impact_medium:
                        priority = PRELUDE_MSG_PRIORITY_MID;
                        break;
                        
                case impact_high:
                        priority = PRELUDE_MSG_PRIORITY_HIGH;
                        break;
                }
        }

        prelude_msgbuf_set_header(msgbuf, PRELUDE_MSG_IDMEF, priority);
        
        packet_to_msg(msgbuf, pkt);

        prelude_msgbuf_set(msgbuf, MSG_ALERT_TAG, 0, NULL);
        
        idmef_send_analyzer(msgbuf, &analyzer);
        
        gettimeofday(&tv, NULL);
        time.sec = tv.tv_sec;
        time.usec = tv.tv_usec;
        idmef_send_create_time(msgbuf, &time);
        
        time.sec = pkt->pcap_hdr->ts.tv_sec;
        time.usec = pkt->pcap_hdr->ts.tv_usec;
        idmef_send_detect_time(msgbuf, &time);
        
        send_additional_data(msgbuf, string, pname, sizeof(pname), plugin_name(p), plugin_name_len(p));
        send_additional_data(msgbuf, string, pauthor, sizeof(pauthor), plugin_author(p), plugin_author_len(p));
        send_additional_data(msgbuf, string, pcontact, sizeof(pcontact), plugin_contact(p), plugin_contact_len(p));
        send_additional_data(msgbuf, string, pdesc, sizeof(pdesc), plugin_desc(p), plugin_desc_len(p));

        return 0;
}




void nids_alert(plugin_generic_t *plugin, packet_container_t *packet,
                nids_alert_t *alert, const char *str, ...)
{
        int ret;
        va_list ap;
        char buf[512];
        idmef_assessment_t assessment;
        const char meaning[] = "Attack description";
        
        if ( ! alert ) {
                log(LOG_ERR, "empty alerted not emited.\n");
                return;
        }
        
	pthread_mutex_lock(&mutex);

        ret = nids_alert_new(plugin, packet, alert);
        if ( ret < 0 ) {
		pthread_mutex_unlock(&mutex);
                log(LOG_ERR, "error creating alert.\n");
                return;
        }
        
        if ( str ) {
                va_start(ap, str);
                ret = vsnprintf(buf, sizeof(buf), str, ap);
                va_end(ap);        

                /*
                 * return -1 if the output was truncated due to this limit. (Thus until
                 * glibc 2.0.6. Since glibc 2.1 these functions follow the
                 * C99 standard and return the number of characters (excluding
                 * the trailing '\0') which would have been written to
                 * the final string if enough space had been available.)
                 */
                if ( ret > sizeof(buf) || ret < 0 )
                        ret = sizeof(buf);
                else
                        ret++;

                send_additional_data(msgbuf, string, meaning, sizeof(meaning), buf, ret);
        }
        
        idmef_send_additional_data_list(msgbuf, &alert->additional_data_list);
        
        if ( alert->impact || alert->confidence ) {
                assessment.impact = alert->impact;
                assessment.confidence = alert->confidence;
                INIT_LIST_HEAD(&assessment.action_list);
                idmef_send_assessment(msgbuf, &assessment);
        }
        
        idmef_send_classification(msgbuf, &alert->classification);

        prelude_msgbuf_set(msgbuf, MSG_END_OF_TAG, 0, NULL);
        prelude_msgbuf_mark_end(msgbuf);

	pthread_mutex_unlock(&mutex);
}



void nids_alert_init(nids_alert_t *alert) 
{
        memset(alert, 0, sizeof(nids_alert_t));
        INIT_LIST_HEAD(&alert->additional_data_list);
}




int nids_alert_init_subsystem(void) 
{
	pthread_mutex_init(&mutex, NULL);

        msgbuf = prelude_msgbuf_new(1);
        if ( ! msgbuf )
                return -1;

        /*
         * we don't use the same msgbuf
         * for heartbeat for asynchronous issue.
         */
        hb_msgbuf = prelude_msgbuf_new(1);
        if ( ! hb_msgbuf )
                return -1;

        /*
         * setup analyzer.
         */
        prelude_analyzer_fill_infos(&analyzer);
        idmef_string_set_constant(&analyzer.version, VERSION);
        idmef_string_set_constant(&analyzer.manufacturer, NIDS_MANUFACTURER);
        idmef_string_set_constant(&analyzer.model, NIDS_MODEL);
        idmef_string_set_constant(&analyzer.class, NIDS_CLASS);

        /*
         * setup analyzer node.
         */
        INIT_LIST_HEAD(&heartbeat.additional_data_list);
        memcpy(&heartbeat.analyzer, &analyzer, sizeof(analyzer));
        prelude_heartbeat_register_cb(&send_heartbeat, NULL);
        
        return 0;
}


















