/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "config.h"
#include "dbus_server.h"
#include "pm_interface.h"
#include "globals.h"
#include "main_loop.h"
#include "powerlib.h"

#include <string>

using namespace std;

#include <liblazy.h>
#include <string.h>

DBusConnection* DBus_Server::_connection = NULL;

using namespace Powersave::Globals;

DBus_Server::DBus_Server()
{
}

DBus_Server::~DBus_Server()
{
	dbus_connection_close(_connection);
	dbus_connection_unref(_connection);
}

gboolean DBus_Server::dbus_reconnect() {
	DBusConnection *dbus_connection = openSystemConnection();
	if (dbus_connection == NULL) {
		return true;
	} else {
		dbus_connection_setup_with_g_main(dbus_connection, NULL);
		pDebug(DBG_ERR,"Reconnected to dbus daemon");
		return false;
	}
}

bool DBus_Server::serviceHasOwner(const string &service)
{
	if (_connection == NULL) {
		pDebug(DBG_INFO, "Connection to dbus not ready, skipping check for service %s",
		       service.c_str());
		return false;
	}

	DBusError dbus_error;
	dbus_error_init(&dbus_error);

	bool has_owner = dbus_bus_name_has_owner(_connection, service.c_str(), &dbus_error);

	if (dbus_error_is_set(&dbus_error)) {
		pDebug(DBG_WARN, "dbus_bus_name_has_owner() failed: %s",
		       dbus_error.message);
		dbus_error_free(&dbus_error);
		return false;
	}
		
	return has_owner;
}

DBusHandlerResult DBus_Server::filter_function(DBusConnection * conn, DBusMessage * msg, void *data)
{
	pDebug(DBG_DEBUG, "");

	/* Be careful, handle error at once and go out, or other functions
	   may fail ungracefully ... */

	if (msg == NULL) {
		return DBUS_HANDLER_RESULT_HANDLED;
		pDebug(DBG_ERR, "Invalid message received");
	}

	/** @todo This means DBus daemon itself threw us out,
	           Need to disconnect gracefully and free all resources
	*/
	if (dbus_message_is_signal(msg, DBUS_INTERFACE_LOCAL, "Disconnected")) {
		pDebug(DBG_ERR, "DBus daemon disconnected. Trying to reconnect...");
		dbus_connection_unref(_connection);
		_connection = NULL;
		// only way I see for now to get this information
		Powersave::Globals::main_loop->addCallback(3000, (GSourceFunc) DBus_Server::dbus_reconnect);
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	int type = dbus_message_get_type(msg);
	if (type == DBUS_MESSAGE_TYPE_ERROR) {
		char *error;
		dbus_message_get_args(msg, NULL, DBUS_TYPE_STRING, &error, DBUS_TYPE_INVALID);
		pDebug(DBG_ERR, "Received error message: %s, %s", dbus_message_get_error_name(msg),
		       error);
		return DBUS_HANDLER_RESULT_HANDLED;
	}
	/* Have to use const char* instead of string here.
	 * Return value might be NULL
	 */
	const char *t = dbus_message_get_interface(msg);
	if (t == NULL) {
		pDebug(DBG_INFO, "Received message from invalid interface");
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
	const string interface = t;
	// We do not compare the object path, I see no need for ...
	//          && strcmp(dbus_message_get_path(msg), DBUS_MESSAGE_BUS_PATH) == 0) {

	/* ToDo: what should be filtered here and what should be filtered
	   in path_message_func (for what is this vtable stuff good for) ?!?
	   -> Deleted the vtable path_message_func stuff for now, all
	   message processing is started here
	 */

	DBusHandlerResult ret = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

	int reply_error_id;

	DBusMessage *reply;

	string msg_member = dbus_message_get_member(msg);
	string sender = dbus_message_get_sender(msg);

	pDebug(DBG_INFO, "Received msg: '%s' from interface: %s", msg_member.c_str(),
	       interface.c_str());
	pDebug(DBG_DEBUG, "Obj.path '%s' from '%s'", dbus_message_get_path(msg), sender.c_str());


	/** @brief checking of interfaces is done here in filter function
	 *         checking for the right member, message contents, etc. must
	 *         be done in pm->handleDBusRequest(..)
	 *         Create the Message here, we must send a reply (sendReply(),
	 *         there the memory for the reply will be freed again.
	 */
	if (interface == PS_DBUS_SCRIPTS_INTERFACE){
		if (type != DBUS_MESSAGE_TYPE_METHOD_CALL){
			pDebug (DBG_ERR, "Invalid msg type sent to Scripts interface");
		}
		else{
			reply = dbus_message_new_method_return(msg);
			pDebug (DBG_DIAG, "Method sent to Scripts interface");
			reply_error_id = pm->handleScriptsMessage(msg, msg_member);
			sendReply(reply, reply_error_id);
		}
		ret = DBUS_HANDLER_RESULT_HANDLED;
	//action interface kept for compatibility
	} else if (interface == PS_DBUS_INTERFACE || interface == "com.novell.powersave.action") {
		if (type != DBUS_MESSAGE_TYPE_METHOD_CALL) {
			pDebug(DBG_ERR, "Invalid msg sent to action interface");
		} else {
			reply = dbus_message_new_method_return(msg);
			pDebug(DBG_INFO, "Method sent to action interface");
			reply_error_id = pm->handleDBusMessage(msg, reply, msg_member);
			sendReply(reply, reply_error_id);
		}
		ret = DBUS_HANDLER_RESULT_HANDLED;
	} else if (interface == DBUS_INTERFACE_DBUS) {
		if (type != DBUS_MESSAGE_TYPE_SIGNAL) {
			pDebug(DBG_ERR, "Invalid signal received");
			/** @todo HANDLED or NOT_YET_HANDLED, does it matter ?!? */
			ret = DBUS_HANDLER_RESULT_HANDLED;
		} else {
			pDebug(DBG_INFO, "Signal from %s with member %s received",
			       DBUS_INTERFACE_DBUS, dbus_message_get_member(msg));

			string msg_member = dbus_message_get_member(msg);
			if (msg_member == "NameOwnerChanged") {

				char *old_owner;
				char *new_owner;
				/** @todo Error catching... */

				/* get the second (the old owner) and the third (the new owner)
				   string of the message (first string: affected owner) */
				if (!liblazy_dbus_message_get_basic_arg(msg, DBUS_TYPE_STRING, &old_owner, 0)
				    && !liblazy_dbus_message_get_basic_arg(msg, DBUS_TYPE_STRING, &new_owner, 2)
				    && old_owner != NULL && new_owner != NULL) {
					/* if the new owner is "" the client broke away -> remove
					   from client list */
					if (!strcmp(new_owner, "")) {
						// test if hal broke away
						if (!strcmp(old_owner, "org.freedesktop.Hal")) {
							pDebug(DBG_WARN, "Hal service stopped. "
							       "Battery information no longer available");
						}
					}
				}
			}
		}
	} else if (interface == DBUS_HAL_DEVICE_INTERFACE) {
 		string msg_member = dbus_message_get_member(msg);
		const char *path = dbus_message_get_path(msg);
		if (msg_member == "PropertyModified") {
			if (liblazy_hal_query_capability((char*)path, "battery")) {
				pm->checkBatteryStateChanges();
			}
			else if (liblazy_hal_query_capability((char*)path, "ac_adapter")) {
				pm->checkACStateChanges();
			}
		}
	} else {
		/* This message is noy for us -> we are the only
		 * server offering com.novell.powersave
		 * -> client probably made a mistake and specified a wrong
		 * interface -> at least warning
		 * We cannot answer to an interface we don't know (I expect).
		 */
		pDebug(DBG_WARN, "Received a message for a foreign interface");
		ret = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
	pDebug(DBG_INFO, "Message handled");
	return ret;
}


DBusConnection *DBus_Server::openSystemConnection()
{
	DBusError error;
	dbus_error_init(&error);
	int ret;

	_connection = dbus_bus_get(DBUS_BUS_SYSTEM, &error);
	if (dbus_error_is_set(&error)) {
		// only throw diag message. Real error message is thrown in main loop
		pDebug(DBG_DIAG, "dbus_connection_open_private(): Error. (%s)", error.message);
		dbus_error_free(&error);
		return NULL;
	} else {
		dbus_connection_set_exit_on_disconnect(_connection, 0);
		pDebug(DBG_INFO, "dbus_bus_get(): Success.");
	}

	dbus_bus_request_name(_connection, PS_DBUS_SERVICE, 0, &error);

	if (dbus_error_is_set(&error)) {
		pDebug(DBG_ERR, "dbus_bus_request_name(): Error. (%s)", error.message);
		if (_connection != NULL) {
			dbus_connection_unref(_connection);
		}
		dbus_error_free(&error);
		return NULL;
	}

	ret = dbus_bus_request_name(_connection, DBUS_SYSTEM_PM_INTERFACE,
				    DBUS_NAME_FLAG_ALLOW_REPLACEMENT, &error);

	if (dbus_error_is_set(&error)) {
		pDebug(DBG_ERR, "dbus_bus_request_name(): Error. (%s)", error.message);
		dbus_error_free(&error);
	}

	switch (ret) {
	case DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER:
		pDebug(DBG_INFO, "Successfully acquired the interface %s",
		       DBUS_SYSTEM_PM_INTERFACE);
		break;
	case DBUS_REQUEST_NAME_REPLY_IN_QUEUE:
		pDebug(DBG_INFO, "Placed in queue for interface %s", DBUS_SYSTEM_PM_INTERFACE);
		break;
	case DBUS_REQUEST_NAME_REPLY_ALREADY_OWNER:
		pDebug(DBG_INFO, "Already owner of interface %s", DBUS_SYSTEM_PM_INTERFACE);
	default:
		break;
	};

	dbus_connection_add_filter(_connection, filter_function, NULL, NULL);

	dbus_bus_add_match(_connection,
			   "type='method_call'," "interface='com.novell.powersave.manager'," "member='connect'",
			   NULL);

	/* keep track o f disconnecting clients */
	dbus_bus_add_match(_connection,
			   "type='signal'," "interface='org.freedesktop.DBus'," "member='NameOwnerChanged'",
			   NULL);

	dbus_bus_add_match(_connection,
			   "type='signal'," "interface=org.freedesktop.Hal.Device," "member=PropertyModified",
			   NULL);

	return _connection;
}

bool DBus_Server::haveClient()
{
	int ret;

	ret = dbus_bus_request_name(_connection, DBUS_SYSTEM_PM_INTERFACE,
				    DBUS_NAME_FLAG_ALLOW_REPLACEMENT, NULL);

	if (ret == DBUS_REQUEST_NAME_REPLY_ALREADY_OWNER) {
		pDebug(DBG_DEBUG, "We own the interface %s", DBUS_SYSTEM_PM_INTERFACE);
		return false;
	}

	pDebug(DBG_DEBUG, "We don't own the interface %s", DBUS_SYSTEM_PM_INTERFACE);
	return true;
}

void DBus_Server::sendReply(DBusMessage * reply, int error_id)
{

	/* We do not generate DBus error messages ourselves
	   If needed, this is the place to do.
	   Problem: we need to generate the message here
	   and do not know whether the
	   message should become an error message
	   Maybe we could convert the message to an error message
	   later, but we should not need that...

	   dbus_message_set_error_name (DBusMessage  *message,
	   const char   *error_name)
	   reply = dbus_message_new_method_return(msg);
	 */
	dbus_uint16_t err_id_u16 = error_id;

	if (reply == NULL) {
		pDebug(DBG_WARN, "Invalid reply");
		return;
	}

	pDebug(DBG_INFO, "Send error_id %d, '%s'", error_id, DBus_Error_Array[error_id]);
	/* attache contents collected from raw_reply_msg to DBusMessage reply */

	/* We only use DBUS_TYPE_UINT16 once: to indicate the error code */
	dbus_message_append_args(reply, DBUS_TYPE_UINT16, &err_id_u16,
				 // don't need to also send the error string ?
				 // or could it be useful dbus-monitor ?
				 // DBUS_TYPE_STRING, &DBus_Error_Array[error_id],
				 DBUS_TYPE_INVALID);

	// check for memory failure? -> then FALSE will be returned
	if (reply == NULL) {
		pDebug(DBG_WARN, "Could not send reply/error messge");
		return;
	}
	dbus_connection_send(_connection, reply, NULL);
	dbus_connection_flush(_connection);
	dbus_message_unref(reply);
}

int DBus_Server::emitSignal(const char *signal, const char *value)
{
	DBusMessage *message;
	
	if (_connection == NULL) {
		pDebug(DBG_INFO, "Connection to dbus not ready, skipping signal %s", value);
		return false;
	}

	message = dbus_message_new_signal( "/com/novell/powersave",
					   "com.novell.powersave.manager",
					   signal );

	/* append the event to the signal */
	dbus_message_append_args(message,
				 DBUS_TYPE_STRING, &value,
				 DBUS_TYPE_INVALID );

	/* and send the message */
	if ( ! dbus_connection_send( _connection,
				     message,
				     NULL ) ) {
		pDebug (DBG_ERR, "Could not send message to D-BUS system bus.\n");
		dbus_message_unref( message );
		return false;
	}

	dbus_message_unref( message );

	return true;
}

bool DBus_Server::sendMethodCall(const char *interface, const char *path,
				 const char *object, const char *method,
				 int first_arg_type, ...)
{
	if (_connection == NULL) {
		pDebug(DBG_INFO, "Connection to dbus not ready, skipping method call %s",
		       method);
		return false;
	}

	DBusMessage *message;
	va_list var_args;

	message = dbus_message_new_method_call( interface, path, object, method );

	va_start(var_args, first_arg_type);
	dbus_message_append_args_valist(message, first_arg_type, var_args);
	va_end(var_args);

	if (!dbus_connection_send(_connection, message, NULL)) {
		pDebug (DBG_ERR, "Could not send method call\n");
		dbus_message_unref( message );
		return false;
	}

	dbus_message_unref( message );
	dbus_connection_flush(_connection);

	return true;
}

bool DBus_Server::sendMethodCallAndBlock(const char *interface, const char *path,
					 const char *object, const char *method,
					 int first_arg_type, ...)
{
	if (_connection == NULL) {
		pDebug(DBG_INFO, "Connection to dbus not ready, skipping method call %s",
		       method);
		return false;
	}

	DBusError dbus_error;
	DBusMessage *message;
	DBusMessage *reply;
	va_list var_args;

	dbus_error_init(&dbus_error);

	message = dbus_message_new_method_call( interface, path, object, method );

	va_start(var_args, first_arg_type);
	dbus_message_append_args_valist(message, first_arg_type, var_args);
	va_end(var_args);

	reply = dbus_connection_send_with_reply_and_block(_connection, message, -1, &dbus_error);

	if (dbus_error_is_set(&dbus_error)) {
		pDebug (DBG_ERR, "Could not send method call: %s\n", dbus_error.message);
		dbus_message_unref( message );
		dbus_error_free(&dbus_error);
		return false;
	}

	dbus_message_unref( message );
	dbus_message_unref(reply);
	dbus_connection_flush(_connection);

	return true;
}

bool DBus_Server::sendMethodCallWithReturn(const char *interface, const char *path,
					   const char* object ,const char *method,
					   int arg_type, void *value)
{
	if (_connection == NULL) {
		pDebug(DBG_INFO, "Connection to dbus not ready, skipping method call %s",
		       method);
		return false;
	}

	DBusMessage	*message;
	DBusError	dbus_error;
	DBusMessage	*reply;

	dbus_error_init(&dbus_error);
	
	message = dbus_message_new_method_call( interface, path, object, method );

	reply = dbus_connection_send_with_reply_and_block( _connection, message, -1, &dbus_error);

	if (dbus_error_is_set(&dbus_error)) {
		pDebug (DBG_WARN, "Could not send method call to D-BUS system bus: %s\n",
			dbus_error.message);
		goto Error;
	}

	if (!dbus_message_get_args(reply, &dbus_error, arg_type, value, DBUS_TYPE_INVALID)){
		if (dbus_error_is_set(&dbus_error))
			pDebug(DBG_WARN, "Could not get argument from reply: %s",
			       dbus_error.message);
		goto NoReply;
	}
			
	dbus_connection_flush(_connection);
	return true;
	
NoReply:
	dbus_message_unref(reply);
Error:
	dbus_message_unref(message);
	dbus_error_free(&dbus_error);

	return false;
}
