/*
 * spawnit.c  Spawn the appropriate protocol.
 *
 * Version:  @(#)spawnit.c  1.30  24-Oct-1997  MvS.
 *
 */

#include "server.h"
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <utmp.h>
#include <time.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <sys/wait.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>

#define MAX_SPAWN_ARGV 256

extern int traffic_stats(int sockfd, struct in_addr ina, struct traffic_stat *t_stat);

/*
 *  Get an utmp entry by process id.
 *  This should be a standard function IMHO.
 */
static struct utmp *getutpid(struct utmp *in)
{
  struct utmp *u;

  while((u = getutent()) != NULL)
  {
    if(u->ut_type != INIT_PROCESS && u->ut_type != LOGIN_PROCESS && u->ut_type != USER_PROCESS)
      continue;
    if(u->ut_pid == in->ut_pid)
      break;
  }
  return u;
}

/*
 *  Relay the SIGHUP to our process group.
 */
static void got_hup(int sig)
{
  struct sigaction sa;
  pid_t pgrp = getpgrp();

  signal(sig, SIG_IGN);
  unblock(sig);

  if(sig == SIGALRM)
    alarm(0);

  if(pgrp > 0)
  {
    if(sig == SIGALRM)
      kill(-pgrp, SIGTERM);
    else
      kill(-pgrp, sig);
  }

  sa.sa_handler = got_hup;
  sa.sa_flags   = SA_RESTART;
  sigaction(sig, &sa, NULL);
}

/*
 *  Strip the realm/passthrough from the login name, if any.
 */
static void striprealm(struct auth *ai)
{
  struct realm_def *realm;

  realm = ckrealm(ai);
  if(realm == NULL || !realm->strip)
    return;
  if(realm->prefix)
  {
    /* strcpy(ai->login, ai->login + strlen(realm->name));
     * parameters for strcpy may not overlap!!! */

    char *tmp = ai->login + strlen(realm->name);
    memmove(ai->login, tmp, strlen(tmp) + 1);
  }
  else
  {
    ai->login[strlen(ai->login) - strlen(realm->name)] = '\0';
  }
}

/*
 *  Execute but split command line args first.
 */
static int doexec(char *cmd, const char *argv0, char *args)
{
  char *argv[MAX_SPAWN_ARGV];
  int argc = 1;
  int rc;
  char *s;

  argv[0] = xstrdup(argv0);
  s = strtok(args, " \t\n");
  while(s && argc < MAX_SPAWN_ARGV - 1)
  {
    argv[argc++] = s;
    s = strtok(NULL, " \t\n");
  }
  argv[argc] = NULL;
  rc = execv(cmd, argv);
  free(argv[0]); /* only happens if we can't exec() */
  return rc;
}

/*
 *  Execute a rlogin session.
 */
static int do_rlogin(const struct auth *ai)
{
  int i = 0;
  const char *args[10];

  args[i++] = "rlogin";

  if(ai->login && ai->login[0] && strcmp(ai->login, "NONE"))
  {
    args[i++] = "-i";
    args[i++] = ai->login;
    args[i++] = "-l";
    args[i++] = ai->login;
  }
  args[i++] = dotted(ai->host);
  args[i] = NULL;

  maketermsane();
  execv(lineconf.rlogin, (char **)args);
  nsyslog(LOG_ERR, "%s: %m", lineconf.rlogin);
  return -1;
}

/*
 *  Execute a telnet session.
 */
static int do_telnet(const struct auth *ai)
{
  int i = 0;
  const char *args[10];

  args[i++] = "telnet";
  args[i++] = "-8";
#ifndef CYCLADES
  args[i++] = "-E";
#endif
  if(ai->login && ai->login[0] && strcmp(ai->login, "NONE"))
  {
    args[i++] = "-l";
    args[i++] = ai->login;
  }
  args[i++] = dotted(ai->host);
  if(ai->loginport)
    args[i++] = dotted(ai->loginport);
  else if(lineconf.socket_port)
    args[i++] = dotted(lineconf.socket_port);
  args[i] = NULL;

  maketermsane();
  execv(lineconf.telnet, (char **)args);
  nsyslog(LOG_ERR, "%s: %m", lineconf.telnet);
  return -1;
}

/*
 *  Execute a secure shell session.
 */
static int do_ssh(const struct auth *ai)
{
  int i = 0;
  const char *args[10];

  args[i++] = "ssh";

  if(ai->proto != P_SSH)
    args[i++] = "-2";
  else
    args[i++] = "-1";

  if(ai->login && ai->login[0] && strcmp(ai->login, "NONE"))
  {
    args[i++] = "-l";
    args[i++] = ai->login;
  }
  if(ai->loginport)
  {
    args[i++] = "-p";
    args[i++] = dotted(ai->loginport);
  }
  args[i++] = dotted(ai->host);
  args[i] = NULL;

  maketermsane();

  execv(lineconf.ssh, (char **)args);
  nsyslog(LOG_ERR, "%s: %m", lineconf.ssh);
  return -1;
}

/*
 *  Execute a PPP session.
 */
static int do_ppp(const struct auth *ai)
{
  char *s;
  char options [1024];

  setenv("PORTSLAVELOGNAME", ai->login, 1);
  setenv("PORTSLAVE_SESSION", ai->acct_session_id, 1);
  setenv("PORTSLAVE_START_TIME", static_num(ai->start), 1);
  setenv("PORTSLAVE_PORT", static_num(GetPortNo()), 1);
  if(ai->do_acct || lineconf.do_acct)
    setenv("PORTSLAVE_DO_ACCT", "1", 1);
  if (ai->conn_info[0]) setenv("CONNECT_INFO", ai->conn_info, 1);
  s = ai->proto == P_AUTOPPP ? lineconf.autoppp : lineconf.pppopt;
  expand_format (options, sizeof(options), s, ai);
  if(setenv_from_rad("PORTSLAVE_FRAMED_ROUTE", (const char **)ai->framed_route, ai->frn))
    return -1;
  if(setenv_from_rad("PORTSLAVE_FILTER", (const char **)ai->filterid, ai->fln))
    return -1;
  if(ai->cli_src[0]) setenv("PORTSLAVE_CLI_SRC", ai->cli_src, 1);
  if(ai->cli_dst[0]) setenv("PORTSLAVE_CLI_DST", ai->cli_dst, 1);
  doexec(lineconf.pppd, "pppd", options);
  nsyslog(LOG_ERR, "%s: %m", lineconf.pppd);
  return -1;
}

/*
 *  Execute a local login.
 */
static int do_local(struct auth *ai, char *id, pid_t parent)
{
  struct utmp ut, *u;
  char *tty;
  char lhost[UT_HOSTSIZE];

  /*
   *  Prevent hacking.
   */
  if(ai->login[0] == '-')
    return -1;

  /*
   *  If we forked we have to pull some tricks to
   *  become process group leader etc.
   */
  if(parent)
  {
    setsid();
    ioctl(0, TIOCSCTTY, (char *)1);
  }

  /*
   *  First fill out an utmp struct.
   *  ttyname() returns a pointer to a static buffer that should not
   *   be free()'d!
   */
  tty = ttyname(0);
  memset(&ut, 0, sizeof(ut));
  if(ai->authenticated)
    ut.ut_type = USER_PROCESS;
  else
    ut.ut_type = LOGIN_PROCESS;
  ut.ut_pid = getpid();
  ut.ut_time = time(NULL);
  strncpy(ut.ut_line, tty, sizeof(ut.ut_line));
  strncpy(ut.ut_user, ai->login, sizeof(ut.ut_user));
  strncpy(ut.ut_id, id, sizeof(ut.ut_id));

  /*
   *  Now try to find the existing struct..
   */
  if(parent)
    ut.ut_pid = parent;
  if((u = getutpid(&ut)) != NULL)
    strncpy(ut.ut_id, u->ut_id, sizeof(ut.ut_id));
  if(parent)
    ut.ut_pid = getpid();
  setutent();
  pututline(&ut);

  if(ai->address)
  {
    const char *p = dotted(ai->address);
    int i = 0;

    while(*p && i < 2)
    {
      if(*p++ == '.')
        i++;
    }

    snprintf(lhost, sizeof (lhost), "%03d:%c.%s", ai->nasport, ai->proto, p);
  }
  else
  {
    snprintf(lhost, sizeof (lhost), "%03d:", ai->nasport);
  }
  striprealm(ai);
  maketermsane();

  /*
   *  If we are already authenticated don't ask for a password.
   */
  if(ai->authenticated)
    execl("/bin/login", "login", "-f", ai->login, "-h", lhost, NULL);
  else
    execl("/bin/login", "login", ai->login, "-h", lhost, NULL);

  nsyslog(LOG_ERR, "/bin/login: %m");
  return -1;
}

static int get_traffic_stats(struct auth *ai)
{
  struct traffic_stat t_stat;
  int sockfd;
  struct in_addr spawned_ip;

  memset(&t_stat, 0, sizeof(t_stat));

  if((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
  {
    nsyslog(LOG_ERR,"socket(sockfd): %m");
    return 1;
  }
    
  spawned_ip.s_addr=ai->address;

  traffic_stats(sockfd, spawned_ip, &t_stat);
  if(t_stat.recv_bytes)  ai->traffic.recv_bytes = t_stat.recv_bytes;
  if(t_stat.sent_bytes)  ai->traffic.sent_bytes = t_stat.sent_bytes;
  if(t_stat.recv_pkts)  ai->traffic.recv_pkts = t_stat.recv_pkts;
  if(t_stat.sent_pkts)  ai->traffic.sent_pkts = t_stat.sent_pkts;
  return 0;
}

/*
 *  Spawn an extra process to do the work, if needed.
 */
int spawnit(struct auth *ai)
{
  pid_t pid;
  int st;
  struct sigaction sa;
  char id[8];
  struct utmp ut, *u;

  /*
   *    Make up an id field if needed.
   */
  snprintf(id, sizeof (id), "T%d", ai->nasport);

  /*
   *    We only fork if we are going to exec another program
   *    to do the work and we need to send a RADIUS logout
   *    packets afterwards.
   */

  pid = -1;
  if(ai->proto != P_LOCAL && ai->proto != P_AUTOPPP && ai->proto != P_PPP)
  {
    if((pid = fork()) < 0)
    {
      nsyslog(LOG_ERR, "fork: %m");
      return -1;
    }
    if(pid > 0)
    {
      /*
       *      Catch SIGHUP.
       */
      sa.sa_handler = got_hup;
      sa.sa_flags   = SA_RESTART;
      sigaction(SIGHUP,  &sa, NULL);
      sigaction(SIGTERM, &sa, NULL);
      sigaction(SIGINT,  &sa, NULL);
      unblock(SIGHUP);
      unblock(SIGTERM);
      unblock(SIGINT);
               
      if(ai->sessiontime > 0)
      {
        alarm(ai->sessiontime);
        sigaction(SIGALRM, &sa, NULL);
        unblock(SIGALRM);
      }
      signal(SIGUSR1, SIG_IGN);       /* Ignore stray SIGUSR1 */

      /*
       *      Wait for the child to exit. Query traffic stats.
       */
      if(ai->proto == P_SLIP || ai->proto == P_CSLIP)
      {
        while(waitpid(pid, &st, WNOHANG) != pid)
        {
          xsleep(STATS_SLEEP);
          get_traffic_stats(ai);
          errno = 0;
        }
        get_traffic_stats(ai);
      }
      else
      {
        while(waitpid(pid, &st, 0) != pid && errno == EINTR)
          errno = 0;
      }

      /*
       *      Readjust the utmp entry for P_SHELL.
       */
      if(ai->proto == P_SHELL)
      {
        setutent();
        ut.ut_pid = pid;
        if((u = getutpid(&ut)) != NULL)
        {
          u->ut_pid = getpid();
          setutent();
          pututline(u);
        }
      }
      return 0;
    }
    /*
     *      If a signal was pending, it will be delivered now.
     */
    unblock(SIGHUP);
    unblock(SIGTERM);
    unblock(SIGINT);
  }

  /*
   *    Set the TERM environment variable.
   */
  if(lineconf.term && lineconf.term[0])
  {
    char aux_speed[16];

    setenv("TERM", lineconf.term, 1);
    sprintf(aux_speed, "%d", lineconf.speed);
    setenv("TSPEED", aux_speed, 1);
  }

  /*
   *    Catch sighup so that we can be killed.
   */
  signal(SIGHUP, SIG_DFL);
  signal(SIGTERM, SIG_DFL);
  signal(SIGINT, SIG_DFL);

  /*
   *      Find the protocol.
   */

  switch(ai->proto)
  {
    case P_TCPCLEAR:
#if 0     /* XXX - unimplemented protocols */
    case P_TCPLOGIN:
      do_tcp(ai);
    break;
    case P_CONSOLE:
      do_console(ai);
    break;
#endif
    case P_SLIP:
    case P_CSLIP:
      do_slip(ai);
    break;
    case P_PPP:
    case P_PPP_ONLY:
    case P_AUTOPPP:
      do_ppp(ai);
    break;
    case P_RLOGIN:
      striprealm(ai);
      do_rlogin(ai);
    break;
    case P_TELNET:
      striprealm(ai);
      do_telnet(ai);
    break;           
    case P_SSH:
    case P_SSH2:
      striprealm(ai);
      do_ssh(ai);
    break;           
    case P_LOCAL:
    case P_SHELL:
      do_local(ai, id, pid == 0 ? getppid() : 0);
    break;
    default:
      printf("protocol `%c' unsupported\n", ai->proto);
      fflush(stdout);
      xusleep(500000);
    break;
  }
  if(pid == 0)
    exit(0);
  return 0;
}
