/*
 * slip.c	Go into SLIP mode.
 *
 * Version:	@(#)slip.c  1.10  14-Jul-1996  MvS.
 *
 */
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <utmp.h>
#include <time.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <sys/wait.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#if !defined(__GLIBC__)
#  include <linux/netdevice.h>
#  include <linux/if_ether.h>
#  include <linux/if_arp.h>
#endif
#include <linux/if_slip.h>

#include "server.h"


static bool got_hup = false;

/*
 *	Note that we got the HUP signal.
 */
static void hup_handler(int unused_sig)
{
	got_hup = true;
	alarm(0);
}

/*
 *	Setup the interface.
 */
static int slip_ifconfig(struct auth *ai, char *slname)
{
	char cmd[100];
	char local[16];
	char remote[16];
	char slip_netmask[16];

	if (ai->netmask == 0) ai->netmask = 0xFFFFFFFF;
	strcpy(remote, dotted(ai->host));
	if(ai->address)
		strcpy(local, dotted(ai->address));
	else
		strcpy(local, dotted(lineconf.loc_host));
	strcpy(slip_netmask, dotted(ai->netmask));

	/* Ifconfig the interface. */
	if (ai->mtu > 0) {
		snprintf(cmd, sizeof (cmd), "exec %s %s %s pointopoint %s "
			"netmask %s mtu %d", PATH_IFCONFIG, slname, local,
			 remote, slip_netmask, ai->mtu);
	} else {
		snprintf(cmd, sizeof (cmd), "exec %s %s %s pointopoint %s netmask %s",
			PATH_IFCONFIG, slname, local, remote, slip_netmask);
	}
	return system(cmd);
}

/*
 *	Add a route to this address.
 */
static int slip_addroute(struct auth *ai, char *slname)
{
	char cmd[100];
	char slip_netmask[16];
	char network[16];
	char remote[16];
	unsigned int nw;
	int ret;

	/* Add a host pointopoint route. */
	snprintf(cmd, sizeof (cmd), "exec %s add -host %s dev %s", PATH_ROUTE,
			dotted(ai->address), slname);
	if ((ret = system(cmd)) != 0)
		return ret;

	/* See if we need to add a route to a whole network. */
	if (ai->netmask && ai->netmask != 0xFFFFFFFF) {
		nw = ai->address & ai->netmask;
		strcpy(slip_netmask, dotted(ai->netmask));
		strcpy(network, dotted(nw));
		strcpy(remote, dotted(ai->address));
		snprintf(cmd, sizeof (cmd), "exec %s add -net %s netmask %s gw %s",
			PATH_ROUTE, network, slip_netmask, remote);
		ret = system(cmd);
	}

	return ret;
}

/*
 *	Add a proxy arp entry.
 *
 *	FIXME: we are very simplistic here. We should loop over
 *	all interfaces and pick the right one (like pppd does).
 *
 */
static int slip_addproxy(struct auth *ai, char *slname_unused)
{
	struct ifreq ifr;
	struct sockaddr *sa;
	int s;
	char ethaddr[16];
	char cmd[100];

	/*
	 *	First we have to find out our ethernet address.
	 */
	strcpy(ifr.ifr_name, "eth0");
	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		nsyslog(LOG_ERR, "slip_addproxy: socket: %m");
		return -1;
	}
	if (ioctl(s, SIOCGIFHWADDR, &ifr) < 0) {
		nsyslog(LOG_ERR, "slip_addproxy: SIOCGIFADDR: %m");
		return -1;
	}
	sa = &(ifr.ifr_hwaddr);
	snprintf(ethaddr, sizeof (ethaddr), "%02x:%02x:%02x:%02x:%02x:%02x",
		(unsigned char)sa->sa_data[0], (unsigned char)sa->sa_data[1],
		(unsigned char)sa->sa_data[2], (unsigned char)sa->sa_data[3],
		(unsigned char)sa->sa_data[4], (unsigned char)sa->sa_data[5]);

	/* Add an proxy arp entry. */
	if (ai->netmask && ai->netmask != 0xFFFFFFFF) {
		snprintf(cmd, sizeof (cmd), "exec %s -s %s %s netmask %s pub",
			PATH_ARP, dotted(ai->address), ethaddr, dotted(ai->netmask));
	} else {
		snprintf(cmd, sizeof (cmd), "exec %s -s %s %s pub", PATH_ARP, dotted(ai->address), ethaddr);
	}
	return system(cmd);
}

/*
 *	Delete proxy arp entry.
 */
static int slip_delproxy(struct auth *ai, char *slname_unused)
{
	char cmd[100];

	snprintf(cmd, sizeof (cmd), "exec %s -d %s pub", PATH_ARP, dotted(ai->address));
	return system(cmd);
}

/*
 *	Put port into SLIP mode.
 */
int do_slip(struct auth *ai) {
	struct termios tty, orig_tty;
	int disc, odisc;
	char slname[16];
	int mode;
	struct sigaction sa;

	/*
	 *	Catch SIGHUP and friends.
	 */
	got_hup = false;
	sa.sa_handler = hup_handler;
	sa.sa_flags   = SA_RESTART;
	sigaction(SIGHUP, &sa, NULL);
	sigaction(SIGINT, &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);

	if (ai->sessiontime > 0) {
		alarm(ai->sessiontime);
		sigaction(SIGALRM, &sa, NULL);
		unblock(SIGALRM);
	}

	unblock(SIGHUP);
	unblock(SIGINT);
	unblock(SIGTERM);

	/*
	 *	First set terminal into raw mode
	 */
	tcgetattr(0, &tty);
	tcgetattr(0, &orig_tty);
	tty.c_lflag = 0;
	tty.c_iflag = 0;
	tty.c_oflag = 0;
	tcsetattr(0, TCSANOW, &tty);

	/*
	 *	Now set line discipline to SLIP.
	 */
	ioctl(0, TIOCGETD, &odisc);
	disc = N_SLIP;
	if (ioctl(0, TIOCSETD, &disc) < 0) {
		tcsetattr(0, TCSANOW, &orig_tty);
		if (errno == EINVAL)
			printf("Sorry - SLIP is not available on this system\r\n");
		else
			perror("SLIP mode");
		nsyslog(LOG_ERR, "TIOCSETD(SLIP): %m");
		return -1;
	}

	/*
	 *	Now set to either SLIP or CSLIP.
	 */
	mode = (ai->proto == P_SLIP ? SL_MODE_SLIP : SL_MODE_CSLIP);
	(void) ioctl(0, SIOCSIFENCAP, &mode);

	/*
	 *	Find out the name of the interface, ifconfig it,
	 *	add routes, and set up proxyarp.
	 */
	ioctl(0, SIOCGIFNAME, slname);
	slip_ifconfig(ai, slname);
	slip_addroute(ai, slname);
	slip_addproxy(ai, slname);

	/*
	 *	Now keep waiting for a SIGHUP.
	 */
	while(!got_hup)
		pause();
	
	ioctl(0, TIOCSETD, &odisc);
	tcsetattr(0, TCSANOW, &orig_tty);
	slip_delproxy(ai, slname);

	return 0;
}
