/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-rendering.h"
#include "polyxedit-ui-seqed-wnd.h"
//#include "polyxmass-polchemdefctxt.h"



GdkPixbuf *
polyxedit_rendering_pixbuf_render_no_modif (gchar *code, 
					    gint monicon_size,
					    PxmEditCtxt *editctxt)
{
  GdkPixbuf *pixbuf = NULL;



  g_assert (code != NULL);
  g_assert (editctxt != NULL);
  g_assert (editctxt->polchemdefctxt != NULL);
  

  /* We are asked to render a pixbuf for a non-modified monomer code.
     We first check if such a pixbuf is not already available to the
     GData of the polymer editing context. 
     
     If such pixbuf is found we just return it without incrementing
     its refcount because we do not consider that it is in use. The
     refcount is automatically incremented when the pixbuf is used to
     make a monicon's canvas_item anyway.
    
     If it is not found, then we simply make it first by trying to
     construct it from the vector graphics file, or from the raster
     file.
   */

  /* Find a pixbuf that has already been prepared for the monomer of
     code 'code'.
   */
  pixbuf = 
    polyxedit_rendering_pixbuf_get_from_data_list (code, 
						   NULL,
						   editctxt);
  
  if (pixbuf != NULL)
    {    
      /* The monomer of code 'code' already has a pixbuf, so just
         return. Note that we do not refcount the pixbuf because it
         will be reffed when the pixbuf is used to create a
         canvas_item. If the caller wants to use this pixbuf for other
         purposes, then it should ref the pixbuf by itself. Indeed, for 
	 the moment all we have is a pointer to it, we do not officially
	 use it.
	 
	 g_object_ref (G_OBJECT (pixbuf));
      */
      
      return pixbuf;
    }  
  

  /* Apparently no pixbuf is already available for monomer code
     'code'. Let's make it. If sucessful, the two functions below
     return a newly allocated pixbuf with a refcount of 1.
   */
  pixbuf = polyxedit_rendering_pixbuf_make_with_vector (code, 
							monicon_size,
							editctxt);
  
  if (pixbuf == NULL)
    pixbuf = polyxedit_rendering_pixbuf_make_with_raster (code, 
							  monicon_size,
							  editctxt);
  if (pixbuf == NULL)
    {
      /* As a last resort we should try to render a file that is in
         the distribution: the interrogation mark.
       */
      g_critical (_("%s@%d: failed rendering a pixbuf for code: '%s'\n"),
	     __FILE__, __LINE__, code);

      return NULL;
    }
  
  /* At this point we have effectively CREATED ex nihilo the pixbuf
     that was asked and we thus let the polymer definition context
     know that we have rendered the pixbuf for the unmodified monomer
     'code'. Note that for the moment we do not ref this newly created
     pixbuf, because it was created with a ref_count of 1
     already. When this function returns the user will know if it is
     to be reffed or not based on what is going to be performed with
     this pixbuf. For example, using the pixbuf to create a
     canvas_item has a side effect of automatically reff the pixbuf
     that's used.
   */
  polyxedit_rendering_pixbuf_set_to_data_list (pixbuf, 
					       code, 
					       NULL, 
					       editctxt);
  return pixbuf;
}



GdkPixbuf *
polyxedit_rendering_pixbuf_render_modif (gchar *code,
					 gchar *modif,
					 gint monicon_size,
					 PxmEditCtxt *editctxt)
{
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  
  PxmMonomerSpec *mnmspec = NULL;
  PxmModifSpec *mdfspec = NULL;

  GdkPixbuf *pixbuf_code = NULL;

  GdkPixbuf *pixbuf_modif = NULL;
  GdkPixbuf *pixbuf_modif_scaled = NULL;

  GError *error = NULL;


  g_assert (code != NULL && strlen (code) > 0);
  g_assert (modif != NULL && strlen (modif) > 0);
  g_assert (editctxt != NULL);

  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  

  /* We are asked to render a pixbuf for a modified monomer code. We
     construct a key (made simply with a concat
     "monomer->code$modif->name" and check using this key if such a
     pixbuf is not already available to the GData of the polymer
     editing context.
     
     If such pixbuf is found we just return it without incrementing
     its refcount because we do not consider that it is in use. The
     refcount is automatically incremented when the pixbuf is used to
     make a monicon's canvas_item anyway.
    
     If it is not found, then we simply make it first by trying to
     construct it from the vector graphics file, or from the raster
     file.
   */

  /* Find a pixbuf that has already been prepared for the monomer of
     code 'code' modified with modif 'modif'.
   */
  pixbuf_code = 
    polyxedit_rendering_pixbuf_get_from_data_list (code, 
						   modif,
						   editctxt);
  
  if (pixbuf_code != NULL)
    {    
      /* The monomer of code 'code' already has a pixbuf, so just
         return. Note that we do not refcount the pixbuf because it
         will be reffed when the pixbuf is used to create a
         canvas_item. If the caller wants to use this pixbuf for other
         purposes, then it should ref the pixbuf by itself.
        
       g_object_ref (G_OBJECT (pixbuf));
      */
      
      return pixbuf_code;
    }  
  
  /* Apparently no pixbuf is already available for monomer code 'code'
     with a modif 'modif'. Let's make it.
   */

  /* We first have to get the specifications about the way the
     modification 'modif' should be rendered. For that we want to get
     the PxmModifSpec instance for the 'modif'.
   */
  mdfspec = libpolyxmass_modifspec_get_ptr_by_name (editctxt->
						    polchemdefctxt->
						    mdfspecGPA,
						    modif);
  
  if (NULL == mdfspec)
    {
      g_critical (_("%s@%d: failed getting modif specification "
	       "for name: '%s'\n"),
	     __FILE__, __LINE__, modif);
      	
	return NULL;
    }
  
  /* Great, at this point we know the modification specifications...
   */

  /* Get the same kind of information but for the monomer.
   */
  mnmspec = libpolyxmass_monomerspec_get_ptr_by_code (editctxt->
						      polchemdefctxt->
						      mnmspecGPA,
						      code);
  
  if (NULL == mnmspec)
    {
      g_critical (_("%s@%d: failed getting monomer specification "
	       "for code: '%s'\n"),
	     __FILE__, __LINE__, code);
      
	return NULL;
    }
  


  /* At this point we should have the name of at least one file
     (vector or raster) that is present for the modif rendering. Note
     that the rendering is more or less difficult, depending on the
     graphical operation that is required.
    
     If the graphical operation is 'O' for "Opaque", that is the easy
     case: we just render the pixbuf corresponding to the file name
     returned either in mdfspec->vector or mdfspec->raster.
     
     If the graphical operation is 'T' for "Transparent", the
     situation is more difficult, as we first must get the pixbuf
     corresponding to the non-modified 'code' (either it exists
     already or we have to load it from file). Next we have to get the
     pixbuf corresponding to the 'modif' and we must composite it onto
     the first pixbuf.
   */
  if (0 == strcmp (mdfspec->action, "O"))
    {
      /* An "Opaque" action is requested, which is we simply have to 
         create a pixbuf by reading a file.
       */

      /* Try to make a pixbuf using the vector file.
       */
      pixbuf_modif = rsvg_pixbuf_from_file_at_size (mdfspec->vector,
						    monicon_size,
						    monicon_size,
						    &error);
      if (pixbuf_modif == NULL)
	{
	  /* Try to make a pixbuf using the raster file.
	   */
	  pixbuf_modif = gdk_pixbuf_new_from_file (mdfspec->raster, &error);
	  
	  if (pixbuf_modif == NULL)
	    {
	      g_error (_("%s@%d: failed rendering a modification pixbuf from "
		       "vector/raster files: '%s'/'%s'\n"),
		     __FILE__, __LINE__, mdfspec->vector, mdfspec->raster);

	      return NULL;
	    }

	  /* Now that we have the pixbuf that was read from raster
	   * file, make sure that we scale it to the requested size.
	   */
	  pixbuf_modif_scaled = 
	    gdk_pixbuf_scale_simple (pixbuf_modif,
				     monicon_size,
				     monicon_size,
				     GDK_INTERP_BILINEAR);
	  
	  /* Since the initial pixbuf was read from file and created
	   * with a refcount of 1 if we call unref on it, it will be
	   * immediately freed, which is exactly what we want.
	   */
	  g_object_unref (G_OBJECT (pixbuf_modif));

	  pixbuf_modif = pixbuf_modif_scaled;
	}
      
      /* At this point we have effectively CREATED ex nihilo the
         pixbuf that was asked and we thus let the polymer definition
         context know that we have rendered the pixbuf for the monomer
         'code' modified with 'modif'. Note that for the moment we do
         not ref this newly created pixbuf, because it was created
         with a ref_count of 1 already. When this function returns the
         user will know if it is to be reffed or not based on what is
         going to be performed with this pixbuf. For example, using
         the pixbuf to create a canvas_item has a side effect of
         automatically reff the pixbuf that's used.
       */
      polyxedit_rendering_pixbuf_set_to_data_list (pixbuf_modif, 
						   code, 
						   modif, 
						   editctxt);
      
      return pixbuf_modif;
    }
  /* end of:
     if (0 == strcmp (mdfspec->action, "O"))
  */
  
  else if (0 == strcmp (mdfspec->action, "T"))
    {
      /* A "Transparent" action is requested, which is we have to:
        
         1. Either get a pointer to or create a pixbuf corresponding
         to the un-modified 'code'.
        
         2. Either get a pointer to or make a pixbuf corresponding to
         the 'modif'.
        
         3. Composite transparently the 'modif' pixbuf onto the 'code'
         pixbuf.
       */

      /* Step 1: Get/Make a pixbuf for the un-modified 'code'. The
         code below will either get a pixbuf to a pre-existing
         un-modified 'code' or will return a pointer to a newly
         created one. Note that whatever the file (vector or raster)
         used to make the pixbuf, the pixbuf is scaled to the good
         dimensions. Also, note that as a good side effect, if the
         pixbuf for 'code' did not pre-exist, it has been created from
         file and is now available in the GData list of the
         polchemdefctxt.
       */
      pixbuf_code =
	polyxedit_rendering_pixbuf_render_no_modif (code,
						    monicon_size,
						    editctxt);
      if (pixbuf_code == NULL)
	{
	  /* FIXME: As a last resort we should try to render a file
	   * that is in the distribution: the interrogation mark.
	   */
	  g_error (_("%s@%d: failed rendering a pixbuf for code: '%s'\n"),
		 __FILE__, __LINE__, code);
	  
	  return NULL;
	}
      /* We have got now a pixbuf that corresponds to an un-modified
         'code'. Since we will paste (composite) the 'modif' pixbuf
         onto this un-modified 'code' pixbuf, we have to make a copy
         of it. Indeed, pixbuf_code is a pointer to the pixbuf that
         corresponds to the un-modified 'code', and we cannot modify
         this pixbuf, otherwise we change the rendering for the
         un-modified 'code'.
       */
      pixbuf_code = gdk_pixbuf_copy (pixbuf_code);
      g_assert (pixbuf_code != NULL);

      /* Note that the product of gdk_pixbuf_copy () is a pixbuf that
	 has a refcount of 1, as if it had been made by reading data
	 from disk.
      */
      
      /* At this point we have a very faithful pixbuf that corresponds
         to an un-modified 'code'. Let's go on with the 'modif'
         pixbuf. Note that the general policy is that 'modif' pixbufs
         can exist in the GData list of the polchemdefctxt: they are
         stored with a key made according to the following call:
         g_strdup_printf ("pxm-raw-modif$%s", modif->name);
       */
      
      /* Step 2: Get/Make a pixbuf for the 'modif'. 
       */
      pixbuf_modif = 
	polyxedit_rendering_pixbuf_get_from_data_list (NULL, 
						       modif,
						       editctxt);
      
      if (pixbuf_modif != NULL)
	{    
	  /* The modif 'modif' already has a pixbuf. We now simply need
	   * to make the compositing onto the modif_code pixbuf.
	   */
	  gdk_pixbuf_composite (/*src*/pixbuf_modif, /*dest*/pixbuf_code,
				/*dest x*/0, /*dest y*/0, 
				/* dest width*/monicon_size, 
				/* dest height*/monicon_size,
				0, 0, /*scale x*/1, /*scale y*/1, 
				GDK_INTERP_NEAREST,
				/*overall_alpha*/128);
	  
	  /* Now that the pixbuf_modif pixbuf has been used for the
	     compositing task, we can free it, as it is no longer
	     required.
	  */
	  g_object_unref (G_OBJECT (pixbuf_modif));
	     
	  /* At this point we have constructed a brand new 
	   * pixbuf corresponding to the 'code' being modified with 'modif'.
	   * Take advantage of all the work we did, and store that
	   * new pixbuf in the polchemdefctxt's GData list.
	   */
	  polyxedit_rendering_pixbuf_set_to_data_list (pixbuf_code, 
						       code, 
						       modif, 
						       editctxt);

	  return pixbuf_code;
	}  
      

      /* Apparently, the 'modif' has not a pre-rendered pixbuf, which
         means that we have to render it from file.
       */

      /* Try to make a pixbuf using the vector file.
       */
      pixbuf_modif = rsvg_pixbuf_from_file_at_size (mdfspec->vector,
						    monicon_size,
						    monicon_size,
						    &error);
      if (pixbuf_modif == NULL)
	{
	  /* Try to make a pixbuf using the raster file.
	   */
	  pixbuf_modif = gdk_pixbuf_new_from_file (mdfspec->raster, &error);
	  
	  if (pixbuf_modif == NULL)
	    {
	      g_error (_("%s@%d: failed rendering a modification pixbuf from "
		       "vector/raster files: '%s'/'%s'\n"),
		     __FILE__, __LINE__, mdfspec->vector, mdfspec->raster);
	      
	      return NULL;
	    }
	}
      
      /* At this point we have a pixbuf corresponding to the 'modif',
         whatever the way it has been rendered (vector or raster). We
         can now make the compositing operation.
       */
      gdk_pixbuf_composite (/*src*/pixbuf_modif, /*dest*/pixbuf_code,
			    /*dest x*/0, /*dest y*/0, 
			    /* dest width*/monicon_size, 
			    /* dest height*/monicon_size,
			    0, 0, /*scale x*/1, /*scale y*/1, 
			    GDK_INTERP_NEAREST,
			    /*overall_alpha*/254);
      
      /* Now that the pixbuf_modif pixbuf has been used for the
	 compositing task, we can free it, as it is no longer
	 required.
      */
      g_object_unref (G_OBJECT (pixbuf_modif));
      
      /* At this point we have constructed a brand new 
         pixbuf corresponding to the 'code' being modified with 'modif'.
         Take advantage of all the work we did, and store that
         new pixbuf in the polchemdefctxt's GData list.
       */
      polyxedit_rendering_pixbuf_set_to_data_list (pixbuf_code, 
						   code, 
						   modif, 
						   editctxt);	  
      return pixbuf_code;
    }  

  return NULL;
}



gboolean
polyxedit_rendering_cursor_monicon_render (PxmEditCtxt *editctxt,
					   gboolean rerender)
{
  GdkPixbuf *pixbuf = NULL;

  PxmMonicon *cursor = NULL;
  
  GError *error = NULL;

  gchar *file = NULL;

  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  

  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  cursor = seqeditorctxt->cursor;
  g_assert (cursor != NULL);
  

  if (rerender == TRUE)
    {
      polyxedit_monicon_free (seqeditorctxt->cursor);

      seqeditorctxt->cursor = polyxedit_monicon_new ();
    }

  /* The cursor is a vector graphics file (svg file) that is named
     cursor.svg and that is in the config dir of the current polymer
     chemistry definition.
   */
  file = 
    g_strdup_printf ("%s/%s", editctxt->polchemdefctxt->polchemdef->dir, 
		     "cursor.svg");

  pixbuf = rsvg_pixbuf_from_file_at_size (file, 
					  seqeditorctxt->monicon_size,
					  seqeditorctxt->monicon_size,
					  &error);

  /* The pixbuf is created from file with a refcount of 1.
   */
  
  if (pixbuf == NULL)
    {
      g_error (_("%s@%d: failed rendering pixbuf from vector file: %s\n"),
	     __FILE__, __LINE__, file);
      
      g_free (file);
      
      return FALSE;
    }

  g_free (file);
  
  cursor->canvas_item =
    gnome_canvas_item_new (seqeditorctxt->main_canvas_group,
			   gnome_canvas_pixbuf_get_type (),
			   "pixbuf", pixbuf,
			   "width", (double) seqeditorctxt->monicon_size,
			   "width_in_pixels", TRUE,
			   "height", (double) seqeditorctxt->monicon_size,
			   "height_in_pixels", TRUE,
			   "x", monicon_rect.x1 
			   - (seqeditorctxt->monicon_size / 2) ,
			   "y", monicon_rect.y1,
			   NULL);

  /* We unref the pixbuf right away, since the making of the canvas_item
     did ref it once. This is done, so that when polyxedit_monicon_free ()
     is called, the pixbuf associated to the cursor is immediately freed.
  */
  g_object_unref (G_OBJECT (pixbuf));

  cursor->x = monicon_rect.x1;
  cursor->y = monicon_rect.y1;

  return TRUE;
}


GdkPixbuf *
polyxedit_rendering_pixbuf_get_from_data_list (gchar *code, 
					       gchar *modif,
					       PxmEditCtxt *editctxt)
{
  gchar *key = NULL;

  GdkPixbuf *pixbuf = NULL;


  g_assert (editctxt != NULL);
  
  if (code != NULL && strlen (code) > 0)
    {
      if (modif != NULL && strlen (modif) > 0)
	/* We want the pixbuf for a 'modif'-modified 'code'.
	 */
	key = g_strdup_printf ("%s$%s", code, modif);
      else
	/* We want the pixbuf for an un-modified 'code'.
	 */
	key = g_strdup (code);
    }

  else if (code == NULL || strlen (code) == 0)
    {
      if (modif != NULL && strlen (modif) > 0)
      /* We want a pixbuf for a "raw modif".
       */
	key = g_strdup_printf ("pxm-raw-modif$%s", modif);
      else
	g_error (_("%s@%d: both 'code' and 'modif' cannot be NULL or empty\n"),
	       __FILE__, __LINE__);
    }
  
  pixbuf = g_datalist_get_data (&editctxt->polchemdefctxt->pixbufGData, key);
  
  g_free (key);
  
  return pixbuf;
}


gboolean
polyxedit_rendering_pixbuf_set_to_data_list (GdkPixbuf *pixbuf, 
					     gchar *code, 
					     gchar *modif,
					     PxmEditCtxt *editctxt)
{
  gchar *key = NULL;
  
  g_assert (pixbuf != NULL);
  g_assert (editctxt != NULL);
  

  if (code != NULL && strlen (code) > 0)
    {
      if (modif != NULL && strlen (modif) > 0)
	/* We want the pixbuf for a 'modif'-modified 'code'.
	 */
	key = g_strdup_printf ("%s$%s", code, modif);
      else
	/* We want the pixbuf for an un-modified 'code'.
	 */
	key = g_strdup (code);
    }

  else if (code == NULL || strlen (code) == 0)
    {
      if (modif != NULL && strlen (modif) > 0)
      /* We want a pixbuf for a "raw modif".
       */
	key = g_strdup_printf ("pxm-raw-modif$%s", modif);
      else
	g_error (_("%s@%d: both 'code' and 'modif' cannot be NULL or empty\n"),
	       __FILE__, __LINE__);
    }
  
  g_datalist_set_data (&editctxt->polchemdefctxt->pixbufGData, key, pixbuf);
  
  g_free (key);
  
  return TRUE;
}


GdkPixbuf *
polyxedit_rendering_pixbuf_make_with_vector (gchar *code, 
					     gint monicon_size,
					     PxmEditCtxt *editctxt)
{
  /* We have the code of a monomer, and with this code we have to:
    
     1. get to know what vector graphics file should be used to render it.
     For this we ask from the polchemdefctxt the monomerspec instance that
     holds the informations for monomer's code 'code'.
     2. make a pixbuf of the right size by using the vector graphics file.
   */
  GError *error = NULL;
  
  PxmMonomerSpec *mnmspec = NULL;
  
  GdkPixbuf *pixbuf = NULL;


  g_assert (code != NULL);

  /* Get a pointer to the PxmMonomerSpec corresponding to 'code'.
   */
  mnmspec = 
    libpolyxmass_monomerspec_get_ptr_by_code (editctxt->
					      polchemdefctxt->mnmspecGPA,
					      code);
  g_assert (mnmspec != NULL);

  /* We now have to ensure that the vector file name is non-NULL, and
     that it points to an existing file.
   */  
  if (NULL == mnmspec->vector)
    {
      g_message (_("%s@%d: mnmspec->vector is NULL for code '%s'\n"),
	     __FILE__, __LINE__, code);
      
      return NULL;
    }
  
  if (FALSE == g_file_test (mnmspec->vector, G_FILE_TEST_EXISTS))
    {
      g_critical (_("%s@%d: mnmspec->vector '%s' does not exist (code: '%s')\n"),
	     __FILE__, __LINE__, mnmspec->vector, code);
      
      return NULL;
    }

  /* Let's do the rendering. The function creates a new pixbuf with a
     ref count of 1. So we do not need to ref it here.
   */
  pixbuf = rsvg_pixbuf_from_file_at_size (mnmspec->vector,
					  monicon_size,
					  monicon_size,
					  &error);

  if (pixbuf == NULL)
    {
      g_critical (_("%s@%d: failed rendering a pixbuf from vector file: %s\n"),
	     __FILE__, __LINE__, mnmspec->vector);
    }
  
  return pixbuf;
}


GdkPixbuf *
polyxedit_rendering_pixbuf_make_with_raster (gchar *code, 
					     gint monicon_size,
					     PxmEditCtxt *editctxt)
{
  /* We have the code of a monomer, and with this code we have to:
    
     1. get to know what raster graphics file should be used to render
     it.  For this we ask from the polchemdefctxt the monomerspec instance
     that holds the informations for monomer's code 'code'.  2. make a
     pixbuf of the right size by using the raster graphics file.
   */
  PxmMonomerSpec *mnmspec = NULL;
  
  GError *error = NULL;

  GdkPixbuf *pixbuf = NULL;
  GdkPixbuf *pixbuf_scaled = NULL;



  g_assert (code != NULL);

  
  /* Get a pointer to the PxmMonomerSpec corresponding to 'code'.
   */
  mnmspec = 
    libpolyxmass_monomerspec_get_ptr_by_code (editctxt->
					      polchemdefctxt->mnmspecGPA,
					      code);
  g_assert (mnmspec != NULL);

  /* We now have to ensure that the rster file name is non-NULL, and
     that it points to an existing file.
   */  
  if (NULL == mnmspec->raster)
    {
      g_message (_("%s@%d: mnmspec->raster is NULL for code '%s'\n"),
	     __FILE__, __LINE__, code);
      
      return NULL;
    }

  if (FALSE == g_file_test (mnmspec->raster, G_FILE_TEST_EXISTS))
    {
      g_critical (_("%s@%d: mnmspec->raster '%s' does not "
		    "exist (code: '%s')\n"),
		  __FILE__, __LINE__, mnmspec->raster, code);
      
      return NULL;
    }

  /* Let's do the rendering. The function creates a new pixbuf with a
     ref count of 1. So we do not need to ref it here.
   */
  pixbuf = gdk_pixbuf_new_from_file (mnmspec->raster, &error);

  if (pixbuf == NULL)
    {
      g_warning (_("%s@%d: failed rendering a pixbuf "
		   "from raster file: '%s'\n"),
		 __FILE__, __LINE__, mnmspec->raster);
  
      return NULL;
    }

  /* Now that we have the pixbuf that was read from file, make sure
     that we scale it to the requested size.
   */
  pixbuf_scaled = gdk_pixbuf_scale_simple (pixbuf,
					   monicon_size,
					   monicon_size,
					   GDK_INTERP_BILINEAR);

  /* Since the initial pixbuf was read from file and created with a
     refcount of 1 if we call unref on it, it will be immediately
     freed, which is exactly what we want.
   */
  g_object_unref (G_OBJECT (pixbuf));
  
  return pixbuf_scaled;
}


gboolean 
polyxedit_rendering_monicon_make_canvas_item (PxmMonicon *monicon,
					      gint monicon_size,
					      GdkPixbuf *pixbuf,
					      gint idx,
					      PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  g_assert (monicon != NULL);
  g_assert (idx >= 0);
  g_assert (pixbuf != NULL);
  
  
  g_assert (editctxt->seqeditorctxt != NULL);
  

  /* First of all, we have to compute the position of the monicon for
     the currently iterated item. Only (x1,y1) are relevant to us here.
   */
  polyxedit_seqed_wnd_monicon_get_pixel_coord (idx, seqeditorctxt, 
					       &monicon_rect, 
					       COORDSYS_NW, COORDSYS_NW);
  
  monicon->canvas_item = 
    gnome_canvas_item_new (editctxt->seqeditorctxt->main_canvas_group,
			   gnome_canvas_pixbuf_get_type (),
			   "pixbuf", pixbuf,
			   "width", (double) monicon_size,
			   "width_in_pixels", TRUE,
		 	   "height", (double) monicon_size,
			   "height_in_pixels", TRUE,
			   "x", monicon_rect.x1,
			   "y", monicon_rect.y1,
			   NULL);

  g_assert (monicon->canvas_item != NULL);

  /* Set the coordinate values for the monicon:
   */
  monicon->x = monicon_rect.x1;
  monicon->y = monicon_rect.y1;
  
  /* PLEASE NOTE THAT LAST PARAM IS NON-NULL, it contains the polymer
     sequence context pointer, so that everything on the seqeditor can
     be accessed in the handler function.
   */
  g_signal_connect (G_OBJECT (monicon->canvas_item),
		    "event",
		    G_CALLBACK (polyxedit_seqed_wnd_canvas_item_event),
		    editctxt);
  
  return TRUE;
}




