// ------------------------------------------------------------------------- //
// $Id: ftdraw.cpp,v 1.34 2003/02/19 19:47:19 pandr Exp $
// ------------------------------------------------------------------------- //

/*
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#if HAVE_CONFIG_H
# include <config.h>
#endif

#if STDC_HEADERS
# include <stdio.h>
#endif

#if HAVE_STRING_H
# include <string.h>
#elif HAVE_STRINGS_H
# include <strings.h>
#endif

#include "common.h"
#include "image.h"
#include "ftdraw.h"

// ------------------------------------------------------------------------- //


FreeTypeFont::handle FreeTypeFont::get_font(const char* fontname, uint size)
{
	FreeTypeFont *f = new FreeTypeFont(fontname, size);
	if (f->font_ok()) {
		return f;
	}
	else {
		delete f;
		return 0;
	}
}

const Glyph* FreeTypeFont::get_tex_glyph(FT_ULong unicode)
{
	FT_UInt glyph_index = FT_Get_Char_Index(_ft_face, unicode);
	if (!glyph_index) return 0;

	if (Glyph* g = _glyphs[glyph_index]) {
		return g;
	}
	_glyphs[glyph_index] = make_tex_glyph(glyph_index);
	return _glyphs[glyph_index];
}

Glyph* FreeTypeFont::make_tex_glyph(FT_UInt glyph_index)
{
	if (FT_Load_Glyph(_ft_face, glyph_index, FT_LOAD_DEFAULT))
		return 0;
	if (FT_Render_Glyph(_ft_face->glyph, ft_render_mode_normal))
		return 0;
	
	FT_Bitmap bitmap = _ft_face->glyph->bitmap;

	iv2 glyph_size(bitmap.width + 1, bitmap.rows + 1);

	Tile::handle tile = TileBank::instance()->get_tile(glyph_size);

	if (!tile) return 0;

	copy_bitmap(tile, _ft_face->glyph);

	Glyph* g = new Glyph(glyph_index);
	g->_tile = tile;
	g->_size.set(bitmap.width, bitmap.rows);
	//g->_advance.set(_ft_face->glyph->advance.x >> 6, 
	//_ft_face->glyph->metrics.vertAdvance >> 6);
	
	// FIXME ???!!! implicit conversion from "signed long" to "float" 
	g->_advance.set(_ft_face->glyph->advance.x * ONE_SIXTYFOURTH,
			_ft_face->glyph->metrics.height * ONE_SIXTYFOURTH);
	g->_offset.set(_ft_face->glyph->metrics.horiBearingX * ONE_SIXTYFOURTH,
		          (_ft_face->glyph->metrics.horiBearingY - 
				   _ft_face->glyph->metrics.height) * ONE_SIXTYFOURTH);
	//g->_line_advance = _ft_face->size->metrics.height * ONE_SIXTYFOURTH;

	return g;
}

void FreeTypeFont::copy_bitmap(Tile::handle tile, FT_GlyphSlot glyph)
{
	FT_Bitmap& bitmap = glyph->bitmap;
	Image& im = *(tile->get_texture()->get_image());
	uint x = tile->_lower_left.x();
	uint y = tile->_lower_left.y();
	GLubyte *bp = im.get_buffer() + 4 * (x + y * im.get_width());
	int i;
	for (int row = bitmap.rows-1; row >= 0 ; row--) {
		i = row * bitmap.pitch;
		for (int col = 0; col < bitmap.width; col++, i++) {
			*bp++ = 255;
			*bp++ = 255;
			*bp++ = 255;
			*bp++ = bitmap.buffer[i];

	
#if 0
			//frame each letter: Smash, this is for you!
			if (row==0 || row == bitmap.rows-1 || col==0 || 
			    col == bitmap.width-1 || row+1 == (glyph->metrics.horiBearingY >> 6)) {
				*(bp-1) = 255;
				*(bp-2) = 0;
				*(bp-3) = 0;
				*(bp-4) = 255;
			}
#endif
			
			
		}
		bp += im.get_width() * 4 - bitmap.width * 4;
	}
}

int FreeTypeFont::render(const char *string, Image *image, int x, int y)
{
	int string_len = strlen(string);
	for (int n = 0; n < string_len; n++) {
		FT_UInt glyph_index;
		glyph_index = FT_Get_Char_Index(_ft_face, string[n]);
		if (FT_Load_Glyph(_ft_face, glyph_index, FT_LOAD_DEFAULT))
			continue;
		if (FT_Render_Glyph(_ft_face->glyph, ft_render_mode_normal))
			continue;

		FT_Bitmap bitmap = _ft_face->glyph->bitmap;
		int i;
		GLubyte *bp = image->get_buffer() + 
			4*(x + _ft_face->glyph->bitmap_left + 
			(y+_ft_face->glyph->bitmap_top)*image->get_width());
		for (int row = 0; row < bitmap.rows; row++) {
			i = row * bitmap.pitch;
			for (int col = 0; col < bitmap.width; col++, i++) {
				GLubyte grayscale = bitmap.buffer[i];
				bp[0+col*4] = 255;
				bp[1+col*4] = 255;
				bp[2+col*4] = 255;
				bp[3+col*4] = grayscale;
			}
			bp -= image->get_width() * 4;
		}
		x += _ft_face->glyph->advance.x >> 6;
	}
	return 0;
}

// ------------------------------------------------------------------------- //

// Static vars of the FreeTypeFont class
FT_Library FreeTypeFont::_ft_lib = 0;

void FreeTypeFont::load_freetype_lib()
{
	if(FT_Init_FreeType(&_ft_lib)) {
			log_abort("Unable to load FreeType library");
	}
}

FreeTypeFont::FreeTypeFont(const char *fontname, uint size)
	: Refcount(true), _texture(0), _ok(false)
{
	// Check if initialized
	if (!_ft_lib) FreeTypeFont::load_freetype_lib();

	if (FT_New_Face(_ft_lib, fontname, 0, &_ft_face)) {
		log_warning("Unable to load font %s", fontname);
		return;
	}

	// Fixme this should probably be a method of class Face
	if (FT_Set_Pixel_Sizes(_ft_face, 0, size)) {
		log_warning("Unable to scale font %s", fontname);
		return;
	}
	_ok = true;
	//std::cout << _("Created font: ") << fontname << _(" size ") << size << std::endl;
}

FreeTypeFont::~FreeTypeFont()
{
	for (_glyphs_::iterator i = _glyphs.begin(); i != _glyphs.end(); ++i)
	{
		delete i->second;
	}
	//std::cout << _("Destroyed font") << std::endl; 
}

uint FreeTypeFont::get_char_index(unsigned char c)
{
	return FT_Get_Char_Index(_ft_face, c);
}

float FreeTypeFont::get_kerning(FT_UInt left, FT_UInt right)
{
	//uint index_left  = get_char_index(left);
	//uint index_right = get_char_index(right);

	FT_Vector k;
	FT_Get_Kerning(_ft_face, left, right, ft_kerning_default, &k);

	if (!FT_HAS_KERNING(_ft_face)) {
		log_warning("No kerning for face");
	}
	
	// FIXME ???!!! implicit conversion from "signed long" to "float" 
	return k.x >> 6;
}

float FreeTypeFont::get_vert_advance()
{
	// FIXME ???!!! implicit conversion from "signed long" to "float" 
	return _ft_face->glyph->metrics.vertAdvance >> 6;
}

float FreeTypeFont::get_baselineskip() const
{
	return _ft_face->size->metrics.height * ONE_SIXTYFOURTH;
	//return (_ft_face->size->metrics.ascender - _ft_face->size->metrics.descender) * ONE_SIXTYFOURTH;
}

float FreeTypeFont::get_underline_position() const
{
	return FT_MulFix(_ft_face->underline_position, _ft_face->size->metrics.y_scale) * ONE_SIXTYFOURTH;
}

float FreeTypeFont::get_underline_thickness() const
{
	return FT_MulFix(_ft_face->underline_thickness, _ft_face->size->metrics.y_scale) * ONE_SIXTYFOURTH;
}
// ------------------------------------------------------------------------- //
