/*
 * $Id: resultform.c,v 1.20 2001/10/27 18:56:41 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "const.h"
#include "control.h"
#include "control.h"
#include "debug.h"
#include "os.h"
#include "prefsdata.h"
#include "resourceids.h"
#include "resultform.h"
#include "search.h"
#include "util.h"


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define DEFAULT_X           2
#define DEFAULT_Y           30
#define DEFAULT_EXTENT_X    156
#define DEFAULT_EXTENT_Y    11
#define MAX_RESULT          9


/***********************************************************************
 *
 *      Internal Types
 *
 ***********************************************************************/
static struct {
    RectangleType   coordinates;
    UInt16          record;
    UInt16          pos;
    UInt16          length;
} SearchResult[ MAX_RESULT ];


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static RectangleType    rect = { { DEFAULT_X, DEFAULT_Y },
                                 { DEFAULT_EXTENT_X, DEFAULT_EXTENT_Y } };
static Int16            count;
static Int16            lineMatch;
static Boolean          newLine;
static FieldType*       fldPtr;
static Char             matches[ 40 ];
static Char             noMatches[ 40 ];
static Char             searching[ 21 ];
static Char             pattern[ MAX_PATTERN_LEN + 1 ];
static FontID           initialFontStyle;



/* Clear result data variables */
static void ClearSearchResult( void )
{
    MemSet( (void*) &SearchResult, sizeof( SearchResult ), 0 );
    count           = 0;
    newLine         = false;
    rect.topLeft.y  = DEFAULT_Y;
}



/* Write header info for search result, return true if end of record has been reached */
Boolean DrawResultHeader
    (
    Char* header    /* header string */
    )
{
    UInt16 len;
    UInt16 deltaX;
    UInt16 coordY;

    if ( newLine ) {
        newLine         = false;
        rect.topLeft.y += DEFAULT_EXTENT_Y;
        count++;
    }

    len     = FntCharsWidth( header, StrLen( header ) );
    deltaX  = ( DEFAULT_EXTENT_X - len ) / 2;
    coordY  = rect.topLeft.y + DEFAULT_EXTENT_Y / 2;

    WinEraseRectangle( &rect, 0 );

    WinDrawLine( DEFAULT_X, coordY, deltaX, coordY );
    WinDrawChars( header, StrLen( header ), DEFAULT_X + deltaX, coordY - 5 );
    WinDrawLine( DEFAULT_X + deltaX + len + 1, coordY, DEFAULT_EXTENT_X - 3, coordY );

    if ( MAX_RESULT - 1 <= count ) {
        count           = 0;
        newLine         = false;
        rect.topLeft.y  = DEFAULT_Y;
        return true;
    }

    return false;
}



/* Write search result, return true if end of record has been reached */
Boolean DrawResultString
    (
    UInt16  record, /* record ID */
    UInt16  pos,    /* poition of search pattern in record */
    UInt16  length, /* length of pattern */
    Char*   result  /* result string */
    )
{
    count++;

    rect.topLeft.y += DEFAULT_EXTENT_Y;

    SearchResult[ count ].coordinates   = rect;
    SearchResult[ count ].record        = record;
    SearchResult[ count ].pos           = pos;
    SearchResult[ count ].length        = length;

    WinDrawChars( result, StrLen( result ), rect.topLeft.x, rect.topLeft.y );

    if ( MAX_RESULT - 1 <= count ) {
        count           = 0;
        newLine         = false;
        rect.topLeft.y  = DEFAULT_Y;
        return true;
    }
    newLine = true;

    return false;
}



/* Display Cancel button */
static void ShowCancel( void )
{
    FormType* resultForm;

    resultForm = FrmGetFormPtr( frmResult );

    FrmHideObject( resultForm, FrmGetObjectIndex( resultForm, frmResultStop ) );
    FrmShowObject( resultForm, FrmGetObjectIndex( resultForm, frmResultCancel ) );
}



/* Display Cancel and Find More buttons */
static void ShowCancelMore( void )
{
    FormType* resultForm;

    resultForm = FrmGetFormPtr( frmResult );

    ShowCancel();
    FrmShowObject( resultForm, FrmGetObjectIndex( resultForm, frmResultFindMore ) );
}



/* Display current status: "Matches for <pattern>" or "No matches for <pattern>" */
static void UpdateStatus
    (
    Boolean findMore    /* true if more records remains to be searched */
    )
{
    if ( ! findMore && count == 0 )
        FldSetTextPtr( fldPtr, noMatches );
    else
        FldSetTextPtr( fldPtr, matches );

    FldDrawField( fldPtr );
}



/* Initialize the result form */
static void ResultFormInit( void )
{
    FormType*   resultForm;
    Char        str[ 30 ];

    GetSearchString( pattern );

    SysCopyStringResource( searching, strResultSearching );

    SysCopyStringResource( str, strResultMatches );
    StrPrintF( matches, "%s \"%s\"", str, pattern );

    SysCopyStringResource( str, strResultNoMatches );
    StrPrintF( noMatches, "%s \"%s\"", str, pattern );

    fldPtr = (FieldType*) GetObjectPtr( frmResultStatus );
    FldSetTextPtr( fldPtr, searching );

    resultForm = FrmGetFormPtr( frmResult );
    FrmDrawForm( resultForm );

    ClearSearchResult();

    initialFontStyle = FntSetFont( stdFont );
}



/* Event handler for the result form */
static Boolean SearchResultHandleEvent
    (
    Boolean startFromBeginning  /* start from the first record in DB */
    )
{
    Boolean handled;
    Boolean done;
    Boolean findMore;

    handled     = false;
    done        = true;
    findMore    = false;

    for ( ;; ) {
        EventType event;

        do {
            EvtGetEvent( &event, evtWaitForever );

            if ( SysHandleEvent( &event ) )
                continue;

            handled = false;

            switch ( event.eType ) {
                case appStopEvent:
                    EvtAddEventToQueue( &event );
                    return false;

                case ctlSelectEvent:
                    if ( event.data.ctlEnter.controlID == frmResultStop ) {
                        UpdateStatus( false );
                        ShowCancelMore();
                        handled = true;
                    }
                    else
                        handled = false;
                    break;

                case penUpEvent:
                    EvtFlushPenQueue();
                    handled = false;
                    break;

                case keyDownEvent:
                    UpdateStatus( false );
                    ShowCancelMore();
                    handled = true;
                    break;

                default:
                    handled = false;
            }

            /* Check if the form can handle the event */
            if ( ! handled )
                FrmHandleEvent( FrmGetFormPtr( frmResult ), &event );
            else
                return true;

        } while ( EvtEventAvail() );

        done = SearchInAllPages( pattern, startFromBeginning, &findMore );
        startFromBeginning = false;

        if ( done ) {
            UpdateStatus( findMore );
            if ( findMore )
                ShowCancelMore();
            else
                ShowCancel();

            return true;
        }
    }
}



/* Event handler for the result form */
Boolean ResultFormHandleEvent
    (
    const EventType* event  /* pointer to an EventType structure */
    )
{
    Boolean handled = false;

    SET_A4_FROM_A5 

    handled = false;

    switch ( event->eType ) {
        case ctlSelectEvent:
            if ( event->data.ctlEnter.controlID == frmResultFindMore ) {
                FormType* resultForm;

                RectangleType resultRect = { { DEFAULT_X, DEFAULT_Y }, 
                                             { DEFAULT_EXTENT_X, 
                                               MAX_RESULT * DEFAULT_EXTENT_Y } };
                WinEraseRectangle( &resultRect, 0 );

                FldSetTextPtr( fldPtr, searching );
                FldDrawField( fldPtr );

                resultForm = FrmGetFormPtr( frmResult );

                FrmHideObject( resultForm, FrmGetObjectIndex( resultForm, frmResultCancel ) );
                FrmHideObject( resultForm, FrmGetObjectIndex( resultForm, frmResultFindMore ) );
                FrmShowObject( resultForm, FrmGetObjectIndex( resultForm, frmResultStop ) );

                ClearSearchResult();

                handled = SearchResultHandleEvent( false );
                break;
            }
            else if ( event->data.ctlEnter.controlID != frmResultCancel )
                break;

            FntSetFont( initialFontStyle );
            FrmReturnToForm( Prefs()->toolbar );
            FrmUpdateForm( Prefs()->toolbar, frmRedrawUpdateCode );
            handled = true;
            break;

        case penDownEvent:
            MSG( _( "pen down at ( %d,%d )\n", event->screenX, event->screenY ) );

            if ( TopLeftY() < event->screenY && event->screenY < 130 ) {
                Int16 i;

                lineMatch = NOT_FOUND;

                for ( i = 0; i < MAX_RESULT; i++ ) {
                    if ( RctPtInRectangle( event->screenX, event->screenY, &SearchResult[ i ].coordinates ) ) {
                        WinInvertRectangle( &SearchResult[ i ].coordinates, 0 );
                        lineMatch = i;
                        break;
                    }
                    handled = true;
                }
            }
            break;

        case penUpEvent:
            MSG( _( "pen up at ( %d,%d )\n", event->screenX, event->screenY ) );

            if ( TopLeftY() < event->screenY && event->screenY < 130 ) {
                if ( lineMatch != NOT_FOUND ) {
                    WinInvertRectangle( &SearchResult[ lineMatch ].coordinates, 0 );

                    if ( RctPtInRectangle( event->screenX, event->screenY, &SearchResult[ lineMatch ].coordinates ) ) {
                        EventType match;

                        FrmReturnToForm( Prefs()->toolbar );

                        MemSet( &match, sizeof( EventType ), 0 );

                        match.eType                         = frmGotoEvent;
                        match.data.frmGoto.formID           = Prefs()->toolbar;
                        match.data.frmGoto.recordNum        = SearchResult[ lineMatch ].record;
                        match.data.frmGoto.matchPos         = SearchResult[ lineMatch ].pos;
                        match.data.frmGoto.matchLen         = SearchResult[ lineMatch ].length;
                        match.data.frmGoto.matchFieldNum    = 0;
                        match.data.frmGoto.matchCustom      = 0;

                        EvtAddEventToQueue( &match );
                    }
                    SndPlaySystemSound( sndClick );
                }
                handled = true;
            }
            break;

        case frmOpenEvent:
            ResultFormInit();
            handled = SearchResultHandleEvent( true );
            break;

        case frmCloseEvent:
            FntSetFont( initialFontStyle );
            handled = false;
            break;

        default:
            handled = false;
            break;
    }

    RESTORE_A4 

    return handled;
}
