/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file player_speaker_zone.cpp
 * \brief Implementation of the ptb::player_speaker_zone class.
 * \author Sbastien Angibaud
 */
#include "ptb/item/player_speaker_zone.hpp"

#include "ptb/controller_config.hpp"
#include "ptb/speaker_item.hpp"
#include "ptb/item/plee/plee.hpp"

#include "engine/export.hpp"

BASE_ITEM_EXPORT( player_speaker_zone, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::player_speaker_zone::player_speaker_zone()
  : m_first_player_spoke(false), m_second_player_spoke(false),
    m_first_player_in_zone(false), m_second_player_in_zone(false),
    m_first_player(NULL), m_second_player(NULL)
{
  set_phantom(true);
  set_can_move_items(false);
} // player_speaker_zone::player_speaker_zone()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 */
void
ptb::player_speaker_zone::progress( bear::universe::time_type elapsed_time )
{
  search_players();

  if ( m_first_player.get() ) 
    test_player_enter(m_first_player,m_first_player_in_zone);
  
  if ( m_second_player.get() ) 
    test_player_enter(m_second_player,m_second_player_in_zone);
} // player_speaker_zone::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type list of <std::string>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::player_speaker_zone::set_string_list_field
( const std::string& name, const std::vector<std::string>& value )
{
  bool result = false;

  if ( name == "player_speaker_zone.speeches" )
    {
      m_speeches = std::vector<std::string>(value);
      result = true;
    }
  else
    result = super::set_string_list_field( name, value );

  return result;
} // player_speaker_zone::set_string_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is correctly initialized.
 */
bool ptb::player_speaker_zone::is_valid() const
{
  return !m_speeches.empty() && super::is_valid();
} // player_speaker_zone::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Search players.
 */
void ptb::player_speaker_zone::search_players()
{
  plee::get_instance_message msg;
  
  if ( m_first_player.get() == NULL )
    {
      get_level_globals().send_message(plee::player_name(1),msg);
      m_first_player = msg.get_instance();
    }
  
  if ( ( game_variables::get_players_count() == 2 ) && 
       ( m_second_player.get() == NULL ) )
    {
      get_level_globals().send_message(plee::player_name(2),msg);
      m_second_player = msg.get_instance();
    }
} // player_speaker_zone::search_players()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if a player enter in the zone.
 */
void ptb::player_speaker_zone::test_player_enter
( bear::universe::derived_item_handle_maker<plee>::handle_type&
  player, bool& player_in_zone )
{
  bear::universe::rectangle_type bounding_box_1
    ( player->get_bounding_box() );
  
  if ( bounding_box_1.intersects(get_bounding_box()) )
    {
      if ( ! player_in_zone )
        {
          ptb::speaker_item* speaker =
            dynamic_cast<ptb::speaker_item*>(player.get());
          
          speaker->reset_speeches();
          
          controller_config cfg;
          const controller_layout& layout =
            cfg.get_layout( player->get_index() );
          
          std::vector<std::string>::iterator it;
          for ( it=m_speeches.begin(); it!=m_speeches.end(); ++it)
            {
              std::vector<std::string> text;
              layout.escape_action_sequence(*it, text);
              
              std::string speech;
              
              for (unsigned int i=0; i!=text.size(); ++i)
                speech += gettext(text[i].c_str());
              
              speaker->add_speech(speech);
            }
          
          player_in_zone = speaker->speak(get_level_globals());
        }
    }
  else
    player_in_zone = false;
} // player_speaker_zone::test_player_enter()
