/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file little_plee.cpp
 * \brief Implementation of the ptb::little_plee class.
 * \author Sbastien Angibaud
 */
#include "ptb/item/little_plee.hpp"
#include "engine/game.hpp"
#include "engine/world.hpp"
#include "universe/derived_item_handle.hpp"
#include "ptb/item/plee/plee.hpp"

#include "engine/export.hpp"

BASE_ITEM_EXPORT( little_plee, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::little_plee::little_plee()
  : m_current_state(run_state), 
    m_life_given(false), m_nb_idle(0)
{
  set_size( 27, 50 );
  set_mass( 100 );
  set_density(2);
  get_rendering_attributes().mirror(false);
} // little_plee::little_plee()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialise the item.
 */
void ptb::little_plee::build()
{
  super::build();

  m_run_animation = get_level_globals().get_animation
    ("animation/powerup/life_bonus/run.canim");

  m_jump_animation = get_level_globals().get_animation
    ("animation/powerup/life_bonus/jump.canim");
} // little_plee::build()

/*---------------------------------------------------------------------------*/
/**
 * \brief Do one iteration in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
void ptb::little_plee::progress( bear::universe::time_type elapsed_time )
{
  super::progress( elapsed_time );

  if ( m_current_state == run_state )
    m_run_animation.next(elapsed_time);
  else
    m_jump_animation.next(elapsed_time);
  
  if ( m_current_state == run_state )
    progress_run();
  else
    progress_jump();
} // little_plee::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite representing the item.
 * \param visuals (out) The sprites of the item, and their positions.
 */
void ptb::little_plee::get_visual
( std::list<bear::engine::scene_visual>& visuals ) const
{
  if ( m_current_state == run_state ) 
    add_visual(m_run_animation, visuals);
  else
    add_visual(m_jump_animation, visuals);
} // little_plee::get_visual()

/*----------------------------------------------------------------------------*/
/**
 * \brief Process a collision.
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
void ptb::little_plee::collision
( bear::engine::base_item& that, bear::universe::collision_info& info )
{
  ptb::plee* p = dynamic_cast<ptb::plee*>(&that);

  if ( ! m_life_given ) 
    if ( (p != NULL) && 
         ( info.get_collision_side() != bear::universe::zone::middle_zone ) )
      {
        game_variables::set_lives_count
          ( p->get_index(), 
            game_variables::get_lives_count(p->get_index()) + 1);
        
        m_life_given = true;
        kill();
      }
} // little_plee::collision()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the orientation of the item.
 * \param orientation The new orientation on x axis.
 */
void ptb::little_plee::set_positive_orientation(bool orientation)
{
  get_rendering_attributes().mirror(orientation);
} // little_plee::set_positive_orientation()

/*----------------------------------------------------------------------------*/
/**
 * \brief Start the action.
 */
void ptb::little_plee::start()
{
  do_jump();
} // little_plee::start()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do a progress when the item run.
 */
void ptb::little_plee::progress_run()
{
  if ( has_bottom_contact() ) 
    {
      if ( has_right_contact() || has_left_contact() )
        {
          m_nb_idle++;
          if ( m_nb_idle >= 2 )
            progress_idle();
        }
      else 
        {
          if ( ! can_go_toward(get_rendering_attributes().is_mirrored()) )
            get_rendering_attributes().mirror
              ( !get_rendering_attributes().is_mirrored() );
          
          if ( get_rendering_attributes().is_mirrored() )
            add_internal_force( bear::universe::force_type(-250000,0) );
          else
            add_internal_force( bear::universe::force_type(250000,0) );
        }
    }
  else
    {
      m_current_state = jump_state;
      m_jump_animation.reset();
    }
} // little_plee::progress_run()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do a progress when the item jump.
 */
void ptb::little_plee::progress_jump()
{
  if ( has_bottom_contact() ) 
    {
      m_current_state = run_state;
      m_run_animation.reset();
    }
  else if ( get_rendering_attributes().is_mirrored() )
    add_internal_force( bear::universe::force_type(-100000, 0) );
  else
    add_internal_force( bear::universe::force_type(100000, 0) );
} // little_plee::progress_jump()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do a jump.
 */
void ptb::little_plee::do_jump()
{
  if ( get_rendering_attributes().is_mirrored() )
    add_internal_force( bear::universe::force_type(-4000000, 7500000) );
  else
    add_internal_force( bear::universe::force_type(4000000, 7500000) );
} // little_plee::jump()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progress when the item is stopped.
 */
void ptb::little_plee::progress_idle()
{
  bool direction_change = false;
  m_nb_idle = 0;
  unsigned int a = (unsigned int)(2.0 * rand() / RAND_MAX);
  
  if ( a == 0 )
    {
      if ( has_right_contact() && can_go_toward(true) )
        {
          direction_change = true;
          get_rendering_attributes().mirror(true);
          add_internal_force( bear::universe::force_type(-250000,0) );
        }
      
      if ( has_left_contact() && can_go_toward(false) )
        {
          direction_change = true;
          get_rendering_attributes().mirror(false);
          add_internal_force( bear::universe::force_type(250000,0) );
        }
    }

  if ( ! direction_change ) 
    do_jump();
} // little_plee::progress_idle()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compute if the item can go toward a given orientation.
 */
bool ptb::little_plee::can_go_toward( bool left_orientation )
{
  bool result = true;
  unsigned int closest = 0;
  bear::universe::coordinate_type dist = 1000;

  plee::get_instance_message msg;
  get_level_globals().send_message(plee::player_name(1),msg);
  bear::universe::const_derived_item_handle_maker<plee>::handle_type
    m_first_player(msg.get_instance());

  if ( m_first_player.get() )
    {
      closest = 1;
      dist = 
        m_first_player->get_center_of_mass().distance(get_center_of_mass());
    }
  
  get_level_globals().send_message(plee::player_name(2),msg);
  bear::universe::const_derived_item_handle_maker<plee>::handle_type
    m_second_player(msg.get_instance());

  if ( m_second_player.get() )
    {
      bear::universe::coordinate_type dist2 = 
        m_second_player->get_center_of_mass().distance(get_center_of_mass());
      if ( (closest == 0) || ((closest == 1) && (dist2 < dist)) )
        {
          closest = 2;
          dist = dist2;
        }
    }
  
  if ( dist <= 250 ) 
    {
      if ( closest == 1 )
        {
          if ( left_orientation ) 
            result = ( m_first_player->get_left() > get_right() );
          else
            result = ( m_first_player->get_right() < get_left() );
        }
      else 
        {
          if ( closest == 2 )
            {
              if ( left_orientation ) 
                result = ( m_second_player->get_left() > get_right() );
              else
                result = ( m_second_player->get_right() < get_left() );
             }
        }
    }

  return result;
} // little_plee::can_go_toward()
