/*
  Bear Engine - Level editor

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/item_selection.cpp
 * \brief Implementation of the bf::item_selection class.
 * \author Julien Jorge
 */
#include "bf/item_selection.hpp"

#include "bf/item_instance.hpp"

/*----------------------------------------------------------------------------*/
bf::item_selection::item_selection()
  : m_item(NULL)
{

} // item_selection::item_selection()

/*----------------------------------------------------------------------------*/
void bf::item_selection::insert( item_instance* item, bool main_selection )
{
  CLAW_PRECOND( item != NULL );

  m_group.insert(item);

  if ( main_selection || (m_item == NULL) )
    m_item = item;
} // item_selection::insert()

/*----------------------------------------------------------------------------*/
void bf::item_selection::insert( const item_selection& s )
{
  if ( !s.empty() )
    {
      m_group.insert(s.m_group.begin(), s.m_group.end());
      m_item = s.m_item;
    }
} // item_selection::insert()

/*----------------------------------------------------------------------------*/
void bf::item_selection::remove( item_instance* item )
{
  CLAW_PRECOND( item != NULL );

  m_group.erase(item);

  if ( m_item == item )
    {
      if ( m_group.empty() )
        m_item = NULL;
      else
        m_item = *m_group.begin();
    }
} // item_selection::remove()

/*----------------------------------------------------------------------------*/
void bf::item_selection::remove( const item_selection& s )
{
  for (const_iterator it=s.begin(); it!=s.end(); ++it)
    m_group.erase(*it);

  if ( s.m_group.find(m_item) != s.m_group.end() )
    {
      if ( m_group.empty() )
        m_item = NULL;
      else
        m_item = *m_group.begin();
    }
} // item_selection::remove()

/*----------------------------------------------------------------------------*/
bool bf::item_selection::is_selected( item_instance const* item ) const
{
  CLAW_PRECOND( item != NULL );

  return m_group.find((item_instance*)item) != m_group.end();
} // item_selection::is_selected()

/*----------------------------------------------------------------------------*/
bool bf::item_selection::is_main_selection( item_instance const* item ) const
{
  CLAW_PRECOND( item != NULL );

  return m_item == item;
} // item_selection::is_main_selection()

/*----------------------------------------------------------------------------*/
void bf::item_selection::clear()
{
  m_item = NULL;
  m_group.clear();
} // item_selection::clear()

/*----------------------------------------------------------------------------*/
bool bf::item_selection::empty() const
{
  CLAW_PRECOND( (m_item != NULL) || m_group.empty() );

  return m_item == NULL;
} // item_selection::empty()

/*----------------------------------------------------------------------------*/
bf::item_instance* bf::item_selection::get_main_selection() const
{
  return m_item;
} // item_selection::get_main_selection()

/*----------------------------------------------------------------------------*/
bf::item_selection::const_iterator bf::item_selection::begin() const
{
  return m_group.begin();
} // item_selection::begin()

/*----------------------------------------------------------------------------*/
bf::item_selection::const_iterator bf::item_selection::end() const
{
  return m_group.end();
} // item_selection::end()
