/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file visual/code/scene_polygon.cpp
 * \brief Implementation of the visual::scene_polygon.
 * \author Julien Jorge
 */
#include "visual/scene_polygon.hpp"

#include "visual/scene_element.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param x X-position of the origin of the polygon on the screen.
 * \param y Y-position of the origin of the polygon on the screen.
 * \param color The color of the polygon.
 * \param p The points of the polygon.
 */
bear::visual::scene_polygon::scene_polygon
( coordinate_type x, coordinate_type y, const claw::graphic::rgba_pixel& color,
  const std::vector<position_type>& p )
  : base_scene_element(x, y), m_color(color), m_points(p)
{

} // scene_polygon::scene_polygon()

/*----------------------------------------------------------------------------*/
/**
 * \brief Allocate a copy of this instance.
 */
bear::visual::base_scene_element* bear::visual::scene_polygon::clone() const
{
  return new scene_polygon(*this);
} // scene_polygon::clone()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a rectangle where the polygon is completely opaque.
 */
bear::visual::rectangle_type bear::visual::scene_polygon::get_opaque_box() const
{
  return rectangle_type(0, 0, 0, 0);
} // scene_polygon::get_opaque_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get a rectangle bounding the polygon.
 */
bear::visual::rectangle_type
bear::visual::scene_polygon::get_bounding_box() const
{
  coordinate_type min_x = m_points[0].x;
  coordinate_type min_y = m_points[0].y;
  coordinate_type max_x = min_x;
  coordinate_type max_y = min_y;

  for (unsigned int i=0; i!=m_points.size(); ++i)
    {
      if ( m_points[i].x < min_x )
        min_x = m_points[i].x;

      if ( m_points[i].y < min_y )
        min_y = m_points[i].y;

      if ( m_points[i].x > max_x )
        max_x = m_points[i].x;

      if ( m_points[i].y > max_y )
        max_y = m_points[i].y;
    }

  return rectangle_type(min_x, min_y, max_x - min_x + 1, max_y - min_y + 1);
} // scene_polygon::get_bounding_box()

/*----------------------------------------------------------------------------*/
/**
 * \brief Split the polygon in sub polygons.
 * \param boxes The boxes describing how to split the polygon.
 * \param output The resulting polygons (they are inserted to the front).
 */
void bear::visual::scene_polygon::burst
( const rectangle_list& boxes, scene_element_list& output ) const
{
  output.push_front( scene_element(*this) );
} // scene_polygon::burst()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the polygon on a screen.
 * \param scr The screen on which we render the polygon.
 */
void bear::visual::scene_polygon::render( base_screen& scr ) const
{
  std::vector<position_type> p(m_points);

  for ( std::size_t i=0; i!=p.size(); ++i )
    {
      p[i].x = get_position().x + p[i].x * get_scale_factor_x();
      p[i].y = get_position().y + p[i].y * get_scale_factor_y();
    }

  scr.draw_polygon(m_color, p);
} // scene_polygon::render()
