/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bitmap_font_loader.cpp
 * \brief Implementation of the bitmap_font_loader class.
 * \author Julien Jorge
 */
#include "engine/bitmap_font_loader.hpp"

#include "engine/level_globals.hpp"

#include <claw/exception.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param f The file from which we load the file.
 * \param glob The level_globals in which we load the resources.
 */
bear::engine::bitmap_font_loader::bitmap_font_loader
( std::istream& f, level_globals& glob )
  : m_file(f), m_level_globals(glob)
{

} // bitmap_font_loader::bitmap_font_loader()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the font.
 */
bear::visual::font bear::engine::bitmap_font_loader::run()
{
  visual::bitmap_font::symbol_table cs;
  bool valid = false;
  unsigned int n;

  if ( m_file >> cs.size.x >> cs.size.y >> n )
    if ( n > 0 )
      {
        std::string image_name;

        cs.font_images.reserve(n);

        for (unsigned int i=0; i!=n; ++i)
          if ( m_file >> image_name )
            cs.font_images.push_back( m_level_globals.get_image(image_name) );

        if ( cs.font_images.size() == n )
          {
            int x, y;
            unsigned int i;
            char c;

            valid = true;

            do
              {
                m_file.ignore(1, '\n');

                if ( m_file.get(c) >> x >> y >> i )
                  if ( i < n )
                    {
                      cs.characters[c].image_index = i;
                      cs.characters[c].position.set(x, y);
                    }
              }
            while ( m_file );
          }
      }

  if (valid)
    return new visual::bitmap_font(cs);
  else
    throw CLAW_EXCEPTION( "Bad font" );
} // bitmap_font_loader::run()
