/*
 * picasm.h
 *
 * Copyright 1995-2005 Timo Rossi, <trossi@iki.fi>
 * See the file LICENSE for license terms.
 * 
 * http://www.iki.fi/trossi/pic/
 * 
 */

#define VERSION "v1.12c"
#define I_VERSION 112

#if defined(__SASC) || defined(__TURBOC__) || defined(__WATCOMC__) || defined(_MSC_VER)
#define strcasecmp stricmp
#define strncasecmp strnicmp
#endif

#define PIC_16BIT_SUPPORT 1

/* hex output end-of-line. if you need CRLFs in hex files
 * when running on a system which does not normally generate
 * them (such as Unix), use "\r\n", else "\n".
 */
#define HEX_EOL "\r\n"

/* maximum number of errors before aborting assembly */
#define MAX_ERRORS 20

/* output formats */
enum {
    IHX8M,
    IHX16
};

/* org mode */
typedef enum {
    O_NONE,
    O_PROGRAM,
    O_REGFILE,
    O_EDATA
} org_mode_t ;

#ifdef PIC_16BIT_SUPPORT
typedef unsigned long pic_instr_t;

#define INVALID_INSTR  0xfffff
#define INVALID_DATA   0xfffff
#define INVALID_CONFIG 0xfffff
#define INVALID_ID     0xfffff
#else
typedef unsigned short pic_instr_t;

#define INVALID_INSTR  0xffff
#define INVALID_DATA   0xffff
#define INVALID_CONFIG 0xffff
#define INVALID_ID     0xffff
#endif /* PIC_16BIT_SUPPORT */

/* list flags */
#define LIST_LOC     1
#define LIST_PROG    2
#define LIST_EDATA   4
#define LIST_FORWARD 8
#define LIST_VAL     0x10
#define LIST_PTR     0x20

/* inc_file types */
typedef enum {
    INC_FILE,
    INC_MACRO
} inctype_t;

/*
 * structure for include files/macros
 */
struct inc_file {
    struct inc_file *next;
    union {
	struct {
	    FILE *fp;
	    char *fname;
	} f; /* file */
	struct {
	    struct symbol *sym;
	    struct macro_line *ml;
	    struct macro_arg *args;
	    int uniq_id;
	} m; /* macro */
    } v;
    inctype_t type;
    int linenum;
    int cond_nest_count;
};

/*
 * structure to hold one macro line
 */
struct macro_line {
  struct macro_line *next;
  char text[1];
};

/* Macro argument */
struct macro_arg {
    struct macro_arg *next;
    char text[1];
};

/*
 * structure for patching forward jumps
 */
typedef enum {
    PATCH11,          /* 14-bit instr. set PICs */
    PATCH9,           /* 12-bit, goto */
    PATCH8,           /* 12-bit, call */
    PATCH_UPPER_BYTE  /* hibyte() */
} patchtype_t;

struct patch {
    struct patch *next;
    struct symbol *label;
    int location;
    patchtype_t type;
};

#define PROGMEM_MAX 16384
#define EEPROM_MAX 256

/*
 * Definitions for different types of PIC processors
 */

typedef enum {
    PIC12BIT=1,
    PIC14BIT,
    PIC16BIT_NOMUL,
    PIC16BIT
} instrset_t;


/*
 * these enums must be kept in sync with config.c and devices.perl
 * (maybe devices.perl could create the other things automatically...)
 */ 
enum { FUSE_CP, FUSE_WDTE, FUSE_PWRTE, FUSE_MCLRE, FUSE_BODEN, FUSE_MPEEN,
       FUSE_LVP, FUSE_CPD, FUSE_WRT, FUSE_BKBUG, N_FUSES };

enum { OSC_LP, OSC_XT, OSC_HS, OSC_RC, OSC_IN, OSC_INTRC, OSC_INTRC_CLKOUT,
       OSC_EXTRC, OSC_EXTRC_CLKOUT, OSC_LF, OSC_EC, N_OSC };

struct pic_type {
    char *name;
    int progmem_size;
    short regfile_limit; /* without banking */
    short eeprom_size;
    instrset_t instr_set;
    int config_def, fuse[N_FUSES];
    int osc_mask, osc[N_OSC];
};

/*
 * truth values for boolean functions
 */
#define EXPR_FALSE (0)
#define EXPR_TRUE (~0)

/* number of bits in an expression value */
#define EXPR_NBITS 32

/*
 * Success/failure return codes for functions
 */
#define OK   (0)
#define FAIL (-1)

/*
 * variable declarations
 */

/* picasm.c */
extern struct inc_file *current_file;
extern char *line_buf_ptr;
extern char line_buffer[256];
extern int unique_id_count;
extern int cond_nest_count;
extern org_mode_t O_Mode;
extern int prog_location, reg_location, edata_location, org_val;
extern int warnlevel;
extern struct patch **local_patch_list_ptr;
extern int prog_mem_size;
extern unsigned short list_flags;
extern struct pic_type *pic_type;
extern pic_instr_t config_fuses;
extern int local_level;

extern int default_radix;

/*
 * function prototypes
 */

/* picasm.c */
void *mem_alloc(int size);
#define mem_free(p) free(p)
void fatal_error(char *, ...), error(int, char *, ...), warning(char *, ...);
void write_listing_line(int cond_flag);
void gen_code(int val);
void add_patch(int tab, struct symbol *sym, patchtype_t type);
int gen_byte_c(int instr_code);

/* config.c */
void parse_config(void);

/* expr.c */
int get_expression(long *);

/* pic12bit.c */
int assemble_12bit_mnemonic(int op);

/* pic14bit.c */
int assemble_14bit_mnemonic(int op);

/* pic16bit.c */
int assemble_16bit_mnemonic(int op);
