/*
 * btb_ia64_smpl.c - shows one branch per line on Itanium & Itanium2 BTB
 *
 * Copyright (C) 2002-2003 Hewlett-Packard Co
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file is part of pfmon, a sample tool to measure performance 
 * of applications on Linux/ia64.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */

#include "pfmon.h"

#include <perfmon/pfmlib_generic_ia64.h>
#include <perfmon/perfmon_default_smpl.h>

/*
 * the name of the output format.
 *
 * You must make sure it is unique and hints the user as to which the format does
 */
#define SMPL_MOD_NAME		"btb"
#define CHECK_VERSION(h)	(PFM_VERSION_MAJOR((h)) != PFM_VERSION_MAJOR(PFM_DEFAULT_SMPL_VERSION))
#define BTB_DFL_SMPL_ENTRIES	2048UL

typedef struct {
	unsigned long smpl_entries;
} btb_options_t;


static btb_options_t btb_options;
static int pmu_is_itanium2;	/* true if host PMU is of type Itanium2 */

/*
 * forward declaration
 */
pfmon_smpl_module_t btb_ia64_smpl_module;

static int
btb_process_samples(pfmon_smpl_desc_t *csmpl)
{
	pfm_default_smpl_hdr_t *hdr;
	pfm_default_smpl_entry_t *ent;
	FILE *fp = csmpl->smpl_fp;
	unsigned long val, pmd16, entry;
	unsigned long i;
	uint64_t count;
	unsigned long *btb;
	unsigned long l,f, last_type;
	int is_itanium2;
	int ret = 0;


	hdr 	   = csmpl->smpl_hdr;
	ent	   = (pfm_default_smpl_entry_t *)(hdr+1);
	entry      = options.opt_aggr ? *csmpl->aggr_count : csmpl->entry_count;
	count      = hdr->hdr_count;
	is_itanium2 = pmu_is_itanium2;

	DPRINT(("hdr_count=%lu hdr=%p\n", count, hdr));

	for(i=0; i < count; i++) {

		/*
		 * overlay btb array
		 */
		btb   = (unsigned long *)(ent+1);
		pmd16 = btb[8];

		/* btb[8]=pmd16, compute first and last element */
		f = pmd16 & 0x8 ? pmd16 & 0x7 : 0;
		l = pmd16 & 0x7;

		DPRINT(("btb_trace: pmd16=0x%lx first=%lu last=%lu bbi=%lu full=%lu\n", 
			pmd16,
			f,
			l, 
			btb[8] & 0x7,
			(btb[8]>>3) & 0x1));

		last_type = 2; /* no branch type, force 0 to be printed */
		do {
			if ((btb[f] & 0x1)) { /* branch instruction */
				if (last_type < 2) {
					if (last_type == 1) fprintf(fp, "0x0000000000000000");
					fprintf(fp, "\n");
				}
			} else { /* branch target */
				if (last_type == 2) fprintf(fp, "0x0000000000000000 ");
			}

			val = btb[f];

			/* correct bundle address on Itanium2-style BTB */
			if (is_itanium2) {
				unsigned long b;
				b       = (pmd16 >> (4 + 4*f)) & 0x1;
				val 	+= b<<4;
			}
			ret = fprintf(fp, "0x%016lx ", val); /* it is enough to capture write error here only */

			last_type = btb[f] & 0x1; /* 1= last PMD contained a branch insn, 0=a branch target */

			f = (f+1) % 8;

			/* fprintf() error detection */
			if (ret == -1) goto error;

		} while (f != l && ret >0);

		if (last_type == 1) fprintf(fp, "0x0000000000000000");

		fputc('\n',fp);

		ent = (pfm_default_smpl_entry_t *)(btb+9);	
		entry++;
	}
	/*
	 * when aggregation is used, for are guaranteed sequential access to
	 * this routine by higher level lock
	 */
	if (options.opt_aggr) {
		*csmpl->aggr_count += hdr->hdr_count;
	} else {
		csmpl->entry_count += hdr->hdr_count;
	}
	csmpl->last_count = count;

	return 0;
error:
	warning("cannot write to sampling file: %s\n", strerror(errno));
	/* not reached */
	return -1;
}

static int
btb_check_new_samples(pfmon_smpl_desc_t *csmpl)
{
	pfm_default_smpl_hdr_t *hdr; 
	uint64_t last_ovfl;

	hdr       = csmpl->smpl_hdr;
	last_ovfl = csmpl->last_ovfl;

	if (hdr->hdr_overflows <= last_ovfl && last_ovfl != ~0UL && hdr->hdr_count == csmpl->last_count) {
		DPRINT(("skipping identical set of samples %lu <= %lu\n", hdr->hdr_overflows, last_ovfl)); 
		return -1;
	}
	/*
	 * XXX: module specific field in generic structure (last_ovfl)
	 */
	csmpl->last_ovfl = hdr->hdr_overflows;

	return 0;
}

/*
 * 000-255   reserved for generic options
 * 400-499   reserved for PMU specific options
 * 500-599   reserved for format specific options
 */
static struct option btb_cmd_options[]={
	{ "smpl-entries", 1, 0, 500},
	{ NULL, 0, 0, 0}
};

static void
btb_show_options(void)
{
	printf("\t--smpl-entries=val\t\tset number of entries for sampling buffer (default %lu)\n", BTB_DFL_SMPL_ENTRIES);
}

/*
 * 0  means we understood the option
 * -1 unknown option
 */
static int
btb_parse_options(int code, char *optarg, pfmon_lib_param_t *evt)
{
	char *endptr = NULL;

	switch(code) {
		case  500:
			if (btb_options.smpl_entries != BTB_DFL_SMPL_ENTRIES) 
				fatal_error("smpl-entries already defined\n");

			btb_options.smpl_entries = strtoul(optarg, &endptr, 0);

			if (*endptr != '\0') 
				fatal_error("invalid number of entries: %s\n", optarg);
			break;
		default:
			return -1;
	}
	return 0;

}


static size_t
btb_get_fmt_arg_size(void)
{
	return sizeof(pfm_default_smpl_ctx_arg_t);
}

#define MAX_PMD_COUNT		64
#define FUDGE_FACTOR		(sizeof(pfm_default_smpl_entry_t)+(sizeof(unsigned long)*MAX_PMD_COUNT))
#define ENTRY_SIZE(npmd)	(sizeof(pfm_default_smpl_entry_t)+((npmd)*sizeof(unsigned long)))
static int
btb_initialize_ctx_arg(pfmon_ctx_arg_t *arg, unsigned max_pmds_sample)
{
	pfm_default_smpl_ctx_arg_t *ctx_arg = (pfm_default_smpl_ctx_arg_t *)arg;
	unsigned long entry_size;

	entry_size = ENTRY_SIZE(max_pmds_sample);
	
	/*
	 * the fudge factor allows us to get exactly the number of entries specified by the
	 * user (or the default). The kernel module is such that it will consider the buffer
	 * full if less than PFM_DEFAULT_MAX_ENTRY_SIZE bytes are left in the buffer. Any
	 * entry size is <= PFM_DEFAULT_MAX_ENTRY_SIZE, therefore we will always record
	 * less than the specified number unless we increase the buffer size slightly.
	 */
	ctx_arg->buf_arg.buf_size = sizeof(pfm_default_smpl_hdr_t)
		                  + (FUDGE_FACTOR-entry_size)
				  + btb_options.smpl_entries*entry_size;

	/*
	 * copy the uuid of the format we are using
	 */
	memcpy(ctx_arg->ctx_arg.ctx_smpl_buf_id, btb_ia64_smpl_module.uuid, sizeof(pfm_uuid_t));

	vbprintf("min buffer entries=%lu buffer_size=%lu max PMD/entry=%u\n", 
		btb_options.smpl_entries, 
		ctx_arg->buf_arg.buf_size, 
		max_pmds_sample);

	DPRINT(("max_pmds_sample=%u buf_size=%lu fudge=%lu buffer_header=%lu entry_header=%lu (max)entry_size=%lu\n", 
		max_pmds_sample, 
		ctx_arg->buf_arg.buf_size, 
		FUDGE_FACTOR, 
		sizeof(pfm_default_smpl_hdr_t), 
		sizeof(pfm_default_smpl_entry_t), 
		entry_size));
		

	return 0;
}

/*
 * module initialization
 */
static int
btb_initialize_module(void)
{
	int pmu_type;

	btb_options.smpl_entries = BTB_DFL_SMPL_ENTRIES;
	/* need to know if Itanium or Itanium2 */
	pfm_get_pmu_type(&pmu_type);

	pmu_is_itanium2 = pmu_type == PFMLIB_ITANIUM2_PMU ? 1 : 0;

	return pfmon_register_smpl_mod_options(btb_cmd_options, sizeof(btb_cmd_options));
}


static int
btb_check_version(pfmon_smpl_desc_t *csmpl)
{
	pfm_default_smpl_hdr_t *hdr;

	hdr   = csmpl->smpl_hdr;

	if (CHECK_VERSION(hdr->hdr_version)) {
		warning("format %s expect format v%u.x not v%u.%u\n", 
				SMPL_MOD_NAME,
				PFM_VERSION_MAJOR(PFM_DEFAULT_SMPL_VERSION),
				PFM_VERSION_MAJOR(hdr->hdr_version),
				PFM_VERSION_MINOR(hdr->hdr_version));
		return -1;
	}
	return 0;
}


static int
btb_print_header(pfmon_smpl_desc_t *csmpl)
{
	fprintf(csmpl->smpl_fp, "# btb only sampling output format (1 branch event/line)\n"
			 	"# line format: source target\n"
			 	"# a value of 0x0000000000000000 indicates source or target not captured\n");
	return 0;
}

static int
btb_validate_events(pfmon_lib_param_t *evt, pfarg_reg_t *pc, pfarg_reg_t *pd)
{
	unsigned int idx;

	/*
	 * must be measuring BTB only
	 */
	if (evt->inp.pfp_event_count > 1) {
		warning("the btb sampling output format with the BRANCH_EVENT only\n");
		return -1;
	}
	/*
	 * find event for host CPU
	 */
	if (pfm_find_event_byname("BRANCH_EVENT", &idx) != PFMLIB_SUCCESS) {
		warning("cannot find BRANCH_EVENT in the event table for this PMU\n");
		return -1;
	}
	/*
	 * verify we have the right event
	 */
	if (evt->inp.pfp_events[0].event != idx) {
		warning("you can only use the BRANCH_EVENT event with the %s sampling module\n", SMPL_MOD_NAME);
		return -1;
	}
	return 0;
}

pfmon_smpl_module_t btb_ia64_smpl_module={
	.name		    = SMPL_MOD_NAME,		
	.pmu_mask	    = PFMON_PMU_MASK(PFMLIB_ITANIUM_PMU)|PFMON_PMU_MASK(PFMLIB_ITANIUM2_PMU),		
	.description	    = "IA-64 column-style BTB raw values. Must use BRANCH_EVENT event only.",
	.check_version	    = btb_check_version,
	.process_samples    = btb_process_samples,
	.get_fmt_arg_size   = btb_get_fmt_arg_size,
	.initialize_ctx_arg = btb_initialize_ctx_arg,
	.check_new_samples  = btb_check_new_samples,
	.show_options       = btb_show_options,
	.parse_options      = btb_parse_options,
	.print_header	    = btb_print_header,
	.initialize_module  = btb_initialize_module,
	.validate_events    = btb_validate_events,
	.uuid		    = PFM_DEFAULT_SMPL_UUID,
};
