#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: unstructured_2d.c,v 1.5 2000/02/01 17:02:51 knepley Exp $";
#endif

static char help[] = "This example deals with 2D unstructured meshes.\n\n";

#include "unstructured_2d.h"

/*
  Here we will create a series of refined 2D unstructured meshes.
*/
#undef  __FUNCT__
#define __FUNCT__ "main"
int main(int argc, char **args)
{
  MeshContext ctx; /* Holds problem specific information */
  int         ierr;

  PetscFunctionBegin;
  ierr = PetscInitialize(&argc, &args, 0, help);                                    CHKERRABORT(PETSC_COMM_WORLD, ierr);

  ierr = MeshContextCreate(PETSC_COMM_WORLD, &ctx);                                 CHKERRABORT(PETSC_COMM_WORLD, ierr);
  ierr = MeshContextRun(ctx);                                                       CHKERRABORT(PETSC_COMM_WORLD, ierr);
  ierr = MeshContextDestroy(ctx);                                                   CHKERRABORT(PETSC_COMM_WORLD, ierr);

  CHKMEMQ;
  PetscFinalize();
  PetscFunctionReturn(0);
}

/*--------------------------------------------- MeshContext Creation -------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MeshContextCreate"
/*@
  MeshContextCreate - This function initializes the Mesh context.

  Collective on MPI_Comm

  Input Parameter:
. comm - The communicator

  Output Parameter:
. mCtx - The MeshContext

  Level: beginner

.keywords: Mesh, context, create
.seealso: MeshContextDestroy(), MeshContextPrint(), MeshContextSetup()
@*/
int MeshContextCreate(MPI_Comm comm, MeshContext *mCtx)
{
  MeshContext ctx;

  PetscFunctionBegin;
  /* Setup context */
  PetscHeaderCreate(ctx, _MeshContext, int, PETSC_VIEWER_COOKIE, -1, "context", comm, 0, 0);
  PetscLogObjectCreate(ctx);
  PetscLogObjectMemory(ctx, sizeof(struct _MeshContext));

  /* Initialize subobjects */
  ctx->mesh = PETSC_NULL;
  /* Set for vertices only */
  ctx->numLocalNodes = 3;
  /* Setup domain */
  ctx->geometryCtx.size[0]  = 2.0;
  ctx->geometryCtx.size[1]  = 2.0;
  ctx->geometryCtx.start[0] = 0.0;
  ctx->geometryCtx.start[1] = 0.0;
  /* Setup refinement */
  ctx->geometryCtx.maxArea  = 0.5;
  ctx->geometryCtx.areaFunc = PointFunctionConstant;
  ctx->geometryCtx.areaCtx  = PETSC_NULL;
  /* Initialize problem loop */
  ctx->numRefine  = 0;
  ctx->numCoarsen = 0;

  *mCtx = ctx;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshContextDestroy"
/*@
  MeshContextDestroy - This function destroys the Mesh context.

  Collective on MeshContext

  Input Parameter:
. ctx - The MeshContext

  Level: beginner

.keywords: Mesh, context, destroy
.seealso: MeshContextCreate()
@*/
int MeshContextDestroy(MeshContext ctx)
{
  PetscFunctionBegin;
  if (--ctx->refct > 0) SETERRQ(PETSC_ERR_PLIB, "Mesh context should not be referenced more than once");
  PetscLogObjectDestroy(ctx);
  PetscHeaderDestroy(ctx);
  PetscFunctionReturn(0);
}

/*------------------------------------------------ Mesh Creation -----------------------------------------------------*/
/*@
  MeshContextSetup - This function initializes all internal variables through options and creates the mesh.

  Collective on MeshContext

  Input Parameter:
. ctx - A MeshContext

  Options Database Keys:
+ num_local_nodes - The number of nodes on an element
. num_refine      - The number of times to refine the mesh
. num_coarsen     - The number of times to coarsen the mesh
- mesh_max_area   - The maximum area of an element after refinement

  Level: beginner

.seealso MeshContextCleanup(), MeshContextCreate()
@*/
#undef  __FUNCT__
#define __FUNCT__ "MeshContextSetup"
int MeshContextSetup(MeshContext ctx)
{
  PetscTruth opt;
  int        ierr;

  PetscFunctionBegin;
  /* The number of local nodes (usually either 3 or 6) */
  ierr = PetscOptionsGetInt(PETSC_NULL, "-num_local_nodes",  &ctx->numLocalNodes,  &opt);                 CHKERRQ(ierr);
  /* The number of times to refine the mesh */
  ierr = PetscOptionsGetInt(PETSC_NULL, "-num_refine",  &ctx->numRefine,  &opt);                          CHKERRQ(ierr);
  /* The number of times to coarsen the mesh */
  ierr = PetscOptionsGetInt(PETSC_NULL, "-num_coarsen", &ctx->numCoarsen, &opt);                          CHKERRQ(ierr);
  /* Setup refinement */
  ierr = PetscOptionsGetReal("mesh", "-max_area", &ctx->geometryCtx.maxArea, &opt);                       CHKERRQ(ierr);

  ierr = MeshContextCreateMesh(ctx);                                                                      CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshContextCreateMesh"
/*@
  MeshContextCreateMesh - This function creates a mesh

  Collective on MeshContext

  Input Parameter:
. ctx - A MeshContext

  Level: beginner

.seealso MeshContextRefineMesh(), MeshContextSetup(), MeshContextCleanup()
@*/
int MeshContextCreateMesh(MeshContext ctx)
{
  MeshGeometryContext *geomCtx = &ctx->geometryCtx;
  int                  meshInfo[4], globalMeshInfo[4];
  char                 name[256];
  int                  ierr;

  PetscFunctionBegin;
  /* Setup mesh boundary */
  ierr = MeshBoundary2DCreateSimple(ctx->comm, &ctx->geometryCtx, &ctx->boundaryCtx);                     CHKERRQ(ierr);
  /* Construct the mesh */
  ierr = MeshCreate(ctx->comm, &ctx->mesh);                                                               CHKERRQ(ierr);
  ierr = MeshSetDimension(ctx->mesh, 2);                                                                  CHKERRQ(ierr);
  ierr = MeshSetPeriodicDimension(ctx->mesh, 0, geomCtx->isPeriodic[0]);                                  CHKERRQ(ierr);
  ierr = MeshSetPeriodicDimension(ctx->mesh, 1, geomCtx->isPeriodic[1]);                                  CHKERRQ(ierr);
  ierr = MeshSetNumCorners(ctx->mesh, 6);                                                                 CHKERRQ(ierr);
  ierr = MeshSetBoundary(ctx->mesh, &ctx->boundaryCtx);                                                   CHKERRQ(ierr);
  sprintf(name, "mesh.r%.4g", geomCtx->maxArea);
  ierr = PetscObjectSetName((PetscObject) ctx->mesh, name);                                               CHKERRQ(ierr);
  ierr = MeshSetFromOptions(ctx->mesh);                                                                   CHKERRQ(ierr);
  if (ctx->geometryCtx.areaCtx != PETSC_NULL) {
    ierr = MeshSetUserContext(ctx->mesh, ctx->geometryCtx.areaCtx);                                       CHKERRQ(ierr);
  } else {
    ierr = MeshSetUserContext(ctx->mesh, &ctx->geometryCtx.maxArea);                                      CHKERRQ(ierr);
  }
  ierr = MeshBoundary2DDestroy(ctx->comm, &ctx->boundaryCtx);                                             CHKERRQ(ierr);
  /* Output mesh information */
  ierr = MeshGetInfo(ctx->mesh, &meshInfo[0], &meshInfo[1], &meshInfo[2], &meshInfo[3]);                  CHKERRQ(ierr);
  ierr = MPI_Allreduce(meshInfo, globalMeshInfo, 4, MPI_INT, MPI_SUM, ctx->comm);                         CHKERRQ(ierr);
  PetscPrintf(ctx->comm, "Elements: %d Vertices: %d Nodes: %d Edges: %d\n",
              globalMeshInfo[3], meshInfo[0], globalMeshInfo[1], globalMeshInfo[2]);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshContextRefineMesh"
/*@
  MeshContextRefineMesh - This function refines the mesh

  Collective on MeshContext

  Input Parameters:
. ctx - A MeshContextContext

  Level: beginner

.seealso MeshContextCoarsenMesh(), MeshContextCreateMesh(), MeshContextCleanup()
@*/
int MeshContextRefineMesh(MeshContext ctx)
{
  MeshGeometryContext *geomCtx = &ctx->geometryCtx;
  Mesh                 refMesh;
  int                  meshInfo[4], globalMeshInfo[4];
  char                 name[256];
  int                  ierr;

  PetscFunctionBegin;
  /* Decrease feature size */
  geomCtx->maxArea *= 0.5;
  /* Construct a refined mesh */
  ierr = MeshRefine(ctx->mesh, geomCtx->areaFunc, &refMesh);                                              CHKERRQ(ierr);
  if (geomCtx->areaCtx != PETSC_NULL) {
    ierr = MeshSetUserContext(refMesh, geomCtx->areaCtx);                                                 CHKERRQ(ierr);
  } else {
    ierr = MeshSetUserContext(refMesh, &geomCtx->maxArea);                                                CHKERRQ(ierr);
  }
  ierr = MeshSetOptionsPrefix(refMesh, "ref_");                                                           CHKERRQ(ierr);
  sprintf(name, "mesh.r%.4g", geomCtx->maxArea);
  ierr = PetscObjectSetName((PetscObject) refMesh, name);                                                 CHKERRQ(ierr);
  ierr = MeshSetFromOptions(refMesh);                                                                     CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
  CHKMEMQ;
#endif
  /* Output new mesh information */
  ierr = MeshGetInfo(refMesh, &meshInfo[0], &meshInfo[1], &meshInfo[2], &meshInfo[3]);                    CHKERRQ(ierr);
  ierr = MPI_Allreduce(meshInfo, globalMeshInfo, 4, MPI_INT, MPI_SUM, ctx->comm);                         CHKERRQ(ierr);
  PetscPrintf(ctx->comm, "Elements: %d Vertices: %d Nodes: %d Edges: %d\n",
              globalMeshInfo[3], meshInfo[0], globalMeshInfo[1], globalMeshInfo[2]);

  /* Replace old mesh with refined mesh */
  ierr = MeshDestroy(ctx->mesh);                                                                          CHKERRQ(ierr);
  ctx->mesh = refMesh;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshContextCoarsenMesh"
/*@
  MeshContextCoarsenMesh - This function coarsen the mesh

  Collective on MeshContext

  Input Parameters:
. ctx - A MeshContextContext

  Level: beginner

.seealso MeshContextRefineMesh(), MeshContextCreateMesh(), MeshContextCleanup()
@*/
int MeshContextCoarsenMesh(MeshContext ctx)
{
  PetscFunctionBegin;
  SETERRQ(PETSC_ERR_SUP, "No coarsening available yet");
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshContextCleanup"
/*@
  MeshContextSetup - This function destroys all internal variables including the mesh.

  Collective on MeshContext

  Input Parameter:
. ctx - A MeshContext

  Level: beginner

.seealso MeshContextSetup(), MeshContextCreate()
@*/
int MeshContextCleanup(MeshContext ctx)
{
  int ierr;

  PetscFunctionBegin;
  ierr = MeshDestroy(ctx->mesh);                                                                          CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*---------------------------------------------- Sanity Checks ------------------------------------------------------*/

/*---------------------------------------------- Problem Callbacks --------------------------------------------------*/

/*---------------------------------------------- Main Computation ---------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MeshContextRun"
int MeshContextRun(MeshContext ctx)
{
  int loop;
  int ierr;

  PetscFunctionBegin;
  ierr = MeshContextSetup(ctx);                                                                           CHKERRQ(ierr);

  for(loop = 0; loop < ctx->numRefine; loop++) {
    ierr = MeshContextRefineMesh(ctx);                                                                    CHKERRQ(ierr);
  }
  for(loop = 0; loop < ctx->numCoarsen; loop++) {
    ierr = MeshContextCoarsenMesh(ctx);                                                                   CHKERRQ(ierr);
  }

  ierr = MeshContextCleanup(ctx);                                                                         CHKERRQ(ierr);
  PetscFunctionReturn(0);
}
