#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: discreg.c,v 1.5 2000/01/10 03:54:25 knepley Exp $";
#endif

#include "src/grid/discretization/discimpl.h"      /*I "discretization.h"  I*/

PetscFList DiscretizationList                       = PETSC_NULL;
PetscTruth DiscretizationRegisterAllCalled          = PETSC_FALSE;
PetscFList DiscretizationSerializeList              = PETSC_NULL;
PetscTruth DiscretizationSerializeRegisterAllCalled = PETSC_FALSE;

#undef __FUNCT__  
#define __FUNCT__ "DiscretizationSetType"
/*@C
  DiscretizationSetType - Sets the creation method for the disc.

  Collective on Discretization

  Input Parameters:
+ disc   - The Discretization context
- method - A known method

  Options Database Command:
. -discretization_type <method> - Sets the method; use -help for a list
                                  of available methods (for instance, tri2dlinear)

  Notes:
  See "petsc/include/discretization.h" for available methods (for instance)
. DISC_TRIANGULAR_2D_LINEAR - Triangular 2D linear discretization

  Normally, it is best to use the DiscretizationSetFromOptions() command and
  then set the Discretization type from the options database rather than by using
  this routine.  Using the options database provides the user with
  maximum flexibility in evaluating the many different solvers.
  The DiscretizationSetType() routine is provided for those situations
  where it is necessary to set the application ordering independently of the
  command line or options database.  This might be the case, for example,
  when the choice of solver changes during the execution of the
  program, and the user's application is taking responsibility for
  choosing the appropriate method.  In other words, this routine is
  not for beginners.

  Level: intermediate

.keywords: Discretization, set, type
.seealso DiscretizationSetSerializeType()
@*/
int DiscretizationSetType(Discretization disc, DiscretizationType method) {
  int      (*r)(Discretization);
  PetscTruth match;
  int        ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(disc, DISCRETIZATION_COOKIE);
  ierr = PetscTypeCompare((PetscObject) disc, method, &match);                                            CHKERRQ(ierr);
  if (match == PETSC_TRUE) PetscFunctionReturn(0);

  /* Get the function pointers for the method requested */
  if (DiscretizationRegisterAllCalled == PETSC_FALSE) {
    ierr = DiscretizationRegisterAll(PETSC_NULL);                                                         CHKERRQ(ierr);
  }
  ierr = PetscFListFind(disc->comm, DiscretizationList, method, (void (**)(void)) &r);                    CHKERRQ(ierr);
  if (!r) SETERRQ1(PETSC_ERR_ARG_WRONG, "Unknown disc type: %s", method);

  if (disc->ops->destroy != PETSC_NULL) {
    ierr = (*disc->ops->destroy)(disc);                                                                   CHKERRQ(ierr);
  }
  ierr = (*r)(disc);                                                                                      CHKERRQ(ierr);

  ierr = PetscObjectChangeTypeName((PetscObject) disc, method);                                           CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "DiscretizationGetType"
/*@C
  DiscretizationGetType - Gets the Discretization method type name (as a string).

  Not collective

  Input Parameter:
. disc - The discretization

  Output Parameter:
. type - The name of Discretization method

  Level: intermediate

.keywords: Discretization, get, type, name
.seealso DiscretizationSetType()
@*/
int DiscretizationGetType(Discretization disc, DiscretizationType *type) {
  int ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(disc, DISCRETIZATION_COOKIE);
  PetscValidPointer(type);
  if (DiscretizationRegisterAllCalled == PETSC_FALSE) {
    ierr = DiscretizationRegisterAll(PETSC_NULL);                                                         CHKERRQ(ierr);
  }
  *type = disc->type_name;
  PetscFunctionReturn(0);
}

#undef __FUNCT__  
#define __FUNCT__ "DiscretizationSetSerializeType"
/*@C
  DiscretizationSetSerializeType - Sets the serialization method for the disc.

  Collective on Discretization

  Input Parameters:
+ disc   - The Discretization context
- method - A known method

  Options Database Command:
. -discretization_serialize_type <method> - Sets the method; use -help for a list
                                            of available methods (for instance, tri2dlinear_binary)

  Notes:
  See "petsc/include/disc.h" for available methods (for instance)
. DISC_SER_TRIANGULAR_2D_LIENAR_BINARY - Triangular 2D linear discretization to binary file

  Normally, it is best to use the DiscretizationSetFromOptions() command and
  then set the Discretization type from the options database rather than by using
  this routine.  Using the options database provides the user with
  maximum flexibility in evaluating the many different solvers.
  The DiscretizationSetSerializeType() routine is provided for those situations
  where it is necessary to set the application ordering independently of the
  command line or options database.  This might be the case, for example,
  when the choice of solver changes during the execution of the
  program, and the user's application is taking responsibility for
  choosing the appropriate method.  In other words, this routine is
  not for beginners.

  Level: intermediate

.keywords: Discretization, set, type, serialization
.seealso DiscretizationSetType()
@*/
int DiscretizationSetSerializeType(Discretization disc, DiscretizationSerializeType method) {
  int      (*r)(MPI_Comm, Discretization *, PetscViewer, PetscTruth);
  PetscTruth match;
  int        ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(disc, DISCRETIZATION_COOKIE);
  ierr = PetscSerializeCompare((PetscObject) disc, method, &match);                                       CHKERRQ(ierr);
  if (match == PETSC_TRUE) PetscFunctionReturn(0);

  /* Get the function pointers for the method requested but do not call */
  if (DiscretizationSerializeRegisterAllCalled == PETSC_FALSE) {
    ierr = DiscretizationSerializeRegisterAll(PETSC_NULL);                                                CHKERRQ(ierr);
  }
  ierr = PetscFListFind(disc->comm, DiscretizationSerializeList, method, (void (**)(void)) &r);           CHKERRQ(ierr);
  if (!r) SETERRQ1(PETSC_ERR_ARG_WRONG, "Unknown discretization serialization type: %s", method);

  ierr = PetscObjectChangeSerializeName((PetscObject) disc, method);                                      CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "DiscretizationGetSerializeType"
/*@C
  DiscretizationGetSerializeType - Gets the Discretization serialization method (as a string).

  Not collective

  Input Parameter:
. disc - The discretization

  Output Parameter:
. type - The name of Discretization serialization method

  Level: intermediate

.keywords: Discretization, get, serialize, type, name
.seealso DiscretizationSetType()
@*/
int DiscretizationGetSerializeType(Discretization disc, DiscretizationSerializeType *type) {
  int ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(disc, DISCRETIZATION_COOKIE);
  PetscValidPointer(type);
  if (DiscretizationSerializeRegisterAllCalled == PETSC_FALSE) {
    ierr = DiscretizationSerializeRegisterAll(PETSC_NULL);                                                CHKERRQ(ierr);
  }
  *type = disc->serialize_name;
  PetscFunctionReturn(0);
}

/*--------------------------------------------------------------------------------------------------------------------*/
/*@C
  DiscretizationRegister - Adds a creation method to the discretization package.

  Synopsis:

  DiscretizationRegister(char *name, char *path, char *func_name, int (*create_func)(Discretization))

  Not Collective

  Input Parameters:
+ name        - The name of a new user-defined creation routine
. path        - The path (either absolute or relative) of the library containing this routine
. func_name   - The name of the creation routine
- create_func - The creation routine itself

  Notes:
  DiscretizationRegister() may be called multiple times to add several user-defined disces.

  If dynamic libraries are used, then the fourth input argument (create_func) is ignored.

  Sample usage:
.vb
  DiscretizationRegisterDynamic("my_disc", "/home/username/my_lib/lib/libO/solaris/libmy.a", "MyDiscretizationCreate", MyDiscretizationCreate);
.ve

  Then, your disc type can be chosen with the procedural interface via
.vb
    DiscretizationCreate(MPI_Comm, Discretization *);
    DiscretizationSetType(vec, "my_disc")
.ve
  or at runtime via the option
.vb
    -discretization_type my_disc
.ve

  Note: $PETSC_ARCH and $BOPT occuring in pathname will be replaced with appropriate values.

  Level: advanced

.keywords: Discretizationretization, register
.seealso: DiscretizationRegisterAll(), DiscretizationRegisterDestroy()
@*/
#undef __FUNCT__  
#define __FUNCT__ "DiscretizationRegister"
int DiscretizationRegister(const char sname[], const char path[], const char name[], int (*function)(Discretization)) {
  char fullname[256];
  int  ierr;

  PetscFunctionBegin;
  ierr = PetscStrcpy(fullname, path);                                                                     CHKERRQ(ierr);
  ierr = PetscStrcat(fullname, ":");                                                                      CHKERRQ(ierr);
  ierr = PetscStrcat(fullname, name);                                                                     CHKERRQ(ierr);
  ierr = PetscFListAdd(&DiscretizationList, sname, fullname, (void (*)(void)) function);                  CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*@C
  DiscretizationSerializeRegister - Adds a serialization method to the disc package.

  Synopsis:

  DiscretizationSerializeRegister(char *name, char *path, char *func_name,
                        int (*serialize_func)(MPI_Comm, Discretization *, PetscViewer, PetscTruth))

  Not Collective

  Input Parameters:
+ name           - The name of a new user-defined serialization routine
. path           - The path (either absolute or relative) of the library containing this routine
. func_name      - The name of the serialization routine
- serialize_func - The serialization routine itself

  Notes:
  DiscretizationSerializeRegister() may be called multiple times to add several user-defined serializers.

  If dynamic libraries are used, then the fourth input argument (serialize_func) is ignored.

  Sample usage:
.vb
  DiscretizationSerializeRegisterDynamic("my_store", "/home/username/my_lib/lib/libO/solaris/libmy.a", "MyStoreFunc", MyStoreFunc);
.ve

  Then, your serialization can be chosen with the procedural interface via
.vb
    DiscretizationSetSerializeType(disc, "my_store")
.ve
  or at runtime via the option
.vb
    -discretization_serialize_type my_store
.ve

  Note: $PETSC_ARCH and $BOPT occuring in pathname will be replaced with appropriate values.

  Level: advanced

.keywords: Discretization, register
.seealso: DiscretizationSerializeRegisterAll(), DiscretizationSerializeRegisterDestroy()
M*/
#undef __FUNCT__  
#define __FUNCT__ "DiscretizationSerializeRegister"
int DiscretizationSerializeRegister(const char sname[], const char path[], const char name[],
                                    int (*function)(MPI_Comm, Discretization *, PetscViewer, PetscTruth))
{
  char fullname[256];
  int  ierr;

  PetscFunctionBegin;
  ierr = PetscStrcpy(fullname, path);                                                                     CHKERRQ(ierr);
  ierr = PetscStrcat(fullname, ":");                                                                      CHKERRQ(ierr);
  ierr = PetscStrcat(fullname, name);                                                                     CHKERRQ(ierr);
  ierr = PetscFListAdd(&DiscretizationSerializeList, sname, fullname, (void (*)(void)) function);         CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*-------------------------------------------------------------------------------------------------------------------*/
#undef __FUNCT__  
#define __FUNCT__ "DiscretizationRegisterDestroy"
/*@C
  DiscretizationRegisterDestroy - Frees the list of creation routines for discretizations that were registered by
  DiscretizationRegister().

  Not collective

  Level: advanced

.keywords: Discretization, register, destroy
.seealso: DiscretizationRegister(), DiscretizationRegisterAll(), DiscretizationSerializeRegisterDestroy()
@*/
int DiscretizationRegisterDestroy(void) {
  int ierr;

  PetscFunctionBegin;
  if (DiscretizationList != PETSC_NULL) {
    ierr = PetscFListDestroy(&DiscretizationList);                                                        CHKERRQ(ierr);
    DiscretizationList = PETSC_NULL;
  }
  DiscretizationRegisterAllCalled = PETSC_FALSE;
  PetscFunctionReturn(0);
}

#undef __FUNCT__  
#define __FUNCT__ "DiscretizationSerializeRegisterDestroy"
/*@C
  DiscretizationSerializeRegisterDestroy - Frees the list of serialization routines for
  discretizations that were registered by DiscretizationSerializeRegister().

  Not collective

  Level: advanced

.keywords: Discretization, serialization, register, destroy
.seealso: DiscretizationSerializeRegisterAll(), DiscretizationRegisterDestroy()
@*/
int DiscretizationSerializeRegisterDestroy() {
  int ierr;

  PetscFunctionBegin;
  if (DiscretizationSerializeList != PETSC_NULL) {
    ierr = PetscFListDestroy(&DiscretizationSerializeList);                                               CHKERRQ(ierr);
    DiscretizationSerializeList = PETSC_NULL;
  }
  DiscretizationSerializeRegisterAllCalled = PETSC_FALSE;
  PetscFunctionReturn(0);
}
