#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: bdquadratic.c,v 1.4 2000/01/31 17:56:20 knepley Exp $";
#endif

/*
   Defines piecewise quadratic function space on a two dimensional 
   grid. Suitable for finite element type discretization of a PDE.
*/

#include "src/grid/discretization/discimpl.h"         /*I "discretization.h" I*/
#include "src/mesh/impls/triangular/triimpl.h"

/* For precomputed integrals, the table is structured as follows:

     precompInt[op,i,j] = \int_{SE} <op \phi^i(\xi), \phi^j(\xi)> |J^{-1}|

   where we recall that |J| is a constant for linear affine maps,
   and the map of any triangle to the standard element is linear.
   The numbering of the nodes in the standard element is

                 1--3--2
*/

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscDestroy_Triangular_2D_Quadratic"
static int BoundaryDiscDestroy_Triangular_2D_Quadratic(Discretization disc) {
  PetscFunctionBegin;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscView_Triangular_2D_Quadratic_File"
static int BoundaryDiscView_Triangular_2D_Quadratic_File(Discretization disc, PetscViewer viewer) {
  PetscFunctionBegin;
  PetscViewerASCIIPrintf(viewer, "Quadratic boundary discretization\n");
  PetscViewerASCIIPrintf(viewer, "    %d shape functions per component\n", disc->funcs);
  PetscViewerASCIIPrintf(viewer, "    %d registered operators\n", disc->numOps);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscView_Triangular_2D_Quadratic"
static int BoundaryDiscView_Triangular_2D_Quadratic(Discretization disc, PetscViewer viewer) {
  PetscTruth isascii;
  int        ierr;

  PetscFunctionBegin;
  ierr = PetscTypeCompare((PetscObject) viewer, PETSC_VIEWER_ASCII, &isascii);                            CHKERRQ(ierr);
  if (isascii == PETSC_TRUE) {
    ierr = BoundaryDiscView_Triangular_2D_Quadratic_File(disc, viewer);                                   CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscEvaluateFunctionGalerkin_Triangular_2D_Quadratic"
static int BoundaryDiscEvaluateFunctionGalerkin_Triangular_2D_Quadratic(Discretization disc, Mesh mesh, PointFunction f,
                                                                        PetscScalar alpha, int edge, PetscScalar *array, void *ctx)
{
  Mesh_Triangular *tri            = (Mesh_Triangular *) mesh->data;
  EdgeContext     *bdCtx          = (EdgeContext *)         ctx;
  void            *uCtx           = bdCtx->ctx;
  double          *nodes          = tri->nodes;
  int             *edges          = tri->edges;
  int              comp           = disc->comp;                 /* The number of components in this field */
  int              funcs          = disc->funcs;                /* The number of shape functions per component */
  PetscScalar     *funcVal        = disc->funcVal;              /* Function value at a quadrature point */
  int              numQuadPoints  = disc->numQuadPoints;        /* Number of points used for Gaussian quadrature */
  double          *quadWeights    = disc->quadWeights;          /* Weights in the standard element for Gaussian quadrature */
  double          *quadShapeFuncs = disc->quadShapeFuncs;       /* Shape function evaluated at quadrature points */
  double          *quadShapeFuncDers = disc->quadShapeFuncDers; /* Shape function derivatives at quadrature points */
  double           jac;                                         /* |J| for map to standard element */
  double           x, y;                                        /* The integration point */
  double           dxxi;                                        /* \PartDer{x}{\xi}  */
  double           dyxi;                                        /* \PartDer{y}{\eta} */
  double           coords[3*2];
  int              rank;
  int              i, j, k, p, func;
#ifdef PETSC_USE_BOPT_g
  PetscTruth       opt;
#endif
  int              ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(disc->comm, &rank);                                                                CHKERRQ(ierr);

  /* For dummy collective calls */
  if (array == PETSC_NULL) {
    for(p = 0; p < numQuadPoints; p++) {
      ierr = (*f)(0, 0, PETSC_NULL, PETSC_NULL, PETSC_NULL, PETSC_NULL, ctx);                             CHKERRQ(ierr);
    }
    PetscFunctionReturn(0);
  }

#ifdef PETSC_USE_BOPT_g
  if ((edge < 0) || (edge >= mesh->numEdges)) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE, "Invalid edge");
#endif
  /* Calculate the determinant of the inverse Jacobian of the map to the standard element */
  if (mesh->isPeriodic == PETSC_TRUE) {
    coords[0*2]   = nodes[edges[edge*2+0]*2];
    coords[0*2+1] = nodes[edges[edge*2+0]*2+1];
    coords[1*2+0] = MeshPeriodicRelativeX(mesh, nodes[edges[edge*2+1]*2],   coords[0*2+0]);
    coords[1*2+1] = MeshPeriodicRelativeY(mesh, nodes[edges[edge*2+1]*2+1], coords[0*2+1]);
    coords[2*2+0] = MeshPeriodicRelativeX(mesh, nodes[bdCtx->midnode *2],   coords[0*2+0]);
    coords[2*2+1] = MeshPeriodicRelativeY(mesh, nodes[bdCtx->midnode *2+1], coords[0*2+1]);
  } else {
    coords[0*2]   = nodes[edges[edge*2+0]*2];
    coords[0*2+1] = nodes[edges[edge*2+0]*2+1];
    coords[1*2]   = nodes[edges[edge*2+1]*2];
    coords[1*2+1] = nodes[edges[edge*2+1]*2+1];
    coords[2*2]   = nodes[bdCtx->midnode *2];
    coords[2*2+1] = nodes[bdCtx->midnode *2+1];
  }
#ifdef PETSC_USE_BOPT_g
  ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                        CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    PetscPrintf(PETSC_COMM_SELF, "[%d]edge: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
                rank, edge, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
  }
#endif

  /* Calculate element vector entries by Gaussian quadrature */
  for(p = 0; p < numQuadPoints; p++)
  {
    /* x                    = \sum^{funcs}_{f=1} x_f \phi^f(p)
       \PartDer{x}{\xi}(p)  = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\xi}
       y                    = \sum^{funcs}_{f=1} y_f \phi^f(p)
       \PartDer{y}{\xi}(p)  = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\xi} */
    x    = 0.0; y    = 0.0;
    dxxi = 0.0; dyxi = 0.0;
    for(func = 0; func < funcs; func++)
    {
      x    += coords[func*2]  *quadShapeFuncs[p*funcs+func];
      dxxi += coords[func*2]  *quadShapeFuncDers[p*funcs+func];
      y    += coords[func*2+1]*quadShapeFuncs[p*funcs+func];
      dyxi += coords[func*2+1]*quadShapeFuncDers[p*funcs+func];
    }
    if (mesh->isPeriodic == PETSC_TRUE) {
      x = MeshPeriodicX(mesh, x);
      y = MeshPeriodicY(mesh, y);
    }
    jac = sqrt(dxxi*dxxi + dyxi*dyxi);
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
                  rank, p, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
    ierr = (*f)(1, comp, &x, &y, PETSC_NULL, funcVal, uCtx);                                              CHKERRQ(ierr);
#ifdef PETSC_USE_BOPT_g
    ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                      CHKERRQ(ierr);
    if (opt == PETSC_TRUE) {
      PetscPrintf(PETSC_COMM_SELF, "[%d]p:%d jac: %g", rank, p, jac);
      for(j = 0; j < comp; j++)
        PetscPrintf(PETSC_COMM_SELF, " func[%d]: %g", j, PetscRealPart(funcVal[j]));
      PetscPrintf(PETSC_COMM_SELF, "\n");
    }
#endif

    for(i = 0, k = 0; i < funcs; i++) {
      for(j = 0; j < comp; j++, k++) {
        array[k] += alpha*funcVal[j]*quadShapeFuncs[p*funcs+i]*jac*quadWeights[p];
#ifdef PETSC_USE_BOPT_g
        ierr = PetscOptionsHasName(PETSC_NULL, "-trace_assembly", &opt);                                  CHKERRQ(ierr);
        if (opt == PETSC_TRUE) {
          PetscPrintf(PETSC_COMM_SELF, "[%d]  array[%d]: %g\n", rank, k, PetscRealPart(array[k]));
        }
#endif
      }
    }
  }
  PetscLogFlops((4 + 8*funcs + 5*funcs*comp) * numQuadPoints);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscEvaluateOperatorGalerkin_Triangular_2D_Quadratic"
static int BoundaryDiscEvaluateOperatorGalerkin_Triangular_2D_Quadratic(Discretization disc, Mesh mesh, int elemSize,
                                                                     int rowStart, int colStart, int op, PetscScalar alpha,
                                                                     int edge, PetscScalar *field, PetscScalar *mat, void *ctx)
{
  Mesh_Triangular *tri        = (Mesh_Triangular *) mesh->data;
  EdgeContext     *bdCtx      = (EdgeContext *)         ctx;
  void            *uCtx       = bdCtx->ctx;
  double          *nodes      = tri->nodes;
  int             *edges      = tri->edges;
  Operator         oper       = disc->operators[op]; /* The operator to discretize */
  Discretization   test       = oper->test;          /* The space of test functions */
  int              rowSize    = test->size;          /* The number of shape functions per element */
  int              colSize    = disc->size;          /* The number of test  functions per element */
  OperatorFunction opFunc     = oper->opFunc;        /* Integrals of operators which depend on J */
  double           coords[MAX_CORNERS*2];            /* Coordinates of the element */
  int              ierr; 

  PetscFunctionBegin;
#ifdef PETSC_USE_BOPT_g
  /* Check for valid operator */
  if ((op < 0) || (op >= disc->numOps) || (!disc->operators[op])) SETERRQ(PETSC_ERR_ARG_WRONG, "Invalid operator");
#endif

  if (opFunc == PETSC_NULL) SETERRQ(PETSC_ERR_ARG_CORRUPT, "Invalid function");
  if (mesh->isPeriodic == PETSC_TRUE) {
    coords[0*2]   = nodes[edges[edge*2+0]*2];
    coords[0*2+1] = nodes[edges[edge*2+0]*2+1];
    coords[1*2+0] = MeshPeriodicRelativeX(mesh, nodes[edges[edge*2+1]*2],   coords[0*2+0]);
    coords[1*2+1] = MeshPeriodicRelativeY(mesh, nodes[edges[edge*2+1]*2+1], coords[0*2+1]);
    coords[2*2+0] = MeshPeriodicRelativeX(mesh, nodes[bdCtx->midnode *2],   coords[0*2+0]);
    coords[2*2+1] = MeshPeriodicRelativeY(mesh, nodes[bdCtx->midnode *2+1], coords[0*2+1]);
  } else {
    coords[0*2]   = nodes[edges[edge*2+0]*2];
    coords[0*2+1] = nodes[edges[edge*2+0]*2+1];
    coords[1*2]   = nodes[edges[edge*2+1]*2];
    coords[1*2+1] = nodes[edges[edge*2+1]*2+1];
    coords[2*2]   = nodes[bdCtx->midnode *2];
    coords[2*2+1] = nodes[bdCtx->midnode *2+1];
  }

  ierr = (*opFunc)(disc, test, rowSize, colSize, rowStart, colStart, elemSize, coords, alpha, field, mat, uCtx);
  CHKERRQ(ierr);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscEvaluateALEOperatorGalerkin_Triangular_2D_Quadratic"
static int BoundaryDiscEvaluateALEOperatorGalerkin_Triangular_2D_Quadratic(Discretization disc, Mesh mesh, int elemSize,
                                                                        int rowStart, int colStart, int op, PetscScalar alpha,
                                                                        int elem, PetscScalar *field, PetscScalar *ALEfield, PetscScalar *mat,
                                                                        void *ctx)
{
  SETERRQ(PETSC_ERR_SUP, " ");
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscEvaluateNonlinearOperatorGalerkin_Triangular_2D_Quadratic"
static int BoundaryDiscEvaluateNonlinearOperatorGalerkin_Triangular_2D_Quadratic(Discretization disc, Mesh mesh,
                                                                                 NonlinearOperator f, PetscScalar alpha, int elem,
                                                                                 int numArgs, PetscScalar **field, PetscScalar *vec, void *ctx)
{
  SETERRQ(PETSC_ERR_SUP, " ");
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscEvaluateNonlinearALEOperatorGalerkin_Triangular_2D_Quadratic"
static int BoundaryDiscEvaluateNonlinearALEOperatorGalerkin_Triangular_2D_Quadratic(Discretization disc, Mesh mesh,
                                                                                    NonlinearOperator f, PetscScalar alpha, int elem,
                                                                                    int numArgs, PetscScalar **field, PetscScalar *ALEfield,
                                                                                    PetscScalar *vec, void *ctx)
{
  SETERRQ(PETSC_ERR_SUP, " ");
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryIdentity_Triangular_2D_Quadratic"
int BoundaryIdentity_Triangular_2D_Quadratic(Discretization disc, Discretization test, int rowSize, int colSize,
                                             int globalRowStart, int globalColStart, int globalSize, double *coords,
                                             PetscScalar alpha, PetscScalar *field, PetscScalar *array, void *ctx)
{
  int     numQuadPoints     = disc->numQuadPoints;     /* Number of points used for Gaussian quadrature */
  double *quadWeights       = disc->quadWeights;       /* Weights in the standard element for Gaussian quadrature */
  double *quadShapeFuncs    = disc->quadShapeFuncs;    /* Shape functions evaluated at quadrature points */
  double *quadTestFuncs     = test->quadShapeFuncs;    /* Test  functions evaluated at quadrature points */
  double *quadShapeFuncDers = disc->quadShapeFuncDers; /* Shape function derivatives evaluated at quadrature points */
  int     comp              = disc->comp;              /* The number of field components */
  int     tcomp             = test->comp;              /* The number of field components */
  int     funcs             = disc->funcs;             /* The number of shape functions */
  int     tfuncs            = test->funcs;             /* The number of test functions */
  double  dxxi;                                        /* \PartDer{x}{\xi}  */
  double  dyxi;                                        /* \PartDer{y}{\xi}  */
  double  jac;                                         /* |J| for map to standard element */
  int     i, j, c, f, p;

  PetscFunctionBegin;
#ifdef PETSC_USE_BOPT_g
  if (comp != tcomp) SETERRQ(PETSC_ERR_ARG_INCOMP, "Shape and test fields must have the same number of components");
#endif
  /* Calculate element matrix entries by Gaussian quadrature */
  for(p = 0; p < numQuadPoints; p++) {
    /* \PartDer{x}{\xi}(p)  = \sum^{funcs}_{f=1} x_f \PartDer{\phi^f(p)}{\xi}
       \PartDer{y}{\xi}(p)  = \sum^{funcs}_{f=1} y_f \PartDer{\phi^f(p)}{\xi} */
    dxxi = 0.0; dyxi = 0.0;
    for(f = 0; f < funcs; f++) {
      dxxi += coords[f*2]  *quadShapeFuncDers[p*funcs+f];
      dyxi += coords[f*2+1]*quadShapeFuncDers[p*funcs+f];
    }
    jac = sqrt(dxxi*dxxi + dyxi*dyxi);
#ifdef PETSC_USE_BOPT_g
    if (jac < 1.0e-14) {
      PetscPrintf(PETSC_COMM_SELF, "p: %d x1: %g y1: %g x2: %g y2: %g x3: %g y3: %g\n",
                  p, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5]);
      SETERRQ(PETSC_ERR_DISC_SING_JAC, "Singular Jacobian");
    }
#endif

    for(i = 0; i < tfuncs; i++) {
      for(j = 0; j < funcs; j++) {
        for(c = 0; c < comp; c++) {
          array[(i*tcomp+c+globalRowStart)*globalSize + j*comp+c+globalColStart] +=
            alpha*quadTestFuncs[p*tfuncs+i]*quadShapeFuncs[p*funcs+j]*jac*quadWeights[p];
        }
      }
    }
  }
  PetscLogFlops((4*funcs + 4 + 5*funcs*funcs*comp) * numQuadPoints);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscInterpolateField_Triangular_2D_Quadratic"
int BoundaryDiscInterpolateField_Triangular_2D_Quadratic(Discretization disc, Mesh oldMesh, int elem, double x, double y, double z,
                                                      PetscScalar *oldFieldVal, PetscScalar *newFieldVal, InterpolationType type)
{
  SETERRQ(PETSC_ERR_SUP, " ");
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscInterpolateElementVec_Triangular_2D_Quadratic"
int BoundaryDiscInterpolateElementVec_Triangular_2D_Quadratic(Discretization disc, ElementVec vec,
                                                              Discretization newDisc, ElementVec newVec)
{
  SETERRQ(PETSC_ERR_SUP, " ");
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscSetupQuadrature_Triangular_2D_Quadratic"
/*
  BoundaryDiscSetupQuadrature_Triangular_2D_Linear - Setup Gaussian quadrature with a 5 point integration rule

  Input Parameter:
. disc - The Discretization
*/
int BoundaryDiscSetupQuadrature_Triangular_2D_Quadratic(Discretization disc) {
  int dim   = disc->dim;
  int funcs = disc->funcs;
  int p;
  int ierr;

  PetscFunctionBegin;
  disc->numQuadPoints = 5;
  ierr = PetscMalloc(disc->numQuadPoints*dim       * sizeof(double), &disc->quadPoints);                  CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints           * sizeof(double), &disc->quadWeights);                 CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints*funcs     * sizeof(double), &disc->quadShapeFuncs);              CHKERRQ(ierr);
  ierr = PetscMalloc(disc->numQuadPoints*funcs*dim * sizeof(double), &disc->quadShapeFuncDers);           CHKERRQ(ierr);
  PetscLogObjectMemory(disc, (disc->numQuadPoints*(funcs*(dim+1) + dim+1)) * sizeof(double));
  disc->quadPoints[0]  = 0.0469101;
  disc->quadWeights[0] = 0.118463;
  disc->quadPoints[1]  = 0.230765;
  disc->quadWeights[1] = 0.239314;
  disc->quadPoints[2]  = 0.500000;
  disc->quadWeights[2] = 0.284444;
  disc->quadPoints[3]  = 0.769235;
  disc->quadWeights[3] = 0.239314;
  disc->quadPoints[4]  = 0.953090;
  disc->quadWeights[4] = 0.118463;
  for(p = 0; p < disc->numQuadPoints; p++) {
    /* \phi^0: (1 - \xi) (1 - 2 \xi) */
    disc->quadShapeFuncs[p*funcs+0]    = (1.0 - disc->quadPoints[p])*(1.0 - 2.0*disc->quadPoints[p]);
    disc->quadShapeFuncDers[p*funcs+0] =  4.0*disc->quadPoints[p] - 3.0;
    /* \phi^1: -\xi (1 - 2 \xi) */
    disc->quadShapeFuncs[p*funcs+1]    = -disc->quadPoints[p]*(1.0 - 2.0*disc->quadPoints[p]);
    disc->quadShapeFuncDers[p*funcs+1] =  4.0*disc->quadPoints[p] - 1.0;
    /* \phi^2: 4 \xi (1 - \xi) */
    disc->quadShapeFuncs[p*funcs+2]    =  4.0*disc->quadPoints[p]*(1.0 - disc->quadPoints[p]);
    disc->quadShapeFuncDers[p*funcs+2] =  4.0 - 8.0*disc->quadPoints[p];
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "BoundaryDiscSetupOperators_Triangular_2D_Quadratic"
/*
  BoundaryDiscSetupOperators_Triangular_2D_Quadratic - Setup the default operators

  Input Parameter:
. disc - The Discretization
*/
int BoundaryDiscSetupOperators_Triangular_2D_Quadratic(Discretization disc) {
  int newOp;
  int ierr;

  PetscFunctionBegin;
  /* The Identity operator I -- the matrix is symmetric */
  ierr = DiscretizationRegisterOperator(disc, BoundaryIdentity_Triangular_2D_Quadratic, &newOp);          CHKERRQ(ierr);
  if (newOp != IDENTITY) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", IDENTITY);
  /* The Laplacian operator \Delta -- the matrix is symmetric */
  ierr = DiscretizationRegisterOperator(disc, PETSC_NULL, &newOp);                                        CHKERRQ(ierr);
  if (newOp != LAPLACIAN) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", LAPLACIAN);
  /* The Gradient operator \nabla -- the matrix is rectangular */
  ierr = DiscretizationRegisterOperator(disc, PETSC_NULL, &newOp);                                        CHKERRQ(ierr);
  if (newOp != GRADIENT) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", GRADIENT);
  /* The Divergence operator \nabla\cdot -- the matrix is rectangular */
  ierr = DiscretizationRegisterOperator(disc, PETSC_NULL, &newOp);                                        CHKERRQ(ierr);
  if (newOp != DIVERGENCE) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", DIVERGENCE);
  /* The weighted Laplacian operator -- the matrix is symmetric */
  ierr = DiscretizationRegisterOperator(disc, PETSC_NULL, &newOp);                                        CHKERRQ(ierr);
  if (newOp != WEIGHTED_LAP) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Default operator %d not setup correctly", WEIGHTED_LAP);
  PetscFunctionReturn(0);
}

static struct _DiscretizationOps DOps = {PETSC_NULL/* DiscretizationSetup */,
                                         BoundaryDiscSetupOperators_Triangular_2D_Quadratic,
                                         PETSC_NULL/* DiscretizationSetFromOptions */,
                                         BoundaryDiscView_Triangular_2D_Quadratic,
                                         BoundaryDiscDestroy_Triangular_2D_Quadratic,
                                         BoundaryDiscEvaluateFunctionGalerkin_Triangular_2D_Quadratic,
                                         BoundaryDiscEvaluateOperatorGalerkin_Triangular_2D_Quadratic,
                                         BoundaryDiscEvaluateALEOperatorGalerkin_Triangular_2D_Quadratic,
                                         BoundaryDiscEvaluateNonlinearOperatorGalerkin_Triangular_2D_Quadratic,
                                         BoundaryDiscEvaluateNonlinearALEOperatorGalerkin_Triangular_2D_Quadratic,
                                         BoundaryDiscInterpolateField_Triangular_2D_Quadratic,
                                         BoundaryDiscInterpolateElementVec_Triangular_2D_Quadratic};

EXTERN_C_BEGIN
#undef   __FUNCT__
#define __FUNCT__ "BoundaryDiscCreate_Triangular_2D_Quadratic"
int BoundaryDiscCreate_Triangular_2D_Quadratic(Discretization disc) {
  int sarg;
  int ierr;

  PetscFunctionBegin;
  if (disc->comp <= 0) {
    SETERRQ(PETSC_ERR_ARG_WRONG, "Discretization must have at least 1 component. Call DiscretizationSetNumComponents() to set this.");
  }
  ierr = PetscMemcpy(disc->ops, &DOps, sizeof(struct _DiscretizationOps));                                CHKERRQ(ierr);
  disc->dim   = 2;
  disc->funcs = 3;
  disc->size  = disc->funcs*disc->comp;

  ierr = DiscretizationSetupDefaultOperators(disc);                                                       CHKERRQ(ierr);
  ierr = BoundaryDiscSetupQuadrature_Triangular_2D_Quadratic(disc);                                       CHKERRQ(ierr);

  /* Storage */
  ierr = PetscMalloc(disc->comp * sizeof(PetscScalar),   &disc->funcVal);                                 CHKERRQ(ierr);
  ierr = PetscMalloc(2          * sizeof(PetscScalar *), &disc->fieldVal);                                CHKERRQ(ierr);
  for(sarg = 0; sarg < 2; sarg++) {
    ierr = PetscMalloc(disc->comp*(disc->dim+1) * sizeof(PetscScalar), &disc->fieldVal[sarg]);             CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}
EXTERN_C_END
