/* $Id: partition.h,v 1.16 2000/10/17 13:49:33 knepley Exp $ */

/*
    Routines for working with mesh partitions
*/

#ifndef __PARTITION_H
#define __PARTITION_H

#include "mesh.h"

/*
  A partition object constructs and applys a partition of a mesh object.
*/
typedef struct _Partition *Partition;

#define PARTITION_TRIANGULAR_1D "tri1d"
#define PARTITION_TRIANGULAR_2D "tri2d"
typedef char *PartitionType;

#define PARTITION_SER_TRIANGULAR_1D_BINARY "tri_1d_binary"
#define PARTITION_SER_TRIANGULAR_2D_BINARY "tri_2d_binary"
typedef char *PartitionSerializeType;

/* Logging support */
extern int PARTITION_COOKIE;

/* Generic Operations */
extern int PartitionView(Partition, PetscViewer);
extern int PartitionSerialize(Mesh, Partition *, PetscViewer, PetscTruth);
extern int PartitionSetUp(Partition);
extern int PartitionCopy(Partition, Partition );
extern int PartitionDuplicate(Partition, Partition *);
extern int PartitionSetFromOptions(Partition);
extern int PartitionPrintHelp(Partition);
extern int PartitionDestroy(Partition);

/* Partition-Specific Operations */
extern int PartitionGhostNodeExchange(Partition, InsertMode, ScatterMode, int *, int *);

/* Partition Query Interface */
  /* partition query functions */
extern int PartitionGetDimension(Partition, int *);
extern int PartitionGetMesh(Partition, Mesh *);
extern int PartitionSetOptionsPrefix(Partition, char *);
extern int PartitionAppendOptionsPrefix(Partition, char *);
extern int PartitionGetOptionsPrefix(Partition, char **);

  /* element query functions */
extern int PartitionGetTotalElements(Partition, int *);
extern int PartitionGetStartElement(Partition, int *);
extern int PartitionGetEndElement(Partition, int *);
extern int PartitionGetNumElements(Partition, int *);
extern int PartitionGetNumOverlapElements(Partition, int *);
extern int PartitionGlobalToLocalElementIndex(Partition, int, int *);
extern int PartitionLocalToGlobalElementIndex(Partition, int, int *);
extern int PartitionGetElementOrdering(Partition, AO *);

  /* node query functions */
extern int PartitionGetTotalNodes(Partition, int *);
extern int PartitionGetStartNode(Partition, int *);
extern int PartitionGetEndNode(Partition, int *);
extern int PartitionGetNumNodes(Partition, int *);
extern int PartitionGetNumOverlapNodes(Partition, int *);
extern int PartitionGlobalToLocalNodeIndex(Partition, int, int *);
extern int PartitionLocalToGlobalNodeIndex(Partition, int, int *);
extern int PartitionGlobalToGhostNodeIndex(Partition, int, int *, int *);
extern int PartitionGhostToGlobalNodeIndex(Partition, int, int *, int *);
extern int PartitionGetNodeOrdering(Partition, AO *);

  /* face query functions */
extern int PartitionGetTotalFaces(Partition, int *);
extern int PartitionGetStartFace(Partition, int *);
extern int PartitionGetEndFace(Partition, int *);
extern int PartitionGetNumFaces(Partition, int *);
extern int PartitionGetNumOverlapFaces(Partition, int *);
extern int PartitionGlobalToLocalFaceIndex(Partition, int, int *);
extern int PartitionLocalToGlobalFaceIndex(Partition, int, int *);
extern int PartitionGetFaceOrdering(Partition, AO *);

  /* edge query functions */
extern int PartitionGetTotalEdges(Partition, int *);
extern int PartitionGetStartEdge(Partition, int *);
extern int PartitionGetEndEdge(Partition, int *);
extern int PartitionGetNumEdges(Partition, int *);
extern int PartitionGetNumOverlapEdges(Partition, int *);
extern int PartitionGetEdgeOrdering(Partition, AO *);

/* Dynamic creation and loading functions */
extern PetscFList PartitionList;
extern int PartitionRegisterAll(const char []);
extern int PartitionRegisterDestroy(void);
extern int PartitionRegisterAllCalled;
extern int PartitionSetType(Partition, PartitionType);
extern int PartitionGetType(Partition, PartitionType *);
extern int PartitionRegister_Private(const char[],const char[],const char[],int(*)(Partition));
extern int PartitionSerializeRegister_Private(const char [], const char [], const char [], int (*)(Mesh, Partition *, PetscViewer, PetscTruth));
#if defined(PETSC_USE_DYNAMIC_LIBRARIES)
#define PartitionRegister(a,b,c,d) PartitionRegister_Private(a,b,c,0)
#define PartitionSerializeRegister(a,b,c,d) PartitionSerializeRegister_Private(a,b,c,0)
#else
#define PartitionRegister(a,b,c,d) PartitionRegister_Private(a,b,c,d)
#define PartitionSerializeRegister(a,b,c,d) PartitionSerializeRegister_Private(a,b,c,d)
#endif

extern PetscFList PartitionSerializeList;
extern int PartitionSerializeRegisterAll(const char []);
extern int PartitionSerializeRegisterDestroy(void);
extern int PartitionSerializeRegisterAllCalled;
extern int PartitionSetSerializeType(Partition, PartitionSerializeType);

/* Partition Creation Functions */
extern int PartitionCreate(Mesh, Partition *);
extern int PartitionCreateTriangular2D(Mesh, Partition *);

#endif /* __PARTITION_H */
