/*
 * STV0680 Vision Camera Chipset Driver
 * Copyright (C) 2000 Adam Harrison <adam@antispin.org> 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA   
 *
 *  Modifications and additions by Kevin Sisson <kjsisson@bellsouth.net>
 *  November, 2001
 *
 */

#include <malloc.h>
#include <stdio.h>
#include <string.h>

#include <stdlib.h>
#include <term.h>
#include <ctype.h>
#include <math.h>
#include <usb.h>

#include "pencam.h"

#define RED 0
#define GREEN 1
#define BLUE 2
#define AD(x, y, w) (((y)*(w)+(x))*3)


void bayer_unshuffle( unsigned char *raw, unsigned char *output, usb_stv *stv680)
{
    int x, y, i;
    int w = stv680->cwidth;
    int vw=stv680->cwidth, vh = stv680->cheight, vstep=1;
    unsigned int p=0;
    int colour = 0, bayer = 0;
//    float R=0, G=0, B=0;
        
    if ((stv680->vwidth == 640) || (stv680->vwidth==644)) {
	vw = 640;  vh = 480;  vstep = 1;
    }
    if ((stv680->vwidth == 322) || (stv680->vwidth==320)) {
	vw = 320;  vh = 240;  vstep = 1;
    }
    if ((stv680->vwidth == 352)) {
	vw = 352;  vh = 288;  vstep = 1;
    }
    if ((stv680->vwidth == 160)) {
	vw = 160;  vh = 120;  vstep = 2;
    }
    if ((stv680->vwidth == 176)) {
	vw = 176;  vh = 144;  vstep = 1;
    }
    memset(output, 0, 3*vw*vh);  /* clear output matrix. Maybe not necessary. */

    /*  raw bayer data: 1st row, 1st half are the odd pixels (same color), 
	2nd half (w/2) are the even pixels (same color) for that row 
	top left corner of sub array is always green */

    for (y=0; y<vh; y++) {
	for (x=0; x<vw; x++) {

	    switch (vstep) {
		case 1: {
		    if (x&1)
	    		p = *(raw + y*w + (x>>1));
		    else
			p = *(raw + y*w + (x>>1) + (w>>1));
		    break;
		}
		case 2: {
		    if (x&1)
	    		p = *(raw + ((y*w)<<1) + x);
		    else
			p = *(raw + ((y*w)<<1) + x + (w>>1));
		    break;
		}
	    }
	    
	    if (y&1)
		bayer = 2;
	    else
		bayer = 0;
	    if (x&1)
		bayer++;

	    switch(bayer) {
		case 0:
		case 3: colour = 1; break;
		case 1: colour = 0; break;
		case 2: colour = 2; break;
	    }
	    i = (y*vw + x)*3;  /* output already zeroed out with memset */

	    /* sample a small region of pic of white object to get correction values */
/*	    if ((y&1) && (!(x&1)) && (x<=4) && (y<=5))
		R += p;		
	    if ((!(y&1)) && (x&1) && (x<=4) && (y<=5))
		B += p;		
	    if ((!(y&1)) && (!(x&1)) && (x<=4) && (y<=5))
		G += p;		
	    if ((y&1) && (x&1) && (x<=4) && (y<=5))
		G += p;		
	    if ((x==5) && (y==5)) {
		R = R/9;
		B = B/9;
		G = G/18;
		fprintf(stdout, "R=%i  G=%i  B=%i\n", (int)R, (int)G, (int)B);
	    }
*/		
	    *(output + i + colour) = (unsigned char) p;
	} /* for x */
    }  /* for y */

}  /* bayer_unshuffle */

void bayer_demosaic(usb_stv *stv680, unsigned char *output)
{
    int x, y, bayer;
    int vw=stv680->vwidth, vh=stv680->vheight;
    
    for (y=1; y<(vh-1); y++) { 
        for (x=1; x<(vw-1); x++) {   /* work out pixel type */
	    if (y&1)
		bayer = 0;
	    else
		bayer = 2;
	    if (!(x&1))
		bayer++;

	    switch(bayer) {

    	    case 0:        /* green. blue lr, red tb */
		*(output + AD(x,y,vw) + BLUE) = ((int)*(output + AD(x-1,y,vw) + BLUE) + (int)*(output + AD(x+1,y,vw) + BLUE))>>1;
		*(output + AD(x,y,vw) + RED) = ((int)*(output + AD(x,y-1,vw) + RED) + (int)*(output + AD(x,y+1,vw) + RED))>>1;
		break;
		    
	    case 1:        /* blue. green lrtb, red diagonals */
		*(output + AD(x,y,vw) + GREEN) = ((int)*(output + AD(x-1,y,vw) + GREEN)+(int)*(output + AD(x+1,y,vw) + GREEN)+(int)*(output + AD(x,y-1,vw) + GREEN)+(int)*(output + AD(x,y+1,vw) + GREEN))>>2;
		*(output + AD(x,y,vw) + RED) = ((int)*(output + AD(x-1,y-1,vw) + RED)+(int)*(output + AD(x-1,y+1,vw) + RED)+(int)*(output + AD(x+1,y-1,vw) + RED)+(int)*(output + AD(x+1,y+1,vw) + RED))>>2;
		break;
		    
	    case 2:        /* red. green lrtb, blue diagonals */
		*(output + AD(x,y,vw) + GREEN) = ((int)*(output + AD(x-1,y,vw) + GREEN)+(int)*(output + AD(x+1,y,vw) + GREEN)+(int)*(output + AD(x,y-1,vw) + GREEN)+(int)*(output + AD(x,y+1,vw) + GREEN))>>2;
		*(output + AD(x,y,vw) + BLUE) = ((int)*(output + AD(x-1,y-1,vw) + BLUE)+(int)*(output + AD(x+1,y-1,vw) + BLUE)+(int)*(output + AD(x-1,y+1,vw) + BLUE)+(int)*(output + AD(x+1,y+1,vw) + BLUE))>>2;
		break;
		    
	    case 3:        /* green. red lr, blue tb */
		*(output + AD(x,y,vw) + RED) = ((int)*(output + AD(x-1,y,vw) + RED)+(int)*(output + AD(x+1,y,vw) + RED))>>1;
		*(output + AD(x,y,vw) + BLUE) = ((int)*(output + AD(x,y-1,vw) + BLUE)+(int)*(output + AD(x,y+1,vw) + BLUE))>>1;
		break;
	    }  /* switch */
	}   /* for x */
    }  /* for y */
}  /* bayer_demosaic */


/****** gamma correction from trans[], plus hardcoded white balance */
/* Thanks to Alexander Schwartx <alexander.schwartx@gmx.net> for this code.
   Gamma correction (trans[] values generated by (pow((i-17)/256, GAMMA)*255)
   where GAMMA=0.45, 1<i<255.  */

#define GAMMA 0.450
#define ZERO 17.0

void light_enhance(usb_stv *stv680, unsigned char *output)
{
    unsigned long int i;
    int vw=stv680->vwidth, vh=stv680->vheight;
    int lt=stv680->light_type;
    float wb[3][3];
    double x, y;
    unsigned int trans[258], v;    
    
    fprintf(stdout," Downloaded picture %i  ", stv680->pic_no);

    /* -1=skip */
    if (stv680->light_type != 3) 
	fprintf(stdout, "\n");
    if (stv680->light_type == -1) 
	return;

    /* 3=auto; decide which type: 0=natural, 1=fluorescent, 2=incandescent */
    if (stv680->light_type == 3) {
    	fprintf(stdout, "(F_Exp=%i  C_Exp=%i  APV=%i => filter = ",
		stv680->FineExp, stv680->CoarseExp, stv680->AvgPixVal); 

	if (stv680->FineExp >= (stv680->CoarseExp<<2)) {
	    lt = 0;
	    fprintf(stdout, "Natural)\n");
//	} else if ((stv680->FineExp <= 5) && (stv680->CoarseExp >= 468)
//		  && ((stv680->AvgPixVal <= 91) || (stv680->AvgPixVal >= 98))) {
//	    lt = 1;
//	    fprintf(stdout, "Fluorescent)\n");
	} else if ((stv680->FineExp <= 5) && (stv680->CoarseExp <= 399)
		    && (stv680->CoarseExp >= 90)) {
	    lt = 2;
	    fprintf(stdout, "Incandescent)\n");
	} else {
	    lt = 1;
	    fprintf(stdout, "Fluorescent)\n");
	}
    }

    x = stv680->brightness;
    /*  white balance filter numbers */
    wb[0][0] = 1.08 * x;  wb[0][1] = 1.00 * x;  wb[0][2] = 0.95 * x;
    wb[1][0] = 1.05 * x;  wb[1][1] = 1.00 * x;  wb[1][2] = 1.00 * x;
    wb[2][0] = 0.90 * x;  wb[2][1] = 1.00 * x;  wb[2][2] = 1.10 * x;
	    
    for(i=1; i<256; ++i)
    {
	x=i; x -= ZERO;
	if (x<1.0)
	    x = 1.0;
	y = pow((x/256.0),GAMMA)*255.0; 	
	trans[i] = (unsigned int) y;
    }
    trans[0] = 0;

    for (i=0;i<(vw*vh*3);i+=3)
    {   
	y = trans[*(output+i)]*wb[lt][0]; v = (unsigned int)y;
	*(output+i) = (unsigned char) (v > 255) ? 255 : v;
	y = trans[*(output+i+1)]*wb[lt][1]; v = (unsigned int)y;
	*(output+i+1) = (unsigned char) (v > 255) ? 255 : v;
	y = trans[*(output+i+2)]*wb[lt][2]; v = (unsigned int)y;
	*(output+i+2) = (unsigned char) (v > 255) ? 255 : v;
    }  /* for */

}  /* light_enhance  */


