/*****************************************************************************
 *                                                                           *
 * Program:   paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     delbordr.c                                                     *
 *            Funktions to cut of the "dark" border of an image              *
 * Author:    Andreas Tille                                                  *
 * Date:      27.02.1998                                                     *
 * Copyright: Andreas Tille, 1999; GNU Public License                        *
 *                                                                           *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "paul.h" 

typedef void (*CountThreshFunc)(PICTURE *, int, int, int *, int *, int, int);

static int TestLeftRight(int *lb, int *rb, int width, int pa, int pleft, int stp, int *found)
/* test, whether left and ap point to different rows
 * --- Parameter: ---
 * int           *lb    : field of left borders
 * int           *rb    : field of right borders
 * int            width : width of image
 * int            pa    : current position
 * int            pleft : beginn of critical (potential) border area
 * int            stp   : store as 3 byte or 1 byte per pixel
 * int            found : number of found valid rows
 * --- Return: ---
 * int           *lb    : field of left borders with new entry
 * int           *rb    : field of right borders with new entry
 * int TestLeftRight()  : if valid border area found assummed offset to a possible next 
 *                        border (half of image width), else 0
 */
{
  register int i, j, storepix = stp;

  if ( storepix != 1 ) {
    pa    /= storepix;
    pleft /= storepix;
  }
  if ( (pa / width) > (pleft / width) ) {
    ++*found;
    i = (*(rb + *found) = pleft % width) -
        (*(lb + *found) = pa    % width);   
    j = (i=MIN(i >> 1, width >> 2)) % storepix;
    return i-j; /* make sure, that a green byte is used */
  }
  return 0;
}

static int WeightedMiddle(int *middle, int abw, int *data, int len)
/* calculate average of data, which doesn't differ more than a certain value
 * from the precalculated average (excludes much to high or much to low values)
 * --- Parameter: ---
 * int *middle : precalculated average
 * int  abw    : use only values with abs(*middle - value) < abw to calculate the weighted middle
 * int *data   : data
 * int  len    : number of data
 * --- Return: ---
 * int *middle : new, "weighted" average
 * int WeightedMiddle(): != 0, if there where less than 25% of the data valid
 */
{
  register int *ap, orimiddle = *middle, oriabw = abw, sum = 0, nsum = 0, *fip;
   
  for ( fip = (ap = data) + len; ap < fip; ap++ ) {
    if ( abs(*ap - orimiddle) < oriabw ) {
      sum += *ap;
      nsum++;
    }
  }
  if ( nsum < (len >> 2) ) return nsum;
  *middle = sum / nsum;
  return 0;
}

static void CountThreshCol(PICTURE *bild, int col, int testcols, 
                    int *nthresh, int *nterr, int thresh, int terr)
/* count all values of the environment of a column, which are greater than thresh and terr
 * --- Parameter: ---
 * PICTURE *bild       : image
 * int      col        : column to test
 * int      testcols   : number of columns to test
 * int     *nthresh    : field to count number of values greater than thresh for each column
 * int     *nterr      : field to count number of values greater than terr for each column
 * unsigned char thresh: threshold to call a value "dark"
 * unsigned char terr  : allowed tolerance for a single pixel
 * --- Return: ---
 * int     *nthresh    : field to count number of values greater than thresh for each column
 * int     *nterr      : field to count number of values greater than terr for each column
 * int CountThreshCol(): new column, -1 in case of error
 */
{
  register int *ip, *jp, *fip, storepix = bild->storepix;
  register unsigned char *ap, *fap;
  int      step = bild->W;

  step = storepix*(step - testcols); 
       /* bild->storepix*testcols byte will be handled in inner loop! */
  for ( fip = (ip = nthresh) + testcols, jp = nterr; 
        ip < fip; ip++, jp++ ) 
    *ip = *jp = 0;

  for ( fap = bild->DATA + storepix*bild->size, 
        ap  = bild->DATA + storepix*(col - (testcols>>1)) + (storepix == 3 ? 1 : 0);  
                                                             /* +1 take GREEN!! */
        ap < fap; ap += step ) {
    for ( fip = (ip = nthresh) + testcols, jp = nterr; 
          ip < fip; ip++, jp++, ap += storepix ) {         /* +3 take GREEN!! */
      if ( *ap > thresh ) {
        ++*ip;
        if ( *ap > terr ) ++*jp;
      }
    }
  }
}

static void CountThreshRow(PICTURE *bild, int row, int testrows, 
                    int *nthresh, int *nterr, int thresh, int terr)
/* count all values of the environment of a row, which are greater than thresh and terr
 * --- Parameter: ---
 * PICTURE *bild       : image
 * int      row        : row to test
 * int      testrows   : number of rows to test
 * int     *nthresh    : field to count number of values greater than thresh for each row
 * int     *nterr      : field to count number of values greater than terr for each row
 * unsigned char thresh: threshold to call a value "dark"
 * unsigned char terr  : allowed tolerance for a single pixel
 * --- Return: ---
 * int     *nthresh    : field to count number of values greater than thresh for each row
 * int     *nterr      : field to count number of values greater than terr for each row
 * int CountThreshRow(): new column, -1 in case of error
 */
{
  register int           *ip, *jp, storepix = bild->storepix;
  register unsigned char *bp, *fbp, *ap, *fap;
  int                    *fip;

  for ( fip = (ip = nthresh) + testrows, jp = nterr; 
        ip < fip; ip++, jp++ ) 
    *ip = *jp = 0;

  for ( ip = nthresh, jp = nterr, ap = bild->DATA + storepix*(row - (testrows>>1)),
        fap = bild->DATA + storepix*bild->size; 
        ip < fip; ip++, jp++ ) 

  ap = bild->DATA + storepix*(row - (testrows>>1)) + (storepix == 3 ? 1 : 0); /*+1 take GREEN!!*/
  for ( fap = ap + storepix*testrows*bild->W, ip = nthresh, jp = nterr; 
        ap < fap; ap = fbp, ip++, jp++ ) {
    for ( fbp = (bp = ap) + storepix * bild->W; bp < fbp; bp += storepix ) { /*+3 take GREEN!!*/
      if ( *bp > thresh ) {
        ++*ip;
        if ( *bp > terr ) ++*jp;
      }
    }
  }
}

static int CheckBorderColumn(PICTURE *bild, int *col, unsigned char thresh, 
                             unsigned char terr, int dir, int flag)
/* test environment of a certain column for "darkness"
 * IF bild->storepix == 3 ONLY GREEN (Byte 2) WILL BE TESTET!!!!!!!!!!!!!!!!!!!
 * --- Parameter: ---
 * PICTURE *bild          : image
 * int     *col           : column to test
 * unsigned char thresh   : threshold to call a value "dark"
 * unsigned char terr     : allowed tolerance for a single pixel
 * int      dir           : direction, where to search for the border 
 *                          (if *col = *left then dir = -1)
 * int      flag          : how often was CheckBorderColumn() called recursively
 * --- Return: ---
 * int     *col           : new column, if it matches border better
 * int CheckBorderColumn(): new column, -1 in case of an error
 */
{
#define TESTCOLS 11        /* how many columns to test in one step? */
#define NITER    10        /* how many recursive iterations         */
  int                    nthresh[TESTCOLS], nterr[TESTCOLS], 
                         tolthr, tolerr, neucol;
  register int           *ip, *jp, *fip, i;

  if ( flag > NITER ) return -1;
    /* No valid border after more than NITER recursive iterations! */

  if ( *col <  TESTCOLS>>1           ) *col = TESTCOLS>>1;             /* exact match of left border */
  if ( *col >= bild->W-(TESTCOLS>>1) ) *col = bild->W-(TESTCOLS>>1)-1; /* exact match of right border */
  CountThreshCol(bild, *col, TESTCOLS, nthresh, nterr, thresh, terr);

  /*   tolthr = bild->W >> 5;   1/32 of data is allowed to fail the threshold test           */
  /*   tolerr = tolthr >> 3;    1/256 of data is allowed to fail the single pixel error test */
  tolthr = bild->W >> 4;  /* 1/16  of data is allowed to fail the threshold test          */
  tolerr = tolthr >> 3;   /* 1/128 of data is allowed to fail the single pixel error test */
  if ( tolthr < 4 ) tolthr = 4;
  if ( tolerr < 1 ) tolerr = 1;
  /* ... useful and sometimes necessary for small images!! */
    
  neucol = -1;
  for ( ip = nthresh, jp = nterr, 
        fip = nthresh + TESTCOLS, i = *col - (TESTCOLS>>1);
        ip < fip; ip++, jp++, i++ ) {
    if ( *ip < tolthr && *jp < tolerr ) {
      if ( dir > 0 ) {
        if ( ip == nthresh )  /* schon am Anfang nicht im Bild => gehe nach links */
          i = (neucol = TESTCOLS) + 1;
        else
          neucol = i;
        break;
      } else {  /* linker Rand */
        neucol = i;
      }
    }
  }

  if ( neucol == --i || neucol < 0 ) {
    neucol = (neucol < 0 ? *col + dir*(TESTCOLS-1) : *col - dir*(TESTCOLS-1));
    if ( CheckBorderColumn(bild, &neucol, thresh, terr, dir, ++flag) < 0 ) return -1;
  }

  return *col = neucol;
#undef TESTCOLS
#undef NITER
}

static int NewBorder(PICTURE *bild, int oldwd, int neuwd, int *left, int *right, 
                     unsigned char thresh, unsigned char terr, CountThreshFunc ctf)
/* Linker/oberer und rechter/unterer Rand eines Bildes werden noch etwas korrigiert,
 * damit bereinstimmung mit anderen Bildern erzielt wird
 * --- Parameter: ---
 * PICTURE         *bild       : Bild
 * int             oldwd       : alte Gesamtbreite/-hhe
 * int             neuwd       : Vorgabe der zu erreichenden Breite/Hhe
 * int             *left       : linke/obere Grenze
 * int             *right      : rechte(untere Grenze
 * unsigned char   thresh      : Schwellwert fr "dunkel"
 * unsigned char   terr        : Toleranz fr einzelne Werte
 * CountThreshFunc *ctf        : Funktion zur Berechnung der Anzahl von Fehlern in Spalten/Zeilen
 * --- Return: ---
 * int             *left       : neue linke/obere Grenze
 * int             *right      : neue rechte/untere Grenze
 * int             NewBorder() : 0 fr OK
 */
{
#define NEXTCOL(bor, a, e, z)  { bor += z; a += z; e += z; } 

   register int  *al, *ar, *el, *er, zw;
   int      delta, *nthl, *ntel, *nthr, *nter, width;
   
   if ( (delta = abs(neuwd - *right + *left)) > (oldwd/20) || 
        *left >= *right || delta == 0 ) /* delta kann nicht == 0, aber Vorsicht */
      return delta;                     /* hat noch nie geschadet               */

   if ( (delta & 1) == 0 ) delta++;  /* ungeradzahlig geht besser! */

   /* delta+2, weil links und rechts von delta gesucht wird!!!! */
   nthl = g_new0(int, (delta+2) << 2 );
   nter = (nthr = (ntel = nthl + delta) + delta) + delta;
   
   if ( (zw = *left - (delta/2) ) < delta/2 ) {
      zw = delta/2 + 1;
      al = nthl + *left;
      el = ntel + *left;
   } else {
      al = nthl + (delta/2) + 1;
      el = ntel + (delta/2) + 1;
   }
   (*ctf)(bild, zw, delta, nthl, ntel, thresh, terr);

   if ( (zw = *right + (delta>>1) ) > (width = oldwd) ) {
      zw = width - (delta>>1) - 1;
      ar = nthr + delta - width + *right;
      er = nter + delta - width + *right;
   } else {
      ar = nthr + (delta>>1) + 1;
      er = nter + (delta>>1) + 1;
   }
   (*ctf)(bild, zw, delta, nthr, nter, thresh, terr);

   if ( neuwd - *right + *left < 0 ) zw = -1;  /* Bild verkleinern */
   else                              zw = +1;  /* Bild vergrern  */
   
   width--;  /* Im Folgenden, mu mit width-1 verglichen werden! */
   do {
      if ( *al == *ar ) {
	 if ( *el <= *er ) { /* in beide Richtungen gleich ==>   *
                              * arbeite vorzugsweise oBdA. links */
	    if ( zw == +1 ) {
               if ( *left > 0 ) {
                  NEXTCOL(*left,  al, el, -1);
	       } else {
		  NEXTCOL(*right, ar, er, +1);
	       }
	    } else {
               NEXTCOL(*left, al, er, +1);
	    }
	 } else {  /* ( *el > *er ) */
            if ( zw == +1 ) {
               if ( *right < width ) {
                  NEXTCOL(*right, ar, er, +1);
               } else {
		  NEXTCOL(*left,  al, el, -1);
	       }
	    } else {
               NEXTCOL(*right, ar, er, -1);
	    }
	 }
      } else {
	 if ( *al < *ar ) {
	    if ( zw == +1 ) {
               if ( *left > 0 ) {
                  NEXTCOL(*left,  al, el, -1);
	       } else {
		  NEXTCOL(*right, ar, er, +1);
	       }
	    } else {
               NEXTCOL(*left, al, er, +1);
	    }
	 } else {  /* ( *al > *ar ) */
            if ( zw == +1 ) {
               if ( *right < width ) {
                  NEXTCOL(*right, ar, er, +1);
               } else {
		  NEXTCOL(*left,  al, el, -1);
	       }
	    } else {
               NEXTCOL(*right, ar, er, -1);
	    }
	 }
      }
   } while ( *right - *left != neuwd );
   FREE(nthl);
   return 0;

#undef NEXTCOL   
}


static int GetBorder(PICTURE *bild, int *left, int *right, unsigned char thresh, 
                   unsigned char terr)
/* Rechter und linker Rand eines Bildes werden abgeschnitten.
 * Als "Rand" werden "gleichmig dunkle" rechteckige Gebiete links und rechts
 * einer tatschlich Information enthaltenden Flche bezeichnet.
 * Dabei heit "dunkel", da die Pixel kleiner sind als die Schwelle thresh.
 * Dabei drfen "einzelne" Werte auch einen Wert bis zu terr erreichen.
 * --- Parameter: ---
 * PICTURE *bild        : Bild
 * int      *left       : Zeiger auf linke Grenze (erste Spalte innerhalb des Bildes)
 * int      *right      : Zeiger auf rechte Grenze (erste Spalte auerhalb des Bildes)
 * unsigned char thresh : Schwellwert fr "dunkel"
 * unsigned char terr   : Toleranz fr einzelne Werte
 * --- Return: ---
 * int      *left       : linke Grenze
 * int      *right      : rechte Grenze
 * int      GetBorder() : RET_OK or RET_ERR
 */
{
   register unsigned char *ap, *fip;
   register int flag = 0, bad = 0, *lbp, *rbp, storepix;
   unsigned char *tleft, *start = bild->DATA;
   int      width = bild->W, *lb, *rb, makesense, dobad, foundlines = 0, *fop;
   char     *nonsense = _("No valid image borders can be found using thresh = %i and terr = %i (%s).");

   storepix   = bild->storepix;
   *left      = *right = 0;
   tleft      = start;
   lb         = g_new0(int, bild->H << 1);
   rb         = lb + (makesense = bild->H);
   makesense *= 100;
   
   if ( storepix == 3 ) 
      ap = start + (width << 1) - (width << 1)%3 + 1;
      /* Anfang erst bei 2/3 => man erhlt nur einen dunklen Bereich, *
       * damit wird auch gleich das grne Byte getroffen              */
   else
      ap = start + (width << 1);

   dobad = (width > 1024 ? 1 : 0);
      /* Wenn die Bilder gro sind, ist auch die Auflsung der Fehler hher! *
       * Daher wird noch ein Pixel weiter geschaut                           */

   for ( fip = bild->DATA + storepix*(bild->size - dobad); 
         ap < fip; ap += storepix ) {
      if ( flag ) {             /* Verdacht auf Punkt im Randstreifen */
         if ( *ap >= thresh ) { /* Fehler im Rand oder Bildpunkt???   */
            if ( bad ) {        /* vorangegangener Punkt auch schon   */
	       if ( dobad )     /* Falls groes Bild, betrachte auch  */
                  if ( *(ap += storepix) < terr ) { /* ... nchsten Punkt noch    */
                     if ( *ap < thresh ) {
                        bad = 0;   /* War doch nix mit Bildanfang     */
                        continue;  /* Also weiter im Rand             */
		     }
		     if ( ++bad < 4 ) {  /* HIER DREHEN !!!!!!!!!!!!   */
                        continue;  /* mal sehen                       */
		     }
		  }

	       ap += (flag = TestLeftRight(lb, rb, width, ap-start, tleft-start, storepix, &foundlines));
	       if ( !flag )     /* Wenn Verdacht auf Rand nicht besttigt ... */
                  if ( !--makesense ) break;  /* merken bzw. Abbruch  */
	       flag = 0;        /* In jedem Fall jetzt wieder im Bild */
	       bad  = 0;        /* ... und kein Verdacht mehr         */
	    } else {            /* erster Punkt der Serie >= thresh   */
	       if ( *ap < terr ) bad = 1;   /* Knnte Fehler sein     */
	       else              flag = 0;  /* Ist doch im Bild       */
	    }
	 }
      } else {                  /* innerhalb des eigentlichen Bildes  */
         if ( *ap < thresh ) {
            flag  = 1;
            tleft = ap;
	 }
      }
   }
   if ( !makesense ) {
      g_warning(nonsense, thresh, terr, _("first scan failed"));
      FREE(lb);
      return RET_ERR;
   }
   if ( bild->H < 3*foundlines ) {
      flag = bad = 0;  /* Registervariablen ausnutzen zum Summen speichern. */   
      for ( fop = (lbp = lb) + foundlines, rbp = rb; lbp < rb; lbp++, rbp++ ) {
         flag += *lbp; 
         bad  += *rbp;
      }
      *left  = flag / foundlines;
      *right = bad  / foundlines;
      if ( !*right || (!*left && *right == width) ) {
         g_warning(nonsense, thresh, terr, _("invalid border"));
         FREE(lb);
         return RET_ERR;
      }

      /* Noch einmal Durchschnitt ermitteln, aber Ausreier herauslassen! */
      flag = MIN(*left, width/20);
      flag = MIN(width-*right, flag);  /* alles auerhalb flag ist Abweichung */
      if ( flag < width/25 ) flag = width / 20;  /* Zahlen sind rein empirisch!!! */
   
      if ( WeightedMiddle(left,  flag, lb, foundlines) )
         WeightedMiddle(left,  flag <<= 1, lb, foundlines);
      if ( WeightedMiddle(right, flag, rb, foundlines) ) 
         WeightedMiddle(right, flag <<=1, rb, foundlines);

      flag >>= 1;  /* und noch einmal mit der halben Abweichung!  Das bringt was!!   *
                    * Ein Fehlertest ist nun aber nicht mehr ntig (eher schdlich!) */
      WeightedMiddle(left,  flag, lb, foundlines);
      WeightedMiddle(right, flag, rb, foundlines);
   } else {
      if ( width > 20 ) {
         *left  = 5;
	 *right = width - 5;
      } else {
         g_warning(nonsense, thresh, terr, _("to less valid rows"));
         FREE(lb);
         return RET_ERR;
      }
   }
   
   /* Nun noch die Randspalten testen */
   flag = *left; bad = *right;

   if ( CheckBorderColumn(bild, left, thresh, terr, -1, 0) < 0 ||
        CheckBorderColumn(bild, right, thresh, terr, 1, 0) < 0 ) 
      return -1;

   if ( *left >= *right ) {
      g_warning(nonsense, thresh, terr, _("vertical control scan"));
      FREE(lb);
      *left = flag; *right = bad;
      return RET_ERR;
   }
   ++*left;    /* Linker Rand soll innerhalb des Bildes sein!! */
   FREE(lb);
   return RET_OK;
}

int CreateCheckLine(PICTURE *bild, int *lb, int *rb)
/* erzeugt eine invertierte Linie an dem neu ermittelten Rand und
 * erweitert beide Rnder um fnf Spalten
 * --- Parameter: ---
 * PICTURE *bild        : Bild
 * int      *lb         : Zeiger auf linke Grenze
 * int      *rb         : Zeiger auf rechte Grenze
 * --- Return: ---
 * int      *lb         : linke Grenze
 * int      *rb         : rechte Grenze
 * int CreateCheckLine(): neue Breite
 */
{
#define REST 10
   register unsigned char *lp, *rp, *fip;
   register int w = (bild->storepix)*bild->W;

   for ( fip = bild->DATA + bild->storepix * bild->size, 
         lp  = bild->DATA + bild->storepix * *lb,
         rp  = bild->DATA + bild->storepix * *rb;
         lp < fip; lp += w, rp += w ) {
      *lp     = ~(*lp);
      *(lp+1) = ~(*(lp+1));
      *(lp+2) = ~(*(lp+2));      
      *rp     = ~(*rp);
      *(rp+1) = ~(*(rp+1));
      *(rp+2) = ~(*(rp+2));      
   }
   if ( *lb > REST ) *lb -= REST;
   else              *lb  = 0;
   if ( *rb < bild->W - REST ) *rb += REST;
   else                        *rb  = bild->W;
   return *rb - *lb;
#undef REST
}

static int GetTopBottom(PICTURE *bild, int *top, int *bot, unsigned char thresh, 
                   unsigned char terr)
/* Oberer und unterer Rand eines Bildes werden abgeschnitten.
 * Als "Rand" werden "gleichmig dunkle" rechteckige Gebiete links und rechts
 * einer tatschlich Information enthaltenden Flche bezeichnet.
 * Dabei heit "dunkel", da die Pixel kleiner sind als die Schwelle thresh.
 * Dabei drfen "einzelne" Werte auch einen Wert bis zu terr erreichen.
 * --- Parameter: ---
 * PICTURE  *bild          : Bild
 * int      *top           : Zeiger auf obere Grenze (erste Zeile innerhalb des Bildes)
 * int      *bot           : Zeiger auf untere Grenze (erste Zeile auerhalb des Bildes)
 * unsigned char thresh    : Schwellwert fr "dunkel"
 * unsigned char terr      : Toleranz fr einzelne Werte
 * --- Return: ---
 * int      *top           : obere Grenze
 * int      *bot           : untere Grenze
 * int      GetTopBottom() : RET_OK or RET_ERR
 */
{
   register unsigned char *bp, *fbp;
   unsigned char     *ap, *fip;
   register int      storepix;
   int               ip, jp, tolthr, tolerr, dobad;

   storepix = bild->storepix;
   /*   tolthr = bild->W >> 5;   1/32 darf drber liegen     */
   /*   tolerr = tolthr >> 3;    1/256 darf ber Fehler sein */
   tolthr = bild->W >> 4;  /* 1/16 darf drber liegen     */
   tolerr = tolthr >> 3;   /* 1/128 darf ber Fehler sein */
   if ( tolthr < 4 ) tolthr = 4;
   if ( tolerr < 1 ) tolerr = 1;
   /* ... damit es bei kleinen Bildern auch klappt!! */

   dobad = 0;
   if ( bild->H > 512 ) {
      if ( bild->H > 1024 ) dobad = 2;
      else                  dobad = 1;
   }
      /* Wenn die Bilder gro sind, ist auch die Auflsung der Fehler hher! *
       * Daher wird noch ein Pixel weiter geschaut                           */
   
   /* oben */
   for ( fip = (ap = bp = bild->DATA + (storepix == 3 ? 1 : 0)) + (bild->storepix * bild->size); 
         ap < fip; ap = fbp) {
      for ( fbp = ( bp = ap ) + storepix*bild->width, ip = jp = 0; bp < fbp; bp += storepix ) 
         if ( *bp > thresh ) {
            if ( dobad ) {
               if ( ((bp += dobad * storepix) < fbp) && (*bp > thresh) ) {
                  ++ip;
                  if ( *bp > terr ) ++jp;
	       }
	    } else {
               ++ip;
               if ( *bp > terr ) ++jp;
	    }
         }
      if ( ip > tolthr || jp > tolerr ) break;
   }
   if ( (*top = ((ap-bild->DATA)/storepix)/bild->W) ) ++*top; /* +1 da innerhalb des Bildes gesucht */

   /* unten */
   for ( ap = bp = (fip = bild->DATA) - (storepix == 3 ? 2 : 1) + (bild->storepix * bild->size); 
         ap > fip; ap = fbp) {
      for ( fbp = ( bp = ap ) - storepix*bild->width, ip = jp = 0; bp > fbp; bp -= storepix ) 
         if ( *bp > thresh ) {
            if ( dobad ) {
               if ( ((bp -= dobad * storepix) > fbp) && (*bp > thresh) ) {
                  ++ip;
                  if ( *bp > terr ) ++jp;
	       }
	    } else {
               ++ip;
               if ( *bp > terr ) ++jp;
	    }
         }
      if ( ip > tolthr || jp > tolerr ) break;
   }
   *bot = ((ap-bild->DATA+2)/storepix)/bild->W;
   if ( *top == 0 && *bot == bild->H ) return RET_ERR;
   return RET_OK;
}

int SelectPictureWithoutBorder(PAUL *p)
/* cut border of images
 * "Borders" are defined as "permanent dark" rectangular areas left and right as well
 * as top and bottom of a information containing rectangular area.
 * "Dark" means all pixels are lass than the threshold "thresh" but single
 * pixels can have values up to the error threshold "terr"
 * --- parameters: ---
 * PAUL *p                           : list of images, options
 *                                     used options:
 *                                     thresh : threshold for dark pixels
 *                                     terr   : threshold for single (noisy) pixels
 *                                     flag   : used, if only left and right border should be cut
 * --- return: ---
 * int   SelectPictureWithoutBorder(): RET_OK or RET_ERR
 */
{
  PICTURE        *bild;
  register GList *pl, *piclist;
  int            *lb, *rb, *wd, *testlr, *tb, *bb, *testtb, i, h, neuwd, neuht = 0;
  char           *lrdesc, *tbdesc, *desc;
   
  g_return_val_if_fail ( IS_PAUL(p), RET_ERR );
  g_return_val_if_fail ( CheckPicList(p->piclist), RET_ERR );
  piclist = p->piclist;

  lb     = g_new0(int, 7*(i = NBILDER(piclist)));
  testtb = (testlr = ( wd = ( bb = ( tb = (rb = lb + i)+i)+i)+i)+i)+i;

  /* Crop left and right border */   
  h = neuwd = 0;

  for ( bild = BILD(pl = piclist), i = 0; pl; bild = BILD(pl = pl->next), i++ ) {
    if ( (*(testlr+i) = GetBorder(bild, lb+i, rb+i, p->opt->thresh, p->opt->terr)) == RET_OK ) {
      neuwd += (*(wd+i) = *(rb+i) - *(lb+i));
      h++;
    }
  }
  if ( h > 1 )   /* let the average be a little bit more if unsure */
    neuwd = (int)( neuwd + ((3*h)>>2) ) / h;   
  for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next)) 
    neuwd = MIN(neuwd, bild->W);
   
  for ( bild = BILD(pl = piclist), i = 0; pl; bild = BILD(pl = pl->next), i++ ) {
    /* delete border left and right */
    if ( *(testlr+i) ) continue;
    if ( *(wd+i) > neuwd ) {
      if ( NewBorder(bild, bild->W, neuwd, lb+i, rb+i, p->opt->thresh, p->opt->terr, CountThreshCol) ) {
        g_warning(_("Width of %s can't be changed to %i."), bild->file, neuwd);
        *(testlr+i) = -1;  /* mark as unchanged in left-right direction */
        continue; 
      }
    } else {
      *(testlr+i) = -1;  /* mark as unchanged in left-right direction */
      continue;
    }

    *(wd+i) = neuwd;  /* WITHOUT CHECKLINE!!!! */
    /*      *(wd+i) = CreateCheckLine(bild, lb+i, rb+i);  */

    CropImage(bild, *(lb+i), 0, *(wd+i), bild->H);
  }

  if ( !OnlyLeftRight(p->opt->f) ) {
    /* Crop top and bottom border */
    h = neuht = 0;
    for ( bild = BILD(pl = piclist), i = 0; pl; bild = BILD(pl = pl->next), i++ ) {
      if ( (*(testtb+i) = GetTopBottom(bild, tb+i, bb+i, p->opt->thresh, p->opt->terr)) == RET_OK ) {
        neuht += (*(wd+i) = *(bb+i) - *(tb+i));
        h++;
      }
    }
    if ( h > 1 )   /* Average and a little bit more ... */
      neuht = (int)( neuht + ((3*h)>>2) ) / h;   
    for ( bild = BILD(pl = piclist), i = 0; pl; bild = BILD(pl = pl->next), i++ ) {
      neuht = MIN(neuht, bild->H);
      if ( *(testtb+i) && (neuht < bild->H) && NBILDER(piclist) > 1 ) {
        g_warning(_("No change of height for all images because %s can't be cut vertically."), bild->file);
        *(testtb+i) = -1;  /* mark as unchanged in top-bottom direction */
        neuht = 0;
        break;
      } else {
        *(testtb+i) = -1;  /* mark as unchanged in left-right direction */
        continue;
      }
    }

    if ( neuht ) {
      for ( bild = BILD(pl = piclist), i = 0; pl; bild = BILD(pl = pl->next), i++ ) {
        if ( *(wd+i) != neuht ) {
          if ( NewBorder(bild, bild->H, neuht, tb+i, bb+i, p->opt->thresh, p->opt->terr, CountThreshRow) ) {
            g_warning(_("Height of %s can't be changed to %i."), bild->file, neuht);
            *(testtb+i) = -1;  /* mark as unchanged in top-bottom direction */
            continue; 
          }
        }
        *(wd+i) = neuht;  /* WITHOUT CHECKLINE!!!! */
        /*      *(wd+i) = CreateVerticalCheckLine(bild, tb+i, bb+i);  */

        CropImage(bild, 0, *(tb+i), bild->W, *(wd+i));
      }
    }
  }

  /* set spezification chunks in the end */
  for ( bild = BILD(pl = piclist), i = 0; pl; bild = BILD(pl = pl->next), i++ ) {
    if ( (*(testlr+i) && !neuht) || (*(testlr+i) && *(testtb+i)) ) {
      /* unterschiedliche Testbedingungen ... mal sehn, ob das gut ist*/
      g_warning(_("%s remains unchanged."), bild->file);
      continue;
    }
    if ( !*(testlr+i) ) lrdesc = g_strdup_printf("(left = %i, right = %i)", *(lb+i), *(rb+i));
    else                lrdesc = g_strdup("");
    if ( !*(testtb+i) ) tbdesc = g_strdup_printf("(top = %i, bottom = %i)", *(tb+i), *(bb+i));
    else                tbdesc = g_strdup("");

    if ( DelScanBorder(p->opt->f) ) {
      desc = g_strdup_printf("White scanner border of %s deleted: %s %s", 
                              ImgFileName(bild), lrdesc, tbdesc);
      ImgChunksUpdate(bild, TypDelScanRand, desc, APPDELSCANBORDER, DELSCANBORDER);
      p->opt->f  &= ~DELSCANBORDER;
    } else {
      desc = g_strdup_printf("New border of %s: (+%i-%i): %s %s", 
                            ImgFileName(bild), p->opt->thresh, p->opt->terr, lrdesc, tbdesc);
      ImgChunksUpdate(bild, TypDelRand, desc, APPDELBORDER, DELBORDER);
      p->opt->f  &= ~DELBORDER;
    }
    FREE(lrdesc);
    FREE(tbdesc);
    FREE(desc);
  }

  FREE(lb);
  return RET_OK;
}

