#include <stdlib.h>


#include "Prefs.h"
#include "SystemColour.h"
#include "mudclient.h"

Prefs::Prefs(MUD * mud) {
  pref_list = NULL;
  initialiseDefaults();

  load(mud);
}

Prefs::~Prefs() {
  struct pref_data * tmp_next;
  for (struct pref_data * tmp = pref_list; tmp; tmp = tmp_next) {
    tmp_next = tmp->next;
    
    free(tmp->name);
    free(tmp->data);
    free(tmp);
  }
}

void Prefs::initialiseDefaults() {

  // Category = Proxies
  setPreferenceBoolean("ProxyTurfHTTPd", false);
  setPreferenceBoolean("ProxyHTTPConnect", false);
  setPreferenceBoolean("ProxySocks", false);
  setPreferenceBoolean("ProxySocks4", false);
  setPreferenceBoolean("HTTPProxy", false);
  setPreference("HttpProxyHost", "wwwcache");
  setPreferenceInteger("HttpProxyPort", 3128);
  setPreference("SocksHost", "socks");
  setPreferenceInteger("SocksPort", 1080);
  setPreference("SocksUser", "Username");
  setPreference("SocksPass", "Password");

  // Category = Connection
  // @@ Add a DISABLE AUTORECONNECT FOR THIS MUD OPTION in AutoReconnect.
  setPreferenceBoolean("ConnectionAutoReconnect", true);
  setPreferenceBoolean("Compression", true);
  setPreference("CompressionOffCommand", "config -mccp");
  // Category = Random Stuff
  setPreference("Browser", "mozilla");

  // Category = Colours
  setPreferenceInteger("DefaultForegroundColour", 7);
  setPreferenceInteger("DefaultBackgroundColour", 8);
  setPreferenceBoolean("EnableColour", true);
  setPreferenceBoolean("BoldColours", true);

  setPreference("White_RGB", "52428 52428 52428");
  setPreference("Red_RGB", "52428 0 0");
  setPreference("Green_RGB", "0 52428 0");
  setPreference("Blue_RGB", "0 0 52428");
  setPreference("Yellow_RGB", "52428 52428 0");
  setPreference("Magenta_RGB", "52428 0 52428");
  setPreference("Cyan_RGB", "0 52428 52428");
  setPreference("Black_RGB", "0 0 0");

  setPreference("White_Bold_RGB", "65535 65535 65535");
  setPreference("Red_Bold_RGB", "65535 0 0");
  setPreference("Green_Bold_RGB", "0 65535 0");
  setPreference("Blue_Bold_RGB", "0 0 65535");
  setPreference("Yellow_Bold_RGB", "65535 65535 0");
  setPreference("Magenta_Bold_RGB", "65535 0 65535");
  setPreference("Cyan_Bold_RGB", "0 65535 65535");
  setPreference("Black_Bold_RGB", "0 0 0");

  setPreference("Background_RGB", "0 0 0");

  setPreference("Echo_RGB", "0 52428 0");

  // Category = Fonts
  setPreference("PromptFont", "monospace 12");
  setPreference("InputFont", "monospace 12");
  setPreference("OutputFont", "monospace 12");
  setPreferenceBoolean("UseBoldFont", true);
  setPreference("BoldFont", "monospace bold 12");
  setPreferenceBoolean("UseItalicFont", true);
  setPreference("ItalicFont", "monospace italic 12");

  // Category = History and Tab Completion
  setPreferenceBoolean("HistoryIgnoreRepeats", false);
  setPreferenceBoolean("HistoryUseLimit", false);
  setPreferenceInteger("HistoryLimit", 100);
  setPreferenceBoolean("HistoryIsCircular", false);
  setPreferenceBoolean("HistoryHonourNull", true);

  // Category = Terminal
  setPreferenceBoolean("LocalEcho", true);
  setPreferenceBoolean("DoublePrompt", false);
  setPreferenceBoolean("SemiColon", false);
  setPreferenceBoolean("clearinput", false);
  setPreferenceBoolean("NegotiateCharMode", true);
  setPreferenceBoolean("FilteredOutput", false); // @@ MOVE TO PLUGIN
  setPreferenceBoolean("PromptOnInputLine", true);
  setPreferenceBoolean("Beep", true);
  setPreference("charset", "ISO-8859-1");
  setPreferenceBoolean("UseMaxLines", false);
  setPreferenceInteger("MaxLines", 1000);
  setPreferenceBoolean("ScrollOnInput", false);
  setPreferenceBoolean("ScrollbackLimit", false);
  setPreferenceInteger("ScrollbackLength", 1000);
  setPreferenceBoolean("UseToolbar", false);
  setPreferenceBoolean("SmoothScrolling", true);
  setPreferenceInteger("WindowX", 800);
  setPreferenceInteger("WindowY", 600);
  setPreferenceBoolean("UsePromptTerminator", false);
  setPreference("PromptTerminator", ">>>");

  // Category - Keepalive
  setPreferenceBoolean("SendKeepalive", false);
  setPreferenceInteger("SendKeepaliveTimer", 10);
  setPreference("SendKeepaliveText", "");

}

struct pref_data * Prefs::findPreference(char * prefName) {
  for (struct pref_data * tmp = pref_list; tmp; tmp = tmp->next) {
    if (!strcmp(prefName, tmp->name))
      return tmp;
  }
  return NULL;
}

void Prefs::setPreference(char * prefName, char * prefData) {
  struct pref_data * data = findPreference(prefName);

  if (!data) {
    data = (struct pref_data *)malloc(sizeof(struct pref_data));
    memset(data, 0, sizeof(struct pref_data));
    data->name = strdup(prefName);
    data->next = pref_list;
    pref_list = data;
  }

  if (data->data)
    free(data->data);
  data->data = strdup(prefData);
}

void Prefs::setPreferenceInteger(char * prefName, int prefData) {
  char buf[1024];

  snprintf(buf, 1024, "%d", prefData);
  setPreference(prefName, buf);
}

void Prefs::setPreferenceLong(char * prefName, unsigned long prefData) {
  char buf[1024];

  snprintf(buf, 1024, "%llu", prefData);
  setPreference(prefName, buf);
}  

void Prefs::setPreferenceBoolean(char * prefName, bool prefData) {
  setPreferenceInteger(prefName, (bool) prefData);
}

char * Prefs::getPreference(char * prefName) {
  struct pref_data * data = findPreference(prefName);
  if (data)
    return data->data;

  return "";
}

int Prefs::getPreferenceInteger(char * prefName) {
  char * tmp = getPreference(prefName);
  if (tmp)
    return atoi(tmp);
  return 0;
}

bool Prefs::getPreferenceBoolean(char * prefName) {
  char * tmp = getPreference(prefName);
  if (tmp)
    return (bool) atoi(tmp);
  return false;
}

unsigned long Prefs::getPreferenceLong(char * prefName) {
  char * tmp = getPreference(prefName);
  if (tmp)
    return (unsigned long) atol(tmp);
  return false;
}

void Prefs::savePrefs(FILE * fp) {
  for (struct pref_data * tmp = pref_list; tmp; tmp = tmp->next) {
    fprintf(fp, "%s %s\n", tmp->name, (char *)tmp->data);
  }
}

bool Prefs::parseLine(FILE * fp) {

  char buffer[4096];
  char key[1024];
  char value[1024];

  while (fgets(buffer, 4096, fp)) {
    value[0] = '\0';
    key[0] = '\0';
    if (sscanf(buffer, "%1023s %1023[^\n]\n", key, value) != 2) {
      // No longer return false, we accept non-matching strings as having
      // null values.  Hopefully only string-based preferences will not match.
      setPreference(key, value);
      continue;
    }

    setPreference(key, value);
  }

  return FALSE;
}

bool Prefs::load(MUD * mud) {

  FILE * fp = (FILE *)NULL;

  char filename[1024];

  if (!mud || !mud->getOwnPrefs())
    snprintf(filename, 1024, "papayarc");
  else
    snprintf(filename, 1024, "prefs/%s", mud->getName());

  // Open papayarc from user's home dir, or system install dir
  fp = openFile(filename, (char *)NULL, "rb");
  if (!fp) {
    debug(_("WARNING: couldn't load preferences."));
    debug(CRLF);
    return FALSE;
  }

  while (parseLine(fp));
  fclose(fp);
  return TRUE;
}

bool Prefs::save(MUD * mud) {
  
  char filename[1024];
  filename[0] = '\0';

  if (mud && mud->getPreferences())
    snprintf(filename, 1024, "prefs/%s", mud->getName());
  else
    snprintf(filename, 1024, "papayarc");

  return save(filename);
}

bool Prefs::save(char * filename) {
  FILE * fp = (FILE *)NULL;

  // Open file in user's ~/.papaya directory to write to
  fp = openFile(filename, (char *)NULL, "wb");
  if (!fp) {
    debug(_("WARNING: couldn't save preferences\n"));
    return FALSE;
  }
  savePrefs(fp);

  fclose(fp);
  return TRUE;
}
