/*
**  LocalStore.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**  
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**  
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#import <Pantomime/LocalStore.h>

#import <Pantomime/Constants.h>
#import <Pantomime/LocalFolder.h>
#import <Pantomime/URLName.h>

@implementation LocalStore

//
//
//
- (void) dealloc
{
  RELEASE(path);
  RELEASE(folders);
  RELEASE(fileManager);
  
  [super dealloc];
}


//
//
//
- (id) initWithPathToDirectory: (NSString *) thePath
{
  BOOL isDirectory;
  
  self = [super init];
  
  [self setPath: thePath];
  
  // We initialize our cache of opened folders
  folders = [[NSMutableDictionary alloc] init];
  
  fileManager = [NSFileManager defaultManager];
  RETAIN(fileManager);

  if ([fileManager fileExistsAtPath: [self path]
		   isDirectory: &isDirectory])
    {
      if (! isDirectory)
	{
	  AUTORELEASE(self);
	  return nil;
	}
    }
  else
    {
      AUTORELEASE(self);
      return nil;
    }


  return self;
}


//
//
//
- (id) initWithURL: (NSString *) theURL;
{
  URLName *urlName;

  urlName = [[URLName alloc] initWithString: theURL];

  self = [self initWithPathToDirectory: [urlName path]];

  RELEASE(urlName);
  
  return self;
}

//
// This method will open automatically Inbox (case-insensitive).
// It may return nil if the opening failed or Inbox wasn't found.
//
- (id) defaultFolder
{
  return [self folderForName: @"Inbox"];
}


//
// This method is used to open the folder theName in the current
// directory of this local store.
//
- (id) folderForName: (NSString *) theName
{
  NSEnumerator *anEnumerator;
  NSString *aString;
  Folder *cachedFolder;

  anEnumerator = [self folderEnumerator];
  cachedFolder = [folders objectForKey: theName];
  
  if ( [self folderForNameIsOpen: theName] )
    {
      return nil;
    }
  
  if (! cachedFolder )
    {
      while ( (aString = [anEnumerator nextObject]) )
	{
	  if ( [aString compare: theName] == NSOrderedSame )
	    {
	      LocalFolder *aFolder;
	      
	      aFolder = [[LocalFolder alloc] initWithPathToFile:
					       [NSString stringWithFormat:@"%@/%@",
							 [self path], aString]];
	      
	      if ( aFolder )
		{
		  [aFolder setStore: (Store *)self];
		  [aFolder setName: theName];
		  [aFolder parse];

		  // We now cache it and return it
		  [folders setObject: AUTORELEASE(aFolder)
			   forKey: theName];
		}
	      
	      return aFolder;
	    }
	}
    return nil; // Never reached?
    }
  else
    {
      return cachedFolder;
    }
}

//
//
//
- (id) folderForURL: (NSString *) theURL;
{
  URLName *urlName;
  id aFolder;

  urlName = [[URLName alloc] initWithString: theURL];

  aFolder = [self folderForName: [urlName foldername]];

  RELEASE(urlName);
  
  return aFolder;
}

//
// This method returns the list of folders contained in 
// a specific directory. It'll currently ignore some things
// like Netscape Mail's summary files and Pantomime's local
// cache files.
//
- (NSEnumerator *) folderEnumerator
{
  NSDirectoryEnumerator *aDictionaryEnumerator;
  NSMutableArray *aMutableArray;
  NSString *aString;
  
  aMutableArray = [[NSMutableArray alloc] initWithArray:
					    [[fileManager enumeratorAtPath: 
							    [self path]] allObjects]];
  AUTORELEASE(aMutableArray);

  aDictionaryEnumerator = [fileManager enumeratorAtPath: [self path]];
  
  // We iterate through our array. If mbox A and .A.summary (or .A.cache) exists, we
  // remove .A.summary (or .A.cache) from our mutable array.
  while ( (aString = [aDictionaryEnumerator nextObject]) )
    {     
      [aMutableArray removeObject: [NSString stringWithFormat: @".%@.summary", aString]];
      [aMutableArray removeObject: [NSString stringWithFormat: @".%@.cache", aString]];
    }
  
  return [aMutableArray objectEnumerator];
}


- (NSString *) path
{
  return path;
}


- (void) setPath: (NSString *) thePath
{
  RETAIN(thePath);
  RELEASE(path);
  path = thePath;
}

- (void) close
{
  NSEnumerator *anEnumerator;
  NSString *aName;

  anEnumerator = [self openedFoldersEnumerator];

  while ( (aName = [anEnumerator nextObject]) )
    {
      LocalFolder *aFolder;

      aFolder = (LocalFolder *)[self folderForName: aName];
      [aFolder close];
    }
}

- (NSEnumerator *) openedFoldersEnumerator
{
  return [folders keyEnumerator];
}

- (void) removeFolderFromOpenedFolders: (LocalFolder *) theFolder
{
  [folders removeObjectForKey: [theFolder name]];
}

- (BOOL) folderForNameIsOpen: (NSString *) theName
{
  NSEnumerator *anEnumerator;
  NSString *aName;
  
  anEnumerator = [self openedFoldersEnumerator];

  while ( (aName = [anEnumerator nextObject]) )
    {
      if ( [aName compare: theName] == NSOrderedSame)
	{
	  return YES;
	}
    }

  return NO;
}


//
//
//
- (BOOL) createFolderWithName: (NSString *) theName
{
  NSString *aName;
  BOOL aBOOL;

  NSEnumerator *anEnumerator;

  anEnumerator = [self folderEnumerator];

  // We verify if the folder with that name does already exist
  while ( (aName = [anEnumerator nextObject]) )
    {
      if ([aName compare: theName
		 options: NSCaseInsensitiveSearch] == NSOrderedSame)
	{
	  return NO;
	}
    }

  aBOOL = [[NSFileManager defaultManager] createFileAtPath: [NSString stringWithFormat: @"%@/%@",
								      [self path], theName]
					  contents: nil 
					  attributes: nil ];
  return aBOOL;
}

//
//
//
- (BOOL) deleteFolderWithName: (NSString *) theName
{
  BOOL aBOOL;
  
  // We remove the mbox
  aBOOL = [[NSFileManager defaultManager] removeFileAtPath: [NSString stringWithFormat: @"%@/%@",
								      [self path], theName]
					  handler: nil];
  
  // We remove the cache
  [[NSFileManager defaultManager] removeFileAtPath: [NSString stringWithFormat: @"%@/.%@.cache",
							      [self path], theName]
				  handler: nil];
  
  return aBOOL;
}

@end
