//
// C++ Implementation: packagesearchimpl
//
// Description: 
//
//
// Author: Benjamin Mesing <bensmail@gmx.net>, (C) 2004
//
// Copyright: See COPYING file that comes with this distribution
//
//

#include "packagesearchimpl.h"

#include <iostream>
#include <algorithm>
#include <cassert>


#include <qaction.h>
#include <qapplication.h>
#include <qclipboard.h>
#include <QCloseEvent>
#include <qcursor.h>
#include <qdir.h>
#include <qicon.h>
#include <QList>
#include <q3listview.h>
#include <qmessagebox.h>
#include <qpixmap.h>
#include <qstatusbar.h>
#include <QStringList>
#include <qtabwidget.h>
#include <QTime>
#include <QToolBar>
#include <QWhatsThis>

#if 0
#include <apt-front/cache/cache.h>
#endif
#include <ept/configuration/apt.h>
#include <ept/cache/apt/packages.h>
#include <wibble/operators.h>


// NApplication
#include "applicationfactory.h"
#include "runcommand.h"
#include "runcommandforoutput.h"


// namespace NPlugin 
#include "plugin.h"
#include "searchplugin.h"
#include "actionplugin.h"
#include "informationplugin.h"
#include "shortinformationplugin.h"
#include "plugincontainer.h"
#include "pluginmanager.h"
#include "packagenotfoundexception.h"
#include "plugincompare.h"
#include "packagenameplugin.h"
#include "scoredisplayplugin.h"

// NXml
#include "xmldata.h"

// NWidget
#include "customizablelistview.h"

// NException
#include "exception.h"

#include "extalgorithm.h"

// NUtil
#include "helpers.h"

#include "packagesearchaboutdlg.h"
#include "packagelistviewitem.h"

#include "eptinstance.h"


using namespace std;

template <typename PluginType> 
bool LesserPriority<PluginType>::operator()(PluginType* p1, PluginType* p2)
{
	return (*p1) < (*p2);
}

template class LesserPriority<NPlugin::SearchPlugin>;
template class LesserPriority<NPlugin::InformationPlugin>;
template class LesserPriority<NPlugin::ShortInformationPlugin>;


PackageSearchImpl::PackageSearchImpl( QWidget* parent, const char* name, Qt::WFlags fl )
: QMainWindow(parent, fl), _DETAILS_INFORMATION_PRIORITY(2)
{
	setName(name);
	setupUi(this);
	// former ui.h initialization
	init();
	_pPackageNamePlugin = 0;
	_pScoreDisplayPlugin = 0;
	_settingsFilename = QDir::homeDirPath()+"/.packagesearch";
	_pInformationDetailsPage = _pInformationContainer->page(0);

	_pPackageListViewControl = new NWidget::CustomizableListView(_pPackageView);
	// this might be overwritten when loading the settings, but this is wanted
	_pPackageListViewControl->setSorting("Score", Qt::Descending);
	connect(_pPackageListViewControl, SIGNAL(shownColumnsChanged()), this, SLOT(onShortColumnsChanged()));
	
	vector<string> pluginDirectories;
	pluginDirectories.push_back("plugins/");
	pluginDirectories.push_back("/usr/lib/packagesearch/");
	_pPluginManager = new NPlugin::PluginManager(pluginDirectories, this);
	_pPluginManager->addPluginUser(this);
	loadSettings();	// must be called after setting _pPluginManager but before calling loadPlugins


	QIcon backIcon(_iconDir + "back.png");
	QIcon forwardIcon(_iconDir + "forward.png");
	_pBackButton->setIcon(backIcon);
	_pForwardButton->setIcon(forwardIcon);
    connect(_pPackageView, SIGNAL(selectionChanged(Q3ListViewItem*)), this, SLOT(onPackageSelectionChanged(Q3ListViewItem*)));
    connect(_pClearSearchButton, SIGNAL(clicked()), this, SLOT(onClearSearch()));
    connect(helpAboutAction, SIGNAL(activated()), this, SLOT(onHelpAboutAction()));
    connect(helpContentsAction, SIGNAL(activated()), this, SLOT(onHelpContentsAction()));
    connect(_pBackButton, SIGNAL(clicked()), this, SLOT(onBack()));
    connect(_pForwardButton, SIGNAL(clicked()), this, SLOT(onForward()));
    connect(_pFileExitAction, SIGNAL(activated()), this, SLOT(close()));
    connect(_pDetailsView, SIGNAL(sourceChanged(const QUrl&)), this, SLOT(onLinkClicked(const QUrl&)));
    connect(_pInformationContainer, SIGNAL(currentChanged(QWidget*)), this, SLOT(onInformationPageChanged(QWidget*)));
    connect(_pControlPluginsAction, SIGNAL(activated()), this, SLOT(onControlPlugins()));
    connect(_pPluginSettingsAction, SIGNAL(activated()), this, SLOT(onPluginSettings()));
    connect(_pPackageSelection, SIGNAL(activated(const QString&)), this, SLOT(selectNewPackage(const QString&)));
}

PackageSearchImpl::~PackageSearchImpl()
{
	delete _pPackageNamePlugin;
	delete _pScoreDisplayPlugin;
}


void PackageSearchImpl::init()
{
	{ // search for the icon and doc pathes
		if  ( QFile("../doc/content.html").exists() )
		{
			QDir docDir = QDir::current();
			docDir.cd("../doc/");
			_docDir = docDir.absPath()+"/";
		}
		else if ( QFile("doc/content.html").exists() )
		{
			QDir docDir = QDir::current();
			docDir.cd("doc/");
			_docDir = docDir.absPath()+"/";
		}
		else if ( QFile("/usr/share/doc/packagesearch/content.html").exists() )
			_docDir = "/usr/share/doc/packagesearch/";
		else
			cerr << "Could not find doc directory. It must be located at "
			"/usr/share/doc/packagesearch/, ../doc/ or doc/."<<endl;
		if ( QFile("../icons/packagesearch.png").exists() )
			_iconDir = "../icons/";
		else if ( QFile("icons/packagesearch.png").exists() )
			_iconDir = "icons/";
		else if ( QFile("/usr/share/pixmaps/packagesearch/packagesearch.png").exists() )
			_iconDir = "/usr/share/pixmaps/packagesearch/";
		else
			cerr << "Could not find icon directory. It must located at "
				"/usr/share/pixmaps/packagesearch, ../icons/ or icons/."<<endl;
	} // search the data paths
	QPixmap appIcon(_iconDir+"packagesearch.png");
	setIcon(appIcon);

	_pWhatsThisAction = QWhatsThis::createAction(this);

	// don't change order, because initMenus relies on an initialized toolbar
	initToolbars();
	initMenus();
		
	
	uint width = 700;
	uint height = 600;
	
	QList<int> sizes;
	sizes.push_back(width/2);
	sizes.push_back((width+1)/2);
	
	resize(width, height);
	_pUpperVSplitter->setSizes(sizes);
	_pLowerVSplitter->setSizes(sizes);


	updateBrowseToolbar();
}


void PackageSearchImpl::initToolbars()
{
	_pWhatsThisAction->setShortcut(QKeySequence("Shift+F1"));
	_pMainToolBar = addToolBar(tr("Main Toolbar"));
	_pMainToolBar->setIconSize(QSize(16,16));
	_pWhatsThisAction->addTo(_pMainToolBar);
	_nameToToolBar["Main"] = _pMainToolBar;
}

void PackageSearchImpl::initMenus()
{
	_pHelpMenu->insertAction(helpContentsAction, _pWhatsThisAction);
	
	QMenu* pViewMenu = new QMenu(tr("View"), _pMenuBar);
	_pMenuBar->insertMenu(_pSystemMenu->menuAction(), pViewMenu);
	pViewMenu->addAction(_pMainToolBar->toggleViewAction());

	QMenu* pPackageMenu = new QMenu(tr("Packages"), _pMenuBar);
	_pMenuBar->insertMenu(_pSystemMenu->menuAction(), pPackageMenu);

	_nameToMenu["File"] = _pFileMenu;
	_nameToMenu["View"] = pViewMenu;
	_nameToMenu["Packages"] = pPackageMenu;
	_nameToMenu["System"] = _pSystemMenu;
	_nameToMenu["Plugins"] = _pPluginsMenu;
	_nameToMenu["Help"] = _pHelpMenu;
}

void PackageSearchImpl::initialize()
{
	// create the hard coded plugins
	_pPackageNamePlugin = new NPlugin::PackageNamePlugin();
	this->addPlugin(_pPackageNamePlugin);
	
	_pScoreDisplayPlugin = new NPlugin::ScoreDisplayPlugin(this);
	_pPluginManager->addPluginUser(_pScoreDisplayPlugin);
	this->addPlugin(_pScoreDisplayPlugin);
	
	_pPluginManager->loadPlugins();
	
	// remove all columns for plugins which are not loaded
	{
		QStringList columns;
		for (ShortInformationPluginContainer::const_iterator it = _shortInformationPlugins.begin();
			it != _shortInformationPlugins.end(); ++it)
		{
			columns.push_back((*it)->shortInformationCaption());
		}
		_pPackageListViewControl->removeColumnsNotInList(columns);
	}
	_pInformationContainer->setCurrentPage(0);
	const set<string>& packages_ = packages();
	// First copy the package names into a stringlist, and then add the whole list
	// to the combobox, because since QT4 _pPackageSelection->insertItem() is totally inefficient
	QStringList packageNames;
	for (set<string>::const_iterator it = packages_.begin(); it != packages_.end(); ++it)
	{
		packageNames.push_back(toQString(*it));
	}
	_pPackageSelection->insertItems(0, packageNames);
	_pPackageSelection->setMinimumWidth(100);
	_pPackageSelection->setCurrentText("");
	setPackageActionsEnabled(false);
}

/////////////////////////////////////////////////////
// IXmlStorable Interface
/////////////////////////////////////////////////////

void PackageSearchImpl::saveSettings()
{
	NXml::XmlData xmlData("packagesearch");
	QDomElement root = xmlData.root();
	xmlData.addAttribute(root, QString("1"), "settingsVersion");
	
	xmlData.addAttribute(root, _pMainToolBar->toggleViewAction()->isChecked(), "showMainToolbar");
	
	QDomElement window = xmlData.addElement(xmlData.root(), "geometry");
	xmlData.addAttribute(window, width(), "width");
	xmlData.addAttribute(window, height(), "height");
	QList<int> sizes = _pHSplitter->sizes();
	xmlData.addAttribute(window, sizes[0], "hSplitterSize1");
	xmlData.addAttribute(window, sizes[1], "hSplitterSize2");

	sizes = _pUpperVSplitter->sizes();
	xmlData.addAttribute(window, sizes[0], "upperVSplitterSize1");
	xmlData.addAttribute(window, sizes[1], "upperVSplitterSize2");

	sizes = _pLowerVSplitter->sizes();
	xmlData.addAttribute(window, sizes[0], "lowerVSplitterSize1");
	xmlData.addAttribute(window, sizes[1], "lowerVSplitterSize2");

	_pPackageListViewControl->saveSettings(xmlData, root);
	_pPluginManager->saveSettings(xmlData, root);
	if ( !xmlData.writeFile(_settingsFilename) )
		reportError(tr("Unable to write settings"), tr("Unable to write settings to") +
			_settingsFilename + "\nPersonal settings were not saved.");
}

void PackageSearchImpl::loadSettings()
{
	NXml::XmlData xmlData;
	if (!xmlData.loadFile(_settingsFilename))
	{
		// sort by the first column by default
		_pPackageView->setSorting(1, false);
		return;
	}
	QDomElement root = xmlData.root();
	
	QString settingsVersion;
	NXml::getAttribute(root, settingsVersion, "settingsVersion", "");
	QDomElement element = NXml::getFirstElement(xmlData.root().firstChild());
	
	bool showMainToolbar;
	NXml::getAttribute(root, showMainToolbar, "showMainToolbar", true);
	// TODO this does not work, even though showMainToobar is read correctly
	_pMainToolBar->toggleViewAction()->setChecked(showMainToolbar);

	if (element.tagName() == "geometry")
	{
		uint width;
		uint height;
		NXml::getAttribute(element, width, "width", 600);
		NXml::getAttribute(element, height, "height", 600);
		
		QList<int> sizes;
		int size;
		NXml::getAttribute(element, size, "hSplitterSize1", height/2);
		sizes.push_back(size);
		NXml::getAttribute(element, size, "hSplitterSize2", height/2);
		sizes.push_back(size);
		_pHSplitter->setSizes(sizes);
		
		sizes.clear();
		NXml::getAttribute(element, size, "upperVSplitterSize1", width/2);
		sizes.push_back(size);
		NXml::getAttribute(element, size, "upperVSplitterSize2", width/2);
		sizes.push_back(size);
		_pUpperVSplitter->setSizes(sizes);
		
		sizes.clear();
		NXml::getAttribute(element, size, "lowerVSplitterSize1", width/2);
		sizes.push_back(size);
		NXml::getAttribute(element, size, "lowerVSplitterSize2", width/2);
		sizes.push_back(size);
		_pLowerVSplitter->setSizes(sizes);
		
		element = NXml::getNextElement(element);
		
		resize(width, height);
	}
	// <customListView> node since version 0.3
	if (settingsVersion >= QString("0.3"))
	{
		element = _pPackageListViewControl->loadSettings(element);
	}
	// dismiss old setting version
	else
	{
		if (element.tagName() == "shownShortInformation")
		{
			//_shownShortInformation = NXml::getTextList(element);
			element = NXml::getNextElement(element);
		}
		if (element.tagName() == "hiddenShortInformation")
		{
			//_hiddenShortInformation = NXml::getTextList(element);
			element = NXml::getNextElement(element);	
		}
	}	
	_pPluginManager->loadSettings(element);
}

/////////////////////////////////////////////////////
// IProvider Interface
/////////////////////////////////////////////////////

void PackageSearchImpl::setEnabled(bool enabled)
{
	QMainWindow::setEnabled(enabled);
}

QPushButton* PackageSearchImpl::createClearButton(QWidget* pParent, const char* name) const
{
	QPushButton* pButton = new QPushButton(pParent, name);
	pButton->setFixedSize(20, 20);
	QPixmap clearIcon(_iconDir+"clear.png");
	pButton->setPixmap(clearIcon);
	return pButton;
}

void PackageSearchImpl::reportError(const QString& title, const QString& message)
{
	QMessageBox::critical(this, title, message, "OK");
}

void PackageSearchImpl::reportWarning(const QString& title, const QString& message)
{
	QMessageBox::warning(this, title, message, "OK");
}

void PackageSearchImpl::reportBusy(NPlugin::Plugin* pPlugin, const QString& message)
{
	qApp->setOverrideCursor(QCursor(Qt::WaitCursor));
	statusBar()->message(message);
}

void PackageSearchImpl::reportReady(NPlugin::Plugin* pPlugin)
{
	qApp->restoreOverrideCursor();
	statusBar()->message(tr("Ready"));
}


NUtil::IProgressObserver* PackageSearchImpl::progressObserver() const
{
	return _pPluginManager->progressObserver();
}

const set<string>& PackageSearchImpl::packages() const
{
	static bool initialized = false;
	if (!initialized)
	{	
		// I used to use the real apt code here, but I gained only around 400ms this is not worth the
		// effort...
		QTime t;
		t.start();
		initialized = true;
		NApplication::RunCommandForOutput ro("");
		ro.run("apt-cache pkgnames --no-all-names");
		QStringList packages = ro.getOutput();
		for(QStringList::iterator it = packages.begin(); it != packages.end(); ++it)
			// const cast OK here, since this is lazy initialization, since const_cast works only
			// for pointers and references, I had to get the address here
			(const_cast< set<string>* >(&_packages))->insert(toString(*it));
	}
	return _packages;
}


NPlugin::IProvider::Aggregator& PackageSearchImpl::aptFrontCache() const
{
	return *EptInstance::aggregator();
}

void PackageSearchImpl::reloadAptFrontCache()
{
	aptFrontCache().invalidate();
}


/////////////////////////////////////////////////////
// IPluginUser Interface
/////////////////////////////////////////////////////

void PackageSearchImpl::addPlugin(NPlugin::Plugin* pPlugin)
{
	using namespace NPlugin;

	SearchPlugin* pSearchPlugin = dynamic_cast<SearchPlugin*>(pPlugin);
	if (pSearchPlugin != 0)
	{
		_searchPlugins.insert(pSearchPlugin);
		updateSearchPluginGui();
		connect(pSearchPlugin, SIGNAL(searchChanged(NPlugin::SearchPlugin*)), 
			SLOT(onSearchChanged(NPlugin::SearchPlugin*)));
		/// @todo connect selectionChanged signal
	}
	InformationPlugin* pInformationPlugin = dynamic_cast<InformationPlugin*>(pPlugin);
	if (pInformationPlugin != 0)
	{
		_informationPlugins.insert(pInformationPlugin);
		updateInformationPluginGui();
	}
	ShortInformationPlugin* pShortInformationPlugin = dynamic_cast<ShortInformationPlugin*>(pPlugin);
	if (pShortInformationPlugin != 0)
	{
		addShortInformationPlugin(dynamic_cast<ShortInformationPlugin*>(pPlugin));
	}
	if (dynamic_cast<NPlugin::ActionPlugin*>(pPlugin))
	{
		addActionPlugin(dynamic_cast<NPlugin::ActionPlugin*>(pPlugin));
	}
}


void PackageSearchImpl::addPlugin(NPlugin::PluginContainer* pPlugin)
{
	assert(pPlugin!= 0);
	// add all plugins offered by this container
	vector<string> offeredPlugins = pPlugin->offeredPlugins();
	for ( vector<string>::iterator it = offeredPlugins.begin(); it != offeredPlugins.end(); ++it)
		addPlugin(pPlugin->requestPlugin(*it));
}

void PackageSearchImpl::removePlugin(NPlugin::Plugin* pPlugin)
{
	qDebug("Removing plugin " + pPlugin->name());
	using namespace NPlugin;
	SearchPlugin* pSPlugin = dynamic_cast<SearchPlugin*>(pPlugin);
	if (pSPlugin)
	{
		_searchPlugins.erase(pSPlugin);
		updateSearchPluginGui();
	}
	InformationPlugin* pIPlugin = dynamic_cast<InformationPlugin*>(pPlugin);
	if (pIPlugin)
	{
		_informationPlugins.erase(pIPlugin);
		updateInformationPluginGui();
	}
	ShortInformationPlugin* pSIPlugin = dynamic_cast<ShortInformationPlugin*>(pPlugin);
	if (pSIPlugin)
	{
		_shortInformationPlugins.erase(pSIPlugin);
		_pPackageListViewControl->removeColumn(pSIPlugin->shortInformationCaption());
	}
	ActionPlugin* pAPlugin = dynamic_cast<ActionPlugin*>(pPlugin);
	if (pAPlugin)
	{
		vector<NPlugin::Action*> actions = pAPlugin->actions();
		for (vector<NPlugin::Action*>::const_iterator it = actions.begin(); it != actions.end(); ++it)
		{
			const NPlugin::Action* pAction = *it;
			if (pAction->packageAction())
				_packageActions.erase( std::find(_packageActions.begin(), _packageActions.end(), pAction->action()) );
		}
	}
}

/////////////////////////////////////////////////////
// IPluginUser Helper Methods ///////////////////////

void PackageSearchImpl::addShortInformationPlugin(NPlugin::ShortInformationPlugin* pPlugin)
{
	const int CHAR_WIDTH = 7;
	const int MARGIN = 10;
	_shortInformationPlugins.insert(pPlugin);
	// if the column is not already in the list, add it before the first column
	// with a higher priority value
	if (!_pPackageListViewControl->isExisting(pPlugin->shortInformationCaption()))
	{
		QStringList columns = _pPackageListViewControl->shownColumns();
		int position = 0;
		int priority = pPlugin->shortInformationPriority();
		// find the position for the first plugin with a higher priority value
		for (QStringList::const_iterator it = columns.begin(); it != columns.end(); ++it)
		{
			NPlugin::ShortInformationPlugin* p = getShortInformationPluginWithCaption(*it);
			if (p && p->shortInformationPriority() > priority)
			{
				break;
			}
			++position;
		}
		_pPackageListViewControl->addColumn
		(
			pPlugin->shortInformationCaption(), 
			position,
			(pPlugin->preferredColumnWidth()==-1) ? -1 : pPlugin->preferredColumnWidth() * CHAR_WIDTH + MARGIN
		);
	}
}

void PackageSearchImpl::addActionPlugin(NPlugin::ActionPlugin* pPlugin)
{
	vector<NPlugin::Action*> actions = pPlugin->actions();
	for (vector<NPlugin::Action*>::const_iterator it = actions.begin(); it != actions.end(); ++it)
	{
		const NPlugin::Action* pAction = *it;
		if (!pAction->menu().isEmpty())
		{
			map<QString, QMenu*>::const_iterator jt = _nameToMenu.find(pAction->menu());
			QMenu* pMenu;
			if (jt != _nameToMenu.end())
			{
				pMenu = jt->second;
			}
			else
			{
				qDebug("Requested to add to an unknown menu \"%s\" in plugin %s.\n"
					"Added to system menu instead.", pAction->menu().ascii(), pPlugin->name().ascii() );
				pMenu = _pSystemMenu;
			}
			pMenu->addAction(pAction->action());
		}
		if (!pAction->toolBar().isEmpty())
		{
			map<QString, QToolBar*>::const_iterator jt = _nameToToolBar.find(pAction->toolBar());
			QToolBar* pToolBar;
			if (jt != _nameToToolBar.end())
			{
				pToolBar = jt->second;
			}
			else
			{
				qDebug("Requested to add to an unknown toolbar \"%s\" in plugin %s.\n"
					"Added to main toolbar instead.", pAction->toolBar().ascii(), pPlugin->name().ascii() );
				pToolBar = _pMainToolBar;
			}
			pToolBar->addAction(pAction->action());
		}
		if (pAction->packageAction())
		{
			_packageActions.push_back(pAction->action());
		}
	}
}


/////////////////////////////////////////////////////
// Helper Methods
/////////////////////////////////////////////////////

void PackageSearchImpl::updateSearchPluginGui()
{
	// remove all pages
	while ( _pInputWidgetsContainer->count() > 0)
	{
		_pInputWidgetsContainer->removePage(_pInputWidgetsContainer->page(0) );
	}
	
#warning QT4 Transition todo
// TODO fix this!! -> I only comented out the next two lines.
	// delete the old layout for the frame
//	delete _pShortSearchFrameLayout;
// 	// replace by a new one
//	_pShortSearchFrameLayout = new QVBoxLayout( _pShortSearchFrame, 5, 10, "_pShortSearchFrameLayout"); 
	_pShortSearchFrame->setLayout(new QVBoxLayout());

	
	//_pShortSearchFrame->layout()
	NExtStd::for_each(_searchPlugins.begin(), _searchPlugins.end(), 
		&PackageSearchImpl::addSearchPluginToGui, this);
	/// @todo this seemed to be neccessary because the tab widget behaved really odd
	_pInputWidgetsContainer->setCurrentPage(1);
	if (_pInputWidgetsContainer->count()>0)
		_pInputWidgetsContainer->setCurrentPage(0);
}

void PackageSearchImpl::addSearchPluginToGui(NPlugin::SearchPlugin* pPlugin)
{
	assert(pPlugin);
	// add the input widget
	if (pPlugin->inputWidget() != 0)
	{
		_pInputWidgetsContainer->addTab(pPlugin->inputWidget(), pPlugin->inputWidgetTitle());
	}
	// add the short input widget
	if (pPlugin->shortInputAndFeedbackWidget()!=0)
	{
		QWidget* pWidget = pPlugin->shortInputAndFeedbackWidget();
		// store if the widget was shown to as reparenting destroys this state
		bool shown = pWidget->isShown();
		// reparenting was the only way which seemed to work to get the widgets where I wanted them :-(
		pWidget->reparent(
			_pShortSearchFrame, 0, QPoint()
		);
		_pShortSearchFrame->layout()->add(pWidget);
		pWidget->setShown(shown);
	}
}

void PackageSearchImpl::updateInformationPluginGui()
{
	QString tmpCurrentPackage = _currentPackage;
	_currentPackage =  "";
	// clear the information plugin GUI except the first page
	while ( _pInformationContainer->count() > 0)
	{
		_pInformationContainer->removePage(_pInformationContainer->page(0) );
	}
	// add a dummy tab, so that the widgets are not always repainted
	bool detailPageAdded = false;
	for (InformationPluginContainer::iterator it = _informationPlugins.begin();
		it != _informationPlugins.end(); ++it)
	{
		if ( !detailPageAdded && 
			((*it)->informationPriority() >= _DETAILS_INFORMATION_PRIORITY) )
		{
			_pInformationContainer->insertTab(_pDetailsView, tr("Details"));
			detailPageAdded = true;
		}
		addInformationPluginToGui(*it);
	}
	if (!detailPageAdded)
		_pInformationContainer->insertTab(_pDetailsView, tr("Details"));
	_pInformationContainer->showPage(_pInformationContainer->page(0));
	setSelectedPackage(tmpCurrentPackage);
/*	{
		QWidget* pParent = dynamic_cast<QWidget*>(_pInformationContainer->parent());
		delete _pInformationContainer;
		_pInformationContainer = new QTabWidget(pParent, "InformationContainer");
		_pDetailsView = new QTextBrowser( _pInformationContainer, "_pDetailsView" );
		bool detailPageAdded = false;
		for (InformationPluginContainer::iterator it = _informationPlugins.begin();
			it != _informationPlugins.end(); ++it)
		{
			if ( !detailPageAdded && 
				((*it)->informationPriority() >= _DETAILS_INFORMATION_PRIORITY) )
			{
				_pInformationContainer->insertTab(_pDetailsView, tr("Details"));
				detailPageAdded = true;
			}
			addInformationPluginToGui(*it);
		}
		_pInformationLayout->add(_pInformationContainer);
	}*/
}

void PackageSearchImpl::addInformationPluginToGui(NPlugin::InformationPlugin* pPlugin)
{
	assert(pPlugin);
	if (pPlugin->informationWidget() != 0)	// if there is a special information widget
	{
		// insert the widget
		_pInformationContainer->insertTab(
			pPlugin->informationWidget(), 
			pPlugin->informationWidgetTitle()
		);
	}
}

NPlugin::ShortInformationPlugin* PackageSearchImpl::getShortInformationPluginWithCaption(
	const QString caption) const
{
	NPlugin::ShortInformationCaptionEquals predicat(caption);
	ShortInformationPluginContainer::const_iterator it = 
		find_if(_shortInformationPlugins.begin(), _shortInformationPlugins.end(), predicat);
	if (it == _shortInformationPlugins.end())
		return 0;
	else 
		return *it;
}

void PackageSearchImpl::onShortColumnsChanged()
{
	onSearchChanged(0);
}


void PackageSearchImpl::onClearSearch()
{
	for (
		SearchPluginContainer::iterator it = _searchPlugins.begin();
		it != _searchPlugins.end();
		++it
	)
 		(*it)->clearSearch();
}


bool PackageSearchImpl::FilterPackages::operator()(const string& package)
{
	return _pPlugin->filterPackage(package);
}


void PackageSearchImpl::onSearchChanged(NPlugin::SearchPlugin* pPlugin)
{
	reportBusy(0, tr("Evaluating searches"));
	using namespace NPlugin;
	using namespace wibble::operators;
	std::set<string> result;
	bool first = true;	// keeps track if this is the first search which produces results
	// evaluate plugins which do not uses filter technique
	for (SearchPluginContainer::iterator it = _searchPlugins.begin(); it != _searchPlugins.end(); ++it)
	{
		SearchPlugin* pPlugin = *it;
		if (!pPlugin->isInactive())
		{
			if (!pPlugin->usesFilterTechnique())
			{
				if (first)
				{
					result = pPlugin->searchResult();
					first = false;
				}
				else
					result &= pPlugin->searchResult();	// create the intersection
			}
		}
	}
	// filter the resulting packages through the filter plugins
	for (SearchPluginContainer::iterator it = _searchPlugins.begin(); it != _searchPlugins.end(); ++it)
	{
		SearchPlugin* pPlugin = *it;
		if (!pPlugin->isInactive())
		{
			// if we had no search which returned a result set
			if (first) 
			{
				// the result set contains all packages available
				for (set<string>::const_iterator it = _packages.begin(); it != _packages.end(); ++it)
					result.insert(*it);
				first = false;
			}
			if (pPlugin->usesFilterTechnique())
			{
				std::set<string> newResult;
				FilterPackages fp(pPlugin);
				// copy the packages which are matched by the package filter to newResult
				NExtStd::copy_if( result.begin(), result.end(), inserter(newResult, newResult.begin()), fp );
				swap(newResult, result);
			}
		}
	}

	// if at least one search was active
	if (!first)
	{
		setPackagesFound(result.size());
	}
	else
	{
		setPackagesFound(-1);
		result.clear();
	}
	updateShortInformation(result);
	reportReady(0);
}


void PackageSearchImpl::updateShortInformation(const set<string>& packages)
{
	string lastPackage;	// it seems like -1 is no valid package id (or seldom used..)
	// if a package is selected
	if ( !_currentPackage.isEmpty() )
		lastPackage = toString(_currentPackage);
	// this pointer holds the new item for the formerly selected package, 
	// 0 if the package is nots in the new result
	Q3ListViewItem* pSelectItem = 0;
	_pPackageView->clear();
	
	
	// collect all shown plugins in a list in order of appearance
	// this holds all shown ShortInformationPlugins in order of appearance 
	// and the section as second entry
	vector< pair<NPlugin::ShortInformationPlugin*, int> > siPlugins;
	const QStringList columns = _pPackageListViewControl->shownColumns();
	int i = 0;
	for (QStringList::const_iterator it = columns.begin(); it != columns.end(); ++it, ++i)
	{
		// if the current column is the score column calculate the scores, this way they are only
		// calculated if the column is not hidden
		if ( _pScoreDisplayPlugin != 0 && _pScoreDisplayPlugin->shortInformationCaption() == *it)
			_pScoreDisplayPlugin->updateScores(packages);

		
		for (ShortInformationPluginContainer::iterator jt = _shortInformationPlugins.begin(); 
			jt != _shortInformationPlugins.end(); ++jt)
		{
			if ((*jt)->shortInformationCaption() == *it)
				siPlugins.push_back( make_pair(*jt, _pPackageListViewControl->mapToSection(i)) );
		}
	}
	
	for (set<string>::const_iterator it = packages.begin(); it != packages.end(); ++it)
	{
		const string& package = *it;
		// add the item with its name
		Q3ListViewItem* pItem = new PackageListViewItem(_pPackageView, toQString(package));
		if (lastPackage == package)
			pSelectItem = pItem;
		// iterate all shown columns
		for ( int i=0; i < siPlugins.size(); ++i)
		{
			try 
			{
				pItem->setText( siPlugins[i].second, siPlugins[i].first->shortInformationText(package) );
			}
			// simply ignore it if the package was not available for this plugin
			catch (NPlugin::PackageNotFoundException& e) {}
		}
	}
	if (pSelectItem)
	{
		_pPackageView->ensureItemVisible(pSelectItem);
		_pPackageView->setSelected(pSelectItem, true);
	}
	// select the first item of the site to be shown
	else if (_pPackageView->firstChild())
	{
		_pPackageView->setSelected(_pPackageView->firstChild(), true);
	}

}

void PackageSearchImpl::showPackageInformation(const QString & package)
{
	// if the package to be shown is not selected, deselect it
	PackageListViewItem* pSelected = dynamic_cast<PackageListViewItem*>(_pPackageView->selectedItem());
	
	if ( pSelected && package != pSelected->package())
		_pPackageView->clearSelection();
	_pDetailsView->clear();
	for ( InformationPluginContainer::iterator it = _informationPlugins.begin();
		it != _informationPlugins.end(); ++it )
	{
		NPlugin::InformationPlugin* pPlugin = (*it);
		if (package.isEmpty())
		{
			pPlugin->clearInformationWidget();
		}
		else
		{
			if ( pPlugin->offersInformationText())
			{
				try 
				{
					_pDetailsView->setText(_pDetailsView->text() + pPlugin->informationText(toString(package)));
				}
				// simply ignore it if the package was not available for this plugin
				catch (NPlugin::PackageNotFoundException& e) {}
			}
			if (_pInformationContainer->currentPage() == pPlugin->informationWidget())
			// show the page only if it is currenly active
				pPlugin->updateInformationWidget(toString(package));
		}
	}
}

void PackageSearchImpl::onInformationPageChanged( QWidget * pPage )
{
	if (_currentPackage.isEmpty())	// if no package was selected
		return;
	for (
		InformationPluginContainer::iterator it = _informationPlugins.begin();
		it != _informationPlugins.end();
		++it
	)
	{
		if (_pInformationContainer->currentPage() == (*it)->informationWidget())
		// show the page only if it is currenly active
			(*it)->updateInformationWidget(toString(_currentPackage));
	}
}

void PackageSearchImpl::onControlPlugins()
{
	_pPluginManager->showControlDialog(this);
}

void PackageSearchImpl::onPluginSettings()
{
	_pPluginManager->showSettingsDialog(this);
}


void PackageSearchImpl::closeEvent(QCloseEvent* pE)
{
	qDebug("closing");
	saveSettings();
	pE->accept();
}


void PackageSearchImpl::on__pPackageView_contextMenuRequested(Q3ListViewItem* pItem, const QPoint& pos)
{
	QList<QAction*> actions;
	
	QAction* pCustomizeColumnsAction = new QAction(tr("Customize columns"), this);
	actions.push_back(pCustomizeColumnsAction);
	
	if (pItem)
		NExtStd::for_each(_packageActions.begin(), _packageActions.end(), &QList<QAction*>::push_back, &actions);

	QAction* pSelectedAction = QMenu::exec(actions, pos);
	if ( pSelectedAction == pCustomizeColumnsAction)
		_pPackageListViewControl->showControlDialog();
}

void PackageSearchImpl::onPackageSelectionChanged(Q3ListViewItem* pItem)
{
	if ( pItem != 0 )
	{
		PackageListViewItem* pPackageItem = dynamic_cast<PackageListViewItem*>(pItem) ;
		selectNewPackage(pPackageItem->package());
		setPackageActionsEnabled(true);
	}
	else
	{
		_pDetailsView->clear();
		setPackageActionsEnabled(false);
	}
}

void PackageSearchImpl::onLinkClicked( const QUrl& link)
{
	selectNewPackage(link.toString());
}

void PackageSearchImpl::onForward()
{
	setSelectedPackage(_viewHistory.forward());
	updateBrowseToolbar();
}

void PackageSearchImpl::onBack()
{
	setSelectedPackage(_viewHistory.back());
	updateBrowseToolbar();
}

void PackageSearchImpl::selectNewPackage(const QString& package)
{
	// reselect the package even if already selected to reflect changes due to a changed search
	setSelectedPackage(package);
	if ( _viewHistory.empty() || _viewHistory.current() != package)
	{
		_viewHistory.append(package);
		updateBrowseToolbar();
	}
}

void PackageSearchImpl::updateBrowseToolbar()
{
	_pForwardButton->setEnabled(_viewHistory.forwardPossible());
	_pBackButton->setEnabled(_viewHistory.backPossible());
}

void PackageSearchImpl::onHelpContentsAction()
{
	QDialog* pDlg = new QDialog(this);
	QVBoxLayout* pMainLayout = new QVBoxLayout(pDlg);
	QTextBrowser *pHelpWindow = new QTextBrowser(pDlg); 
	pHelpWindow->setCaption(tr("Package Search Help Page"));
	pHelpWindow->setSource(_docDir+"content.html");
	pMainLayout->add(pHelpWindow);
	pDlg->resize(520,560);
	pDlg->show();
}

void PackageSearchImpl::onHelpAboutAction()
{
	NPackageSearch::PackageSearchAboutDlg dlg;
	dlg.exec();
}

void PackageSearchImpl::setPackagesFound(int number)
{
	QString output("<font size=\"-1\">");
	if (number == -1)
		output += "No Search Active";
	else
		output += QString().setNum(number)+" Packages Found";
	output += ("</font>");
	_pPackagesFoundDisplay->setText(output);
}

void PackageSearchImpl::setSelectedPackage( const QString & package )
{
	if (package != _currentPackage)
	{
		_currentPackage = package;
		// this triggers a selectNewPackage() call  but it returns immidiately because 
		// package == _currentPackage
		_pPackageSelection->setCurrentText(package);	
	}
	// reselect the package even if already selected to reflect changes due to a changed search
	showPackageInformation(package);
}


void PackageSearchImpl::setPackageActionsEnabled(bool enabled)
{
	for (list<QAction*>::iterator it = _packageActions.begin(); it != _packageActions.end(); ++it)
		(*it)->setEnabled(enabled);
}

// Include template instantiations as well
// Need to undef emit because it's been defined by qt as something else
#undef emit
#include <ept/cache/apt/packages.tcc>
#include <ept/cache/debtags/vocabulary.tcc>
// Luckily, qt defines emit as an empty symbol, so it's easy to redefine it here
#define emit


