
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: playlist.c 2471 2007-07-09 11:04:18Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "environment.h"
#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "mutex.h"
#include "oxine.h"
#include "playlist.h"
#include "playlist_m3u.h"
#include "playlist_pls.h"
#include "playlist_xml.h"
#include "utils.h"

extern oxine_t *oxine;

static playitem_length_cb_t playitem_length_cb = NULL;


/// Calls the playlist-changed callback if allowed.
static void
playlist_changed (playlist_t * playlist)
{
    if (playlist->call_change_cb && playlist->change_cb) {
        playlist->change_cb (playlist);
    }
}


/**
 * This is the callback passed to l_list_free and l_list_clear to free the
 * memory of the playitem.
 */
static void
playitem_free_cb (void *data)
{
    playitem_t *playitem = (playitem_t *) data;

    assert (playitem);

    ho_free (playitem->title);
    ho_free (playitem->mrl);
    ho_free (playitem->subtitle_mrl);
    ho_free (playitem->thumbnail_mrl);

    ho_free (playitem);
}


playlist_t *
playlist_new (const char *mrl, playlist_change_cb_t cb)
{
    playlist_t *playlist = ho_new (playlist_t);

    playlist->list = l_list_new ();

    pthread_mutexattr_init (&playlist->mutex_attr);
    pthread_mutexattr_settype (&playlist->mutex_attr,
                               PTHREAD_MUTEX_RECURSIVE);
    pthread_mutex_init (&playlist->mutex, &playlist->mutex_attr);

    playlist->mrl = mrl ? ho_strdup (mrl) : NULL;
    playlist->current = NULL;
    playlist->playmode = PLAYBACK_MODE_NORMAL;
    playlist->top_position = 0;
    playlist->cur_position = 0;
    playlist->change_cb = cb;
    playlist->call_change_cb = true;
    playlist->playback_length = 0;

    return playlist;
}


void
playlist_clear (playlist_t * playlist)
{
    assert (playlist);

    playlist_lock (playlist);

    l_list_clear (playlist->list, playitem_free_cb);
    playlist->current = NULL;
    playlist->playmode = PLAYBACK_MODE_NORMAL;
    playlist->top_position = 0;
    playlist->cur_position = 0;
    playlist->playback_length = 0;

    playlist_changed (playlist);
    playlist_unlock (playlist);
}


void
playlist_free (playlist_t * playlist)
{
    assert (playlist);

    playlist_lock (playlist);

    l_list_free (playlist->list, playitem_free_cb);

    playlist_unlock (playlist);

    pthread_mutex_destroy (&playlist->mutex);
    pthread_mutexattr_destroy (&playlist->mutex_attr);

    ho_free (playlist->mrl);
    ho_free (playlist);
}


static char *
try_subtitles (const char *mrl)
{
    const char *endings[] =
        { "sub", "srt", "asc", "smi", "ssa", "txt", NULL };

    /* Allocate enough memory to hold the MRL plus a suffix of four
     * characters plus the terminating \0. */
    int length = strlen (mrl) + 5;
    char *subtitle = ho_malloc (length);

    /* Try to find a subtitle file that is named like the MRL with 
     * an appended suffix. */
    int i = 0;
    for (i = 0; endings[i]; i++) {
        snprintf (subtitle, length, "%s.%s", mrl, endings[i]);

        if (file_exists (subtitle)) {
            return subtitle;
        }
    }

    ho_free (subtitle);
    return NULL;
}


static char *
guess_subtitle (const char *mrl)
{
    char *subtitle = NULL;

    if (starts_with (mrl, "file:")) {
        return guess_subtitle (mrl + 5);
    }

    if ((subtitle = try_subtitles (mrl)) == NULL) {
        char *mrl_no_suffix = ho_strdup (mrl);
        char *suffix = NULL;
        if ((suffix = rindex (mrl_no_suffix, '.')) != NULL)
            suffix[0] = '\0';
        subtitle = try_subtitles (mrl_no_suffix);
        ho_free (mrl_no_suffix);
    }
#ifdef DEBUG
    if (subtitle) {
        char *mrl_bn = get_basename (mrl);
        char *sub_bn = get_basename (subtitle);
        debug ("found subtitle file");
        debug ("         mrl: %s", mrl_bn);
        debug ("    subtitle: %s", sub_bn);
        ho_free (mrl_bn);
        ho_free (sub_bn);
    }
#endif

    return subtitle;
}


playitem_t *
playlist_insert (playlist_t * playlist, int position,
                 const char *title, const char *mrl)
{
    assert (playlist);

    playlist_lock (playlist);

    playitem_t *playitem = ho_new (playitem_t);
    playitem->subtitle_mrl = NULL;
    playitem->thumbnail_mrl = NULL;
    playitem->played = false;
    playitem->start_time = 0;
    if (playitem_length_cb) {
        playitem->playback_length = playitem_length_cb (mrl);
    }
    else {
        playitem->playback_length = 0;
    }

    playitem->title = title ? ho_strdup (title) : NULL;
    playitem->mrl = mrl ? ho_strdup (mrl) : NULL;

    if (config_get_bool ("subtitles.autoload")) {
        playitem->subtitle_mrl = guess_subtitle (mrl);
    }

    l_list_insert (playlist->list, position, playitem);
    playlist->playback_length += playitem->playback_length;

    playlist_changed (playlist);
    playlist_unlock (playlist);

    return playitem;
}


playitem_t *
playlist_add (playlist_t * playlist, const char *title, const char *mrl)
{
    assert (playlist);

    int length = l_list_length (playlist->list);

    return playlist_insert (playlist, length, title, mrl);
}


playitem_t *
playlist_add_fileitem (playlist_t * playlist, fileitem_t * fileitem)
{
    playitem_t *item = NULL;

    assert (playlist);
    assert (fileitem);

    playlist_lock (playlist);
    playlist->call_change_cb = false;

    switch (fileitem->type) {
    case FILE_TYPE_REGULAR:
        {
            const char *thumbnail_mrl = fileitem_get_thumbnail (fileitem);
            item = playlist_add (playlist, fileitem->title, fileitem->mrl);
            if (thumbnail_mrl) {
                item->thumbnail_mrl = ho_strdup (thumbnail_mrl);
            }
        }
        break;
    case FILE_TYPE_DIRECTORY:
    case FILE_TYPE_MOUNTPOINT:
    case FILE_TYPE_MEDIAMARKS:
    case FILE_TYPE_CDDA_VFOLDER:
#ifdef HAVE_SHOUTCAST
    case FILE_TYPE_SHOUTCAST_STATION:
#endif
#ifdef HAVE_YOUTUBE
    case FILE_TYPE_YOUTUBE_VFOLDER:
    case FILE_TYPE_YOUTUBE_VIDEO:
#endif
#ifdef HAVE_TVLINKS
    case FILE_TYPE_TVLINKS_VFOLDER:
    case FILE_TYPE_TVLINKS_VIDEO:
#endif
        {
            filelist_t *sl = filelist_expand (fileitem);
            fileitem_t *si = filelist_first (sl);
            while (si) {
                playlist_add_fileitem (playlist, si);
                si = filelist_next (sl, si);
            }
        }
        break;
    case FILE_TYPE_PLAYLIST_M3U:
    case FILE_TYPE_PLAYLIST_PLS:
        {
            playlist_t *pl = playlist_new (fileitem->mrl, NULL);
            playlist_load (pl, NULL);
            playitem_t *pi = playlist_first (pl);
            while (pi) {
                playlist_add (playlist, pi->title, pi->mrl);
                pi = playlist_next (pl, pi);
            }
            playlist_free (pl);
        }
    default:
        error (_("Unknown filetype: %d!"), fileitem->type);
        break;
    }

    playlist->call_change_cb = true;
    playlist_changed (playlist);
    playlist_unlock (playlist);

    return item;
}


void
playlist_move_up (playlist_t * playlist, playitem_t * playitem)
{
    assert (playlist);
    assert (playitem);

    playlist_lock (playlist);

    l_list_move_up (playlist->list, playitem);

    playlist_changed (playlist);
    playlist_unlock (playlist);
}


void
playlist_move_down (playlist_t * playlist, playitem_t * playitem)
{
    assert (playlist);
    assert (playitem);

    playlist_lock (playlist);

    l_list_move_down (playlist->list, playitem);

    playlist_changed (playlist);
    playlist_unlock (playlist);
}


void
playlist_remove (playlist_t * playlist, playitem_t * playitem)
{
    assert (playlist);
    assert (playitem);

    playlist_lock (playlist);

    if (playlist->current == playitem)
        playlist->current = NULL;

    l_list_remove (playlist->list, playitem);
    playlist->playback_length -= playitem->playback_length;

    ho_free (playitem->title);
    ho_free (playitem->mrl);
    ho_free (playitem->subtitle_mrl);
    ho_free (playitem->thumbnail_mrl);
    ho_free (playitem);

    playlist_changed (playlist);
    playlist_unlock (playlist);
}


static playitem_t *
playlist_random_next (playlist_t * playlist)
{
    assert (playlist);

    if (!l_list_length (playlist->list))
        return NULL;

    int num_played = 0;
    playitem_t *item = l_list_first (playlist->list);
    while (item) {
        if (item->played)
            num_played++;
        item = l_list_next (playlist->list, item);
    }
    if (num_played == l_list_length (playlist->list))
        return NULL;

    int found = 0;
    while (!found) {
        int next =
            (int) ((double) l_list_length (playlist->list) *
                   ((double) random () / (double) RAND_MAX));
        playitem_t *item = l_list_first (playlist->list);
        while (item && next > 0) {
            next -= 1;
            item = l_list_next (playlist->list, item);
        }
        if (next == 0) {
            if (!item->played) {
                return item;
            }
        }
    }

    return NULL;
}


playitem_t *
playlist_get_first (playlist_t * playlist)
{
    assert (playlist);

    playlist_lock (playlist);

    playitem_t *item = NULL;
    switch (playlist->playmode) {
    case PLAYBACK_MODE_NORMAL:
    case PLAYBACK_MODE_REPEAT:
        item = l_list_first (playlist->list);
        break;
    case PLAYBACK_MODE_RANDOM:
        srandom ((unsigned int) time (NULL));
        item = l_list_first (playlist->list);
        while (item) {
            item->played = false;
            item = l_list_next (playlist->list, item);
        }
        item = playlist_random_next (playlist);
        break;
    }

    playlist_set_current (playlist, item);

    playlist_unlock (playlist);

    return playlist->current;
}


playitem_t *
playlist_get_prev (playlist_t * playlist)
{
    assert (playlist);

    playlist_lock (playlist);

    if (!playlist->current)
        return NULL;

    playitem_t *item = NULL;
    switch (playlist->playmode) {
    case PLAYBACK_MODE_NORMAL:
        item = l_list_prev (playlist->list, playlist->current);
        break;
    case PLAYBACK_MODE_REPEAT:
        item = l_list_prev (playlist->list, playlist->current);
        if (!item)
            item = l_list_last (playlist->list);
        break;
    case PLAYBACK_MODE_RANDOM:
        item = playlist_random_next (playlist);
        break;
    }

    playlist_set_current (playlist, item);

    playlist_unlock (playlist);

    return playlist->current;
}


static playitem_t *
playlist_get_next_to_play (playlist_t * playlist)
{
    if (!playlist->current) {
        return NULL;
    }

    playitem_t *item = NULL;
    switch (playlist->playmode) {
    case PLAYBACK_MODE_NORMAL:
        item = l_list_next (playlist->list, playlist->current);
        break;
    case PLAYBACK_MODE_REPEAT:
        item = l_list_next (playlist->list, playlist->current);
        if (!item) {
            item = l_list_first (playlist->list);
        }
        break;
    case PLAYBACK_MODE_RANDOM:
        item = playlist_random_next (playlist);
        break;
    }

    return item;
}


playitem_t *
playlist_get_next (playlist_t * playlist)
{
    assert (playlist);

    playlist_lock (playlist);

    playlist_set_current (playlist, playlist_get_next_to_play (playlist));

    playlist_unlock (playlist);

    return playlist->current;
}


playitem_t *
playlist_set_current (playlist_t * playlist, playitem_t * current)
{
    assert (playlist);

    playlist_lock (playlist);

    if (playlist->current != current) {
        playlist->current = current;

        if (playlist->current) {
            playlist->current->played = true;
        }

        playlist_changed (playlist);
    }

    playlist_unlock (playlist);

    return playlist->current;
}


playitem_t *
playlist_get_current (playlist_t * playlist)
{
    assert (playlist);

    return playlist->current;
}


bool
playlist_current_is_last (playlist_t * playlist)
{
    assert (playlist);

    playlist_lock (playlist);

    bool result = (playlist_get_next_to_play (playlist) == NULL);

    playlist_unlock (playlist);

    return result;
}


playitem_t *
playlist_set_current_pos (playlist_t * playlist, int pos)
{
    assert (playlist);

    playlist_lock (playlist);

    playitem_t *cur = NULL;
    if ((pos >= 0) && (pos < playlist_length (playlist))) {
        int i = 0;
        cur = l_list_first (playlist->list);
        while (cur) {
            if (i == pos) {
                break;
            }
            cur = l_list_next (playlist->list, cur);
            i++;
        }
    }

    playlist_set_current (playlist, cur);

    playlist_unlock (playlist);

    return playlist->current;
}


int
playlist_get_current_pos (playlist_t * playlist)
{
    assert (playlist);

    playlist_lock (playlist);

    int p = -1;
    int i = 0;
    playitem_t *cur = l_list_first (playlist->list);
    while (cur) {
        if (cur == playlist->current) {
            p = i;
            break;
        }
        cur = l_list_next (playlist->list, cur);
        i++;
    }

    playlist_unlock (playlist);

    return p;
}


playitem_t *
playlist_first (playlist_t * playlist)
{
    assert (playlist);

    return l_list_first (playlist->list);
}


playitem_t *
playlist_next (playlist_t * playlist, playitem_t * current)
{
    assert (playlist);
    assert (current);

    return l_list_next (playlist->list, current);
}


int
playlist_length (playlist_t * playlist)
{
    assert (playlist);

    return l_list_length (playlist->list);
}


playback_mode_t
playlist_get_playmode (playlist_t * playlist)
{
    assert (playlist);

    return playlist->playmode;
}


void
playlist_set_playmode (playlist_t * playlist, playback_mode_t playmode)
{
    assert (playlist);

    playlist_lock (playlist);

    playlist->playmode = playmode;

    playlist_changed (playlist);
    playlist_unlock (playlist);
}


bool
playlist_load (playlist_t * playlist, const char *mrl)
{
    assert (playlist);

    playlist_lock (playlist);
    playlist->call_change_cb = false;

    bool res = false;
    const char *lmrl = mrl ? mrl : playlist->mrl;
    if (is_playlist_oxp (lmrl)) {
        res = playlist_xml_load (playlist, lmrl);
    }
    else if (is_playlist_m3u (lmrl)) {
        res = playlist_m3u_load (playlist, lmrl);
    }
    else if (is_playlist_pls (lmrl)) {
        res = playlist_pls_load (playlist, lmrl);
    }

    playlist->call_change_cb = true;
    playlist_changed (playlist);
    playlist_unlock (playlist);

    return res;
}


bool
playlist_save (playlist_t * playlist, const char *mrl)
{
    assert (playlist);

    playlist_lock (playlist);

    bool res = false;
    const char *lmrl = mrl ? mrl : playlist->mrl;
    if (is_playlist_oxp (lmrl)) {
        res = playlist_xml_save (playlist, lmrl);
    }
    else if (is_playlist_m3u (lmrl)) {
        res = playlist_m3u_save (playlist, lmrl);
    }
    else if (is_playlist_pls (lmrl)) {
        res = playlist_pls_save (playlist, lmrl);
    }

    playlist_unlock (playlist);

    return res;
}


const char *
playitem_get_thumbnail (playitem_t * playitem)
{
    const char *mrl = NULL;

    if (!playitem) {
        /* Nothing to do. */
    }
    else if (playitem->thumbnail_mrl) {
        mrl = playitem->thumbnail_mrl;
    }
    else if (is_file_image (playitem->mrl)) {
        playitem->thumbnail_mrl = ho_strdup (playitem->mrl);
        mrl = playitem->thumbnail_mrl;
    }
    else {
        playitem->thumbnail_mrl = get_thumbnail (playitem->mrl);
        mrl = playitem->thumbnail_mrl;
    }

    return mrl;
}


void
playlist_set_length_cb (playitem_length_cb_t cb)
{
    playitem_length_cb = cb;
}


int
playlist_get_playback_length (playlist_t * playlist)
{
    assert (playlist);

    return playlist->playback_length;
}
