
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: menulist.c 2556 2007-07-21 07:58:57Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <stdio.h>

#include "codeset.h"
#include "environment.h"
#include "heap.h"
#include "i18n.h"
#include "list.h"
#include "logger.h"
#include "menulist.h"
#include "otk.h"
#include "oxine.h"
#include "utils.h"
#include "xmlparser.h"

#include "menu_help.h"
#include "menu_extractor.h"
#include "menu_filelist.h"
#include "menu_main.h"
#include "menu_playback.h"
#include "menu_playlist.h"
#include "menu_settings.h"
#include "menu_weather.h"

extern oxine_t *oxine;

static menuitem_t *
menuitem_load (xml_node_t * node, menulist_t * menulist)
{
    menuitem_t *item = ho_new (menuitem_t);

    item->title = NULL;
    item->submenu = NULL;
    item->is_uplink = false;
    item->cb = NULL;
    item->cb_data = oxine;

    node = node->child;
    while (node) {
        if (!strcasecmp (node->name, "title")) {
            char *codeset = get_system_encoding ();
            recode_t *xr = recode_init ("utf8", codeset);
            item->title = recode (xr, node->data);
            recode_done (xr);
            ho_free (codeset);
        }

        else if (!strcasecmp (node->name, "action")) {
            const char *type = xml_parser_get_property (node, "type");
            const char *param = xml_parser_get_property (node, "parameter");
            if (type) {
                /* Play a MRL */
                if ((strcasecmp (type, "mrl_play") == 0) && param) {
                    item->cb = play_mrl_cb;
                    item->cb_data = ho_strdup (param);
                }
                /* Play TV, DVD, VCD or CDDA */
                else if ((strcasecmp (type, "auto_play") == 0) && param) {
                    if (strcasecmp (param, "tv") == 0) {
                        if (config_get_bool ("television.enable")) {
                            item->cb = play_tv_cb;
                        }
                        else {
                            debug ("Hiding autoplay entry 'Television' in "
                                   "mainmenu because configuration "
                                   "television.enable is FALSE.");
                        }
                    }
#ifdef HAVE_HAL
                    else if (strcasecmp (param, "dvd") == 0) {
                        debug ("Hiding autoplay entry 'DVD' in mainmenu "
                               "because HAL is enabled.");
                    }
                    else if (strcasecmp (param, "vcd") == 0) {
                        debug ("Hiding autoplay entry 'Video CD' in mainmenu "
                               "because HAL is enabled.");
                    }
                    else if (strcasecmp (param, "cdda") == 0) {
                        debug ("Hiding autoplay entry 'Audio CD' in mainmenu "
                               "because HAL is enabled.");
                    }
#else
                    else if (strcasecmp (param, "dvd") == 0) {
                        item->cb = play_dvd_cb;
                    }
                    else if (strcasecmp (param, "vcd") == 0) {
                        item->cb = play_vcd_cb;
                    }
                    else if (strcasecmp (param, "cdda") == 0) {
                        item->cb = play_cdda_cb;
                    }
#endif
                    else {
                        error (_("Unknown parameter '%s' for "
                                 "type 'autoplay'!"), param);
                    }
                }
                /* Show main menu. */
                else if (strcasecmp (type, "main_menu") == 0) {
                    item->cb = show_menu_main;
                }
                /* Show media menu. */
                else if (strcasecmp (type, "media_menu") == 0) {
                    item->cb = show_menu_media;
                }
                /* Show music menu. */
                else if (strcasecmp (type, "music_menu") == 0) {
                    item->cb = show_menu_music;
                }
                /* Show video menu. */
                else if (strcasecmp (type, "video_menu") == 0) {
                    item->cb = show_menu_video;
                }
                /* Show image menu. */
                else if (strcasecmp (type, "image_menu") == 0) {
#ifdef HAVE_IMAGE_PLAYBACK
                    item->cb = show_menu_image;
#else
                    debug ("Hiding entry for image menu in mainmenu "
                           "because image support is disabled.");
#endif
                }
                /* Show playlist menu. */
                else if (strcasecmp (type, "playlist_menu") == 0) {
                    item->cb = show_menu_playlist;
                }
                /* Show help menu. */
                else if (strcasecmp (type, "help_menu") == 0) {
                    item->cb = show_menu_help;
                }
                /* Show settings menu. */
                else if (strcasecmp (type, "settings_menu") == 0) {
                    item->cb = show_menu_settings;
                }
                /* Show the weather menu. */
                else if (strcasecmp (type, "weather_menu") == 0) {
#ifdef HAVE_WEATHER
                    item->cb = show_menu_weather;
#else
                    debug ("Hiding entry for weather menu in mainmenu "
                           "because weather support is disabled.");
#endif
                }
                /* Show playback menu. */
                else if (strcasecmp (type, "playback_menu") == 0) {
                    item->cb = show_menu_playback;
                }
                /* Show extractor menu. */
                else if (strcasecmp (type, "extractor_menu") == 0) {
#ifdef HAVE_EXTRACTOR
                    item->cb = show_menu_extractor;
#else
                    debug ("Hiding entry for extractor menu in mainmenu "
                           "because extractor support is disabled.");
#endif
                }
                /* Show a submenu */
                else if (strcasecmp (type, "sub_menu") == 0) {
                    item->cb = show_menu_sub;
                    item->submenu = menulist_new (param, menulist);
                    item->cb_data = item->submenu;
                }
                /* Go back up one menu level */
                else if (strcasecmp (type, "uplink_menu") == 0) {
                    item->cb = show_menu_sub;
                    item->submenu = menulist->parent;
                    item->cb_data = menulist->parent;
                    item->is_uplink = true;
                }
                /* Show the shutdown menu */
                else if (strcasecmp (type, "shutdown") == 0) {
                    item->cb = shutdown_cb;
                }
                /* Execute a shell command. */
                else if (strcasecmp (type, "shellcommand") == 0) {
                    item->cb = execute_cb;
                    item->cb_data = ho_strdup (param);
                }
                /* Unknown action */
                else {
                    error (_("Unknown menu action: '%s'!"), type);
                }
            }
        }

        else {
            error (_("Unknown name '%s' for node!"), node->name);
        }

        node = node->next;
    }

    if (item && !item->cb) {
        ho_free (item->title);
        ho_free (item);
        item = NULL;
    }

    return item;
}

static void
menuitem_free_cb (void *data)
{
    menuitem_t *menuitem = (menuitem_t *) data;
    assert (menuitem);

    if (menuitem->title)
        ho_free (menuitem->title);
    if (menuitem->cb_data == oxine)
        menuitem->cb_data = NULL;
    if (menuitem->cb_data == menuitem->submenu)
        menuitem->cb_data = NULL;
    if (menuitem->cb_data)
        ho_free (menuitem->cb_data);
    if (menuitem->submenu && !menuitem->is_uplink)
        menulist_free (menuitem->submenu);

    ho_free (menuitem);
}


static void
read_nodes (menulist_t * menulist, xml_node_t * node)
{
    while (node) {
        if (!strcasecmp (node->name, "button")) {
            menuitem_t *item = menuitem_load (node, menulist);
            if (item && item->cb) {
                l_list_append (menulist->list, item);
            }
        }

        else {
            error (_("Unknown name '%s' for node!"), node->name);
        }

        node = node->next;
    }
}


static int
main_menu_parse (menulist_t * menulist, const char *mrl)
{
    int size;
    char *file = read_entire_file (mrl, &size);
    xml_node_t *node;

    if (!file)
        return 0;

    xml_parser_init (file, strlen (file), XML_PARSER_CASE_INSENSITIVE);

    if (xml_parser_build_tree (&node) < 0) {
        error (_("Parsing '%s' failed!"), mrl);
        return 0;
    }

    if (strcasecmp (node->name, "oxine_menu")) {
        error (_("Root node of '%s' must be '%s'!"), mrl, "oxine_menu");
        return 0;
    }

    const char *px = xml_parser_get_property (node, "x");
    const char *py = xml_parser_get_property (node, "y");
    const char *pw = xml_parser_get_property (node, "w");
    const char *ph = xml_parser_get_property (node, "h");

    menulist->x = px ? atoi (px) : -1;
    menulist->y = py ? atoi (py) : -1;
    menulist->w = pw ? atoi (pw) : -1;
    menulist->h = ph ? atoi (ph) : -1;

    read_nodes (menulist, node->child);
    xml_parser_free_tree (node);
    ho_free (file);

    return 1;
}


menulist_t *
menulist_new (const char *mrl, menulist_t * parent)
{
    menulist_t *menulist = ho_new (menulist_t);

    menulist->list = l_list_new ();
    menulist->parent = parent;

    info (_("Loading menu file '%s'..."), mrl);
    if (!main_menu_parse (menulist, mrl)) {
        menulist_free (menulist);
        return NULL;
    }

    return menulist;
}


void
menulist_free (menulist_t * menulist)
{
    assert (menulist);

    if (menulist->list)
        l_list_free (menulist->list, menuitem_free_cb);
    ho_free (menulist);
}
