
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: heap.c 2587 2007-07-23 09:36:39Z mschwerin $
 *
 */
#include "config.h"

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "heap.h"
#include "list.h"
#include "logger.h"

#ifdef USE_OXINE_HEAP_MANAGEMENT

/*
 * Heap objects are aligned on sizeof(int) boundaries
 */
#define ALIGNMENT       (sizeof(int))
#define DOALIGN(num)    (((num)+ALIGNMENT-1)&~(ALIGNMENT-1))

static unsigned long heap_size_cur = 0;
static unsigned long heap_size_max = 0;

/*
 * tag datastructures
 */
typedef struct prefix_tag_s prefix_tag_t;
typedef struct postfix_tag_s postfix_tag_t;

struct prefix_tag_s {
    prefix_tag_t *prev;                                         /* previous object in heap      */
    prefix_tag_t *next;                                         /* next object in heap          */
    postfix_tag_t *postfix;                                     /* ptr to postfix object        */
    char *filename;                                             /* filename ptr or NULL         */
    long line;                                                  /* line number or 0             */
    size_t size;                                                /* size of allocated block      */
    void *content;                                              /* _gen_malloc() ptr of object  */
};

struct postfix_tag_s {
    prefix_tag_t *prefix;
};

/*
 * GLOBAL: Points to first object in linked list of heap objects
 */

static prefix_tag_t *heap_head = NULL;

static void ho_verify (void *content);
static void AddToLinkedList (prefix_tag_t *);
static void RemoveFromLinkedList (prefix_tag_t *);
static void RenderDesc (prefix_tag_t *, char *);

void *
_gen_malloc (char *lpFile, int nLine, size_t wSize)
{

    prefix_tag_t *prefix;

    wSize = DOALIGN (wSize);
    prefix = (prefix_tag_t *) malloc (sizeof (prefix_tag_t) + wSize +
                                      sizeof (postfix_tag_t));
    if (prefix) {
        AddToLinkedList (prefix);
        prefix->postfix = (postfix_tag_t *) ((char *) (prefix + 1) + wSize);
        prefix->postfix->prefix = prefix;
        prefix->filename = lpFile;
        prefix->line = nLine;
        prefix->content = prefix + 1;
        prefix->size = wSize;
        memset (prefix->content, 0, wSize);
    }
    else {
        _log (LEVEL_FATAL, lpFile, nLine, "Could not allocate memory!");
        abort ();
    }

    heap_size_cur += prefix->size;
    if (heap_size_max < heap_size_cur)
        heap_size_max = heap_size_cur;

    return (prefix ? prefix + 1 : NULL);
}

void *
_gen_realloc (char *lpFile, int nLine, void *lpOld, size_t wSize)
{
    void *lpNew = NULL;

    /* Try to realloc */
    if (lpOld) {
        ho_verify (lpOld);
        prefix_tag_t *prefix = (prefix_tag_t *) lpOld - 1;
        prefix_tag_t *lpNewPrefix;
        prefix_tag_t *lpPre;

        heap_size_cur -= prefix->size;

        /* Try to reallocate block */
        RemoveFromLinkedList (prefix);
        memset (prefix->postfix, 0, sizeof (postfix_tag_t));
        wSize = DOALIGN (wSize);
        lpNewPrefix = (prefix_tag_t *) realloc (prefix,
                                                sizeof (prefix_tag_t) +
                                                wSize +
                                                sizeof (postfix_tag_t));

        /* Add new (or failed old) back in */
        lpPre = (lpNewPrefix ? lpNewPrefix : prefix);
        AddToLinkedList (lpPre);
        lpPre->postfix = (postfix_tag_t *) ((char *) (lpPre + 1) + wSize);
        lpPre->postfix->prefix = lpPre;
        lpPre->content = lpPre + 1;
        lpPre->size = wSize;

        heap_size_cur += lpPre->size;
        if (heap_size_max < heap_size_cur)
            heap_size_max = heap_size_cur;

        /* Finish */
        lpNew = (lpNewPrefix ? &lpNewPrefix[1] : NULL);
        if (!lpNew) {
            _log (LEVEL_FATAL, lpFile, nLine, "Could not allocate memory!");
            abort ();
        }
    }

    /* Else try new allocation */
    else {
        lpNew = _gen_malloc (lpFile, nLine, wSize);
    }

    /* Return address to object */
    return lpNew;

}


void *
_gen_free (void *content)
{
    if (!content) {
        return NULL;
    }

    ho_verify (content);

    prefix_tag_t *prefix = (prefix_tag_t *) content - 1;
    size_t wSize = (char *) (prefix->postfix + 1) - (char *) prefix;

    heap_size_cur -= prefix->size;

    RemoveFromLinkedList (prefix);
    memset (prefix, 0, wSize);
    free (prefix);

    return NULL;
}


char *
_gen_strdup (char *lpFile, int nLine, const char *lpS)
{
    void *lpReturn = NULL;

    if (!lpS) {
        _log (LEVEL_FATAL, lpFile, nLine,
              "You tried to duplicate a NULL string!");
        abort ();
    }

    size_t wSize = (size_t) (strlen (lpS) + 1);

    lpReturn = _gen_malloc (lpFile, nLine, wSize);
    if (lpReturn) {
        memcpy (lpReturn, lpS, wSize);
    }

    return lpReturn;

}


void
heapstat (void)
{
#ifdef DEBUG
    unsigned long total = 0;
    unsigned long chunks = 0;
    if (heap_head) {
        prefix_tag_t *lpCur = heap_head;

        do {
            ho_verify (&lpCur[1]);

            char buffer[100];

            RenderDesc (lpCur, buffer);

            debug ("%s", buffer);
            total += lpCur->size;
            chunks++;
            lpCur = lpCur->next;
        } while (lpCur != heap_head);

        if (total)
            debug ("memory usage: %li words in %li chunks", total, chunks);
    }

    debug ("maximum heap size: %8li kByte", heap_size_max / 1024);
    debug ("current heap size: %8li kByte", heap_size_cur / 1024);
#endif /* DEBUG */
}


void static
AddToLinkedList (prefix_tag_t * lpAdd)
{

    /* Add before current head of list */
    if (heap_head) {
        lpAdd->prev = heap_head->prev;
        (lpAdd->prev)->next = lpAdd;
        lpAdd->next = heap_head;
        (lpAdd->next)->prev = lpAdd;
    }
    /* Else first node */
    else {
        lpAdd->prev = lpAdd;
        lpAdd->next = lpAdd;
    }

    /* Make new item head of list */
    heap_head = lpAdd;
}

static void
RemoveFromLinkedList (prefix_tag_t * lpRemove)
{

    /* Remove from doubly linked list */
    (lpRemove->prev)->next = lpRemove->next;
    (lpRemove->next)->prev = lpRemove->prev;

    /* Possibly correct head pointer */
    if (lpRemove == heap_head) {
        heap_head = ((lpRemove->next == lpRemove) ? NULL : lpRemove->next);
    }
}

static int
ho_is_ok (void *content)
{
    return !((long) content & (ALIGNMENT - 1));
}


static void
ho_verify (void *content)
{
    if (!content) {
        fatal ("Heap object was NULL!");
        abort ();
    }

    if (!ho_is_ok (content)) {
        fatal ("Heap object 0x%08X was not correctly aligned!", content);
        abort ();
    }

    prefix_tag_t *prefix = (prefix_tag_t *) content - 1;
    if (prefix->content != content) {
        fatal ("Heap object 0x%08X does not have a valid prefix!", content);
        abort ();
    }

    if (prefix->postfix->prefix != prefix) {
        fatal ("Heap object 0x%08X (%s:%d) is not valid!",
               content, prefix->filename, prefix->line);
        abort ();
    }
}


void static
RenderDesc (prefix_tag_t * prefix, char *lpBuffer)
{
    if (prefix->content == &prefix[1]) {
        sprintf (lpBuffer, "%3lu words @ 0x%08lx:",
                 (unsigned long) prefix->size,
                 (unsigned long) prefix->content);
        if (prefix->filename) {
            sprintf (lpBuffer + strlen (lpBuffer), "%s:%ld ",
                     prefix->filename, prefix->line);
        }
    }
    else {
        strcpy (lpBuffer, "(bad pointer given)");
    }
}

#else /* USE_OXINE_HEAP_MANAGEMENT */

inline void *
_gen_malloc (size_t size)
{
    void *ptr = malloc (size);

    if (ptr)
        memset (ptr, 0, size);

    return ptr;
}

inline void *
_gen_free (void *ptr)
{
    free (ptr);

    return NULL;
}

inline void
heapstat (void)
{

}

#endif /* USE_OXINE_HEAP_MANAGEMENT */

char *
_gen_strdup_printf (char *lpFile, int nLine, const char *format, ...)
{
    char out[4096];

    va_list args;
    va_start (args, format);
    vsnprintf (out, 4096, format, args);
    va_end (args);

#ifdef USE_OXINE_HEAP_MANAGEMENT
    return _gen_strdup (lpFile, nLine, out);
#else
    return strdup (out);
#endif
}


char *
_gen_strdup_strftime (char *lpFile, int nLine, const char *format,
                      const struct tm *tm)
{
    char out[4096];

    strftime (out, 4096, format, tm);

#ifdef USE_OXINE_HEAP_MANAGEMENT
    return _gen_strdup (lpFile, nLine, out);
#else
    return strdup (out);
#endif
}
