
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Thanks goes to Florian Echtler for adding Xinerama support.
 *
 * $Id: x11.c,v 1.69 2006/01/26 15:12:43 mschwerin Exp $
 *
 */

#include "config.h"

#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <xine.h>
#include <sys/types.h>

#include <X11/Xlib.h>
#include <X11/cursorfont.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/Xatom.h>

#ifdef HAVE_XTEST
#include <X11/extensions/XTest.h>
#endif

#ifdef HAVE_XINERAMA
#include <X11/extensions/Xinerama.h>
#endif

#include "event.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "odk_plugin.h"
#include "scheduler.h"

#define WINDOW_TITLE                PACKAGE_NAME" version "PACKAGE_VERSION

typedef struct {
    odk_window_t window_class;
    x11_visual_t visual;

    /* Keep going in the event loop. */
    int keep_going;

    /* The event handler. */
    odk_cb event_handler;
    void *event_handler_data;

    /* Is the window in fullscreen mode. */
    int is_fullscreen;

    /* The X display and screen number */
    Display *display;
    int screen;

    /* The output window. */
    Window window;
    Atom wm_del_win;

    double pixel_aspect;
    double fullscreen_underscan;

    /* The real width and height of the display. */
    int display_width;
    int display_height;

    /* The position and size of the window. */
    int window_xpos;
    int window_ypos;

    int window_width;
    int window_height;

    /* The saved position and size of the non-fullscreen window. */
    int saved_xpos;
    int saved_ypos;

    int saved_width;
    int saved_height;

    /* Stuff needed to hide the mouse cursor. */
    Cursor no_cursor;
    Cursor on_cursor;
    int cursor_visible;
    int mouse_cursor_job;

    /* Stuff needed to disable the screensaver. */
    int have_xtest;
    int xtest_keycode;
    int keep_screensaver;
    int screensaver_job;
} x11_window_t;

static Atom XA_WIN_STATE = None;
static Atom XA_WIN_LAYER = None;
static Atom XA_NET_WM_STATE_FULLSCREEN = None;
static Atom XA_MOTIF_WM_HINTS = None;

/*
 * Callbacks
 */
static void
dest_size_cb (void *x11_win_p,
              int video_width, int video_height,
              double video_pixel_aspect,
              int *dest_width, int *dest_height, double *dest_pixel_aspect)
{
    x11_window_t *x11_win = (x11_window_t *) x11_win_p;
    assert (x11_win);

    *dest_width = x11_win->window_class.output_width;
    *dest_height = x11_win->window_class.output_height;
    *dest_pixel_aspect = x11_win->pixel_aspect;
}


static void
frame_output_cb (void *x11_win_p,
                 int video_width, int video_height,
                 double video_pixel_aspect,
                 int *dest_x, int *dest_y,
                 int *dest_width, int *dest_height,
                 double *dest_pixel_aspect, int *win_x, int *win_y)
{
    x11_window_t *x11_win = (x11_window_t *) x11_win_p;
    assert (x11_win);

    *win_x = x11_win->window_xpos;
    *win_y = x11_win->window_ypos;

    *dest_x = x11_win->window_class.output_xpos;
    *dest_y = x11_win->window_class.output_ypos;

    *dest_width = x11_win->window_class.output_width;
    *dest_height = x11_win->window_class.output_height;

    *dest_pixel_aspect = x11_win->pixel_aspect;
}


/*
 * Connect to the X server.
 * returns 1 on success, 0 on error
 */
static int
connect_x11_server (x11_window_t * win)
{
    if (!XInitThreads ()) {
        error (_("Could not initialize X threads!"));
        return 0;
    }

    /* Try to open the X display. */
    win->display = XOpenDisplay (NULL);
    if (!win->display) {
        error (_("Could not open display!"));
        return 0;
    }

    XLockDisplay (win->display);

    /* Get the default screen. */
    win->screen = DefaultScreen (win->display);

    win->display_width = DisplayWidth (win->display, win->screen);
    win->display_height = DisplayHeight (win->display, win->screen);

    /* Determine display aspect ratio. */
    double res_h = (DisplayWidth (win->display, win->screen) * 1000
                    / DisplayWidthMM (win->display, win->screen));
    double res_v = (DisplayHeight (win->display, win->screen) * 1000
                    / DisplayHeightMM (win->display, win->screen));
    win->pixel_aspect = res_v / res_h;
    double aspect_diff = win->pixel_aspect - 1.0;
    if ((aspect_diff < 0.01) && (aspect_diff > -0.01)) {
        win->pixel_aspect = 1.0;
    }

    /* We want to get notified if user closes the window. */
    win->wm_del_win = XInternAtom (win->display, "WM_DELETE_WINDOW", False);

    XUnlockDisplay (win->display);

    return 1;
}


/*
 * Close the connection to the X server.
 */
static int
disconnect_x11_server (x11_window_t * win)
{

    if (win->display) {
        XLockDisplay (win->display);
        XCloseDisplay (win->display);
    }

    win->display = NULL;
    win->screen = 0;
    win->pixel_aspect = 0;

    return 1;
}


/*
 * Job to hide the mouse cursor.
 */
static void
hide_mouse_cursor (void *x11_win_p)
{
    x11_window_t *win = (x11_window_t *) x11_win_p;

    XLockDisplay (win->display);
    XDefineCursor (win->display, win->window, win->no_cursor);
    XFlush (win->display);
    XUnlockDisplay (win->display);
}


/*
 * Shows the mouse cursor.
 */
static void
show_mouse_cursor (void *x11_win_p)
{
    x11_window_t *win = (x11_window_t *) x11_win_p;

    XLockDisplay (win->display);
    XDefineCursor (win->display, win->window, win->on_cursor);
    XFlush (win->display);
    XUnlockDisplay (win->display);

    cancel_job (win->mouse_cursor_job);
    win->mouse_cursor_job = schedule_job (6000, hide_mouse_cursor, win);
}


/*
 * Job to disable the screensaver.
 */
static void
screensaver_job (void *x11_win_p)
{
    x11_window_t *win = (x11_window_t *) x11_win_p;

    if (win->is_fullscreen) {
        XLockDisplay (win->display);
#ifdef HAVE_XTEST
        if (win->have_xtest == True) {
            XTestFakeKeyEvent (win->display, win->xtest_keycode, True,
                               CurrentTime);
            XTestFakeKeyEvent (win->display, win->xtest_keycode, False,
                               CurrentTime);
            XSync (win->display, False);
        } else
#endif
        {
            XResetScreenSaver (win->display);
        }
        XUnlockDisplay (win->display);
    }

    win->screensaver_job = schedule_job (6000, screensaver_job, win);
}


/*
 * This creates a new X window.
 * returns 1 on success, 0 on error
 */
static int
create_x11_window (x11_window_t * win, int x, int y, int w, int h)
{
    win->window_xpos = x;
    win->window_ypos = x;
    win->window_width = w;
    win->window_height = h;

    win->window_class.output_xpos = 0;
    win->window_class.output_ypos = 0;
    win->window_class.output_width = win->window_width;
    win->window_class.output_height = win->window_height;

    XLockDisplay (win->display);

    /* Create the window */
    unsigned long black = BlackPixel (win->display, win->screen);
    win->window = XCreateSimpleWindow (win->display,
                                       RootWindow (win->display, win->screen),
                                       win->window_xpos,
                                       win->window_ypos,
                                       win->window_width,
                                       win->window_height, 0, black, black);
    if (!win->window) {
        XUnlockDisplay (win->display);
        return 0;
    }

    /* Set window manager properties */
    XSizeHints hint;
    hint.flags = PSize | PMinSize | PWinGravity;
    hint.width = win->window_width;
    hint.height = win->window_height;
    hint.min_width = hint.width;
    hint.min_height = hint.height;
    hint.win_gravity = StaticGravity;
    XmbSetWMProperties (win->display, win->window, WINDOW_TITLE, WINDOW_TITLE,
                        NULL, 0, &hint, NULL, NULL);

    /* Set the window layer. */
    long propvalue[1];
    if (XA_WIN_LAYER == None)
        XA_WIN_LAYER = XInternAtom (win->display, "_WIN_LAYER", False);
    propvalue[0] = 10;
    XChangeProperty (win->display, win->window,
                     XA_WIN_LAYER, XA_CARDINAL, 32, PropModeReplace,
                     (unsigned char *) propvalue, 1);

    XSetTransientForHint (win->display, win->window, None);

    /* Set the input masks */
    XSelectInput (win->display, win->window,
                  StructureNotifyMask |
                  ExposureMask |
                  ButtonPressMask | PointerMotionMask | KeyPressMask);

    XSetWMProtocols (win->display, win->window, &(win->wm_del_win), 1);

    /* Create the mouse cursors (on and off) */
    static unsigned char bm_no_data[] = { 0, 0, 0, 0, 0, 0, 0, 0 };
    Pixmap bm_no = XCreateBitmapFromData (win->display, win->window,
                                          bm_no_data, 8, 8);
    win->no_cursor = XCreatePixmapCursor (win->display, bm_no, bm_no,
                                          (XColor *) & black,
                                          (XColor *) & black, 0, 0);
    win->on_cursor = XCreateFontCursor (win->display, XC_left_ptr);

    XFlush (win->display);
    XUnlockDisplay (win->display);

    win->visual.display = win->display;
    win->visual.screen = win->screen;
    win->visual.d = win->window;
    win->visual.dest_size_cb = dest_size_cb;
    win->visual.frame_output_cb = frame_output_cb;
    win->visual.user_data = win;

    show_mouse_cursor (win);

    return 1;
}


/*
 * This destroys the window.
 */
static int
destroy_x11_window (x11_window_t * win)
{
    XLockDisplay (win->display);
    XDestroyWindow (win->display, win->window);
    XUnlockDisplay (win->display);

    win->window_class.output_xpos = 0;
    win->window_class.output_ypos = 0;
    win->window_class.output_width = 0;
    win->window_class.output_height = 0;

    win->visual.display = NULL;
    win->visual.screen = 0;
    win->visual.dest_size_cb = NULL;
    win->visual.frame_output_cb = NULL;
    win->visual.user_data = NULL;

    return 1;
}

#define MWM_HINTS_DECORATIONS   (1L << 1)
#define PROP_MWM_HINTS_ELEMENTS 5
typedef struct {
    uint32_t flags;
    uint32_t functions;
    uint32_t decorations;
    int32_t input_mode;
    uint32_t status;
} MWMHints;

static int
is_fullscreen (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;

    return win->is_fullscreen;
}

/*
 * This sets and unsets the fullscreen mode.
 */
static int
fullscreen (odk_window_t * this, int want_fullscreen)
{
    x11_window_t *win = (x11_window_t *) this;

    if (win->is_fullscreen == want_fullscreen)
        return 1;

    XLockDisplay (win->display);

    if (want_fullscreen) {
        /* Save size and position of non-fullscreen window. */
        win->saved_xpos = win->window_xpos;
        win->saved_ypos = win->window_ypos;
        win->saved_width = win->window_width;
        win->saved_height = win->window_height;

        /* Set size and position of the output window. */
#ifdef HAVE_XINERAMA
        int event_base;
        int error_base;

        int screens;
        XineramaScreenInfo *screeninfo;

        if (XineramaQueryExtension (win->display, &event_base, &error_base)
            && (screeninfo = XineramaQueryScreens (win->display, &screens))
            && (XineramaIsActive (win->display))) {

            debug ("The display is using xinerama with %d screens.", screens);
            debug ("We're going to assume we are using the first screen.");
            debug ("The size of the first screen is %dx%d.\n",
                   screeninfo[0].width, screeninfo[0].height);

            win->window_width = screeninfo[0].width;
            win->window_height = screeninfo[0].height;
            win->window_xpos = screeninfo[0].x_org;
            win->window_ypos = screeninfo[0].y_org;
        } else
#endif
        {
            win->window_width = win->display_width;
            win->window_height = win->display_height;
            win->window_xpos = 0;
            win->window_ypos = 0;
        }

        /* Set size and position of the output area inside the window. */
        win->window_class.output_width =
            win->fullscreen_underscan * win->display_width;
        win->window_class.output_height =
            win->fullscreen_underscan * win->display_height;
        win->window_class.output_xpos =
            (win->display_width - win->window_class.output_width) / 2;
        win->window_class.output_ypos =
            (win->display_height - win->window_class.output_height) / 2;
    }

    else {
        /* Set size and position of the output window. */
        win->window_width = win->saved_width;
        win->window_height = win->saved_height;
        win->window_xpos = win->saved_xpos;
        win->window_ypos = win->saved_ypos;

        /* Set size and position of the output area inside the window. */
        win->window_class.output_width = win->window_width;
        win->window_class.output_height = win->window_height;
        win->window_class.output_xpos = 0;
        win->window_class.output_ypos = 0;
    }

    /* Go to fullscreen the modern (e.g. metacity way) */
    if (XA_WIN_STATE == None)
        XA_WIN_STATE = XInternAtom (win->display, "_NET_WM_STATE", False);
    if (XA_NET_WM_STATE_FULLSCREEN == None)
        XA_NET_WM_STATE_FULLSCREEN = XInternAtom (win->display,
                                                  "_NET_WM_STATE_FULLSCREEN",
                                                  False);
    long propvalue[1];
    if (want_fullscreen) {
        propvalue[0] = XA_NET_WM_STATE_FULLSCREEN;
    } else {
        propvalue[0] = 0;
    }
    XChangeProperty (win->display, win->window,
                     XA_WIN_STATE, XA_ATOM,
                     32, PropModeReplace, (unsigned char *) propvalue, 1);

    /* Set/ unset window borders. */
    if (XA_MOTIF_WM_HINTS == None)
        XA_MOTIF_WM_HINTS = XInternAtom (win->display,
                                         "_MOTIF_WM_HINTS", False);
    MWMHints mwmhints;
    mwmhints.flags = MWM_HINTS_DECORATIONS;
    if (want_fullscreen) {
        mwmhints.decorations = 0;
    } else {
        mwmhints.decorations = 1;
    }
    XChangeProperty (win->display, win->window,
                     XA_MOTIF_WM_HINTS, XA_MOTIF_WM_HINTS,
                     32, PropModeReplace,
                     (unsigned char *) &mwmhints, PROP_MWM_HINTS_ELEMENTS);

    /* Set the input masks */
    if (want_fullscreen) {
        XSelectInput (win->display, win->window,
                      ExposureMask | KeyPressMask | ButtonPressMask |
                      StructureNotifyMask | FocusChangeMask |
                      PointerMotionMask);
    } else {
        XSelectInput (win->display, win->window,
                      ExposureMask | KeyPressMask | ButtonPressMask |
                      StructureNotifyMask | PointerMotionMask);
    }

    /* Resize and move the window. */
    XMoveResizeWindow (win->display, win->window, win->window_xpos,
                       win->window_ypos, win->window_width,
                       win->window_height);

    /* Raise the window */
    XRaiseWindow (win->display, win->window);

    /* Set the input focus */
    XSetInputFocus (win->display, win->window, RevertToNone, CurrentTime);

    XFlush (win->display);
    XUnlockDisplay (win->display);

    /* Send size change event to xine. */
    xine_port_send_gui_data (win->window_class.video_port,
                             XINE_GUI_SEND_DRAWABLE_CHANGED,
                             (void *) win->window);

    show_mouse_cursor (win);

    win->is_fullscreen = want_fullscreen;

    return 1;
}

static void
handle_event (x11_window_t * win, XEvent * event)
{
    switch (event->type) {
        case Expose:
            if (event->xexpose.count == 0) {
                xine_port_send_gui_data (win->window_class.video_port,
                                         XINE_GUI_SEND_EXPOSE_EVENT, event);
            }
            break;
        case ConfigureNotify:
            {
                XConfigureEvent *cev = (XConfigureEvent *) event;
                Window tmp_win;
                if (win->is_fullscreen)
                    break;
                win->window_width = cev->width;
                win->window_height = cev->height;
                if ((cev->x == 0) && (cev->y == 0)) {
                    XLockDisplay (cev->display);
                    XTranslateCoordinates (cev->display, cev->window,
                                           RootWindow (cev->display,
                                                       win->screen), 0, 0,
                                           &win->window_xpos,
                                           &win->window_ypos, &tmp_win);
                    XUnlockDisplay (cev->display);
                } else {
                    win->window_xpos = cev->x;
                    win->window_ypos = cev->y;
                }

                win->window_class.output_width = win->window_width;
                win->window_class.output_height = win->window_height;
                if (win->event_handler) {
                    oxine_event_t ev;
                    ev.type = OXINE_EVENT_FRAME_FORMAT_CHANGED;
                    win->event_handler (win->event_handler_data, &ev);
                }
            }
            break;
        case ClientMessage:
            /* We got a window deletion message from our window manager. */
            if (((XClientMessageEvent *) event)->data.l[0] == win->wm_del_win)
                win->keep_going = 0;
            break;
        case MotionNotify:
            if (win->event_handler) {
                XMotionEvent *mevent = (XMotionEvent *) event;

                oxine_event_t ev;
                ev.type = OXINE_EVENT_MOTION;
                ev.data.where.x = mevent->x - win->window_class.output_xpos;
                ev.data.where.y = mevent->y - win->window_class.output_ypos;

                win->event_handler (win->event_handler_data, &ev);
            }

            show_mouse_cursor (win);
            break;
        case ButtonPress:
            if (win->event_handler) {
                XButtonEvent *bevent = (XButtonEvent *) event;

                oxine_event_t ev;
                ev.type = OXINE_EVENT_BUTTON;
                ev.data.where.x = bevent->x - win->window_class.output_xpos;
                ev.data.where.y = bevent->y - win->window_class.output_ypos;
                switch (bevent->button) {
                    case Button1:
                        ev.source.button = OXINE_BUTTON1;
                        break;
                    case Button2:
                    case Button3:
                        ev.source.key = OXINE_KEY_MENU_OSD;
                        ev.type = OXINE_EVENT_KEY;
                        break;
                    case Button4:
                        ev.source.button = OXINE_BUTTON4;
                        break;
                    case Button5:
                        ev.source.button = OXINE_BUTTON5;
                        break;
                    default:
                        ev.source.button = OXINE_BUTTON_NULL;
                        break;
                }
                win->event_handler (win->event_handler_data, &ev);
            }

            show_mouse_cursor (win);
            break;
        case KeyPress:
            if (win->event_handler) {
                XKeyEvent *kevent = (XKeyEvent *) event;

                KeySym key;
                char kbuf[256];
                XLockDisplay (kevent->display);
                XLookupString (kevent, kbuf, sizeof (kbuf), &key, NULL);
                XUnlockDisplay (kevent->display);

#ifdef DEBUG_MODIFIERS
                if (event->xkey.state & ControlMask)
                    debug ("detected modifier: CTRL");
                if (event->xkey.state & Mod1Mask)
                    debug ("detected modifier: MOD1");
                if (event->xkey.state & Mod2Mask)
                    debug ("detected modifier: MOD2");
                if (event->xkey.state & Mod3Mask)
                    debug ("detected modifier: MOD3");
#endif

                oxine_event_t ev;
                ev.type = OXINE_EVENT_KEY;
                switch (key) {
                    case XK_Q:
                        win->keep_going = 0;
                        break;
                    case XK_F1:
                        ev.source.key = OXINE_KEY_MENU1;
                        break;
                    case XK_F2:
                        ev.source.key = OXINE_KEY_MENU2;
                        break;
                    case XK_F3:
                        ev.source.key = OXINE_KEY_MENU3;
                        break;
                    case XK_F4:
                        ev.source.key = OXINE_KEY_MENU4;
                        break;
                    case XK_F5:
                        ev.source.key = OXINE_KEY_MENU5;
                        break;
                    case XK_F6:
                        ev.source.key = OXINE_KEY_MENU6;
                        break;
                    case XK_F7:
                        ev.source.key = OXINE_KEY_MENU7;
                        break;
                    case XK_0:
                        ev.source.key = OXINE_KEY_0;
                        break;
                    case XK_1:
                        ev.source.key = OXINE_KEY_1;
                        break;
                    case XK_2:
                        ev.source.key = OXINE_KEY_2;
                        break;
                    case XK_3:
                        ev.source.key = OXINE_KEY_3;
                        break;
                    case XK_4:
                        ev.source.key = OXINE_KEY_4;
                        break;
                    case XK_5:
                        ev.source.key = OXINE_KEY_5;
                        break;
                    case XK_6:
                        ev.source.key = OXINE_KEY_6;
                        break;
                    case XK_7:
                        ev.source.key = OXINE_KEY_7;
                        break;
                    case XK_8:
                        ev.source.key = OXINE_KEY_8;
                        break;
                    case XK_9:
                        ev.source.key = OXINE_KEY_9;
                        break;
                    case XK_z:
                    case XK_y:
                        if ((event->xkey.state & ControlMask)
                            && (event->xkey.state & Mod1Mask)) {
                            ev.source.key = OXINE_KEY_ZOOM_RESET;
                        } else if (event->xkey.state & ControlMask) {
                            ev.source.key = OXINE_KEY_ZOOM_X;
                            ev.data.how = +1;
                        } else if (event->xkey.state & Mod1Mask) {
                            ev.source.key = OXINE_KEY_ZOOM_Y;
                            ev.data.how = +1;
                        } else {
                            ev.source.key = OXINE_KEY_ZOOM;
                            ev.data.how = +1;
                        }
                        break;
                    case XK_Z:
                    case XK_Y:
                        if ((event->xkey.state & ControlMask)
                            && (event->xkey.state & Mod1Mask)) {
                            ev.source.key = OXINE_KEY_ZOOM_RESET;
                        } else if (event->xkey.state & ControlMask) {
                            ev.source.key = OXINE_KEY_ZOOM_X;
                            ev.data.how = -1;
                        } else if (event->xkey.state & Mod1Mask) {
                            ev.source.key = OXINE_KEY_ZOOM_Y;
                            ev.data.how = -1;
                        } else {
                            ev.source.key = OXINE_KEY_ZOOM;
                            ev.data.how = -1;
                        }
                        break;
                    case XK_x:
                        ev.source.key = OXINE_KEY_AUDIO_CHANNEL;
                        ev.data.how = +1;
                        break;
                    case XK_X:
                        ev.source.key = OXINE_KEY_AUDIO_CHANNEL;
                        ev.data.how = -1;
                        break;
                    case XK_c:
                        ev.source.key = OXINE_KEY_AUDIO_OFFSET;
                        ev.data.how = +9;
                        break;
                    case XK_C:
                        ev.source.key = OXINE_KEY_AUDIO_OFFSET;
                        ev.data.how = -9;
                        break;
                    case XK_r:
                        ev.source.key = OXINE_KEY_SPU_CHANNEL;
                        ev.data.how = +1;
                        break;
                    case XK_R:
                        ev.source.key = OXINE_KEY_SPU_CHANNEL;
                        ev.data.how = -1;
                        break;
                    case XK_t:
                        ev.source.key = OXINE_KEY_SPU_OFFSET;
                        ev.data.how = +9;
                        break;
                    case XK_T:
                        ev.source.key = OXINE_KEY_SPU_OFFSET;
                        ev.data.how = -9;
                        break;
                    case XK_v:
                    case XK_plus:
                    case XK_KP_Add:
                        ev.source.key = OXINE_KEY_VOLUME;
                        ev.data.how = +5;
                        break;
                    case XK_V:
                    case XK_minus:
                    case XK_KP_Subtract:
                        ev.source.key = OXINE_KEY_VOLUME;
                        ev.data.how = -5;
                        break;
                    case XK_H:
                    case XK_h:
                    case XK_question:
                        ev.source.key = OXINE_KEY_HELP;
                        break;
                    case XK_S:
                    case XK_s:
                        ev.source.key = OXINE_KEY_STOP;
                        break;
                    case XK_q:
                        ev.source.key = OXINE_KEY_SHUTDOWN;
                        break;
                    case XK_O:
                    case XK_o:
                        ev.source.key = OXINE_KEY_MENU_OSD;
                        break;
                    case XK_E:
                    case XK_e:
                        ev.source.key = OXINE_KEY_EJECT;
                        break;
                    case XK_A:
                    case XK_a:
                        ev.source.key = OXINE_KEY_TOGGLE_ASPECT_RATIO;
                        break;
                    case XK_I:
                    case XK_i:
                        ev.source.key = OXINE_KEY_VO_DEINTERLACE;
                        break;
                    case XK_F:
                    case XK_f:
                        ev.source.key = OXINE_KEY_FULLSCREEN;
                        break;
                    case XK_M:
                    case XK_m:
                        ev.source.key = OXINE_KEY_VOLMUTE;
                        break;
                    case XK_Pause:
                        ev.source.key = OXINE_KEY_PAUSE;
                        break;
                    case XK_space:
                        ev.source.key = OXINE_KEY_SELECT;
                        break;
                    case XK_Page_Up:
                    case XK_p:
                        ev.source.key = OXINE_KEY_PREV;
                        break;
                    case XK_Page_Down:
                    case XK_n:
                        ev.source.key = OXINE_KEY_NEXT;
                        break;
                    case XK_Escape:
                        ev.source.key = OXINE_KEY_MENU_MAIN;
                        break;
                    case XK_Delete:
                        ev.source.key = OXINE_KEY_REMOVE;
                        break;
                    case XK_Up:
                        ev.source.key = OXINE_KEY_SPEED_UP;
                        ev.data.how = +1;
                        break;
                    case XK_Down:
                        ev.source.key = OXINE_KEY_SPEED_DOWN;
                        ev.data.how = -1;
                        break;
                    case XK_Right:
                        ev.source.key = OXINE_KEY_FFORWARD;
                        ev.data.how = +30;
                        break;
                    case XK_Left:
                        ev.source.key = OXINE_KEY_REWIND;
                        ev.data.how = -30;
                        break;
                    case XK_KP_Home:
                    case XK_Home:
                        ev.source.key = OXINE_KEY_FIRST;
                        break;
                    case XK_KP_End:
                    case XK_End:
                        ev.source.key = OXINE_KEY_LAST;
                        break;
                    case XK_KP_Page_Up:
                        ev.source.key = OXINE_KEY_PAGE_UP;
                        break;
                    case XK_KP_Page_Down:
                        ev.source.key = OXINE_KEY_PAGE_DOWN;
                        break;
                    case XK_BackSpace:
                        ev.source.key = OXINE_KEY_BACK;
                        break;
                    case XK_KP_8:
                    case XK_KP_Up:
                        ev.source.key = OXINE_KEY_UP;
                        break;
                    case XK_KP_2:
                    case XK_KP_Down:
                        ev.source.key = OXINE_KEY_DOWN;
                        break;
                    case XK_KP_4:
                    case XK_KP_Left:
                        ev.source.key = OXINE_KEY_LEFT;
                        break;
                    case XK_KP_6:
                    case XK_KP_Right:
                        ev.source.key = OXINE_KEY_RIGHT;
                        break;
                    case XK_KP_0:
                    case XK_KP_Insert:
                        ev.source.key = OXINE_KEY_SELECT;
                        break;
                    case XK_Return:
                    case XK_KP_Enter:
                        ev.source.key = OXINE_KEY_ACTIVATE;
                        break;
                    default:
                        ev.source.key = OXINE_KEY_NULL;
                }
                win->event_handler (win->event_handler_data, &ev);
            }
            break;
        default:
            break;
    }

}


/*
 * The X window event loop
 */
static void
event_loop (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;

    win->keep_going = 1;
    while (win->keep_going) {
        XEvent event;
        XNextEvent (win->display, &event);
        handle_event (win, &event);
    }
    win->keep_going = 0;
}


static void
set_event_handler (odk_window_t * this, odk_cb cb, void *data)
{
    x11_window_t *win = (x11_window_t *) this;
    win->event_handler = cb;
    win->event_handler_data = data;
}


/*
 * Makes the window visible.
 * Returns 1 on success and 0 on error.
 */
static int
show_window (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;
    XLockDisplay (win->display);
    XMapWindow (win->display, win->window);
    XUnlockDisplay (win->display);
    return 1;
}


/*
 * This hides the current window.
 * Returns 1 on success and 0 on error.
 */
static int
hide_window (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;
    XLockDisplay (win->display);
    XUnmapWindow (win->display, win->window);
    XUnlockDisplay (win->display);
    return 1;
}


/*
 * This changes the size and place of the current window.
 * Returns 1 on success and 0 on error.
 */
static int
displace_window (odk_window_t * this, int x, int y, int w, int h)
{
    fatal ("Tell the developers to write this function!!");
    abort ();
    return 0;
}


static void
stop_event_loop (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;
    XEvent event;
    event.type = Expose;
    event.xexpose.display = win->display;
    event.xexpose.window = win->window;
    event.xexpose.x = 0;
    event.xexpose.y = 0;
    event.xexpose.width = 1;
    event.xexpose.height = 1;
    win->keep_going = 0;
    XLockDisplay (win->display);
    XSendEvent (win->display, win->window, False, ExposureMask, &event);
    XFlush (win->display);
    XUnlockDisplay (win->display);
}


/*
 * This disposes the current window.
 * Returns 1 on success and 0 on error.
 */
static int
dispose_window (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;
    /* Stop the event loop. */
    if (win->keep_going) {
        stop_event_loop (this);
    }
    /* We need to lock the job mutex, because we modify the screensaver and
     * the mouse cursor jobs. */
    lock_job_mutex ();
    cancel_job (win->screensaver_job);
    cancel_job (win->mouse_cursor_job);
    unlock_job_mutex ();

    /* Close the video port and destroy the window. */
    xine_close_video_driver (this->xine, this->video_port);
    this->video_port = NULL;
    destroy_x11_window (win);
    disconnect_x11_server (win);

    ho_free (win);

    return 1;
}

odk_window_t *x11_construct (xine_t * xine, const char *driver);

/*
 * Constructor
 */
odk_window_t *
x11_construct (xine_t * xine, const char *driver)
{
    x11_window_t *win = ho_new (x11_window_t);
    if (!connect_x11_server (win)) {
        ho_free (win);
        return NULL;
    }

    int window_width =
        xine_config_register_num (xine, "gui.window_width", 800,
                                  _("non-fullscreen window width"),
                                  _("The width of the non-fullscreen window"),
                                  10, NULL, NULL);
    int window_height =
        xine_config_register_num (xine, "gui.window_height", 600,
                                  _("non-fullscreen window height"),
                                  _
                                  ("The height of the non-fullscreen window"),
                                  10, NULL, NULL);
    int fullscreen_underscan =
        xine_config_register_range (xine, "gui.fullscreen_underscan", 100, 0,
                                    100,
                                    _("percentage of fullscreen underscan"),
                                    _("percentage of fullscreen underscan"),
                                    10, NULL, NULL);

    create_x11_window (win, 0, 0, window_width, window_height);
    win->window_class.xine = xine;
    win->fullscreen_underscan = (double) fullscreen_underscan / (double) 100;

    /* Create video port */
    win->window_class.video_port =
        xine_open_video_driver (xine, driver, XINE_VISUAL_TYPE_X11,
                                (void *) &(win->visual));
    if (!win->window_class.video_port) {
        warn (_("Failed to open video port '%s', falling back to 'auto'."),
              driver);
        win->window_class.video_port =
            xine_open_video_driver (xine, NULL, XINE_VISUAL_TYPE_X11,
                                    (void *) &(win->visual));
    }

    if (!win->window_class.video_port) {
        error (_("Failed to open video port!"));
        destroy_x11_window (win);
        ho_free (win);
        return NULL;
    }

    win->window_class.show = show_window;
    win->window_class.hide = hide_window;
    win->window_class.displace = displace_window;
    win->window_class.dispose = dispose_window;
    win->window_class.fullscreen = fullscreen;
    win->window_class.is_fullscreen = is_fullscreen;
    win->window_class.event_loop = event_loop;
    win->window_class.stop_event_loop = stop_event_loop;
    win->window_class.set_event_handler = set_event_handler;

    /* Screensaver disabling stuff */
#ifdef HAVE_XTEST
    {
        int a;
        int b;
        int c;
        int d;
        win->have_xtest = XTestQueryExtension (win->display, &a, &b, &c, &d);
        if (win->have_xtest == True)
            win->xtest_keycode = XKeysymToKeycode (win->display, XK_Shift_L);
    }
#endif

    win->screensaver_job = schedule_job (6000, screensaver_job, win);
    win->mouse_cursor_job = schedule_job (6000, hide_mouse_cursor, win);

    return (odk_window_t *) win;
}
