
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: playlist.c,v 1.34 2006/01/19 09:31:32 mschwerin Exp $
 *
 */
#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "oxine.h"
#include "playlist.h"

extern oxine_t *oxine;

/* 
 * ***************************************************************************
 * Name:            playitem_free_cb
 * Access:          private
 *
 * Description:     This is the callback passed to l_list_free and
 *                  l_list_clear to free the memory of the playitem.
 * ***************************************************************************
 */
static void
playitem_free_cb (void *data)
{
    playitem_t *playitem = (playitem_t *) data;

    assert (playitem);

    if (playitem->title)
        ho_free (playitem->title);
    if (playitem->mrl)
        ho_free (playitem->mrl);
    if (playitem->subtitle_mrl)
        ho_free (playitem->subtitle_mrl);

    ho_free (playitem);
}


/* 
 * ***************************************************************************
 * Name:            playlist_new
 * Access:          public
 *
 * Description:     Creates a new playlist.
 * ***************************************************************************
 */
playlist_t *
playlist_new (void)
{
    playlist_t *playlist = ho_new (playlist_t);

    playlist->list = l_list_new ();
    playlist->current = NULL;
    playlist->playmode = PLAYLIST_MODE_NORMAL;

    return playlist;
}


/* 
 * ***************************************************************************
 * Name:            playlist_clear
 * Access:          public
 *
 * Description:     Removes all playitems.
 * ***************************************************************************
 */
void
playlist_clear (playlist_t * playlist)
{
    assert (playlist);

    l_list_clear (playlist->list, playitem_free_cb);
    playlist->current = NULL;
}


/* 
 * ***************************************************************************
 * Name:            playlist_free
 * Access:          public
 *
 * Description:     Removes all playitems and frees the list.
 * ***************************************************************************
 */
void
playlist_free (playlist_t * playlist)
{
    assert (playlist);

    l_list_free (playlist->list, playitem_free_cb);
    ho_free (playlist);
}


static char *
guess_subtitle_mrl (const char *mrl)
{
    char *know_subs = "sub,srt,asc,smi,ssa,txt";

    char *_mrl = (char *) mrl;
    if (!strncasecmp (_mrl, "file:", 5))
        _mrl += 5;

    char autosub[1024];
    memset (&autosub, 0, sizeof (autosub));
    snprintf (autosub, sizeof (autosub), "%s", _mrl);

    char *ending;
    if ((ending = strrchr (autosub, '.')))
        ending++;
    else {
        ending = autosub + strlen (autosub);
        *ending++ = '.';
    }

    char *sub = NULL;
    char *subs = ho_strdup (know_subs);
    char *ext = strtok (subs, ",");
    while (ext) {
        sprintf (ending, "%s", ext);
        *(ending + strlen (ext) + 1) = '\0';

        struct stat pstat;
        if (((stat (autosub, &pstat)) > -1)
            && (S_ISREG (pstat.st_mode))
            && strcmp (autosub, _mrl)) {
            sub = ho_strdup (autosub);

#ifdef DEBUG
            char *__mrl = ho_strdup (mrl);
            char *__sub = ho_strdup (sub);
            debug ("found subtitle file");
            debug ("         mrl: %s", basename (__mrl));
            debug ("    subtitle: %s", basename (__sub));
            ho_free (__mrl);
            ho_free (__sub);
#endif

            break;
        }

        ext = strtok (NULL, ",");
    }
    ho_free (subs);

    return sub;
}


/* 
 * ***************************************************************************
 * Name:            playlist_insert
 * Access:          public
 *
 * Description:     Adds a new item to the list.
 * ***************************************************************************
 */
playitem_t *
playlist_insert (playlist_t * playlist, int position,
                 const char *title, const char *mrl)
{
    assert (playlist);

    playitem_t *playitem = ho_new (playitem_t);

    if (title)
        playitem->title = ho_strdup (title);
    if (mrl)
        playitem->mrl = ho_strdup (mrl);

    int subtitle_autoload =
        xine_config_register_bool (oxine->xine, "gui.subtitle_autoload", 1,
                                   _("Subtitle autoloading"),
                                   _("Automatically load subtitles "
                                     "if they exist."), 10, NULL, NULL);

    if (subtitle_autoload)
        playitem->subtitle_mrl = guess_subtitle_mrl (mrl);

    l_list_insert (playlist->list, position, playitem);

    return playitem;
}


/* 
 * ***************************************************************************
 * Name:            playlist_add
 * Access:          public
 *
 * Description:     Adds a new item to the list.
 * ***************************************************************************
 */
playitem_t *
playlist_add (playlist_t * playlist, const char *title, const char *mrl)
{
    assert (playlist);

    playitem_t *playitem = ho_new (playitem_t);

    if (title)
        playitem->title = ho_strdup (title);
    if (mrl)
        playitem->mrl = ho_strdup (mrl);

    int subtitle_autoload =
        xine_config_register_bool (oxine->xine, "gui.subtitle_autoload", 1,
                                   _("Subtitle autoloading"),
                                   _("Automatically load subtitles "
                                     "if they exist."), 10, NULL, NULL);

    if (subtitle_autoload)
        playitem->subtitle_mrl = guess_subtitle_mrl (mrl);

    l_list_append (playlist->list, playitem);

    return playitem;
}


/* 
 * ***************************************************************************
 * Name:            playlist_add_fileitem
 * Access:          public
 *
 * Description:     Adds a fileitem to the playlist.
 * ***************************************************************************
 */
playitem_t *
playlist_add_fileitem (playlist_t * playlist, fileitem_t * fileitem)
{
    assert (playlist);
    assert (fileitem);

    playitem_t *playitem = NULL;

    switch (fileitem->type) {
        case FILE_TYPE_REGULAR:
            playitem =
                playlist_add (playlist, fileitem->title, fileitem->mrl);
            break;
        case FILE_TYPE_DIRECTORY:
        case FILE_TYPE_MOUNTPOINT:
        case FILE_TYPE_MEDIAMARKS:
        case FILE_TYPE_M3U:
        case FILE_TYPE_AUTODISC:
            if (fileitem->sublist) {
                fileitem_t *subitem = filelist_first (fileitem->sublist);
                while (subitem) {
                    filelist_expand (fileitem->sublist, subitem);
                    playlist_add_fileitem (playlist, subitem);
                    subitem = filelist_next (fileitem->sublist, subitem);
                }
            } else {
                fatal ("You should call have filelist_expand!");
                abort ();
            }
            break;
        default:
            break;
    }

    return playitem;
}


/* 
 * ***************************************************************************
 * Name:            playlist_move_up
 * Access:          public
 *
 * Description:     Moves the item one position towards the top of list.
 * ***************************************************************************
 */
void
playlist_move_up (playlist_t * playlist, playitem_t * playitem)
{
    assert (playlist);
    assert (playitem);

    l_list_move_up (playlist->list, playitem);
}


/* 
 * ***************************************************************************
 * Name:            playlist_move_down
 * Access:          public
 *
 * Description:     Moves the item one position towards the end of list.
 * ***************************************************************************
 */
void
playlist_move_down (playlist_t * playlist, playitem_t * playitem)
{
    assert (playlist);
    assert (playitem);

    l_list_move_down (playlist->list, playitem);
}


/* 
 * ***************************************************************************
 * Name:            playlist_remove
 * Access:          public
 *
 * Description:     Removes the item from the list.
 * ***************************************************************************
 */
void
playlist_remove (playlist_t * playlist, playitem_t * playitem)
{
    assert (playlist);
    assert (playitem);

    l_list_remove (playlist->list, playitem);

    if (playitem->title) {
        ho_free (playitem->title);
    }
    if (playitem->mrl) {
        ho_free (playitem->mrl);
    }
    ho_free (playitem);
}


/* 
 * ***************************************************************************
 * Name:            playlist_random_next
 * Access:          private
 *
 * Description:     Get the next entry to play. 
 * ***************************************************************************
 */
static playitem_t *
playlist_random_next (playlist_t * playlist)
{
    assert (playlist);

    if (!playlist->list->length)
        return NULL;

    int num_played = 0;
    playitem_t *item = l_list_first (playlist->list);
    while (item) {
        if (item->played)
            num_played++;
        item = l_list_next (playlist->list, item);
    }
    if (num_played == playlist->list->length)
        return NULL;

    int found = FALSE;
    while (!found) {
        int next =
            (int) ((double) playlist->list->length *
                   ((double) random () / (double) RAND_MAX));
        playitem_t *item = l_list_first (playlist->list);
        while (item && next > 0) {
            next -= 1;
            item = l_list_next (playlist->list, item);
        }
        if (next == 0) {
            if (!item->played) {
                return item;
            }
        }
    }

    return NULL;
}


/* 
 * ***************************************************************************
 * Name:            playlist_get_first
 * Access:          public
 *
 * Description:     Returns the first entry to play. If we're currently in
 *                  random mode it may not be the first entry in the list.
 * ***************************************************************************
 */
playitem_t *
playlist_get_first (playlist_t * playlist)
{
    assert (playlist);

    switch (playlist->playmode) {
        case PLAYLIST_MODE_NORMAL:
        case PLAYLIST_MODE_REPEAT:
            playlist->current = l_list_first (playlist->list);
            break;
        case PLAYLIST_MODE_RANDOM:
            srandom ((unsigned int) time (NULL));
            playlist->current = l_list_first (playlist->list);
            while (playlist->current) {
                playlist->current->played = FALSE;
                playlist->current =
                    l_list_next (playlist->list, playlist->current);
            }
            playlist->current = playlist_random_next (playlist);
            break;
    }
    if (playlist->current)
        playlist->current->played = TRUE;
    return playlist->current;
}


/* 
 * ***************************************************************************
 * Name:            playlist_get_prev
 * Access:          public
 *
 * Description:     Get previous entry to play.
 * ***************************************************************************
 */
playitem_t *
playlist_get_prev (playlist_t * playlist)
{
    assert (playlist);

    switch (playlist->playmode) {
        case PLAYLIST_MODE_NORMAL:
        case PLAYLIST_MODE_REPEAT:
            playlist->current =
                l_list_prev (playlist->list, playlist->current);
            break;
        default:
            break;
    }
    if (playlist->current)
        playlist->current->played = TRUE;
    return playlist->current;
}


/* 
 * ***************************************************************************
 * Name:            playlist_get_next
 * Access:          public
 *
 * Description:     Get next entry to play. In random mode this may not be the
 *                  next entry in the list.
 * ***************************************************************************
 */
playitem_t *
playlist_get_next (playlist_t * playlist)
{
    assert (playlist);

    switch (playlist->playmode) {
        case PLAYLIST_MODE_NORMAL:
            playlist->current =
                l_list_next (playlist->list, playlist->current);
            break;
        case PLAYLIST_MODE_REPEAT:
            playlist->current =
                l_list_next (playlist->list, playlist->current);
            if (!playlist->current)
                playlist->current = l_list_first (playlist->list);
            break;
        case PLAYLIST_MODE_RANDOM:
            playlist->current = playlist_random_next (playlist);
            break;
    }
    if (playlist->current)
        playlist->current->played = TRUE;
    return playlist->current;
}


/* 
 * ***************************************************************************
 * Name:            playlist_set_current
 * Access:          public
 *
 * Description:     Sets the current pointer.
 * ***************************************************************************
 */
playitem_t *
playlist_set_current (playlist_t * playlist, playitem_t * current)
{
    assert (playlist);

    playlist->current = current;
    if (playlist->current)
        playlist->current->played = TRUE;

    return playlist->current;
}


/* 
 * ***************************************************************************
 * Name:            playlist_set_current_pos
 * Access:          public
 *
 * Description:     Sets the current pointer.
 * ***************************************************************************
 */
playitem_t *
playlist_set_current_pos (playlist_t * playlist, int pos)
{
    assert (playlist);
    assert (pos < playlist_length (playlist));

    int i = 0;
    playitem_t *cur = l_list_first (playlist->list);
    while (cur) {
        if (i == pos) {
            playlist->current = cur;
            playlist->current->played = TRUE;
            break;
        }
        cur = l_list_next (playlist->list, cur);
        i++;
    }

    return playlist->current;
}


/* 
 * ***************************************************************************
 * Name:            playlist_current_pos
 * Access:          public
 *
 * Description:     Returns the position of the currently playing entry.
 * ***************************************************************************
 */
int
playlist_current_pos (playlist_t * playlist)
{
    assert (playlist);

    int i = 0;
    playitem_t *cur = l_list_first (playlist->list);
    while (cur) {
        if (cur == playlist->current)
            return i;
        cur = l_list_next (playlist->list, cur);
        i++;
    }
    return -1;
}


/* 
 * ***************************************************************************
 * Name:            playlist_first
 * Access:          public
 *
 * Description:     Returns the real first entry.
 * ***************************************************************************
 */
playitem_t *
playlist_first (playlist_t * playlist)
{
    assert (playlist);

    return l_list_first (playlist->list);
}


/* 
 * ***************************************************************************
 * Name:            playlist_next
 * Access:          public
 *
 * Description:     Returns the real next entry.
 * ***************************************************************************
 */
playitem_t *
playlist_next (playlist_t * playlist, playitem_t * current)
{
    assert (playlist);
    assert (current);

    return l_list_next (playlist->list, current);
}


/* 
 * ***************************************************************************
 * Name:            playlist_length
 * Access:          public
 *
 * Description:     Returns the number of entries in the list.
 * ***************************************************************************
 */
int
playlist_length (playlist_t * playlist)
{
    assert (playlist);

    return l_list_length (playlist->list);
}
