///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file ViewportConfiguration.h
 * \brief Contains the definition of the Core::ViewportRecord and the Core::ViewportConfiguration class.
 */

#ifndef __VIEWPORT_CONFIGURATION_H
#define __VIEWPORT_CONFIGURATION_H

#include <core/Core.h>
#include <core/viewport/Viewport.h>

namespace Core {


/**
 * \brief Stores the configuration for a single viewport.
 *
 * This class is used to serialize the viewing configuration of a Viewport
 * in the scene file. The configurations for all viewports are centralized in
 * an instance of the ViewportConfiguration class.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT ViewportRecord : public RefTarget
{
	Q_OBJECT

public:

	/// Default constructor.
	ViewportRecord(bool isLoading = false);

	/// Initialization constructor.
	ViewportRecord(Viewport::ViewportType viewType, Viewport::ViewportShadingMode shadingMode, bool showGrid,
		FloatType fieldOfView, const AffineTransformation& viewMatrix, bool showRenderFrame);

	/// Viewport constructor.
	ViewportRecord(Viewport* viewport) : RefTarget(false), _viewport(viewport) {
		INIT_PROPERTY_FIELD(ViewportRecord, _viewNode);
	}

	/// \brief Returns the view type of the viewport.
	/// \return The type of view used in the viewport.
	/// \sa setViewType()
	Viewport::ViewportType viewType() const { return _viewType; }

	/// \brief Changes the view type.
	/// \param type The new view type.
	/// \note if \a type is set to Viewport::ViewportType::VIEW_SCENENODE then a view node should be set
	///       using setViewNode().
	void setViewType(Viewport::ViewportType type);

	/// \brief Returns the current shading mode used for scene rendering.
	/// \return The current shading mode.
	/// \sa setShadingMode()
	Viewport::ViewportShadingMode shadingMode() const { return _shadingMode; }

	/// \brief Sets the shading mode to use for scene rendering.
	/// \param mode The new shading mode.
	/// \note This method should not be called while the scene is being rendered.
	///       It causes the viewport to be updated.
	/// \sa shadingMode()
	void setShadingMode(Viewport::ViewportShadingMode mode);

	/// \brief Returns whether the grid display is currently enabled.
	/// \return \c true if the grid is displayed in the viewport; \c false otherwise.
	/// \sa setGridShown()
	bool isGridShown() const { return _showGrid; }

	/// \brief Turns the grid display on or off.
	/// \param visible Controls the display of the construction grid.
	/// \sa setVisible()
	void setGridShown(bool visible);

	/// \brief Sets the orientation of the grid plane.
	/// \param gridmat The transformation matrix that defines the grid orientation.
	///                It transforms from grid coordinates to world coordinates.
	void setGridMatrix(const AffineTransformation& gridmat);

	/// \brief Returns the field of view value of the viewport.
	/// \return Horizontal camera angle in radians if the viewport uses a perspective projection or
	///         the field of view in the horizontal direction in world units if the viewport
	///         uses an orthogonal projection.
	FloatType fieldOfView() const;

	/// \brief Sets the zoom of the viewport.
	/// \param fov Horizontal camera angle in radians if the viewport uses a perspective projection or
	///            the field of view in the horizontal direction in world units if the viewport
	///            uses an orthogonal projection.
	void setFieldOfView(FloatType fov);

	/// \brief Returns the world to camera (view) transformation without projection.
	const AffineTransformation& viewMatrix() const;

	/// \brief Returns the camera (view) to world transformation without the projection part.
	AffineTransformation inverseViewMatrix() const;

	/// \brief Sets the view matrix of the viewport.
	void setViewMatrix(const AffineTransformation& tm);

	/// \brief Returns whether the render frame is shown in the viewport.
	/// \sa setRenderFrameShown()
	bool renderFrameShown() const { return _showRenderFrame; }

	/// \brief Sets whether the render frame is shown in the viewport.
	/// \param show Specifies whether the render frame is shown or not.
	/// \sa renderFrameShown()
	void setRenderFrameShown(bool show);

	/// \brief Gets the scene node used as camera for the viewport.
	/// \return The scene node or \c NULL if no scene node has been set.
	/// \sa setViewNode()
	ObjectNode* viewNode() const { return _viewNode; }

	/// \brief Sets the scene node used as camera for the viewport.
	/// \param node The scene node to be used as view point. The scene node must be a camera object and the
	///             viewport type must have been set to ViewportType::VIEW_SCENENODE using setViewType()
	///             to enable camera mode for this viewport.
	/// \sa viewNode(), setViewType()
	void setViewNode(ObjectNode* node) { _viewNode = node; }

	/// \brief Returns a description the viewport's view at the given animation time.
	/// \param time The animation time for which the view description is requested.
	/// \param aspectRatio Specifies the desired aspect ratio (height/width) of the output image.
	/// \param sceneBoundingBox The bounding box of the scene in world coordinates. This is used to calculate the near and far z-clipping planes.
	/// \return This structure can be used by a PluginRenderer to render the scene as it is currently displayed in the viewport.
	CameraViewDescription getViewDescription(TimeTicks time, FloatType aspectRatio, const Box3& sceneBoundingBox = Box3());

	/// \brief Returns whether a explicitely set center point is used by the orbit navigation mode.
	/// \return \c true if a center point is set.
	/// \sa orbitCenter(), setUseOrbitCenter()
	bool useOrbitCenter() const { return _useOrbitCenter; }

	/// \brief Sets whether the explicitely set center point is used by the orbit navigation mode.
	/// \sa setOrbitCenter(), useOrbitCenter()
	void setUseOrbitCenter(bool enable) { _useOrbitCenter = enable; }

	/// \brief Returns the current center point for the orbit navigation mode.
	/// \return The center point in world space.
	/// \sa useOrbitCenter()
	///
	/// The center point is only used if it has been activated with a call to setUseOrbitCenter().
	const Point3& orbitCenter() const { return _orbitCenter; }

	/// \brief Sets the current center point to use for the orbit navigation mode.
	/// \param center The center point in world space.
	/// \sa setUseOrbitCenter()
	///
	/// The center point is only used if it is activated with a call to setUseOrbitCenter().
	void setOrbitCenter(const Point3& center) { _orbitCenter = center; }

protected:

	/// Saves the class' contents to the given stream.
	virtual void saveToStream(ObjectSaveStream& stream);
	/// Loads the class' contents from the given stream.
	virtual void loadFromStream(ObjectLoadStream& stream);
	/// Creates a copy of this object.
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

	/// This method is called when a reference target has sent a message.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg);

	/// Is called when the value of a reference field of this RefMaker changes.
	virtual void onRefTargetReplaced(const PropertyFieldDescriptor& field, RefTarget* oldTarget, RefTarget* newTarget);

private:

	/// The viewport this helper object belongs to.
	Viewport* _viewport;

	/// The type of the viewport (top, left, perspective etc.)
	Viewport::ViewportType _viewType;

	/// Shading mode of viewport (shaded, wireframe etc..)
	Viewport::ViewportShadingMode _shadingMode;

	/// Indicates whether the grid is activated.
	bool _showGrid;

	/// The zoom or field of view.
	FloatType _fieldOfView;

	/// World to camera (view) transformation without projection.
	AffineTransformation _viewMatrix;

	/// Indicates whether the rendering frame is shown.
	bool _showRenderFrame;

	/// The scene node (camera, light, etc.) that has been selected as the view node.
	ReferenceField<ObjectNode> _viewNode;

	/// The center point in world space used for the orbit navigation mode.
	Point3 _orbitCenter;

	/// Controls whether the explicit center point is used by the orbit navigation mode.
	bool _useOrbitCenter;

	DECLARE_SERIALIZABLE_PLUGIN_CLASS(ViewportRecord)
	DECLARE_REFERENCE_FIELD(_viewNode)

	friend class ViewportConfiguration;
};


/**
 * \brief This helper class is used to save the state of the viewports into the scene file.
 *
 * This class holds a collection of ViewportRecord objects, one for each viewport.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT ViewportConfiguration : public RefTarget
{
public:
	/// Constructor.
	ViewportConfiguration(bool isLoading = false) : RefTarget(isLoading), _activeViewport(-1), _maximizedViewport(-1) {
		INIT_PROPERTY_FIELD(ViewportConfiguration, _viewRecords);
	}

	/// Returns the records for each of the viewports.
	const QVector<ViewportRecord*>& records() const { return _viewRecords; }

	/// Add a record for a new viewport.
	void addViewport(const ViewportRecord::SmartPtr& newRecord) { _viewRecords.push_back(newRecord); }

	/// Return the index of the active viewport or -1 if there is no active viewport.
	int activeViewport() const { return _activeViewport; }

	/// Sets the index of the active viewport.
	void setActiveViewport(int index) { OVITO_ASSERT(index < _viewRecords.size()); _activeViewport = index; }

	/// Return the settings of the active viewport or NULL if there is no active viewport.
	ViewportRecord::SmartPtr activeViewportSettings() const {
		if(_activeViewport < 0 || _activeViewport >= _viewRecords.size()) return NULL;
		return _viewRecords[_activeViewport];
	}

	/// Return the index of the maximized viewport ot -1 if there is no maximized viewport.
	int maximizedViewport() const { return _maximizedViewport; }

	/// Sets the maximized viewport.
	void setMaximizedViewport(int index) { OVITO_ASSERT(index < _viewRecords.size()); _maximizedViewport = index; }

	/// The takes the current configuration of the viewports and saves it into this object.
	void saveConfiguration();

	/// This applies the saved configuration to the viewports in the viewport panel.
	void restoreConfiguration();

protected:

	/// Saves the class' contents to the given stream.
	virtual void saveToStream(ObjectSaveStream& stream);
	/// Loads the class' contents from the given stream.
	virtual void loadFromStream(ObjectLoadStream& stream);
	/// Creates a copy of this object.
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

private:

	/// One record for every viewport.
	VectorReferenceField<ViewportRecord> _viewRecords;

	/// The index of the active viewport. Can be -1 if no viewport is active.
	int _activeViewport;

	/// The index of the maximized viewport or -1.
	int _maximizedViewport;

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(ViewportConfiguration)
	DECLARE_VECTOR_REFERENCE_FIELD(_viewRecords)
};

};

#endif		// __VIEWPORT_CONFIGURATION_H
