/*
 * part-item.c
 *
 * 
 * Author: 
 *  Richard Hult <rhult@hem.passagen.se>
 * 
 *  http://www.dtek.chalmers.se/~d4hult/oregano/ 
 * 
 * Copyright (C) 1999,2000  Richard Hult 
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the 
 * License, or (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include <config.h>
#include <math.h>
#include <gnome.h>
#include <glade/glade.h>
#include <math.h>
#include "main.h"
#include "schematic-view.h"
#include "sheet-private.h"
#include "sheet-item.h"
#include "part-item.h"
#include "part-private.h"
#include "part-property.h"
#include "load-library.h"
#include "load-common.h"
#include "netlist.h"
#include "part-label.h"
#include "stock.h"
#include "dialogs.h"


#define NORMAL_COLOR "medium sea green"
#define SELECTED_COLOR "red"
#define O_DEBUG 0
	
static void part_item_class_init (PartItemClass *klass);
static void part_item_init (PartItem *gspart);
static void part_item_set_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void part_item_get_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void part_item_destroy (GtkObject *object);
static void part_item_moved (SheetItem *sheet_item);

static void edit_properties (SheetItem *object);
static void properties_cmd (GtkWidget *widget, SchematicView *sv);
static void edit_properties_unselect_row (GtkCList *clist, int row, int column, GdkEventButton *event, GladeXML *gui);
static void edit_properties_select_row (GtkCList *clist, int row, int column, GdkEventButton *event, GladeXML *gui);

static void selection_changed (PartItem *item, gboolean select, gpointer user_data);
static int select_idle_callback (PartItem *item);
static int deselect_idle_callback (PartItem *item);

static void update_canvas_labels (PartItem *part_item);

static gboolean is_in_area (SheetItem *object, SheetPos *p1, SheetPos *p2);
inline static void get_cached_bounds (PartItem *item, SheetPos *p1, SheetPos *p2);
static void show_labels (SheetItem *sheet_item, gboolean show);
static void part_item_paste (SchematicView *sv, ItemData *data);

static void part_rotated_callback (ItemData *data, int angle, SheetItem *item);
static void part_flipped_callback (ItemData *data, gboolean horizontal, SheetItem *sheet_item);

static void part_moved_callback (ItemData *data, SheetPos *pos, SheetItem *item);

static void part_item_place (SheetItem *item, SchematicView *sv);
static void part_item_place_ghost (SheetItem *item, SchematicView *sv);

static void create_canvas_items (GnomeCanvasGroup *group, LibraryPart *library_part);
static void create_canvas_labels (PartItem *item, Part *part);


enum {
	ARG_0,
	ARG_NAME,
	ARG_SYMNAME,
	ARG_LIBNAME,
	ARG_REFDES,
	ARG_TEMPLATE,
	ARG_MODEL
};

struct _PartItemPriv {
	guint cache_valid : 1;
	guint highlight : 1;
	
	GnomeCanvasGroup *label_group;
	GSList *label_items;

	/*
	 * Cached bounding box. This is used to make
	 * the rubberband selection a bit faster.
	 */
	SheetPos bbox_start;
	SheetPos bbox_end;
};

typedef struct {
	GnomeDialog *dialog;
	GtkCList *clist;
	GtkWidget *name_entry;
	GtkWidget *value_entry;
	int selected_row;
} PartPropDialog;

static PartPropDialog *prop_dialog = NULL;
static SheetItemClass *part_item_parent_class = NULL;

/*
 * This is the lower part of the object popup menu. It contains actions
 * that are specific for parts.
 */
static GnomeUIInfo part_popup_menu [] = {
	GNOMEUIINFO_SEPARATOR,
	
	{ GNOME_APP_UI_ITEM, N_("Properties"), N_("Edit the part's properties"), properties_cmd, NULL, NULL,
	  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP, 0, 0 },
	GNOMEUIINFO_END
};

GtkType
part_item_get_type ()
{
	static GtkType part_item_type = 0;
	
	if (!part_item_type) {
		GtkTypeInfo part_item_info = {
			"PartItem",
			sizeof (PartItem),
			sizeof (PartItemClass),
			(GtkClassInitFunc) part_item_class_init,
			(GtkObjectInitFunc) part_item_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL,
			NULL
		};
		part_item_type = gtk_type_unique (sheet_item_get_type (), &part_item_info);
	}
	return part_item_type;
}

static void
part_item_class_init (PartItemClass *part_item_class)
{
	GtkObjectClass *object_class;
	SheetItemClass *sheet_item_class;
	
	object_class = (GtkObjectClass *) part_item_class;
	sheet_item_class = (SheetItemClass *) part_item_class;
	part_item_parent_class = gtk_type_class (sheet_item_get_type ());

	gtk_object_add_arg_type ("PartItem::name", GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_NAME);
	gtk_object_add_arg_type ("PartItem::symbol_name", GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_SYMNAME);
	gtk_object_add_arg_type ("PartItem::library_name", GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_LIBNAME);
	gtk_object_add_arg_type ("PartItem::refdes", GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_REFDES);
	gtk_object_add_arg_type ("PartItem::template", GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_TEMPLATE);
	gtk_object_add_arg_type ("PartItem::model", GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_MODEL);

	object_class->set_arg = part_item_set_arg;
	object_class->get_arg = part_item_get_arg;
	object_class->destroy = part_item_destroy;

	sheet_item_class->moved = part_item_moved;
	sheet_item_class->is_in_area = is_in_area;
	sheet_item_class->show_labels = show_labels;
	sheet_item_class->paste = part_item_paste;
	sheet_item_class->edit_properties = edit_properties;
	sheet_item_class->selection_changed = (gpointer) selection_changed;

	sheet_item_class->place = part_item_place;
	sheet_item_class->place_ghost = part_item_place_ghost;

	sheet_item_class->context_menu = g_new0 (SheetItemMenu, 1);
	sheet_item_class->context_menu->menu = part_popup_menu;
	sheet_item_class->context_menu->size = sizeof (part_popup_menu) / sizeof (part_popup_menu[0]);
}

static void
part_item_init (PartItem *item)
{
	PartItemPriv *priv;

	priv = g_new0 (PartItemPriv, 1);

	priv->highlight = FALSE;
	priv->cache_valid = FALSE;

	item->priv = priv;
}

static void
part_item_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	PartItem *part_item;
	PartItemPriv *priv;

	g_return_if_fail (object != NULL);	
	g_return_if_fail (IS_PART_ITEM (object));
	
	part_item = PART_ITEM (object);
	priv = part_item->priv;

	switch (arg_id) {
	default:
		g_warning ("PartItem: Invalid argument.\n");

	}
}

static void
part_item_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	PartItem *part_item;
	PartItemPriv *priv;
	
	g_return_if_fail (object != NULL);	
	g_return_if_fail (IS_PART_ITEM (object));
	
	part_item = PART_ITEM (object);
	priv = part_item->priv;

	switch (arg_id) {
	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}

static void
part_item_destroy (GtkObject *object)
{
	PartItem *item;
	PartItemPriv *priv;
//	Part *part;
	Sheet *sheet;
	ArtPoint *old;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_PART_ITEM (object));

	/*
	 * Free the stored coordinate that lets us rotate circles.
	 */
	old = gtk_object_get_data (object, "hack");
	g_free (old);

	sheet = sheet_item_get_sheet (SHEET_ITEM (object));
	item = PART_ITEM (object);
	priv = item->priv;

	gtk_object_destroy (GTK_OBJECT (priv->label_group));

/*	part = PART (sheet_item_get_data (SHEET_ITEM (object)));
	if (part)
		gtk_object_unref (GTK_OBJECT (part));
*/
	if (GTK_OBJECT_CLASS (part_item_parent_class)->destroy)
		(* GTK_OBJECT_CLASS (part_item_parent_class)->destroy) (object);
}

static void
part_item_set_label_items (PartItem *item, GSList *item_list)
{
	PartItemPriv *priv;

	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_PART_ITEM (item));

	priv = item->priv;

	if (priv->label_items)
		g_slist_free (priv->label_items);

	priv->label_items = item_list;
}

/*
 * part_item_moved
 *
 * "moved" signal handler. Invalidates the bounding box cache.
 */
static void
part_item_moved (SheetItem *sheet_item)
{
	PartItem *part_item;

	part_item = PART_ITEM (sheet_item);
	part_item->priv->cache_valid = FALSE;
}

PartItem *
part_item_new (Sheet *sheet, Part *part)
{
	PartItem *item;
	PartItemPriv *priv;

	g_return_val_if_fail (sheet != NULL, NULL);
	g_return_val_if_fail (IS_SHEET (sheet), NULL);
	g_return_val_if_fail (part != NULL, NULL);
	g_return_val_if_fail (IS_PART (part), NULL);

	gtk_object_ref (GTK_OBJECT (part));
	gtk_object_sink (GTK_OBJECT (part));

	item = PART_ITEM (gnome_canvas_item_new (
		sheet->object_group,
		part_item_get_type (),
		"data", part,
		NULL));

	priv = item->priv;

	priv->label_group = GNOME_CANVAS_GROUP (gnome_canvas_item_new (
		GNOME_CANVAS_GROUP (item),
		gnome_canvas_group_get_type (),
		"x", 0.0,
		"y", 0.0,
		NULL));

	gtk_signal_connect_while_alive (GTK_OBJECT (part), "rotated", part_rotated_callback, item, GTK_OBJECT (item));
	gtk_signal_connect_while_alive (GTK_OBJECT (part), "flipped", part_flipped_callback, item, GTK_OBJECT (item));
	gtk_signal_connect_while_alive (GTK_OBJECT (part), "moved", part_moved_callback, item, GTK_OBJECT (item));

	return item;
}

static void
update_canvas_labels (PartItem *item)
{
	PartItemPriv *priv;
	Part *part;
	GSList *labels, *label_items;
	GnomeCanvasItem *canvas_item;
	
	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_PART_ITEM (item));

	priv = item->priv;
	part = PART (sheet_item_get_data (SHEET_ITEM (item)));

	label_items = priv->label_items;

 	for (labels = part_get_labels (part); labels; labels = labels->next, label_items = label_items->next) {
 		char *text;
 		PartLabel *label = (PartLabel*) labels->data;

		g_assert (label_items != NULL);
		canvas_item = label_items->data;
 
 		text = part_property_expand_macros (part, label->text);
 		gnome_canvas_item_set (canvas_item, "text", text, NULL);
 		g_free (text);
	}	
}

static void
prop_dialog_close (void)
{
	gnome_dialog_close (prop_dialog->dialog);
	prop_dialog = NULL;
}

static void
prop_dialog_cancel_cb (GtkWidget *widget, gpointer data)
{
	prop_dialog_close ();
}

/*
 * Go through the properties and commit the changes.
 */
static void
prop_dialog_ok_cb (GtkWidget *widget, PartItem *item)
{
	GSList *properties;
	Property *prop;		
	PartItemPriv *priv;
	Part *part;
	int row;
	char *value;

	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_PART_ITEM (item));

	priv = item->priv;
	part = PART (sheet_item_get_data (SHEET_ITEM (item)));

	row = 0;
	for (properties = part_get_properties (part); properties; properties = properties->next) {
		prop = properties->data;

		if (!g_strcasecmp (prop->name, "internal")) {
			continue;
		}

		if (row == prop_dialog->selected_row) {
			value = gtk_editable_get_chars (GTK_EDITABLE (prop_dialog->value_entry), 0, -1);
			prop->value = g_strdup (value);
			g_free (value);
		} else if (gtk_clist_get_text (prop_dialog->clist, row, 1, &value)) {
			g_free (prop->value);
			prop->value = g_strdup (value);
		}

		row++;
	}

	update_canvas_labels (item);

	prop_dialog_close ();
}

static void
edit_properties_select_row (GtkCList *clist, int row, int column, GdkEventButton *event, GladeXML *gui)
{
	char *value;

	prop_dialog->selected_row = row;

	if (gtk_clist_get_text (clist, row, 0, &value))
		gtk_entry_set_text (GTK_ENTRY (prop_dialog->name_entry), value);

	if (gtk_clist_get_text (clist, row, 1, &value))
		gtk_entry_set_text (GTK_ENTRY (prop_dialog->value_entry), value);
	
}

static void
edit_properties_unselect_row (GtkCList *clist, int row, int column, GdkEventButton *event, GladeXML *gui)
{
	char *value;

	value = gtk_editable_get_chars (GTK_EDITABLE (prop_dialog->value_entry), 0, -1);
	gtk_clist_set_text (clist, row, 1, value);
	g_free (value);
}

static void 
edit_properties (SheetItem *object)
{       
	GSList *properties;
	Sheet *sheet;
	PartItem *item;
	PartItemPriv *priv;
	Part *part;
	char *internal, *msg;
	GladeXML *gui;
	char *row[2];
	
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_PART_ITEM (object));

	item = PART_ITEM (object);
	priv = item->priv;
	part = PART (sheet_item_get_data (SHEET_ITEM (item)));

	internal = part_get_property (part, "internal");
	if (internal) {
		if (g_strcasecmp (internal, "ground") == 0) {
			g_free (internal);
			return;
		}
	}

	g_free (internal);

	if (!g_file_exists (OREGANO_GLADEDIR "/part-properties-dialog.glade")) {
		msg = g_strdup_printf (_("Could not find the required file:\n%s"), 
				       OREGANO_GLADEDIR "/part-properties-dialog.glade");
		oregano_error (msg);
		g_free (msg);
		return;
	}

	gui = glade_xml_new (OREGANO_GLADEDIR "/part-properties-dialog.glade", NULL);
	if (!gui) {
		oregano_error (_("Could not create part properties dialog."));
		return;
	}

	prop_dialog = g_new0 (PartPropDialog, 1);
	prop_dialog->dialog = GNOME_DIALOG (glade_xml_get_widget (gui, "part-properties-dialog"));

	prop_dialog->clist = GTK_CLIST (glade_xml_get_widget (gui, "properties-list"));

	prop_dialog->value_entry = glade_xml_get_widget (gui, "value-entry");
	prop_dialog->name_entry = glade_xml_get_widget (gui, "name-entry");
	
	sheet = sheet_item_get_sheet (SHEET_ITEM (item));
	sheet_dialog_set_parent (sheet, prop_dialog->dialog);

	gnome_dialog_button_connect (prop_dialog->dialog, 0, prop_dialog_ok_cb, item);
	gnome_dialog_button_connect (prop_dialog->dialog, 1, prop_dialog_cancel_cb, item);

	for (properties = part_get_properties (part); properties; properties = properties->next) {
		Property *prop;
		prop = properties->data;

		if (prop->name) {
			if (!g_strcasecmp (prop->name, "internal"))
				continue;
			
			row[0] = prop->name;
			row[1] = prop->value;

			gtk_clist_append (GTK_CLIST (prop_dialog->clist), row);
		}
	}	
	
	gnome_dialog_set_default (prop_dialog->dialog, 0);
	gnome_dialog_editable_enters (prop_dialog->dialog, GTK_EDITABLE (prop_dialog->value_entry));
	gtk_widget_set_sensitive (prop_dialog->name_entry, FALSE);

	gtk_signal_connect (GTK_OBJECT (prop_dialog->clist), "select-row",
			    GTK_SIGNAL_FUNC (edit_properties_select_row), gui);

	gtk_clist_select_row (GTK_CLIST (prop_dialog->clist), 0, 0);

	gtk_signal_connect (GTK_OBJECT (prop_dialog->clist), "unselect-row",
			    GTK_SIGNAL_FUNC (edit_properties_unselect_row), gui);

	gnome_dialog_run_and_close (prop_dialog->dialog);
}

static void
properties_cmd (GtkWidget *widget, SchematicView *sv)
{
	GList *list;

	list = schematic_view_get_selection (sv);
	if ((list != NULL) && IS_PART_ITEM (list->data)) 
		edit_properties (list->data);
}

static void
canvas_text_affine (GnomeCanvasItem *canvas_item, double *affine)
{
        ArtPoint src, dst;
        GtkArg arg[2];

        arg[0].name = "x";
        arg[1].name = "y";
        gtk_object_getv (GTK_OBJECT (canvas_item), 2, arg);

        src.x = GTK_VALUE_DOUBLE (arg[0]);
        src.y = GTK_VALUE_DOUBLE (arg[1]);
        art_affine_point (&dst, &src, affine);

        gnome_canvas_item_set (canvas_item, "x", dst.x,"y",dst.y, NULL);
}

static void
canvas_line_affine (GnomeCanvasItem *canvas_item, double *affine)
{
	GnomeCanvasPoints *points;
	ArtPoint src, dst;
	GtkArg arg[2];
	int i;

	arg[0].name = "points";
	gtk_object_getv (GTK_OBJECT (canvas_item), 1, arg);
	points = GTK_VALUE_POINTER (arg[0]);

	for (i = 0; i < points->num_points * 2; i += 2) {
		src.x = points->coords[i];
		src.y = points->coords[i + 1];
		art_affine_point (&dst, &src, affine);
		if (fabs (dst.x) < 1e-2) 
			dst.x = 0;
		if (fabs (dst.y) < 1e-2) 
			dst.y = 0;
		points->coords[i] = dst.x;
		points->coords[i + 1] = dst.y;
	}
	
	gnome_canvas_item_set (canvas_item, "points", points, NULL);
	gnome_canvas_points_unref (points);
}	

static void
part_rotated_callback (ItemData *data, int angle, SheetItem *sheet_item)
{
	double affine[6];
	double x1, y1, x2, y2, x0, y0;
	double left, top, right, bottom, dx, dy;
	GList *list;
	GSList *label_items;
	GtkAnchorType anchor;
	ArtPoint src, dst;
	GnomeCanvasGroup *group;
	GnomeCanvasItem *canvas_item;
	GtkArg arg[4];
	ArtPoint *old = NULL;
	PartItem *item;
	PartItemPriv *priv;
	Part *part;

	g_return_if_fail (sheet_item != NULL);
	g_return_if_fail (IS_PART_ITEM (sheet_item));

	item = PART_ITEM (sheet_item);
	group = GNOME_CANVAS_GROUP (item);
	part = PART (data);
	
	priv = item->priv;

	art_affine_rotate (affine, angle);

	for (list = group->item_list; list; list = list->next) {
		canvas_item = GNOME_CANVAS_ITEM (list->data);
		if (GNOME_IS_CANVAS_LINE (canvas_item)) {
			canvas_line_affine (canvas_item, affine);
                }
                else if ( GNOME_IS_CANVAS_TEXT (canvas_item)) {
                        //   canvas_text_affine (canvas_item,affine);           
                }
                else if (GNOME_IS_CANVAS_ELLIPSE (canvas_item)) {
			/* 
			 * Big great hack. Needed to rotate circles in the non aa-canvas. 
			 */
			arg[0].name = "x1";
			arg[1].name = "y1";
			arg[2].name = "x2";
			arg[3].name = "y2";
			gtk_object_getv (GTK_OBJECT (list->data), 4, arg);
			x1 = GTK_VALUE_DOUBLE(arg[0]);
			y1 = GTK_VALUE_DOUBLE(arg[1]);
			x2 = GTK_VALUE_DOUBLE(arg[2]);
			y2 = GTK_VALUE_DOUBLE(arg[3]);
			
			left = MIN (x1, x2);
			right = MAX (x1, x2);
			top = MIN (y1, y2);
			bottom = MAX (y1, y2);

			old = gtk_object_get_data (GTK_OBJECT (canvas_item), "hack");
			if (old  != NULL) {
				x0 = src.x = old->x;
				y0 = src.y = old->y;
			} else {
				x0 = src.x = left + (right - left) / 2;
				y0 = src.y = top + (bottom - top) / 2;
				old = g_new (ArtPoint, 1);
				gtk_object_set_data (GTK_OBJECT (canvas_item), "hack", old);
			}
			
			art_affine_point (&dst, &src, affine);
			old->x = dst.x;
			old->y = dst.y;
			
			dx = dst.x - x0;
			dy = dst.y - y0;
			
			gnome_canvas_item_move (canvas_item, dx, dy);
		}
	}		

	/*
	 * Get the right anchor for the labels. This is needed since the
	 * canvas don't know how to rotate text and since we rotate the
	 * label_group instead of the labels directly.
	 */
	switch (part_get_rotation (part)) {
	case 0:
		anchor = GTK_ANCHOR_SOUTH_WEST;
		break;
	case 90:
		anchor = GTK_ANCHOR_NORTH_WEST;
		break;
	case 180:
		anchor = GTK_ANCHOR_NORTH_EAST;
		break;
	case 270:
		anchor = GTK_ANCHOR_SOUTH_EAST;
		break;
	default:
		anchor = GTK_ANCHOR_SOUTH_WEST;
		break;
	}
	
	for (label_items = priv->label_items; label_items; label_items = label_items->next) {
                canvas_text_affine (label_items->data,affine);
        
		gnome_canvas_item_set (
			GNOME_CANVAS_ITEM (label_items->data),
			"anchor", anchor,
			NULL);
                        
	}

	/*
	 * Invalidate the bounding box cache.
	 */
	priv->cache_valid = FALSE;
}

static void
part_flipped_callback (ItemData *data, gboolean horizontal, SheetItem *sheet_item)
{
	GList *list;
	GSList *label_items;
	GtkAnchorType anchor;
	GnomeCanvasGroup *group;
	GnomeCanvasItem *canvas_item;
	GtkArg arg[4];
	PartItem *item;
	PartItemPriv *priv;
	Part *part;
	double affine[6], x1, y1, x2, y2, left, top, right, bottom;
	ArtPoint src, dst;

	g_return_if_fail (sheet_item != NULL);
	g_return_if_fail (IS_PART_ITEM (sheet_item));

	item = PART_ITEM (sheet_item);
	group = GNOME_CANVAS_GROUP (item);
	part = PART (data);
	
	priv = item->priv;

	if (horizontal)
		art_affine_scale (affine, -1, 1);
	else
		art_affine_scale (affine, 1, -1);

	for (list = group->item_list; list; list = list->next) {
		canvas_item = GNOME_CANVAS_ITEM (list->data);
		if (GNOME_IS_CANVAS_LINE (canvas_item)) {
			/*
			 * Flip the line.
			 */
			canvas_line_affine (canvas_item, affine);
		} else if (GNOME_IS_CANVAS_ELLIPSE (canvas_item)) {
			/*
			 * Flip the ellipse.
			 */
			arg[0].name = "x1";
			arg[1].name = "y1";
			arg[2].name = "x2";
			arg[3].name = "y2";
			gtk_object_getv (GTK_OBJECT (canvas_item), 4, arg);

			src.x = GTK_VALUE_DOUBLE(arg[0]);
			src.y = GTK_VALUE_DOUBLE(arg[1]);
			art_affine_point (&dst, &src, affine);
			if (fabs (dst.x) < 1e-2) 
				dst.x = 0;
			if (fabs (dst.y) < 1e-2) 
				dst.y = 0;
			x1 = dst.x;
			y1 = dst.y;

			src.x = GTK_VALUE_DOUBLE(arg[2]);
			src.y = GTK_VALUE_DOUBLE(arg[3]);
			art_affine_point (&dst, &src, affine);
			if (fabs (dst.x) < 1e-2) 
				dst.x = 0;
			if (fabs (dst.y) < 1e-2) 
				dst.y = 0;
			x2 = dst.x;
			y2 = dst.y;

			left = MIN (x1, x2);
			right = MAX (x1, x2);
			top = MIN (y1, y2);
			bottom = MAX (y1, y2);

			gnome_canvas_item_set (
				canvas_item, 
				"x1", left,
				"y1", top,
				"x2", right,
				"y2", bottom,
				NULL);
		}
	}		

	/*
	 * Flip the labels as well.
	 */
	for (label_items = priv->label_items; label_items; label_items = label_items->next) {
		GnomeCanvasItem *label;

		label = label_items->data;
		arg[0].name = "x";
		arg[1].name = "y";
		gtk_object_getv (GTK_OBJECT (label), 2, arg);
		src.x = GTK_VALUE_DOUBLE (arg[0]);		
		src.y = GTK_VALUE_DOUBLE (arg[1]);
		
		art_affine_point (&dst, &src, affine);

		if (fabs (dst.x) < 1e-2) 
			dst.x = 0;
		if (fabs (dst.y) < 1e-2) 
			dst.y = 0;
		
		gnome_canvas_item_set (
			GNOME_CANVAS_ITEM (label_items->data),
			"x", dst.x,
			"y", dst.y,
			NULL);
	}

	/*
	 * Invalidate the bounding box cache.
	 */
	priv->cache_valid = FALSE;
}

void
part_item_signal_connect_floating_group (Sheet *sheet, SchematicView *sv)
{
	g_return_if_fail (sheet != NULL);
	g_return_if_fail (IS_SHEET (sheet));
	g_return_if_fail (sv != NULL);
	g_return_if_fail (IS_SCHEMATIC_VIEW (sv));

	sheet->state = SHEET_STATE_FLOAT_START;

	/* FIXME: clean all this mess with floating groups etc... */
	if (sheet->priv->float_handler_id != 0)
		return;

	sheet->priv->float_handler_id = gtk_signal_connect (
		GTK_OBJECT (sheet), 
		"event",
		GTK_SIGNAL_FUNC (sheet_item_floating_event),
		sv);
}

void
part_item_signal_connect_floating (PartItem *item)
{
	Sheet *sheet;

	sheet = sheet_item_get_sheet (SHEET_ITEM (item));
	sheet->state = SHEET_STATE_FLOAT_START;

	gtk_signal_connect (
		GTK_OBJECT (item), 
		"double_clicked",
		GTK_SIGNAL_FUNC (edit_properties),
		item);
}

static void
selection_changed (PartItem *item, gboolean select, gpointer user_data)
{
	gtk_object_ref (GTK_OBJECT (item));		
	if (select)
		gtk_idle_add ((gpointer) select_idle_callback, item);
	else 
		gtk_idle_add ((gpointer) deselect_idle_callback, item);
}

static int
select_idle_callback (PartItem *item)
{
	PartItemPriv *priv;
	GnomeCanvasItem *canvas_item;
	GList *list;

	priv = item->priv;

	if (priv->highlight) {
		gtk_object_unref (GTK_OBJECT (item));			
		return FALSE;
	}

	for (list = GNOME_CANVAS_GROUP (item)->item_list; list; list = list->next){
		canvas_item = GNOME_CANVAS_ITEM (list->data);
		if (GNOME_IS_CANVAS_LINE (canvas_item))
			gnome_canvas_item_set (canvas_item, "fill_color", SELECTED_COLOR, NULL);
		else if (GNOME_IS_CANVAS_ELLIPSE (canvas_item))
			gnome_canvas_item_set (canvas_item, "outline_color", SELECTED_COLOR, NULL);
                else if (GNOME_IS_CANVAS_TEXT  (canvas_item))
                        gnome_canvas_item_set (canvas_item, "fill_color", SELECTED_COLOR, NULL);
	}

	priv->highlight = TRUE;

	gtk_object_unref (GTK_OBJECT (item));	
	return FALSE;
}

static int
deselect_idle_callback (PartItem *item)
{
	GList *list;
	GnomeCanvasItem *canvas_item;
	PartItemPriv *priv;

	priv = item->priv;

	if (!priv->highlight) {
		gtk_object_unref (GTK_OBJECT (item));			
		return FALSE;
	}
	
	for (list = GNOME_CANVAS_GROUP (item)->item_list; list; list = list->next){
		canvas_item = GNOME_CANVAS_ITEM (list->data);
		if (GNOME_IS_CANVAS_LINE (canvas_item))
			gnome_canvas_item_set (canvas_item, "fill_color", NORMAL_COLOR, NULL);
		else if (GNOME_IS_CANVAS_ELLIPSE (canvas_item))
			gnome_canvas_item_set (canvas_item, "outline_color", NORMAL_COLOR, NULL);
                else if (GNOME_IS_CANVAS_TEXT  (canvas_item))
                        gnome_canvas_item_set (canvas_item, "fill_color", "blue", NULL);
	}

	priv->highlight = FALSE;

	gtk_object_unref (GTK_OBJECT (item));	
	return FALSE;
}

static gboolean
is_in_area (SheetItem *object, SheetPos *p1, SheetPos *p2)
{
	PartItem *item;
	SheetPos bbox_start, bbox_end;

	item = PART_ITEM (object);

	get_cached_bounds (item, &bbox_start, &bbox_end);

	if (p1->x < bbox_start.x &&
	    p2->x > bbox_end.x &&
	    p1->y < bbox_start.y &&
	    p2->y > bbox_end.y)
		return TRUE;

	return FALSE;
}

static void
show_labels (SheetItem *sheet_item, gboolean show)
{
	PartItem *item;
	PartItemPriv *priv;
	
	g_return_if_fail (sheet_item != NULL);
	g_return_if_fail (IS_PART_ITEM (sheet_item));
	
	item = PART_ITEM (sheet_item);
	priv = item->priv;

	if (show)
	        gnome_canvas_item_show (GNOME_CANVAS_ITEM (priv->label_group));
	else
	        gnome_canvas_item_hide (GNOME_CANVAS_ITEM (priv->label_group));
}

/*
 * Retrieves the bounding box. We use a caching scheme for this
 * since it's too expensive to calculate it every time we need it.
 */
inline static void
get_cached_bounds (PartItem *item, SheetPos *p1, SheetPos *p2)
{
	PartItemPriv *priv;
	priv = item->priv;

	if (!priv->cache_valid) {
		SheetPos start_pos, end_pos;
		gdouble x1, y1, x2, y2;

		/*
		 * Hide the labels, then get bounding box, then show them again.
		 */
		gnome_canvas_item_hide (GNOME_CANVAS_ITEM (priv->label_group));
		gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (item), &x1, &y1, &x2, &y2);
		gnome_canvas_item_show (GNOME_CANVAS_ITEM (priv->label_group));

		start_pos.x = x1;
		start_pos.y = y1;
		end_pos.x = x2;
		end_pos.y = y2;

		priv->bbox_start = start_pos;
		priv->bbox_end = end_pos;
		priv->cache_valid = TRUE;
	}
	
	memcpy (p1, &priv->bbox_start, sizeof (SheetPos));
	memcpy (p2, &priv->bbox_end, sizeof (SheetPos));
}

static void
part_item_paste (SchematicView *sv, ItemData *data)
{
	g_return_if_fail (sv != NULL);
	g_return_if_fail (IS_SCHEMATIC_VIEW (sv));
	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_PART (data));

	schematic_view_add_ghost_item (sv, data);
}

/*
 * FIXME: make this the default constructor for PartItem (rename to part_item_new).
 */
PartItem *
part_item_new_from_part (Sheet *sheet, Part *part)
{
        Library *library;
	LibraryPart *library_part;
	PartPriv *priv;
	PartItem *item;
	int angle;
	IDFlip flip;

	priv = part->priv;

	library = priv->library;
	library_part = library_get_part (library, priv->name);

	/*
	 * Create the PartItem canvas item.
	 */
	item = part_item_new (sheet, part);
	
	create_canvas_items (GNOME_CANVAS_GROUP (item), library_part);
	create_canvas_labels (item, part);

	angle = part_get_rotation (part);
	part_rotated_callback (ITEM_DATA (part), angle, SHEET_ITEM (item));
	
	flip = part_get_flip (part);
	if (flip & ID_FLIP_HORIZ)
		part_flipped_callback (ITEM_DATA (part), TRUE, SHEET_ITEM (item));
	if (flip & ID_FLIP_VERT)
		part_flipped_callback (ITEM_DATA (part), FALSE, SHEET_ITEM (item));

	return item;
}

void
part_item_create_canvas_items_for_preview (GnomeCanvasGroup *group, LibraryPart *library_part)
{
	g_return_if_fail (group != NULL);
	g_return_if_fail (GNOME_IS_CANVAS_GROUP (group));
	g_return_if_fail (library_part != NULL);

	create_canvas_items (group, library_part);
}

static void
create_canvas_items (GnomeCanvasGroup *group, LibraryPart *library_part)
{
	GnomeCanvasItem   *item;
	GnomeCanvasPoints *points;
	GSList            *objects;
	LibrarySymbol     *symbol;
	SymbolObject      *object;
	
	g_return_if_fail (group != NULL);
	g_return_if_fail (GNOME_IS_CANVAS_GROUP (group));
	g_return_if_fail (library_part != NULL);
 
	symbol = library_get_symbol (library_part->symbol_name);
	if (symbol ==  NULL){
		g_warning ("Couldn't find the requested symbol %s for part %s in library.\n", 
			   library_part->symbol_name, 
			   library_part->name);
		return;
	}

        for (objects = symbol->symbol_objects; objects; objects = objects->next) {
		object = (SymbolObject *)(objects->data);
		switch (object->type){
		case SYMBOL_OBJECT_LINE:
                        points = object->u.uline.line;
			item = gnome_canvas_item_new (
				group,
				gnome_canvas_line_get_type (),
				"points", points,
				"fill_color", "medium sea green",
				"width_pixels", 1,
				"cap_style", GDK_CAP_BUTT,
				NULL);
                        if (object->u.uline.spline) {
                                gnome_canvas_item_set (
                                        item,
                                        "smooth", TRUE,
                                        "spline_steps", 5,
                                        NULL);                  
                        }
			break;
			
		case SYMBOL_OBJECT_ARC:
			item = gnome_canvas_item_new (
				group,
				gnome_canvas_ellipse_get_type (),
				"x1", object->u.arc.x1,
				"y1", object->u.arc.y1,
				"x2", object->u.arc.x2,
				"y2", object->u.arc.y2,
				"outline_color", "medium sea green",
				"width_pixels", 1,
				NULL);
			break;

                case SYMBOL_OBJECT_TEXT:
                        item = gnome_canvas_item_new (
                                group,
                                gnome_canvas_text_get_type (),
                                "text",object->u.text.str,
                                "x", (double) object->u.text.x,
                                "y", (double) object->u.text.y,
                                "fill_color", "blue",
                                "font", "-*-helvetica-medium-r-*-*-*-100-*-*-*-*-*-*",
                                NULL);
                        break;

		default:
			g_warning ("Unknown symbol object.\n");
			continue;
		}
	}
}

static void
create_canvas_labels (PartItem *item, Part *part)
{
	GnomeCanvasItem *canvas_item;
	GSList *list, *labels, *item_list;
	GnomeCanvasGroup *group;
	
	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_PART_ITEM (item));
	g_return_if_fail (part != NULL);
	g_return_if_fail (IS_PART (part));

	labels = part_get_labels (part);
	group = item->priv->label_group;
	item_list = NULL;

	for (list = labels; list; list = list->next) {
		PartLabel *label = list->data;
		char *text;

		text = part_property_expand_macros (part, label->text);

		canvas_item = gnome_canvas_item_new (
			group,
			gnome_canvas_text_get_type (),
			"x", (double) label->pos.x,
			"y", (double) label->pos.y,
			"text", text,
			"anchor", GTK_ANCHOR_SOUTH_WEST,
			"fill_color", "blue",
			"font", "-*-helvetica-medium-r-*-*-*-100-*-*-*-*-*-*",
			NULL);

		item_list = g_slist_prepend (item_list, canvas_item);

		g_free (text);
	}
	item_list = g_slist_reverse (item_list);
	part_item_set_label_items (item, item_list);
}

/*
 * This is called when the part data was moved. Update the view accordingly.
 */
static void
part_moved_callback (ItemData *data, SheetPos *pos, SheetItem *item)
{
	PartItem *part_item;

	g_return_if_fail (data != NULL);
	g_return_if_fail (IS_ITEM_DATA (data));
	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_PART_ITEM (item));

	if (pos == NULL)
		return;

	part_item = PART_ITEM (item);

	/* 
	 * Move the canvas item and invalidate the bbox cache.
	 */
	gnome_canvas_item_move (GNOME_CANVAS_ITEM (item), pos->x, pos->y);
	part_item->priv->cache_valid = FALSE;
}

static void
part_item_place (SheetItem *item, SchematicView *sv)
{
	gtk_signal_connect (
		GTK_OBJECT (item), 
		"event",
		GTK_SIGNAL_FUNC (sheet_item_event),
		sv);
	
	gtk_signal_connect (
		GTK_OBJECT (item), 
		"double_clicked",
		GTK_SIGNAL_FUNC (edit_properties),
		item);
}

static void
part_item_place_ghost (SheetItem *item, SchematicView *sv)
{
//	part_item_signal_connect_placed (PART_ITEM (item));
}


