/*
 * part-browser.c
 *
 * 
 * Author: 
 *  Richard Hult <rhult@hem.passagen.se>
 * 
 *  http://www.dtek.chalmers.se/~d4hult/oregano/ 
 * 
 * Copyright (C) 1999-2001  Richard Hult 
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the 
 * License, or (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <gnome.h>
#include <glade/glade.h>
#include <math.h>
#include "main.h"
#include "load-library.h"
#include "schematic.h"
#include "schematic-view.h"
#include "part-browser.h"
#include "part-item.h"
#include "dialogs.h"
#include "sheet-pos.h"

typedef struct _Browser Browser;
struct _Browser {
	SchematicView    *schematic_view;
        GtkWidget        *viewport; 
	GtkWidget        *list;
	GtkWidget        *canvas;
	GnomeCanvasText  *description;
	GnomeCanvasGroup *preview;
	Library          *library;
	GtkWidget        *library_menu_item;     /* The currently selected option menu item. */
	gboolean          hidden;
	int               selected;      /* The currently selected row in the clist. */
	int               prev_selected; /* Previous part, to revert to if errors occur. */
};

typedef struct {
	Browser *br;
	SchematicView *schematic_view;
	char *library_name;
	char *part_name;
} DndData;

#define PREVIEW_WIDTH 100
#define PREVIEW_HEIGHT 100
#define PREVIEW_TEXT_HEIGHT 25

static void update_preview (Browser *br);
static void add_part (gpointer key, LibraryPart *part, Browser *br);
static int part_selected (GtkCList *list, GdkEvent *event, Browser *br);
static void part_browser_setup_libs (Browser *br, GladeXML *gui);
static void library_switch_cb (GtkWidget *item, Browser *br);
static void preview_realized (GtkWidget *widget, Browser *br);
static void wrap_string(char* str, int width);

static void
place_cmd (GtkWidget *widget, Browser *br)
{
	LibraryPart *library_part;
	char *part_name;
	SheetPos pos;
	Sheet *sheet;
	Part *part;

	schematic_view_reset_tool (br->schematic_view);
	sheet = schematic_view_get_sheet (br->schematic_view);

	pos.x = 0;
	pos.y = 0;

	gtk_clist_get_text (GTK_CLIST (br->list),
			    br->selected,
			    0,
			    &part_name);

	library_part = library_get_part (br->library, part_name);
	part = part_new_from_library_part (library_part);
	if (!part)
		return;

	item_data_set_pos (ITEM_DATA (part), &pos);

	schematic_view_select_all (br->schematic_view, FALSE);
	schematic_view_clear_ghosts (br->schematic_view);
	schematic_view_add_ghost_item (br->schematic_view, ITEM_DATA (part));

	part_item_signal_connect_floating_group (sheet, br->schematic_view);
}

static int 
part_selected (GtkCList *list, GdkEvent *event, Browser *br)
{
	int row, col;
	if (!gtk_clist_get_selection_info (list, event->button.x, event->button.y, &row, &col))
		return FALSE;

	switch (event->type) {
	case GDK_BUTTON_PRESS:
		if (event->button.button == 1 || event->button.button == 3) {
			br->prev_selected = br->selected;
			br->selected = row;
			update_preview (br);
		}
		break;
	case GDK_2BUTTON_PRESS:
		if (event->button.button == 1) {
			/*
			 * Make sure we really clicked on the same row both times. 
			 * Stupid GtkCList.
			 */
			if (row == br->prev_selected)
				place_cmd (NULL, br);
		}
		break;
	case GDK_3BUTTON_PRESS:
		gtk_signal_emit_stop_by_name (GTK_OBJECT (list), "event");
		break;
	default:
		return FALSE;
	}

	return TRUE;
}

static void
update_preview (Browser *br)
{
	LibraryPart *library_part;
	gdouble new_x, new_y, x1, y1, x2, y2;
	gdouble scale_x, scale_y;
	double affine[6];
	gdouble width, height;
	char *part_name;
	char *description;
	gdouble text_width;
	GtkArg arg;

	/*
	 * If there already is a preview part, destroy its group and create a new one.
	 */
	if (br->preview != NULL)
		gtk_object_destroy (GTK_OBJECT (br->preview));
	
	br->preview = GNOME_CANVAS_GROUP (gnome_canvas_item_new (
		gnome_canvas_root (GNOME_CANVAS (br->canvas)),
		gnome_canvas_group_get_type(),
		"x", 0.0,
		"y", 0.0,
		NULL));

	gtk_clist_get_text (GTK_CLIST (br->list),
			    br->selected,
			    0,
			    &part_name);

	library_part = library_get_part (br->library, part_name);

	if (!library_part)
		return;

	part_item_create_canvas_items_for_preview (br->preview, library_part);

	width = br->canvas->allocation.width;
	height = br->canvas->allocation.height - PREVIEW_TEXT_HEIGHT;

	gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (br->preview), &x1, &y1, &x2, &y2);

	if ((x2 - x1) < 0.50 * width || (y2 - y1) < 0.50 * height) {
		scale_x = 0.50 * width / (x2 - x1);
		scale_y = 0.50 * height / (y2 - y1);

		scale_x = MIN (scale_x, scale_y);
		art_affine_scale (affine, scale_x, scale_x);

		gnome_canvas_item_affine_absolute (GNOME_CANVAS_ITEM (br->preview), affine);

		/* Get the scaled bounds. */
		gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (br->preview), &x1, &y1, &x2, &y2);

	} else if ((x2 - x1) > width || (y2 - y1) > height) {
		scale_x = 0.85 * width / (x2 - x1);
		scale_y = 0.85 * height / (y2 - y1);

		scale_x = MIN (scale_x, scale_y);
		art_affine_scale (affine, scale_x, scale_x);

		gnome_canvas_item_affine_absolute (GNOME_CANVAS_ITEM (br->preview), affine);

		/* Get the scaled bounds. */
		gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (br->preview), &x1, &y1, &x2, &y2);
	}

	new_x = PREVIEW_WIDTH / 2 - (x2 - x1) / 2;
	new_y = PREVIEW_HEIGHT / 2 - (y2 - y1) / 2;

	gnome_canvas_item_move (GNOME_CANVAS_ITEM (br->preview), new_x - x1, new_y - y1);

	description = g_strdup (library_part->description);
	wrap_string (description, 20);
	gnome_canvas_item_set (GNOME_CANVAS_ITEM (br->description), "text", description, NULL);
	g_free (description);

	arg.type = GTK_TYPE_DOUBLE;
	arg.name = "text_width";
	gtk_object_getv (GTK_OBJECT(br->description), 1, &arg);
	text_width = GTK_VALUE_DOUBLE (arg);
	gtk_object_set (GTK_OBJECT(br->description), "x", PREVIEW_WIDTH / 2 - text_width / 2, NULL);		
	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (br->description));
}

static void
select_row (GtkCList *clist,
	    gint row,
	    gint column,
	    GdkEventButton *event,
	    Browser *br)
{
	br->selected = row;
	update_preview (br);
}

static void
add_part (gpointer key, LibraryPart *part, Browser *br)
{
	g_return_if_fail (part != NULL);
	g_return_if_fail (part->name != NULL);
	g_return_if_fail (br != NULL);
	g_return_if_fail (br->list != NULL);

	gtk_clist_append (GTK_CLIST (br->list), &(part->name));
}


/*
 * Read the available parts from the library and put them in the browser clist.
 */
static void
update_list (Browser *br)
{
	gtk_clist_clear (GTK_CLIST (br->list));
	g_hash_table_foreach (br->library->part_hash, (GHFunc) add_part, br);
	gtk_clist_sort (GTK_CLIST (br->list));
}


/*
 * part_browser_toggle_show
 *
 * Show a part browser. If one already exists, just bring it up, otherwise create it.
 * We can afford to keep it in memory all the time, and we don't have to delete it and
 * build it every time it is needed. If already shown, hide it.
 */
void
part_browser_toggle_show (SchematicView *schematic_view)
{
	Browser *browser = schematic_view_get_browser (schematic_view);

	if (browser == NULL){
		part_browser_create (schematic_view);
	} else {
		if (browser->hidden){
			gtk_widget_show (browser->viewport);
			browser->hidden = FALSE;
		} else{
			gtk_widget_hide (browser->viewport);
			browser->hidden = TRUE;
		}
	}
}

void
part_browser_dnd (GtkSelectionData *selection_data, int x, int y)
{
	LibraryPart *library_part;
	SheetPos pos;
	DndData *data;
	Sheet *sheet;
	Part *part;

	data = (DndData *) (selection_data->data);

	g_return_if_fail (data != NULL);

	pos.x = x;
	pos.y = y;

	sheet = schematic_view_get_sheet (data->schematic_view);

	snap_to_grid (sheet->grid, &pos.x, &pos.y);

	library_part = library_get_part (data->br->library, data->part_name);
	part = part_new_from_library_part (library_part);
	if (!part)
		return;

	item_data_set_pos (ITEM_DATA (part), &pos);

	schematic_view_select_all (data->schematic_view, FALSE);
	schematic_view_clear_ghosts (data->schematic_view);
	schematic_view_add_ghost_item (data->schematic_view, ITEM_DATA (part));

	part_item_signal_connect_floating_group (sheet, data->schematic_view);
}

static void
drag_data_get  (GtkWidget          *widget,
		GdkDragContext     *context,
		GtkSelectionData   *selection_data,
		guint               info,
		guint               time,
		Browser            *br)
{
	DndData *data;
	Library *library;

	schematic_view_reset_tool (br->schematic_view);

	data = g_new0 (DndData, 1);

	data->schematic_view = br->schematic_view;
	data->br = br;

	library = g_list_nth_data (oregano.libraries, 0);
	data->library_name = library->name;
	
	gtk_clist_get_text (GTK_CLIST (br->list),
			    br->selected,
			    0,
			    &(data->part_name));

	gtk_selection_data_set (selection_data,
				selection_data->target,
				8,
				(gpointer) data,
				sizeof (DndData));

	/*
	 * gtk_selection_data_set copies the information so we can free it now.
	 */
	g_free (data);
}

/*
 * part_browser_create
 * 
 * Creates a new part browser. This is only called once per schematic window.
 */
GtkWidget *
part_browser_create (SchematicView *schematic_view)
{
	Browser *br;
	GladeXML *gui;
	char *msg;
	GtkWidget *w;
	GtkStyle *style;
	GdkColormap *colormap;
	GnomeCanvasPoints *points;
	static GtkTargetEntry dnd_types[] = { { "x-application/oregano-part", 0, 0 } };
	static int dnd_num_types = sizeof (dnd_types) / sizeof (dnd_types[0]);

	br = g_new0 (Browser, 1);
	br->prev_selected = 0;
	br->selected = 0;
	br->preview = NULL;
	br->schematic_view = schematic_view;
	br->hidden = FALSE;

	if (!g_file_exists (OREGANO_GLADEDIR "/part-browser.glade")) {
		msg = g_strdup_printf (_("Could not find the required file:\n%s"), 
				       OREGANO_GLADEDIR "/part-browser.glade");
		oregano_error (msg);
		g_free (msg);
		return NULL;
	}

	gui = glade_xml_new (OREGANO_GLADEDIR "/part-browser.glade", "part_browser_viewport");
	if (!gui) {
		oregano_error (_("Could not create part browser."));
		return NULL;
	}

	w = glade_xml_get_widget (gui, "preview_canvas");
	br->canvas = w;

	gtk_signal_connect (GTK_OBJECT (w), "realize", preview_realized, br);
	
	style =  gtk_style_new ();
	colormap = gtk_widget_get_colormap (GTK_WIDGET (w));
	gdk_color_white (colormap, &style->bg[GTK_STATE_NORMAL]);
	gtk_widget_set_style (GTK_WIDGET (w), style);
	gtk_widget_set_usize (w, PREVIEW_WIDTH, PREVIEW_HEIGHT + PREVIEW_TEXT_HEIGHT);
	gnome_canvas_set_scroll_region (GNOME_CANVAS (w), 0, 0, PREVIEW_WIDTH, PREVIEW_HEIGHT + PREVIEW_TEXT_HEIGHT);

	points = gnome_canvas_points_new (2);
	points->coords[0] = - 10;
	points->coords[1] = PREVIEW_HEIGHT;
	points->coords[2] = PREVIEW_WIDTH + 10;
	points->coords[3] = PREVIEW_HEIGHT;

	gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS(br->canvas)),
			       gnome_canvas_line_get_type (),
			       "fill_color", "gray",
			       "line_style", GDK_LINE_ON_OFF_DASH,
			       "points", points,
			       NULL);
	
	gnome_canvas_points_unref (points);

	br->description = GNOME_CANVAS_TEXT (
		gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (br->canvas)),
				       gnome_canvas_text_get_type (),
				       "x", 0.0,
				       "y", PREVIEW_HEIGHT + 4.0,
				       "anchor", GTK_ANCHOR_NW,
				       "justification", GTK_JUSTIFY_CENTER,
				       "font", "-*-helvetica-medium-r-*-*-10-*-*-*-*-*-*-*",
				       "text", "",
				       NULL));


	/*
	 * Set up dnd.
	 */
	gtk_signal_connect (GTK_OBJECT (br->canvas), "drag_data_get",
			    GTK_SIGNAL_FUNC (drag_data_get), br);
	
	gtk_drag_source_set (br->canvas,
			     GDK_BUTTON1_MASK | GDK_BUTTON3_MASK,
			     dnd_types, dnd_num_types,
			     GDK_ACTION_MOVE);

	/* Buttons. */
	w = glade_xml_get_widget (gui, "place_button");
	gtk_signal_connect (GTK_OBJECT (w), "clicked",
			    GTK_SIGNAL_FUNC (place_cmd), br);

	/* Update the libraries option menu */
	br->library = g_list_nth_data (oregano.libraries, 0);
	part_browser_setup_libs (br, gui);

	/* Parts list. */
	w = glade_xml_get_widget (gui, "parts_list");
	br->list = w;
	gtk_clist_freeze (GTK_CLIST (w));
	update_list (br);
	gtk_signal_connect (GTK_OBJECT (w), "event",
			    GTK_SIGNAL_FUNC (part_selected), br);

	gtk_signal_connect (GTK_OBJECT (w), "select_row",
			    GTK_SIGNAL_FUNC (select_row), br);

	gtk_clist_thaw (GTK_CLIST (w));
	gtk_clist_select_row (GTK_CLIST (w), 0, 0);

	br->viewport = glade_xml_get_widget (gui, "part_browser_viewport");
	
	return br->viewport;
}

static void
part_browser_setup_libs (Browser *br, GladeXML *gui) {
	GtkWidget *w, *menu, *item;
	GList *libs;
	gboolean first;

	w = glade_xml_get_widget (gui, "library_optionmenu");
	gtk_option_menu_remove_menu(GTK_OPTION_MENU(w));
	
	menu = gtk_menu_new ();
	gtk_widget_show (menu);
	libs = oregano.libraries;

	first = TRUE;
	while (libs) {
		Library *lib;
		int selected = 0;
		
		lib = (Library *) libs->data;
		item = gtk_menu_item_new_with_label (lib->name);

		gtk_object_set_data (GTK_OBJECT (item), "selected", GINT_TO_POINTER (selected));	
		
		if (first) {
			br->library_menu_item = item;
			first = FALSE;
		}

		gtk_menu_append (GTK_MENU (menu), item);
		gtk_widget_show (item);
		gtk_object_set_user_data (GTK_OBJECT(item), lib);
	
		gtk_signal_connect (GTK_OBJECT (item), "activate", 
			GTK_SIGNAL_FUNC (library_switch_cb), br);
	
		libs = libs->next;
	}
	
	gtk_option_menu_set_menu (GTK_OPTION_MENU (w), menu);
	gtk_option_menu_set_history (GTK_OPTION_MENU (w), 0);
}

static void
library_switch_cb (GtkWidget *item, Browser *br) 
{
	Library *lib;
	int selected;

	/* 1) Update the browser. */
	lib = (Library *) gtk_object_get_user_data (GTK_OBJECT (item));
	br->library = lib;
	
	/* 2) Update the clist. */
	gtk_clist_freeze (GTK_CLIST (br->list));
	update_list (br);
	gtk_clist_thaw (GTK_CLIST (br->list));

        /* 3) Select the part that was selected the last time this item was chosen. */
	gtk_object_set_data (GTK_OBJECT (br->library_menu_item), "selected", GINT_TO_POINTER (br->selected));
	br->library_menu_item = item;

	selected = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (item), "selected"));
        br->prev_selected = selected;
	br->selected = selected;
	update_preview (br);
	gtk_clist_select_row (GTK_CLIST (br->list), selected, 0);
}

static void
wrap_string (char* str, int width)
{
   const int minl = width/2;
   char *lnbeg, *sppos, *ptr;
   int te = 0;

   g_return_if_fail (str != NULL);

   lnbeg= sppos = ptr = str;

   while (*ptr) {
      if (*ptr == '\t') 
	      te += 7;

      if (*ptr == ' ')
	      sppos = ptr;

      if(ptr - lnbeg > width - te && sppos >= lnbeg + minl) {
	      *sppos = '\n';
	      lnbeg = ptr;
	      te = 0;
      }
      
      if(*ptr=='\n') {
	      lnbeg = ptr; 
	      te = 0;
      }
      ptr++;
   }
}

static void
preview_realized (GtkWidget *widget, Browser *br)
{
	update_preview (br);
}



