/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: b3dprint.hxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 02:27:10 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef _B3D_B3DPRINT_HXX
#define _B3D_B3DPRINT_HXX

#ifndef _TOOLS_DEBUG_HXX
#include <tools/debug.hxx>
#endif

#ifndef _SV_OUTDEV_HXX
#include <vcl/outdev.hxx>
#endif

#ifndef _B3D_B3DCOMMN_HXX
#include "b3dcommn.hxx"
#endif

#ifndef _B3D_BUCKET_HXX
#include "bucket.hxx"
#endif

/*************************************************************************
|*
|* Typen von moeglichen Primitiven
|*
\************************************************************************/

enum B3dPrimitiveType
{
	B3dPrimitivePoint = 0,
	B3dPrimitiveLine,
	B3dPrimitiveTriangle
};

/*************************************************************************
|*
|* Klasse, um sich ein Primitiv zu merken
|*
\************************************************************************/

class B3dPrimitive
{
private:
	// Index des ersten Punktes im PointBucket; die Anzahl Punkte steht
	// durch den Typ implizit fest
	ULONG					nIndex;

	// Zum Testen die Gewichte setzen
#ifdef DBG_UTIL
	ULONG					nWeight;
#endif

	// fuer den BSPTree
	B3dPrimitive*			pLeft;
	B3dPrimitive*			pRight;
	B3dPrimitive*			pSame;

	// fuer dessen Ausgabe
	B3dPrimitive*			pParent;

	// Normale des Primitivs, falls es ein Dreieck ist
	// es geht hier um die GEOMETRISCHE Normale,
	// nicht um die Beleuchtungsnormalen in den Eckpunkten
	Vector3D				aNormal;

	// Material, falls eine Linie oder ein Dreieck
	ULONG					nMaterialIndex;

	// Typ des gemerkten Primitivs
	B3dPrimitiveType		eType;

	// Primitiv ausgeben?
	BOOL					bIsVisible		: 1;

	// Schon besucht?
	BOOL					bIsLeftDone		: 1;
	BOOL					bIsRightDone	: 1;
	BOOL					bIsOutputDone	: 1;
	BOOL					bIsSameDone		: 1;

public:
	B3dPrimitive(B3dPrimitiveType eTy, ULONG nInd);

	// Typ des Primitivs
	B3dPrimitiveType GetType() { return eType; }
	void SetType(B3dPrimitiveType eNew) { eType = eNew; }

	// Index in die Punktmenge
	ULONG GetIndex() { return nIndex; }
	void SetIndex(ULONG nNew) { nIndex = nNew; }

	// Bei Wiederverwendung dieses Elements ist diese
	// Funktion zu Verwenden VOR Einordnung in den BSPTree
	void Reset();

	// Zugriffe auf die Zeiger des BSPTrees
	B3dPrimitive* GetLeft() { return pLeft; }
	void SetLeft(B3dPrimitive* pNew) { pLeft = pNew; }
	B3dPrimitive* GetRight() { return pRight; }
	void SetRight(B3dPrimitive* pNew) { pRight = pNew; }
	B3dPrimitive* GetSame() { return pSame; }
	void SetSame(B3dPrimitive* pNew) { pSame = pNew; }
	B3dPrimitive* GetParent() { return pParent; }
	void SetParent(B3dPrimitive* pNew) { pParent = pNew; }

	// Zugriffe auf Normale und Skalar
	Vector3D& Normal() { return aNormal; }
	double Scalar(B3dEntityBucket& rBucket)
		{ return -(rBucket[nIndex].Point().GetVector3D().Scalar(aNormal)); }

	// Material setzen/lesen
	ULONG GetMaterialIndex() { return nMaterialIndex; }
	void SetMaterialIndex(ULONG nNew) { nMaterialIndex = nNew; }

	// Ist das Primitiv sichtbar?
	BOOL IsVisible() { return bIsVisible; }
	void SetVisible(BOOL bNew=TRUE) { bIsVisible = bNew; }

	// Ist das Primitiv schon besucht?
	BOOL IsLeftDone() { return bIsLeftDone; }
	void SetLeftDone(BOOL bNew=TRUE) { bIsLeftDone = bNew; }
	BOOL IsRightDone() { return bIsRightDone; }
	void SetRightDone(BOOL bNew=TRUE) { bIsRightDone = bNew; }
	BOOL IsOutputDone() { return bIsOutputDone; }
	void SetOutputDone(BOOL bNew=TRUE) { bIsOutputDone = bNew; }
	BOOL IsSameDone() { return bIsSameDone; }
	void SetSameDone(BOOL bNew=TRUE) { bIsSameDone = bNew; }
	void SetNothingDone(BOOL bNew=FALSE);
	BOOL IsAllDone() { return (bIsLeftDone && bIsRightDone && bIsSameDone && bIsOutputDone); }

#ifdef DBG_UTIL
	ULONG GetWeight() { return nWeight; }
	void SetWeight(ULONG nNew) { nWeight = nNew; }
#endif
};

/*************************************************************************
|*
|* Bucket fuer Primitive
|*
\************************************************************************/

BASE3D_DECL_BUCKET(B3dPrimitive, Bucket)

/*************************************************************************
|*
|* Klasse, um lokale Parameter in rekursiven Aufrufen fuer den Aufbau
|* von BSPTrees zu halten
|*
\************************************************************************/

class Base3DBSPLocal
{
public:
	Vector3D				aVec1;
	Vector3D				aVec2;
	Vector3D				aVec3;
	double					fScalar;
	double					fZwi;
	double					fCut;
	double					fCut1;
	double					fCut2;
	double					fCut3;
	ULONG					nIndex;
	UINT16					nNumCuts;
	BOOL					bSamePlane	: 1;
	BOOL					bSameSide	: 1;
	BOOL					bSide		: 1;
	BOOL					bSide1		: 1;
	BOOL					bSide2		: 1;
	BOOL					bSide3		: 1;
	BOOL					bCut1		: 1;
	BOOL					bCut2		: 1;
	BOOL					bCut3		: 1;

	Base3DBSPLocal() {}
};

/*************************************************************************
|*
|* Bucket fuer Base3DBSPLocal
|*
\************************************************************************/

BASE3D_DECL_BUCKET(Base3DBSPLocal, Bucket)

/*************************************************************************
|*
|* Definition fuer die kleinste Dreiecksgroesse bzw. Linienlaenge
|*
\************************************************************************/

#define	MINIMAL_SPLIT_SIZE			(3)

/*************************************************************************
|*
|* Die Basisklasse fuer Standard 3D Ausgaben auf dem Drucker
|*
\************************************************************************/

#define NO_MATERIAL_INDEX		(0xffffffff)

class Base3DPrinter : public Base3DCommon
{
private:
	// Behaelter fuer beliebig viele Punkte der gesamten Geometrie
	B3dEntityBucket			aEntityBucket;

	// Behaelter fuer beliebig viele Primitive
	B3dPrimitiveBucket		aPrimitiveBucket;

	// Behaelter fuer beliebig viele Materialien
	B3dMaterialBucket		aMaterialBucket;

	// wurzel des BSPTrees
	B3dPrimitive*			pBSPTreeRoot;

	// Minimale Flaeche eines Dreiecks in DeviceKoordinaten
	// (logischen Einheiten des zugehoerigen OutputDevice)
	// 0.0 bedeutet: deaktiviert
	double					fMinTriangleSize;

	// Minimale Laenge einer Linie in DeviceKoordinaten
	// (logischen Einheiten des zugehoerigen OutputDevice)
	// 0.0 bedeutet: deaktiviert
	double					fMinLineLength;

	// Quadrat des Minimalen Abstands der Farben im RGB Farbraum
	// mit r,g,b im Bereich [0..255]. 0 bedeutet: deaktiviert
	ULONG					nMinColorDistance;

	// Indexe fuer aktuelle Materialien in den MaterialBucket
	ULONG					nMaterialFrontIndex;
	ULONG					nMaterialBackIndex;

	// Bucket fuer LocalVars und Index in diesen
	Base3DBSPLocalBucket	aLocalVarBucket;
	ULONG					nLocalVarIndex;

	// Gueltigkeit der Indexe fuer aktuelle Materialien
	// in den MaterialBucket
	BOOL					bBackMaterialChanged	: 1;
	BOOL					bFrontMaterialChanged	: 1;

public:
	Base3DPrinter(OutputDevice* pOutDev);
	virtual ~Base3DPrinter();

	// Typbestimmung
	virtual UINT16 GetBase3DType();

	// Szenenverwaltung
	virtual void StartScene();
	virtual void EndScene();

	// Setze die minimale Groesse eines Primitivs vom Typ Dreieck
	// fuer die Druckausgabe. Der Wert 0.0 bedeutet, dass kein Dreieck
	// gesplittet wird und ist der Default-Wert. Falls ein
	// Dreieck im Zuge der Ausgabe gesplittet wird, wird spaetestens
	// bei einer Flaeche von fMinTriangleSize aufgehoert.
	void SetMinimalTriangleSize(double fNew) { fMinTriangleSize = fNew; }
	double GetMinimalTriangleSize() { return fMinTriangleSize; }

	// Setze die minimale Laenge eines Primitivs vom Typ Linie
	// fuer die Druckausgabe. Der Wert 0.0 bedeutet, dass keine Linie
	// gesplittet wird und ist der Default-Wert. Falls eine
	// Linie im Zuge der Ausgabe gesplittet wird, wird spaetestens
	// bei einer Laenge von fMinLineLength aufgehoert.
	void SetMinimalLineLength(double fNew) { fMinLineLength = fNew; }
	double GetMinimalLineLength() { return fMinLineLength; }

	// Setze den minimalen Abstand im Farbraum (RGB, s.o) bei der
	// Ausgabe von smooth shaded faces bei der Druckerausgabe. Die
	// Primitive werde solange weiter unterteilt, bis der Farbabstand
	// kleiner gleich diesem Wert ist, oder die Parameter
	// fMinTriangleSize  bzw. fMinLineLength unterschritten werden
	void SetMinimalColorDistance(ULONG nNew)
		{ nMinColorDistance = nNew * nNew; }
	ULONG GetMinimalColorDistance()
		{ return (ULONG)(sqrt((double)nMinColorDistance)); }

	// Materials ueberladen, um auf Aenderungen reagieren zu
	// koennen. Es werden die Flags bBackMaterialChanged (bzw. Front)
	// gesetzt
	virtual void SetMaterial(Color rNew,
		Base3DMaterialValue=Base3DMaterialAmbient,
		Base3DMaterialMode=Base3DMaterialFrontAndBack);
	virtual void SetShininess(UINT16 nExponent,
		Base3DMaterialMode=Base3DMaterialFrontAndBack);

protected:
	// Hierkommen die von Base3DCommon geclippten Daten an
	virtual void Clipped3DPoint(UINT32 nInd);
	virtual void Clipped3DLine(UINT32 nInd1, UINT32 nInd2);
	virtual void Clipped3DTriangle(UINT32 nInd1, UINT32 nInd2, UINT32 nInd3);

	// Funktionen, um die aktuellen Materialien zwischenzuspeichern
	ULONG GetMaterialIndex(Base3DMaterialMode=Base3DMaterialFrontAndBack);

	// Ausgabefunktionen fuer die Primitive unter
	// StarView im Vektorformat, um die Daten in das Metafile
	// zu schreiben
	void PrintPrimitive(B3dPrimitive* pPrimitive);
	void Print3DPoint(B3dPrimitive* pPrimitive, ULONG nInd);
	void Print3DLine(B3dPrimitive* pPrimitive, ULONG nInd1, ULONG nInd2);
	void Print3DLine(B3dPrimitive* pPrimitive, B3dEntity& rEntity1,
		B3dEntity& rEntity2);
	BOOL DoSplitLine(B3dPrimitive* pPrimitive, B3dEntity& rEntity1,
		B3dEntity& rEntity2);
	void Print3DTriangle(B3dPrimitive* pPrimitive, ULONG nInd1, ULONG nInd2,
		ULONG nInd3);
	void Print3DTriangle(B3dPrimitive* pPrimitive, B3dEntity& rEntity1,
		B3dEntity& rEntity2, B3dEntity& rEntity3);
	double CalcSizeOfTriangle(B3dEntity& rEntity1, B3dEntity& rEntity2,
		B3dEntity& rEntity3);
	double CalcLengthOfLine(B3dEntity& rEntity1, B3dEntity& rEntity2);
	BOOL DoSplitTriangle(B3dPrimitive* pPrimitive, B3dEntity& rEntity1,
		B3dEntity& rEntity2, B3dEntity& rEntity3);

	// Funktionen zur Bufferverwaltung und Aufbau des BSPTrees
	void EmptyBuckets();

	void AddPointToBSPTree(B3dPrimitive* pCurrent, B3dPrimitive* pNew);

	void AddLineToBSPTree(B3dPrimitive* pCurrent, B3dPrimitive* pNew);
	void AddPartialLine(B3dPrimitive* pCurrent, BOOL bSide,
		B3dPrimitive* pPart);

	void AddTriangleToBSPTree(B3dPrimitive* pCurrent, B3dPrimitive* pNew);
	void AddPartialTriangle(B3dPrimitive* pCurrent, BOOL bSide,
		B3dPrimitive* pPart);

	// Anlegen von Primitiven im Ausgabebuffer, Rueckgabe des
	// Indexes auf das neu angelegte Objekt
	ULONG NewPointPrimitive(B3dEntity& rEnt1, ULONG nMatInd=NO_MATERIAL_INDEX);
	ULONG NewLinePrimitive(B3dEntity& rEnt1,
		B3dEntity& rEnt2, ULONG nMatInd=NO_MATERIAL_INDEX);
	ULONG NewTrianglePrimitive(B3dEntity& rEnt1,
		B3dEntity& rEnt2, B3dEntity& rEnt3, ULONG nMatInd=NO_MATERIAL_INDEX);
	ULONG NewQuadPrimitive(B3dEntity& rEnt1, B3dEntity& rEnt2,
		B3dEntity& rEnt3, B3dEntity& rEnt4, ULONG nMatInd);

	// diverse Hilfsfunktionen zur Berechnung
	BOOL GetCutFactor(double& rFac1, double& rFac2, const Vector3D& rPnt1,
		const Vector3D& rVec1, const Vector3D& rPnt2, const Vector3D& rVec2);
};



#endif          // _B3D_B3DPRINT_HXX
