/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: TocTree.java,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 16:41:10 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/
package com.sun.xmlsearch.tree;

import java.io.*;
import java.util.Enumeration;
import java.util.Vector;
import javax.swing.tree.TreeNode;
import com.jclark.xsl.om.*;
import com.jclark.xsl.dom.Transform;
import com.jclark.xsl.dom.TransformException;
import com.sun.xmlsearch.util.IntegerArray;

public final class TocTree implements Externalizable {
  private String _titles;	// all titles, concatenated
  private String _xPaths;	// all paths
  
  private int[] _parent;
  private int[] _titleStart;
  private int[] _xPathStart;
  
  // not externalized
  private TocNode[] _nodes;
  private int _size;
  private IntegerArray[] _children;
  
  public TocTree() { /* for Externalizable */ }
  
  public TocNode getRoot() {
    return getNode(0);
  }

  public TocNode[] getNodes(int[] indices) {
    TocNode[] result = new TocNode[indices.length];
    for (int i = 0; i < indices.length; i++)
      result[i] = getNode(indices[i]);
    return result;
  }

  public TocNode getNode(int index) {
    if (_nodes == null) {
      _nodes = new TocNode[_size = _parent.length];
      _children = new IntegerArray[_size];
      for (int i = 0; i < _size; i++)
	(_nodes[i] = new TocNode())._index = i;
      for (int i = 1; i < _size; i++) {
	int parent = _parent[i];
	if (_children[parent] == null)
	  _children[parent] = new IntegerArray(4);
	_children[parent].add(i);
      }
    }
    return _nodes[index];
  }

  public void readExternal(ObjectInput in)
    throws ClassNotFoundException, IOException {
      _titles = (String) in.readObject();
      _xPaths = (String) in.readObject();
      _parent = (int[]) in.readObject();
      _titleStart = (int[]) in.readObject();
      _xPathStart = (int[]) in.readObject();
      _size = _parent.length;
  }
  
  public void writeExternal(ObjectOutput out) throws IOException {
    out.writeObject(_titles);
    out.writeObject(_xPaths);
    out.writeObject(_parent);
    out.writeObject(_titleStart);
    out.writeObject(_xPathStart);
  }

  public static TocTree makeTocTree(Node docRoot,
				    Transform tocTransform,
				    Names names) 
    throws TransformException, XSLException {
      TocTree tree = new TocTree();
      tree.build(docRoot, tocTransform, names);
      return tree;
  }

  private void build(Node docRoot, Transform tocTransform, Names names)
    throws TransformException, XSLException {
      TocResult content = new TocResult(names);
      tocTransform.transform(docRoot, content);
      content.finish();
  }

  private final class TocResult extends ResultAdapter {
  
    private final class TemporaryTocNode {
      int parent;
      int titleStart;
      int xPathStart;
    }

    private StringBuffer _titleBuffer = new StringBuffer(2048);
    private int _titlesOffset;
    private StringBuffer _xPathBuffer = new StringBuffer(2048);
    private int _pathsOffset;
    
    private int _currentNodeIndex;
    private IntegerArray _stack = new IntegerArray(32);
    private TemporaryTocNode _currentNode;
    private Vector _tempNodes = new Vector(512);

    private final Name _title_Name;
    private final Name _path_Name;
    private final Name _TocNode_Name;

    public TocResult(Names names) throws XSLException {
      _title_Name   = names.getAttributeName("title");
      _path_Name   = names.getAttributeName("path");
      _TocNode_Name = names.getElementName("TocNode");
    }

    public void startElement(Name elementType, NamespacePrefixMap nsMap) {
      //      System.out.println("startElement " + elementType.toString());
      _currentNode = new TemporaryTocNode();
      _currentNode.parent = _stack.cardinality() > 0 ? _stack.last() : -1;
      _stack.add(_currentNodeIndex++);
      _tempNodes.addElement(_currentNode);
    }
  
    public void attribute(Name name, String value) throws XSLException {
      if (name == _title_Name) {
	//	System.out.println("title " + value);
	_titleBuffer.append(value);
	_currentNode.titleStart = _titlesOffset;
	_titlesOffset += value.length();
      }
      else if (name == _path_Name) {
	//	System.out.println("path " + value);
	_xPathBuffer.append(value);
	_currentNode.xPathStart = _pathsOffset;
	_pathsOffset += value.length();
      }
      else
	throw new XSLException("unknown TocNode attribute: " + name.toString());
    }

    public void endElement(Name elementType) {
      _stack.pop();
    }

    public void finish() throws XSLException {
      _titles = _titleBuffer.toString();
      _xPaths = _xPathBuffer.toString();
      
      int size = _tempNodes.size();
      
      _parent = new int[size];
      _titleStart = new int[size + 1];
      _xPathStart = new int[size + 1];
      for (int i = 0; i < size; i++) {
	TemporaryTocNode tn = (TemporaryTocNode)_tempNodes.elementAt(i);
	_parent[i]      = (int) tn.parent;
	_titleStart[i]  = (int) tn.titleStart;
	_xPathStart[i]  = (int) tn.xPathStart;
      }
      _titleStart[size] = (int) _titles.length();
      _xPathStart[size] = (int) _xPaths.length();
    }
  }

  public final class TocNode implements TreeNode {
    private int _index;

    public String toString() {
      return getTitle() + '\t' + getXPath();
    }

    public int index() {
      return _index;
    }

    public String getTitle() {
      return _titles.substring(_titleStart[_index],
			       _titleStart[_index + 1]);
    }

    public String getXPath() {
      return _xPaths.substring(_xPathStart[_index],
			       _xPathStart[_index + 1]);
    }

    public TreeNode getParent() {
      return _nodes[_parent[_index]];
    }

    public boolean getAllowsChildren() {
      return true;
    }

    public boolean isLeaf() {
      return _children[_index] == null;
    }

    public int getChildCount() {
      return isLeaf() ? 0 : _children[_index].cardinality();
    }

    public TreeNode getChildAt(int i) {
      return _nodes[_children[_index].at(i)];
    }

    public int getIndex(final TreeNode node) {
      return isLeaf() ? -1 : _children[_index].indexOf(((TocNode)node)._index);
    }

    public Enumeration children() {
      return new Enumeration() {
	private int _index;
	public boolean hasMoreElements() {
	  return _index < getChildCount();
	}
	public Object nextElement() {
	  return getChildAt(_index++);
	}
      };
    }
  }
}
