'encoding UTF-8  Do not remove or change this line!
'*************************************************************************
'*
'*  OpenOffice.org - a multi-platform office productivity suite
'*
'*  $RCSfile: docfuncs.inc,v $
'*
'*  $Revision: 1.31 $
'*
'*  last change: $Author: jsk $ $Date: 2006/06/27 13:44:01 $
'*
'*  The Contents of this file are made available subject to
'*  the terms of GNU Lesser General Public License Version 2.1.
'*
'*
'*    GNU Lesser General Public License Version 2.1
'*    =============================================
'*    Copyright 2005 by Sun Microsystems, Inc.
'*    901 San Antonio Road, Palo Alto, CA 94303, USA
'*
'*    This library is free software; you can redistribute it and/or
'*    modify it under the terms of the GNU Lesser General Public
'*    License version 2.1, as published by the Free Software Foundation.
'*
'*    This library is distributed in the hope that it will be useful,
'*    but WITHOUT ANY WARRANTY; without even the implied warranty of
'*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
'*    Lesser General Public License for more details.
'*
'*    You should have received a copy of the GNU Lesser General Public
'*    License along with this library; if not, write to the Free Software
'*    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
'*    MA  02111-1307  USA
'*
'/************************************************************************
'*
'* owner : joerg.skottke@sun.com
'*
'* short description : misc functions to handle documents
'*
'*******************************************************************************
' **
' #1 hCloseDocs ' closes all open documents and checks the documentcount
' #1 hUseImpressAutopilot ' enables/disables the autopilot in tools/options
' #1 hCheckDocCount ' check if the correct number of documents is open
' #1 hCloseNavigator ' closes the navigator in writer
' #1 hNumericDocType ' translates a given number to a gApplication
' #1 hNameGen_append ' creates a name that ends on a given char (decimal)
' #1 hNameGen_lead ' creates a name that begins with a given char (decimal)
' #1 hChangeDoc ' changes any documenttype (to trigger document-is-changed-flag)
' #1 exitToolsOptions ' exit Tools/options
' #1 enterToolsOptionsOOoSecurity ' go to the security page
' #1 enterToolsOptionsOOoView ' go to the view page
' #1 hIdentifyWriterDoc ' verify that the correct document has the focus
' #1 hDestroyDocument ' close a document, return true on success
' #1 hCreateDocument ' open new document, return true on success
' #1 hDocumentTypeKeys ' type some chars/sequences to any document
' **
'\******************************************************************************

function hCloseDocs( iDocCount as integer ) as integer

    '///<h3>Close a specified number of documents</h3>
    '///<i>Starting point: Any open document</i><br>
    '///<u>Use of this function is not encouraged</u>
    '///<ul>

    const CFN = "hCloseDocs::"
    
    '///+<li>Verify function parameter</li>
    if ( iDocCount < 1 ) then 
        warnlog( CFN & "Invalid parameter passed to function" )
        hCloseDocs() = 0
        exit function
    endif
    
    dim iCurrentDoc as integer
    
    dim iDocsFound  as integer
        iDocsFound = getDocumentCount()
    
    dim iRealDocCount as integer
        iRealDocCount = iDocCount + 1
    
    
    '///+<li>check the expected number of documents against the current number</li>
    if ( iDocsFound <> iRealDocCount ) then
    
        qaerrorlog( CFN & "The number of currently open documents is incorrect" )
        printlog( CFN & "Expected: " & iRealDocCount )
        printlog( CFN & "Found...: " & iDocsFound    )
    
    endif
    
    '///+<li>close the documents</li>
    for iCurrentDoc = 2 to iDocsFound
        hDestroyDocument()
    next iCurrentDoc
    
    '///+<li>Return the number of documents found <u>before</u> closing</li>
    '///</ul>
    hCloseDocs() = iDocsFound

end function

'*******************************************************************************

function hUseImpressAutopilot( bEnable as boolean ) as boolean

    '///<h3>Enable/disable the Impress Autopilot in Tools/Options</h3>
    '///<i>Starting point: Any plain document</i>
    '///<ul>
    const CFN = "hUseImpressAutopilot::"

    '///+<li>Create a new IMPRESS document</li>
    gApplication = "IMPRESS"
    hCreateDocument()

    '///+<li>Open Tools/Options, go to Presentataion/General page</li>
    ToolsOptions
    hToolsOptions( "Presentation" , "General" )

    '///+<li>Check or uncheck to start with autopilot</li>
    Kontext "TabSonstigesDraw"
    if ( mitAutopilotStarten.exists() ) then

        if ( bEnable ) then
            mitAutopilotStarten.check()
            printlog( CFN & "Enable Impress Autopilot" )
        else
            mitAutopilotStarten.uncheck()
            printlog( CFN & "Disable Impress Autopilot" )
        endif

    else

        warnlog( CFN & "Cannot find Autopilot Checkbox" )

    endif
   
    '///+<li>Return Status of Checkbox (checked=TRUE)</li>
    hUseImpressAutopilot() = mitAutopilotStarten.isChecked()

    '///+<li>Close Tools/Options</li>
    Kontext "OptionenDlg"
    OptionenDlg.OK()

    '///+<li>Close IMPRESS document</li>
    hDestroyDocument()

    '///+<li>Returncode is undefined</li>
    '///</ul>
    
end function

'*******************************************************************************

function hCheckDocCount( iDocsExpected as integer, bReset as boolean ) as boolean

    const CFN = "hCheckDocCount::"
    
    '///<h3>Function to verify the correct number of documents is open</h3>
    '///<i>Starting point: Any plain document</i>
    '///<ul>
    
    '///+<li>Verify function parameter</li>
    if ( iDocsExpected < 1 ) then
        warnlog( CFN & "Invalid parameter passed to function: " & iDocsExpected )
        hCheckDocCount() = false
        exit function
    endif
    
    
    dim iDocsPresent as integer
        iDocsPresent = getDocumentCount()
    
    '///+<li>Test present documents against the expected number</li>
    if ( iDocsExpected = iDocsPresent ) then
    
        printlog( CFN & "Correct number of documents is open" )
        hCheckDocCount() = true
    
    else
    
        warnlog( CFN & "Incorrect document count." )
        printlog( CFN & "Expected: " & iDocsExpected )
        printlog( CFN & "Found...: " & iDocsPresent  )
        hCheckDocCount() = false
    
        if ( bReset ) then
            printlog( CFN & "Restart the application" )
            all exitRestartTheOffice()
        endif
    
    endif
    '///+<li>Return TRUE if the expected number of documents are open</li>
    '///</ul>

end function

'*******************************************************************************

function hCloseNavigator() as boolean

    '///<h3>Function to close the navigator</h3>
    '///<i>Starting point: Any document</i>
    '///<ul>
    const CFN = "hCloseNavigator::"
    
    '///+<li>close the navigator if found</li>
    printlog( CFN & "Closing Navigator if present" )
    Kontext "Navigator"
    if ( Navigator.Exists() ) then
        Navigator.Close()
    end if
    
    '///+<li>verify that the navigator is indeed closed, return true on success</li>
    Kontext "Navigator"
    if ( Navigator.exists() ) then
        hCloseNavigator() = false
        warnlog( CFN & "Failed to close Navigator, it is still open." )
    else
        hCloseNavigator() = true
    endif
    '///+<li>Returns TRUE on success</li>
    '///</ul>

end function

'*******************************************************************************

function hNumericDocType( doctype as integer ) as string

    '///<h3>Convert a numeric expression to a gApplication</h3>
    '///<i>BEWARE: This is a core function used by many testcases!</i><br>
    '///<u>Using file: framework/tools/input/applications.txt as reference</u>
    '///<ul>
    
    const CFN = "hNumericDocType::"
    
    dim sFile as string
        sFile = gTesttoolPath & "framework\tools\input\applications.txt"
        sFile = convertpath( sFile )
       
    dim sList( 100 ) as string
    dim iDocTypes as integer
    
    '///+<li>Retrieve matching list from file</li>
    hGetDataFileSection( sFile , sList() , "application" , "" , "" )
    iDocTypes = val( sList( 0 ) )
    
    '///+<li>Verify that the functionparameter (documenttype) is valid</li>
    if ( ( doctype < 0 ) or ( doctype > iDocTypes ) ) then
        warnlog( CFN & "Undefined numeric doctype: " & doctype )
        hNumericDocType() = ""
        exit function
    endif
       
    '///+<li>Set gApplication to the requested document type</li>
    gApplication = hGetValueForKeyAsString( sList() , doctype )
    
    '///+<li>Set a plain text name as returnvalue for the function</li>
    hGetDataFileSection( sFile , sList() , "application_names" , "" , "" )
    hNumericDocType() = hGetValueForKeyAsString( sList() , gApplication )
    '///</ul>

end function

'*******************************************************************************

function hNameGen_append( iDecChar as long , bSuf as boolean ) as string

    '///<h3>Create a filename with appended special unicode character</h3>
    '///<ul>

    ' returns a filename that contains an ASCII/UNICODE character before the
    ' (optional) dot + suffix
    
    '///+<li>Append a given UNICODE character to a filename</li>
    dim cFile as string
        cFile = "test" & CHR$( iDecChar )
    
    '///+<li>Optionally add a suffix</li>
    if ( bSuf = TRUE ) then
        cFile = cFile & hGetSuffix( "current" )
    endif
    
    '///+<li>Return the filename</li>
    hNameGen_append() = cFile
    '///</ul>

end function

'*******************************************************************************

function hNameGen_lead( iDecChar as long , bSuf as boolean ) as string

    '///<h3>Create a filename with prepended special unicode character</h3>
    '///<ul>

    ' returns a filename that contains a leading ASCII/UNICODE character infront
    ' of the optional dot + suffix
    
    '///+<li>Prepend a given UNICODE character to a filename</li>
    dim cFile as string
        cFile = CHR$( iDecChar ) & "test"
    
    '///+<li>Optionally add a suffix</li>
    if ( bSuf = TRUE ) then
        cFile = cFile & hGetSuffix( "current" )
    endif
    
    '///+<li>Return the filename</li>
    hNameGen_lead() = cFile
    '///</ul>

end function

'*******************************************************************************

function hChangeDoc() as string

    '///<h3>Function to modify all documenttypes to set  the "changed" flag</h3>
    '///<i>Starting point: Any plain document</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Nothing</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>String, ready to use in printlog</li>
    '///<ul>
    '///+<li>&quot;Changed &lt;Documenttype&gt;&quot;</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>

    select case gApplication
   
      '///+<li>WRITER: Type some text</li>
      case "WRITER", "HTML", "MASTERDOC"
                      Kontext "DocumentWriter"
                      DocumentWriter.TypeKeys "<Return>This doc is changed!"
                      Sleep 1
                      hChangeDoc() = "Changed WRITER/HTML/MASTERDOC" 

      '///+<li>CALC: Write some text into the current location (usually A1)</li>
      case "CALC"   : Kontext "DocumentCalc"
                      DocumentCalc.TypeKeys "<Down>This doc is changed!<Return>"
                      Sleep 1
                      hChangeDoc() = "Changed CALC" 

      '///+<li>DRAW: Draw a rectangle</li>
      case "DRAW"   : Kontext "DocumentDraw"
                      hRechteckErstellen ( 20, 20, 25, 25 )
                      gMouseClick ( 1, 1 )
                      Sleep 1
                      hChangeDoc() = "Changed DRAW" 

      '///+<li>IMPRESS: Draw a rectangle</li>
      case "IMPRESS": Kontext "DocumentImpress"
                      hRechteckErstellen ( 20, 20, 25, 25 )
                      gMouseClick ( 1, 1 )
                      Sleep 1
                      hChangeDoc() = "Changed IMPRESS" 

      '///+<li>MATH: Create a simple formaula</li>
      case "MATH"   : SchreibenInMathdok( "b over c" )
                      hChangeDoc() = "Changed MATH" 

   end select
   '///</ul>
   
end function

'*******************************************************************************

function exitToolsOptions( bRestart as boolean )

    '///<h3>Exit Tools/Options dialog, optionally restart the office</h3>
    '///<i>Starting point: Tools/Options dialog</i>

    Kontext "OptionenDLG"
    OptionenDLG.OK()
   
    if( bRestart ) then
        call exitRestartTheOffice()
    endif
   
end function

'*******************************************************************************

function enterToolsOptionsOOoSecurity() as boolean

    '///<h3>Go to the Tools/Options - OpenOffice.org/Security page directly</h3>
    '///<i>Starting point: Any plain document</i>
    '///<ul>
    dim brc as boolean

    '///+<li>Open a new document</li>
    hCreateDocument()
   
    '///+<li>Open Tools/Options</li>
    ToolsOptions
    
    '///+<li>Go to the requested tabpage</li>
    call hToolsOptions( "StarOffice" , "Security" )
    
    '///+<li>Test whether the Macro Security button exists or not</li>
    if ( MacroSecurity.exists() ) then
        brc = true
    else
        brc = false
    endif
    
    '///+<li>Return TRUE if - and only if - the Macro Security Button exists</li>
    enterToolsOptionsOOoSecurity() = brc
    '///</ul>
   
end function

'*******************************************************************************

function enterToolsOptionsOOoView()

    '///<h3>Go to the Tools/Options - OpenOffice.org/Security page directly</h3>
    '///<i>Starting point: Any plain document</i><br>
    '///<u>There is no returncode defined yet</u>
    '///<ul>

    '///+<li>Open a new document</li>
    hCreateDocument()
   
    '///+<li>Open Tools/Options</li>
    ToolsOptions
    
    '///+<li>Go to the requested tabpage</li>
    call hToolsOptions( "StarOffice" , "View" )
    '///</ul>
   
end function

'*******************************************************************************

function hIdentifyWriterDoc( cString as string, bWarn as boolean ) as boolean

    '///<h3>Function to validate that the expected WRITER doc has the focus</h3>
    '///<i>Starting point: Any WRITER document containing some text</i>
    '///<ul>

    const CFN = "hIdentifyWriterDoc::"
    dim cIdentifier as string
    dim irc as integer ' some temporary returnvalue

    '///+<li>Test the function parameter</li>
    if ( cString = "" ) then
        qaerrorlog( CFN & "Function parameter is empty string. Aborting" )
        hIdentifyWriterDoc() = false
        exit function
    endif

    '///+<li>Verify function prerequisites</li>
    kontext "DocumentWriter"
    if ( not DocumentWriter.exists() ) then
        printlog( CFN & "There is no Writer document. Aborting" )
        hIdentifyWriterDoc() = false
        exit function
    endif
    
    hIdentifyWriterDoc() = true

    '///+<li>Go to the very beginning of the current writer document</li>
    kontext "DocumentWriter"
    DocumentWriter.typeKeys( "<MOD1 HOME>" )
    
    '///+<li>Select the entire line</li>
    DocumentWriter.typeKeys( "<SHIFT END>" )
   
    '///+<li>copy the string to the clipboard and store it into a variable</li>
    editcopy
    cIdentifier = getClipboardText()
   
    '///+<li>compare the string to the reference (given as parameter)</li>
    irc = hCompareSubstrings( cIdentifier, cString )

    '///+<li>Write the result to the printlog if desired</li>
    if ( irc = 0 ) then
   
        if ( bWarn ) then
            warnlog ( CFN & "The focus is not on the expected document" )
        else
            printlog( CFN & "The focus is not on the expected document" )
        endif
        
        printlog( CFN & "Expected: " & cString )
        printlog( CFN & "Found...: " & cIdentifier )
        hIdentifyWriterDoc() = false
      
    else
        printlog( CFN & "OK: The document contains the requested string" )
    endif
    '///+<li>Return TRUE only if the current document is the expected one</li>
    '///</ul>

end function

'*******************************************************************************

function hCloseDocumentNew() as boolean
    '///<h3>Legacy wrapper for hDestroyDocument</h3>
    hCloseDocumentNew() = hDestroyDocument()
end function

function hDestroyDocument() as boolean

    '///<h3>Improved version of hCloseDocument()</h3>
    '///<i>Starting point: Any document</i><br>
    '///<i>Note: Using this function on the backing window triggers a 
    '///+ restart of the office. If gApplication is not "BACKGROUND" the
    '///+ function assumes that this is unintentional, warns about it and
    '///+ returns FALSE!</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Nothing, valid gApplication must exist (refer to hNumericDocType())</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Errorcondition (boolean)</li>
    '///<ul>
    '///+<li>TRUE if document was closed cleanly</li>
    '///+<li>FALSE if some error occurred.</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    
    const CFN = "hDestroyDocument::"
    dim iDocCountBeforeClose as integer
    dim iDocCountAfterClose as integer
    dim iCounter as integer
    dim irc as integer
    dim brc as boolean
    dim cMessage as string
    
    '///+<li>If current application is the backing window, restart the office</li>
    if ( ucase( gApplication ) = "BACKGROUND" ) then
    
        printlog( CFN & "Attempting to restart the office from backing window" )
        call ExitRestartTheOffice
        if ( getDocumentCount = 1 ) then
            brc = hIdentifyWriterDoc( "The first doc!" )
        else
            brc = false
        endif
        hDestroyDocument() = brc
        
        ' leave the function, else we end up with too deeply nested if-statements
        exit function
        
    endif
        
    '///+<li>Get the current number of open documents</li>
    iDocCountBeforeClose = getDocumentCount()
    
    '///+<li>Close leftover open dialogs</li>
    irc = hHandleInitialDialogs()
    if ( irc <> 0 ) then
        warnlog( CFN & "Closed unexpected dialog(s)" )
    endif
    
    '///+<li>Sometimes we end up with getDocumentCount reporting 0 open documents</li>
    if ( iDocCountBeforeClose = 0 ) then
        qaerrorlog( CFN & "getDocumentCount reported 0 open documents" )
        printlog( CFN & "Verify that the office is ready to accept commands" )
        sleep( 5 )
        iDocCountBeforeClose = getDocumentCount()
    endif
    
    '///+<li>If only one document is left, we have a problem that needs fixing</li>
    if ( iDocCountBeforeClose = 1 ) then
    
        brc = hIdentifyWriterDoc( "The first doc!", false )

        ' if this is "The first doc!", we simply don't close it
        if ( brc ) then
            printlog( CFN & "Refusing to close <The first doc!>" )
        else
            ' if this is an unknown document, we cannot determine the status 
            ' so we better do a restart.
            warnlog( CFN & "Office status is undefined, restarting" )
            call ExitRestartTheOffice
        endif
        
        hDestroyDocument() = false
        exit function

    endif
    
    ' If we end here, this is always an error because gApplication is not backing window
    '///+<li>If Backing Window is only document but gApplication says otherwise
    '///+ we restart the office. This is always due to previous errors</li>
    if ( iDocCountBeforeClose = 0 ) then
        warnlog( CFN & "Restarting office, there are no more documents left" )
        call ExitRestartTheOffice()
        hDestroyDocument() = false
        exit function
    endif
         
    '///+<li>Under normal conditions: Try to close the document max 3 times</li>
    '///<ol>
    for iCounter = 1 to 3
    
        '///+<li>Close fileopen dialog if present</li>
        kontext "OeffnenDlg"
        if ( OeffnenDlg.exists() ) then
            printlog( CFN & "Closed FileOpen dialog" )
            OeffnenDlg.cancel()
        endif
        
        '///+<li>Close filesave dialog if present</li>
        kontext "SpeichernDlg"
        if ( SpeichernDlg.exists() ) then
             printlog( CFN & "Closed FileSave dialog" )
             SpeichernDlg.cancel()
        endif    
    
        '///+<li>Trigger FileClose slot</li>
        try
            FileClose
        catch
        endcatch
       
        sleep( 3 )
 
        '///+<li>Handle dialogs such as file has not been saved ...</li>
        kontext "active"
        if ( active.exists() ) then
            cMessage = active.getText()
            cMessage = hRemoveLineBreaks( cMessage )
            printlog( CFN & cMessage )
            try
                active.no()
            catch
            endcatch
            
            try
                active.click( 202 )
            catch
            endcatch
        endif
        
        '///+<li>Verify the document is indeed closed</li>
        iDocCountAfterClose = getDocumentCount()
    
        if ( iDocCountAfterClose = ( iDocCountBeforeClose - 1 ) ) then
            printlog( CFN & "Document closed. Open: " & iDocCountAfterClose )
            hDestroyDocument() = true
            exit for
        else
            warnlog( CFN & "Document not closed" )
            printlog( CFN & "Before: " & iDocCountBeforeClose )
            printlog( CFN & "After.: " & iDocCountAfterClose )
            hDestroyDocument() = false
            wait( 1000 )
        endif
        
    next iCounter
    '///</ol>
    '///</ul>
    
end function
       
'*******************************************************************************

function hCreateDocument() as boolean

    '///<h3>Create anew document, extends hNewDocument()</h3>
    '///<i>Starting point: Any document</i><br>
    '///<i>Note: When calling this function with gApplication "BACKGROUND" the
    '///+ function will actually close one document to get to the backing window</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Nothing, valid gApplication must be set</li>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Errorstatus (boolean)</li>
    '///<ul>
    '///+<li>TRUE if a new document was created successfully</li>
    '///+<li>FALSE if anything went wrong</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    

    const CFN = "hCreateDocument::"

    dim iDocCountBefore as integer 
    dim iDocCountAfter as integer
    dim iDocIndex as integer

    '///+<li>Retrieve the number of open documents</li>
    iDocCountBefore = getDocumentCount()
    
    if ( ucase( gApplication ) = "BACKGROUND" ) then
        if ( iDocCountBefore = 1 ) then
            call hCloseDocument()
            iDocIndex = -1
        else
            qaerrorlog( "Too many windows open for backing mode" )
        endif
    else
        '///+<li>Open a new document using hNewDocument()</li>
        call hNewDocument()
        iDocIndex = +1
    endif

    '///+<li>Retrieve the number of open documents</li>
    iDocCountAfter = getDocumentCount()

    '///+<li>Verify that there is exactly one more document than before</li>
    if ( iDocCountAfter = ( iDocCountBefore + iDocIndex ) ) then
        printlog( CFN & "New " & gApplication & " document. Open: " & iDocCountAfter )
        hCreateDocument() = true
    else
        printlog( CFN & "Documentcount unchanged: " & iDocCountAfter )
        hCreateDocument() = false
    endif

    '///+<li>Return TRUE if a new document has been created</li>
    '///</ul>
    
end function

'*******************************************************************************

function hDocumentTypeKeys( cString as string ) as boolean

    '///<h3>Type something to a document of type iApp</h3>
    '///<i>See hNumericDocType for details and number&eq;gApplication matching</i><br>
    '///<u>Input</u>:
    '///<ol>
    '///+<li>Keystrokes to be sent to the document (string), e.g.</li>
    '///<ul>
    '///+<li>&quot;&lt;SHIFT HOME&gt;&quot</li>
    '///+<li>&quot;&lt;MOD1 A&gt;&quot</li>
    '///+<li>&quot;&lt;MOD2 Q&gt;&quot</li>
    '///+<li>&quot;&lt;Hello&gt;&quot</li>
    '///</ul>
    '///</ol>
    '///<u>Returns</u>:
    '///<ol>
    '///+<li>Errorstatus (boolean)</li>
    '///<ul>
    '///+<li>TRUE if all went well</li>
    '///+<li>FALSE on any error</li>
    '///</ul>
    '///</ol>
    '///<u>Description</u>:
    '///<ul>
    
    const CFN = "hDocumentTypeKeys::"
    dim brc as boolean
        brc = true
    
    '///+<li>Set focus to the current documenttype determined by gApplication</li>
    '///+<li>Type the string passed as function parameter</li>
    select case gApplication
    case "WRITER"   :   kontext "DocumentWriter"
                        DocumentWriter.typeKeys( cString )
    case "CALC"     :   kontext "DocumentCalc"
                        DocumentCalc.typeKeys( cString )
    case "IMPRESS"  :   kontext "DocumentImpress"
                        DocumentImpress.typeKeys( cString )
    case "DRAW"     :   kontext "DocumentDraw"
                        DocumentDraw.typeKeys( cString )
    case "MATH"     :   kontext "DocumentMath"
                        DocumentMath.typeKeys( cString )
    case "MASTERDOC":   kontext "DocumentWriter"
                        DocumentWriter.typeKeys( cString )
    case "HTML"     :   kontext "DocumentWriter"
                        DocumentWriter.typeKeys( cString )
    case else
        warnlog( CFN & "Incorrect parameter passed to function (iApp): " & iApp )
        brc = false
    end select
    
    '///+<li>Print some message to the log on success</li>
    if ( brc ) then
        printlog( CFN & "Sent keystroke to " & gApplication )
    endif
    
    hDocumentTypeKeys() = brc
    '///</ul>
    
end function
