;;;-*- Mode: Lisp; Package: CCL -*-
;;;
;;;   Copyright (C) 1994-2001 Digitool, Inc
;;;   This file is part of Opensourced MCL.
;;;
;;;   Opensourced MCL is free software; you can redistribute it and/or
;;;   modify it under the terms of the GNU Lesser General Public
;;;   License as published by the Free Software Foundation; either
;;;   version 2.1 of the License, or (at your option) any later version.
;;;
;;;   Opensourced MCL is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;   Lesser General Public License for more details.
;;;
;;;   You should have received a copy of the GNU Lesser General Public
;;;   License along with this library; if not, write to the Free Software
;;;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;


(in-package "CCL")

(eval-when (:compile-toplevel :load-toplevel :execute)
  (require "SPARC-ARCH")
  (require "RISC-LAP")
  (require "SPARC-ASM")
)

(defvar *sparc-lap-macros* (make-hash-table :test #'equalp))

(defparameter *sparc-lap-register-aliases*
  `((%nfn . ,sparc::%nfn)
    (%fname . ,sparc::%fname)
    (%next-method-context . ,sparc::%next-method-context)))

(defun sparc-lap-macro-function (name)
  (gethash (string name) *sparc-lap-macros*))

(defun (setf sparc-lap-macro-function) (def name)
  (let* ((s (string name)))
    (when (gethash s sparc::*sparc-opcode-numbers*)
      (error "~s already defines a Sparc instruction . " name))

    (setf (gethash s *sparc-lap-macros*) def)))

(defmacro defsparclapmacro (name arglist &body body)
  `(progn
     (setf (sparc-lap-macro-function ',name)
           (nfunction (sparc-lap-macro ,name) ,(parse-macro name arglist body)))
     (record-source-file ',name 'sparc-lap)
     ',name))

(defvar *sparc-lap-constants* ())
(defvar *sparc-lap-regsave-reg* ())
(defvar *sparc-lap-regsave-addr* ())
(defvar *sparc-lap-regsave-label* ())






(defun sparc-lap-macroexpand-1 (form)
  (unless (and (consp form) (atom (car form)))
    (values form nil))
  (let* ((expander (sparc-lap-macro-function (car form))))
    (if expander
      (values (funcall expander form nil) t)
      (values form nil))))


#|
(defun sparc-lap-encode-regsave-info (maxpc)
  (declare (fixnum maxpc))
  (if *sparc-lap-regsave-label*
    (let* ((regsave-pc (ash (the fixnum (lap-label-address *sparc-lap-regsave-label*)) -2)))
      (declare (fixnum regsave-pc))
      (if (< regsave-pc #x80)
        (let* ((instr (sparc-emit-lap-instruction *sparc-lwz-instruction*
                                                (list *sparc-lap-regsave-reg*
                                                      (dpb (ldb (byte 2 5) regsave-pc) 
                                                           (byte 2 0) 
                                                           *sparc-lap-regsave-addr*)
                                                      (ldb (byte 5 0) regsave-pc)))))
          (setf (lap-instruction-address instr) maxpc)
          (incf maxpc 4))
        (warn "Can't encode register save information."))))
  maxpc)
|#

(defun sparc-lap-encode-regsave-info (maxpc)
  maxpc)

(defun %define-sparc-lap-function (name body &optional (bits 0))
  (with-dll-node-freelist (*lap-instructions* *lap-instruction-freelist*)
      (let* ((*lap-labels* ())
             (*sparc-lap-regsave-label* ())
             (*sparc-lap-regsave-reg* ())
             (*sparc-lap-regsave-addr* ())
             (*sparc-lap-constants* ()))
        (dolist (form body)
          (sparc-lap-form form))
        #+sparc-lap-scheduler (sparc-schedule-instuctions)       ; before resolving branch targets
        (sparc-lap-generate-code name (sparc-lap-encode-regsave-info (sparc-lap-do-labels)) bits))))

            

(defun sparc-lap-assign-addresses (delete-labels-p)
  (let* ((pc 0))
    (declare (fixnum pc))
    (do-dll-nodes (node *lap-instructions*)
      (setf (instruction-element-address node) pc)
      (if (not (typep node 'lap-instruction))
        (if delete-labels-p (remove-dll-node node))
        (incf pc 4)))
    (if (>= pc (ash 1 13)) (compiler-function-overflow))
    pc))


;;  Try to move useful instructions into branch delay slots.
;;  Be careful.
(defparameter *sparc-lap-optimize-delay-slots* t)

(defun sparc-lap-optimize-branch-delay-slots ()
  (when *sparc-lap-optimize-delay-slots*
  ;; We can't move something if it's not an instruction (e.g., it's a label)
  ;; of if it's in the delay slot of a branch.
    (labels
	(
	 (branch-target-label (branch)
	   (svref (lap-instruction-parsed-operands branch)
		  (1- (arch::opcode-min-args
		       (lap-instruction-opcode branch)))))
	 (is-delayed-instruction (element)
	   (if (typep element 'lap-instruction)
	       (let* ((flags (arch::opcode-flags
			      (lap-instruction-opcode element))))
		 (if (logbitp sparc::f-delayed flags)
		     flags))))
	 (is-delay-slot (element)
	   (and (typep element 'lap-instruction)
		(is-delayed-instruction (lap-instruction-pred element))))
	 ;; Simply because something isn't immovable doesn't imply that
	 ;; it -should- be moved; "non-immovability" is a necesary but
	 ;; not sufficient condition for moving something.  An instruction
	 ;; that affects whether a branch is taken (or where it goes) should
	 ;; stay put.
	 ;; If the branch is unconditional and the immediately preceding
	 ;; element is not immovable, we can move that element into the
	 ;; branch delay slot
	 (immovable (element)
	   (or (not (typep element 'lap-instruction))
	       (is-delayed-instruction element)
	       (is-delay-slot element)))
         ;; Is thing a NOP that's preceded by (some form of) delayed branch ?
         ;; If so, return the preceding branch instruction.
	 (empty-delay-slot-p (element)
	   (and (is-delay-slot element)
		(eq (lap-instruction-opcode element)
		    sparc::*sparc-nop-opcode*)
		(lap-instruction-pred element)))
	 ;; We know that branch reaches label.  If no other branches
	 ;; do, the label uniquely denotes an address, and we can't
	 ;; "fall into" the label, then the branch is the only path
	 ;; to the label and we might be able to move the instruction
	 ;; following the label into the branch delay slot.  If we
	 ;; do so, we have to annul the branch if it's conditional,
	 ;; so the delayed instruction would only be executed if the
	 ;; branch is taken.
	 (only-path-to-label (label)
	   (if (null (cdr (lap-label-refs label)))
	       (let* ((prev (lap-label-pred label)))
		 (if (is-delay-slot prev)
		     (let* ((prevcti (lap-instruction-pred prev))
			    (flags (arch::opcode-flags
				    (lap-instruction-opcode prevcti))))
		       (declare (fixnum flags))
		       ;; A call or conditional-branch can fall through
		       (and (not (logbitp sparc::f-condbr flags))
			    (not (logbitp sparc::f-jsr flags))
			    (let* ((succ (lap-label-succ label)))
			      (unless (immovable succ)
				succ))))))))	   
         (replace-delay-slot (current candidate)
           (let* ((delayed-branch (dll-node-pred current)))
             (remove-dll-node candidate)
             (remove-dll-node current)
             (insert-dll-node-after candidate delayed-branch)))
         )
    (do* ((header *lap-instructions*)
          (current (dll-header-last header) (dll-node-pred current)))
         ((eq current header))
      (let* ((delayed-branch (empty-delay-slot-p current)))
        (when delayed-branch
          (let* ((opcode (lap-instruction-opcode delayed-branch))
                 (flags (arch::opcode-flags opcode)))
            (cond
	      ((and (logbitp sparc::f-unbr flags)
		    (not (logbitp sparc::f-branch-computed flags))) ; truly an absolute jmp
	       (let* ((candidate (lap-instruction-pred delayed-branch)))
		 (cond ((not (immovable candidate))
			(setq current (replace-delay-slot current candidate ))))))
	      ((logbitp sparc::f-pcrel flags)
	       (let* ((target-label (branch-target-label delayed-branch))
		      (target-instruction (only-path-to-label target-label)))
		 (when target-instruction
		   (setq current (replace-delay-slot current target-instruction))
		   (when (logbitp sparc::f-condbr flags)
		     (sparc-annul-branch-instruction delayed-branch)))))))))))))

(defun sparc-annul-branch-instruction (b)
  (let* ((old-opcode (lap-instruction-opcode b))
	 (parsed-operands (lap-instruction-parsed-operands b))
	 (nold (arch::opcode-min-args old-opcode)))
    (unless (eql nold 2)
      (setf (svref parsed-operands 1)
	    (svref parsed-operands 0)
	    (svref parsed-operands 0)
	    (ldb (byte 4 9) (arch::opcode-op-high old-opcode))))
    (setf (lap-instruction-opcode b) sparc::*sparc-bicc.a-opcode*)))

    
                        
(defun sparc-lap-do-labels ()
  (dolist (lab *lap-labels*)
    (if (and (lap-label-refs lab) (not (lap-label-emitted-p lab)))
        (error "Label ~S was referenced but never defined. " 
               (lap-label-name lab)))
    ;; Repeatedly iterate through label's refs, until none of them is the preceding
    ;; instruction.  This eliminates
    ;; (b @next)
    ;;    (nop)
    ;; @next
    ;;
    ;; but can probably be fooled by hairier nonsense.

    (loop
        (when (dolist (ref (lap-label-refs lab) t)
                (let* ((next (lap-instruction-succ ref)))
                  (when (and (typep next 'lap-instruction)
                             (eq sparc::*sparc-nop-opcode* (lap-instruction-opcode next))
                             (eq lab (lap-instruction-succ next)))
                    
                    (setf (lap-label-refs lab) (delete ref (lap-label-refs lab)))
                    (remove-dll-node ref 2) ; remove the branch & the nop
                    (return))))
          (return))))
  (sparc-lap-optimize-branch-delay-slots)
  ;; Assign pc to emitted labels, splice them out of the list.
  (sparc-lap-assign-addresses t))

; Replace each label with the difference between the label's address
; and the referencing instruction's address.
(defun sparc-lap-resolve-labels ()
  (dolist (label *lap-labels*)
    (let* ((label-address (lap-label-address label)))
      (declare (fixnum label-address))          ; had BETTER be ...
      (dolist (insn (lap-label-refs label))
        (let* ((diff (- label-address (lap-instruction-address insn))))
          (declare (fixnum diff))
          (let* ((opvals (lap-instruction-parsed-operands insn))
                 (pos (position label opvals)))
            (unless pos
              (error "Bug: label ~s should be referenced by instruction ~s, but isn't."))
            (setf (svref opvals pos) diff)))))))

(defun sparc-lap-generate-instruction (code-vector index insn &optional vinsn)
  (let* ((op (lap-instruction-opcode insn))
         (vals (lap-instruction-parsed-operands insn))
         (high (arch::opcode-op-high op))
         (low (arch::opcode-op-low op))
         (idx -1))
    (dolist (operand (if vinsn (arch::opcode-vinsn-operands op) (arch::opcode-operands op)))
      (let* ((val (svref vals (incf idx)))
             (insert-function (arch::operand-insert-function operand)))
        (multiple-value-setq (high low)
          (if insert-function
            (funcall insert-function high low val)
            (sparc::insert-default operand high low val)))
        (if (null high)
          (error "Invalid operand for ~s instruction: ~d" (arch::opcode-name op) val))))
    (setf (lap-instruction-parsed-operands insn) nil)
    (free-lap-operand-vector vals)
    (locally (declare (type (simple-array (unsigned-byte 16) (*)) code-vector)
                      (optimize (speed 3) (safety 0)))
      (setf (aref code-vector (+ index index)) high
            (aref code-vector (+ index index 1)) low)
     nil)))


  
(defun sparc-lap-generate-code (name maxpc bits)
  (declare (fixnum maxpc))
  (let* ((code-vector (%alloc-misc (ash maxpc -2)
                                   #+sparc-target arch::subtag-code-vector
                                   #-sparc-target arch::subtag-xcode-vector))
         (constants-size (+ 3 (length *sparc-lap-constants*)))
         (constants-vector (%alloc-misc constants-size
                                        #+sparc-target arch::subtag-function
                                        #-sparc-target arch::subtag-xfunction))
         (i 0))
    (declare (fixnum i constants-size))
    (sparc-lap-resolve-labels)            ; all operands fully evaluated now.
    (do-dll-nodes (insn *lap-instructions*)
      (sparc-lap-generate-instruction code-vector i insn)
      (incf i))
    (dolist (immpair *sparc-lap-constants*)
      (let* ((imm (car immpair))
             (k (cdr immpair)))
        (declare (fixnum k))
        (setf (uvref constants-vector (ash (- k sparc::misc-data-offset) -2))
              imm)))
    (setf (uvref constants-vector (1- constants-size)) bits       ; lfun-bits
          (uvref constants-vector (- constants-size 2)) name
          (uvref constants-vector 0) code-vector)
    #+sparc-target (%make-code-executable code-vector)
    constants-vector))

(defun sparc-lap-pseudo-op (form)
  (case (car form)
    (:regsave
     )))


       
(defun sparc-lap-form (form)
  (if (and form (symbolp form))
    (emit-lap-label form)
    (if (or (atom form) (not (symbolp (car form))))
      (error "~& unknown SPARC-LAP form: ~S ." form)
      (multiple-value-bind (expansion expanded)
                           (sparc-lap-macroexpand-1 form)
        (if expanded
          (sparc-lap-form expansion)
          (let* ((name (car form)))
            (if (keywordp name)
              (sparc-lap-pseudo-op form)
              (case name
                ((progn) (dolist (f (cdr form)) (sparc-lap-form f)))
                ((let) (sparc-lap-equate-form (cadr form) (cddr form)))
                (t
		 (sparc-lap-instruction name (cdr form)))))))))))

;;; (let ((name val) ...) &body body)
;;; each "val" gets a chance to be treated as a SPARC register name
;;; before being evaluated.
(defun sparc-lap-equate-form (eqlist body)
  (let* ((*sparc-lap-register-aliases* *sparc-lap-register-aliases*)
	 (symbols (mapcar #'(lambda (x)
                              (let* ((name (car x)))
                                (or
                                 (and name 
                                      (symbolp name)
                                      (not (constant-symbol-p name))
                                      name)
                                 (error 
                                  "~S is not a bindable symbol name ." name))))
                          eqlist))
	 (valforms (mapcar #'cadr eqlist))
	 (pairs ())
	 (bindsyms ())
	 (bindvals ()))
    (dolist (val valforms)
      (let* ((s (pop symbols))
	     (reg (sparc-register-name val 0)))
	(if reg
	  (push (cons s reg) pairs)
	  (progn
	    (push s bindsyms)
	    (push val bindvals)))))
    (dolist (pair (nreverse pairs))
      (push pair *sparc-lap-register-aliases*))
    (progv (nreverse bindsyms) (nreverse bindvals)
      (dolist (form body)
	(sparc-lap-form form)))))

(defun sparc-lap-constant-offset (x)
  (or (cdr (assoc x *sparc-lap-constants* :test #'equal))
      (let* ((n (+ sparc::misc-data-offset (ash (1+ (length *sparc-lap-constants*)) 2))))
        (push (cons x n) *sparc-lap-constants*)
        n)))

; Evaluate an arbitrary expression; warn if the result isn't a fixnum.
(defun sparc-lap-evaluated-expression (x)
  (if (and (consp x) (eq (car x) 'quote))
    (let* ((quoted-form (cadr x)))
      (if (typep quoted-form 'fixnum)
	(ash quoted-form arch::fixnumshift)
	(sparc-lap-constant-offset quoted-form)))
    (if (typep x 'fixnum)
	x
	(let* ((val (handler-case (eval x)          ; Look! Expression evaluation!
				  (error (condition) (error "~&Evaluation of ~S signalled assembly-time error ~& ~A ."
							    x condition)))))
	  (unless (typep val 'fixnum)
	    (warn "assembly-time evaluation of ~S returned ~S, which may not have been intended ."
		  x val))
	  val))))



(defparameter *sparc-lap-fp-register-aliases*
  ())

(defun sparc-register-name (x flags)
  (let* ((r (and (or (symbolp x) (stringp x))
		 (or
		  (position (string x) sparc::*gpr-register-names* :test #'string-equal)
		  (cdr (assoc x *sparc-lap-register-aliases* :test #'string-equal))))))
    (and r
	 (or (not (logbitp sparc::sparc-operand-double flags))
	     (evenp r))
	 (or (not (logbitp sparc::sparc-operand-quad flags))
	     (= 0 (mod r 4)))
	 r)))

(defun sparc-fp-register-name  (x flags)
  (let* ((r (and (or (symbolp x) (stringp x))
		 (or
		  (position (string x) sparc::*fpr-register-names* :test #'string-equal)
		  (cdr (assoc x *sparc-lap-fp-register-aliases* :test #'string-equal))))))
    (and r
	 (or (not (logbitp sparc::sparc-operand-double flags))
	     (evenp r))
	 (or (not (logbitp sparc::sparc-operand-quad flags))
	     (= 0 (mod r 4)))
	 r)))

(defun sparc-op2-expression (op2 flags width)
  (or (sparc-register-name op2 flags)
      (let* ((imm (sparc-lap-evaluated-expression op2)))
	(if (typep imm
		   (if (= width 13)
		     '(signed-byte 13)
		     (if (logbitp sparc::sparc-operand-signed flags)
			 `(signed-byte ,(1- width))
			 `(unsigned-byte ,width))))
	    (logior sparc::immed
		    (logand imm (1- (ash 1 13))))))))

;;; This accepts:
;;; (rs1)
;;; (rs1 rs2)
;;; (rs1 simm13)
;;; and nothing but.  Gas and as are more permissive.
(defun sparc-address-expression (x flags)
  (if (consp x)
    (destructuring-bind (rs1 &optional op2) x
      (let* ((rs1val (sparc-register-name rs1 flags))
	     (op2val (if op2
			 (sparc-op2-expression
			  op2
			  (logior
			   (ash 1 sparc::sparc-operand-signed)
			   flags) 13)
			 0)))
	(and rs1val op2val (logior (ash rs1val 14) op2val))))))

(defun sparc-regaddr-expression (x flags)
  (if (consp x)
    (destructuring-bind (rs1 &optional rs2) x
      (let* ((rs1val (sparc-register-name rs1 flags))
	     (rs2val (if rs2 (sparc-register-name rs2 flags) 0)))
	(and rs1val rs2val (logior (ash rs1val 14) rs2val))))))

(defun sparc-lap-instruction (name opvals)
  (let* ((opnum (gethash (string name) sparc::*sparc-opcode-numbers*))
         (opcode (and opnum 
                          (< -1 opnum (length sparc::*sparc-opcodes*))
                          (svref sparc::*sparc-opcodes* opnum))))
    (unless opcode
          (error "Unknown SPARC opcode: ~a in ~s" name `(,name ,@opvals)))
    ;; Unless either
    ;;  a) The number of operand values in the macro call exactly
    ;;      matches the number of operands accepted by the instruction or
    ;;  b) The number of operand values is one less, and the instuction
    ;;     takes an optional operand
    ;;  we've got a wrong-number-of-args error.
    ;;  In case (b), there's at most one optional argument per instruction;
    ;;   provide 0 for the missing value.
    (let* ((operands (arch::opcode-operands opcode))
           (nwant (length operands))
           (nhave (length opvals)))
      (declare (fixnum nwant nhave))
      (if (= nhave nwant)
        (sparc-emit-lap-instruction opcode opvals)
        (if (> nhave nwant)
          (error "Too many operands in ~s (~a accepts at most ~d)"
                 opvals name nwant)
	  (error "Too few operands in ~s : (~a requires at least ~d)"
                   opvals name nwant))))))

; This is pretty rudimentary: if the operand has the "sparc::$sparc-operand-relative" bit
; set, we demand a label name and note the fact that we reference the label in question.
; Otherwise, we use the "register-name-or-expression" thing.
; Like most SPARC assemblers, this lets you treat everything as an expression, even if
; you've got the order of some arguments wrong ...

(defun sparc-parse-lap-operand (opvalx operand insn &optional parsed)
  (let* ((flags (arch::operand-flags operand)))
    
    (declare (fixnum flags))
    (if (logbitp sparc::sparc-operand-relative flags)
      (lap-note-label-reference opvalx insn)        
      (if parsed
          opvalx
          (cond ((logbitp sparc::sparc-operand-fpr flags)
                 (sparc-fp-register-name opvalx flags))
                ((logbitp sparc::sparc-operand-address flags)
                 (sparc-address-expression opvalx flags))
                ((logbitp sparc::sparc-operand-op2 flags)
                 (sparc-op2-expression opvalx flags 13))
                ((logbitp sparc::sparc-operand-regaddr flags)
                 (sparc-regaddr-expression opvalx flags))
                ((logbitp sparc::sparc-operand-gpr flags)
                 (sparc-register-name opvalx flags))
                (t
                 (sparc-lap-evaluated-expression opvalx)))))))




; We've checked that the number of operand values match the number expected
; Labels - and some constructs that might someday do arithmetic on them -
; are about the only class of forward references we need to deal with.  This whole
; two-pass scheme seems overly general, but if/when we ever do instruction scheduling
; it'll probably make it simpler.
(defun sparc-emit-lap-instruction (opcode opvals &optional parsed)
  (let* ((operands (if parsed (arch::opcode-vinsn-operands opcode) (arch::opcode-operands opcode)))
         (parsed-values (alloc-lap-operand-vector))
         (insn (make-lap-instruction opcode))
	 (bad ())
	 (orig opvals)
         (idx -1))
    (declare (fixnum idx))
    (dolist (op operands)
      (let* ((opval (pop opvals))
	     (val  (sparc-parse-lap-operand opval op insn parsed)))
        (declare (fixnum flags))
	(if (null val)
	  (push opval bad))	   
        (setf (svref parsed-values (incf idx)) val)))
    (when bad
      (warn "Invalid operand~p ~s in ~& ~a"
	    (length bad) bad (cons (arch::opcode-name opcode) orig)))
    (setf (lap-instruction-parsed-operands insn) parsed-values)
    (append-dll-node insn *lap-instructions*)))

(defmacro defsparclapfunction (&environment env name arglist &body body)
  (let* ((targeting-sparc (eq *target-backend* *sparc-backend*))
	 (cross-compiling (not (eq *host-backend* *sparc-backend*))))
    `(progn
      (eval-when (:compile-toplevel)
	(note-function-info ',name t ,env))
      ,@(if cross-compiling
	    `((progn
		(eval-when (:load-toplevel)
		  (%defun (nfunction ,name (lambda (&lap 0) (sparc-lap-function ,name ,arglist ,@body)))))    
		(eval-when (:execute)
		  (%define-sparc-lap-function ',name '((let ,arglist ,@body)))))))
      ,@(if (and targeting-sparc (not cross-compiling))	; just shorthand for defun
	    `((%defun (nfunction ,name (lambda (&lap 0) (sparc-lap-function ,name ,arglist ,@body)))))))))
 


(provide "SPARC-LAP")
