/*
 * vidreps.cc --
 *
 *      Video frame memory representation method definitions.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "vidreps.h"

/*
 * PUT_BUF_NUM used to construct Dali buffername strings from buffer ids.
 */
#define PUT_BUF_NUM(str,prefix,buf) sprintf(str, "%s%d", prefix, buf)

/*
 * Constructor for VidRep class.
 */
VidRep::VidRep() :
  TclObject(),
  w_(0),
  h_(0),
  h_subsample_(1),
  v_subsample_(1),
  resized_(0),
  true_w_(0),
  true_h_(0),
  xoff_(0),
  yoff_(0),
  ssrc_(0),
  ts_(0),
  source_id_(0)
{
  bind("w_", (int*)&w_);
  bind("h_", (int*)&h_);
  bind("h_subsample_", (int*)&h_subsample_);
  bind("v_subsample_", (int*)&v_subsample_);
  bind("true_w_", (int*)&true_w_);
  bind("true_h_", (int*)&true_h_);
  bind("xoff_", (int*)&xoff_);
  bind("yoff_", (int*)&yoff_);
  bind("ssrc_", (unsigned int*)&ssrc_);
  bind("ts_", (unsigned int*)&ts_);
};

/*
 * Destructor for VidRep class. Currently empty.
 */
VidRep::~VidRep() {
};

/*
 * Copies geometry for representation passed in as template.
 * Does no allocation or deallocation, simply changes geometry variables.
 */
void
VidRep::copy_geometry(VidRep *temp)
{
  w_ = temp->w_;
  h_ = temp->h_;
  h_subsample_ = temp->h_subsample_;
  v_subsample_ = temp->v_subsample_;
  true_w_ = temp->true_w_;
  true_h_ = temp->true_h_;
  xoff_ = temp->xoff_;
  yoff_ = temp->yoff_;
}

/*
 * Tcl command interface for VidRep.
 * Commands supported:
 *     copy_geometry <VidRepObject>
 */
int
VidRep::command(int argc, const char*const* argv)
{
  if (argc == 3) {
    if (strcmp(argv[1], "copy_geometry") == 0) {
      VidRep *temp;

      temp = (VidRep *) TclObject::lookup(argv[2]);
      if (temp != 0) {
	copy_geometry(temp);
      }
      return TCL_OK;
    }
    if (strcmp(argv[1], "set_source_id") == 0) {
      if (source_id_)
	free(source_id_);
      source_id_ = (char *)malloc(strlen(argv[2]) + 1);
      strcpy(source_id_, argv[2]);
      return TCL_OK;
    }
  }
  return (TclObject::command(argc, argv));
}

/*
 * OTcl class binding for Uncompressed.
 * OTcl class is "VidRep/Uncompressed"
 */
static class UncompressedClass : public TclClass {
public:
  UncompressedClass() : TclClass("VidRep/Uncompressed") {}
  TclObject *create(int argc, const char*const* argv) {
    return (new Uncompressed());
  }
} uncompressed_class_;

/*
 * Constructor for Uncompressed
 */
Uncompressed::Uncompressed()
  : VidRep(),
    self_allocated_(0),
    lum_(0),
    cr_(0),
    cb_(0),
    lum_id_(-1),
    cr_id_(-1),
    cb_id_(-1),
    bp_(0)
{
};

/*
 * Destructor for Uncompressed.
 */
Uncompressed::~Uncompressed() {
  dealloc();
};

/*
 * Removes Dali buffers and deallocates Dali structures.
 * Also deallocates buffer memory if originally self-allocated.
 */
void Uncompressed::dealloc()
{
  if (lum_ != 0) {
    RemoveBufByNum(lum_id_);
    ckfree((char *)lum_);
  }
  if (cr_ != 0) {
    RemoveBufByNum(cr_id_);
    ckfree((char *)cr_);
  }
  if (cb_ != 0) {
    RemoveBufByNum(cb_id_);
    ckfree((char *)cb_);
  }
  lum_ = 0;
  cr_ = 0;
  cb_ = 0;
  lum_id_ = cr_id_ = cb_id_ = -1;

  if (self_allocated_) {
    if (bp_ != 0) {
      ckfree((char *)bp_);
    }
    bp_ = 0;
  }
  if (source_id_)
    free((void *)source_id_);
}

/*
 * Given specific geometry and pointer to buffer memory, creates
 * Dali structures and registers Dali buffers. Deallocates any
 * existing structures first.
 */
void Uncompressed::init(int w, int h, int h_subsample, int v_subsample,
			int true_w, int true_h, int xoff, int yoff,
			u_char *ldata, u_char *crdata,
			u_char *cbdata)
{
  Tcl& tcl = Tcl::instance();

  w_ = w;
  h_ = h;
  h_subsample_ = h_subsample;
  v_subsample_ = v_subsample;
  true_w_ = true_w;
  true_h_ = true_h;
  xoff_ = xoff;
  yoff_ = yoff;
  self_allocated_ = 0;

  lum_ = (ByteImage *) ckalloc(sizeof(ByteImage));
  cr_ = (ByteImage *) ckalloc(sizeof(ByteImage));
  cb_ = (ByteImage *) ckalloc(sizeof(ByteImage));


  lum_->parentWidth = lum_->width = w_;
  lum_->height = h_;
  lum_->x = 0;
  lum_->y = 0;
  lum_->isVirtual = 0;
  lum_->firstByte = ldata;
  lum_id_ = PutByteImage(tcl.interp(), lum_);

  cr_->parentWidth = cr_->width = w_ / h_subsample_;
  cr_->height = h_ / v_subsample_;
  cr_->x = 0;
  cr_->y = 0;
  cr_->isVirtual = 0;
  cr_->firstByte = crdata;
  cr_id_ = PutByteImage(tcl.interp(), cr_);

  cb_->parentWidth = cb_->width = w_ / h_subsample_;
  cb_->height = h_ / v_subsample_;
  cb_->x = 0;
  cb_->y = 0;
  cb_->isVirtual = 0;
  cb_->firstByte = cbdata;
  cb_id_ = PutByteImage(tcl.interp(), cb_);
}

/*
 * Tcl command interface to Uncompressed class.
 * Commands supported:
 *    get_lum_name - returns Dali buffer name of lum. plane
 *    get_cr_name -    "       "    "      "   "  cr    "
 *    get_cb_name -    "       "    "      "   "  cb    "
 *    allocate - creates new structures based on current geometry and
 *               allocates buffer memory.
 */
int
Uncompressed::command(int argc, const char*const* argv)
{
  Tcl& tcl=Tcl::instance();
  char *wrk = tcl.buffer();

  if (argc == 2) {
    if (strcmp(argv[1], "get_lum_name") == 0) {
      if (lum_ == 0) {
	*wrk = 0;
      } else {
	PUT_BUF_NUM(wrk, BYTE_PREFIX, lum_id_);
      }
      tcl.result(wrk);
      return (TCL_OK);
    }
    if (strcmp(argv[1], "get_cr_name") == 0) {
      if (cr_ == 0) {
	*wrk = 0;
      } else {
	PUT_BUF_NUM(wrk, BYTE_PREFIX, cr_id_);
      }
      tcl.result(wrk);
      return (TCL_OK);
    }
    if (strcmp(argv[1], "get_cb_name") == 0) {
      if (cb_ == 0) {
	*wrk = 0;
      } else {
	PUT_BUF_NUM(wrk, BYTE_PREFIX, cb_id_);
      }
      tcl.result(wrk);
      return (TCL_OK);
    }
    if (strcmp(argv[1], "allocate") == 0) {
      u_char *ldata, *crdata, *cbdata;

      dealloc();
      bp_ = (u_char *) ckalloc(w_*h_+2*((w_/h_subsample_)*(h_/v_subsample_)));
      self_allocated_ = 1;

      ldata = bp_;
      crdata = bp_ + (w_*h_);
      cbdata = bp_ + (w_*h_) + ((w_/h_subsample_)*(h_/v_subsample_));

      init(w_, h_, h_subsample_, v_subsample_, true_w_, true_h_, xoff_, yoff_,
	   ldata, crdata, cbdata);
      return (TCL_OK);
    }
  }
  return (VidRep::command(argc, argv));
}

/*
 * OTcl class binding for JPEGCompressed.
 * OTcl class is "VidRep/JPEGCompressed"
 */
static class JPEGCompressedClass : public TclClass {
public:
  JPEGCompressedClass() : TclClass("VidRep/JPEGCompressed") {}
  TclObject *create(int argc, const char*const* argv) {
    return (new JPEGCompressed());
  }
} jpeg_compressed_class_;

/*
 * Constructor for JPEGCompressed
 * Changes h_subsample_ and v_subsample_ to reflect default 422 geometry.
 */
JPEGCompressed::JPEGCompressed()
  : VidRep(),
    self_allocated_(0),
    q_(0),
    len_(0),
    bp_(0)
{
  h_subsample_ = 2;
  v_subsample_ = 1;
  bind("q_", (int*)&q_);
  bind("len_", (int*)&len_);
}

/*
 * Destructor for JPEGCompressed.
 */
JPEGCompressed::~JPEGCompressed() {
  dealloc();
}

/*
 * Deallocates buffer memory if originally self-allocated.
 */
void JPEGCompressed::dealloc() {
  if (self_allocated_) {
    if (bp_ != 0) {
      free(bp_);
      bp_ = 0;
      self_allocated_ = 0;
    }
  }
}

/*
 * Given geometry info and pointer to data, initializes structure
 * appropriately.
 */
void JPEGCompressed::set(int w, int h, int decimation, int q, int len,
			   u_int32_t ssrc, u_int32_t ts, u_char *bp)
{
  dealloc();
  true_w_ = w_ = w;
  true_h_ = h_ = h;
  q_ = q;
  len_ = len;

  if (decimation == 422) {
    h_subsample_ = 2;
    v_subsample_ = 1;
  } else {
    h_subsample_ = 2;
    v_subsample_ = 2;
  }
  xoff_ = yoff_ = 0;

  ssrc_ = ssrc;
  ts_ = ts;

  bp_ = bp;
  self_allocated_ = 0;
}

/*
 * Tcl command interface to JPEGCompressed object.
 * The following commands are supported:
 *       copy <JPEGCompressed> - copies geometry and buffer of
 *                               passed in source object.
 */
int
JPEGCompressed::command(int argc, const char*const* argv)
{
  Tcl& tcl=Tcl::instance();

  if (argc == 3) {
    if (strcmp(argv[1], "copy") == 0) {
      JPEGCompressed *src = (JPEGCompressed *) TclObject::lookup(argv[2]);
      if (src != 0) {
	dealloc();
	copy_geometry((VidRep *)src);
	q_ = src->q_;
	len_ = src->len_;
	ssrc_ = src->ssrc_;
	ts_ = src->ts_;
	self_allocated_ = 1;
	bp_ = (u_char *) malloc(len_);
	memcpy(bp_, src->bp_, len_);
      }
      return TCL_OK;
    }
  }
  return (VidRep::command(argc, argv));
}

/*
 * OTcl class binding for Semicompressed.
 * OTcl class is "VidRep/Semicompressed"
 */
static class SemicompressedClass : public TclClass {
public:
  SemicompressedClass() : TclClass("VidRep/Semicompressed") {}
  TclObject *create(int argc, const char*const* argv) {
    return (new Semicompressed());
  }
} semi_compressed_class_;

/*
 * Constructor for Semicompressed.
 */
Semicompressed::Semicompressed()
  : VidRep(),
    self_allocated_(0),
    lum_(0),
    cr_(0),
    cb_(0),
    lum_id_(-1),
    cr_id_(-1),
    cb_id_(-1)
{
};

/*
 * Destructor for Semicompressed.
 */
Semicompressed::~Semicompressed()
{
  dealloc();
}

/*
 * Deallocates Dali structures and buffer memory if self-allocated.
 */
void Semicompressed::dealloc()
{
  if (lum_ != 0) {
    RemoveBufByNum(lum_id_);
    if (self_allocated_) {
      ScFree(lum_);
    }
  }
  if (cr_ != 0) {
    RemoveBufByNum(cr_id_);
    if (self_allocated_) {
      ScFree(cr_);
    }
  }
  if (cb_ != 0) {
    RemoveBufByNum(cb_id_);
    if (self_allocated_) {
      ScFree(cb_);
    }
  }
  lum_ = 0;
  cr_ = 0;
  cb_ = 0;
  lum_id_ = cr_id_ = cb_id_ = -1;
}

/*
 * Tcl interface to Semicompressed objects.
 * Current commands supported:
 *      print
 *      print <x> <y>
 *      get_lum_name
 *      get_cr_name
 *      get_cb_name
 *      allocate
 */
int
Semicompressed::command(int argc, const char*const* argv)
{
  Tcl& tcl=Tcl::instance();
  char *wrk = tcl.buffer();

  if (argc == 2) {
    if (strcmp(argv[1], "print") == 0) {
      print();
      return TCL_OK;
    }
    if (strcmp(argv[1], "get_lum_name") == 0) {
      if (lum_ == 0) {
	*wrk = 0;
      } else {
	PUT_BUF_NUM(wrk, SC_PREFIX, lum_id_);
	tcl.result(wrk);
      }
      return (TCL_OK);
    }
    if (strcmp(argv[1], "get_cr_name") == 0) {
      if (cr_ == 0) {
	*wrk = 0;
      } else {
	PUT_BUF_NUM(wrk, SC_PREFIX, cr_id_);
	tcl.result(wrk);
      }
      return (TCL_OK);
    }
    if (strcmp(argv[1], "get_cb_name") == 0) {
      if (cb_ == 0) {
	*wrk = 0;
      } else {
	PUT_BUF_NUM(wrk, SC_PREFIX, cb_id_);
	tcl.result(wrk);
      }
      return (TCL_OK);
    }
    if (strcmp(argv[1], "allocate") == 0) {
      allocate();
      return (TCL_OK);
    }
  }
  if (argc == 4) {
    if (strcmp(argv[1], "print") == 0) {
      int x, y;

      x = atoi(argv[2]);
      y = atoi(argv[3]);

      print(x,y);
      return TCL_OK;
    }
  }
  return (VidRep::command(argc, argv));
}

/*
 * Allocates Dali structures for current geometry. Deallocates first.
 */
void
Semicompressed::allocate()
{
  int i;
  Tcl& tcl=Tcl::instance();

  dealloc();
  lum_ = ScNew(w_/8,h_/8);
  cr_ = ScNew(w_/(8*h_subsample_),h_/(8*v_subsample_));
  cb_ = ScNew(w_/(8*h_subsample_),h_/(8*v_subsample_));

  for(i=0; i<(w_/8)*(h_/8); i++) {
    ScBlock *scb = &(lum_->firstBlock[i]);
    scb->intracoded = 1;
    scb->skipMB = 0;
    scb->skipBlock = 0;
    scb->dc = 0;
    scb->numOfAC = 0;
  }

  for(i=0; i<(w_/(8*h_subsample_))*(h_/(8*v_subsample_)); i++) {
    ScBlock *scb = &(cr_->firstBlock[i]);
    scb->intracoded = 1;
    scb->skipMB = 0;
    scb->skipBlock = 0;
    scb->dc = 0;
    scb->numOfAC = 0;

    scb = &(cb_->firstBlock[i]);
    scb->intracoded = 1;
    scb->skipMB = 0;
    scb->skipBlock = 0;
    scb->dc = 0;
    scb->numOfAC = 0;
  }

  lum_id_ = PutScImage(tcl.interp(), lum_);
  cr_id_ = PutScImage(tcl.interp(), cr_);
  cb_id_ = PutScImage(tcl.interp(), cb_);
  self_allocated_ = 1;
}

/*
 * Given geometry, sets and allocates appropriately.
 */
void
Semicompressed::set(int w, int h, int h_subsample, int v_subsample,
		    int tw, int th, int xoff, int yoff)
{
  w_ = w;
  h_ = h;
  h_subsample_ = h_subsample;
  v_subsample_ = v_subsample;
  true_w_ = tw;
  true_h_ = th;
  xoff_ = xoff;
  yoff_ = yoff;
  allocate();
}

/*
 * Print values for all block x,y in all planes.
 * Mostly a debugging function.
 */
void
Semicompressed::print(int x, int y)
{
  if (lum_ != 0) {
    fprintf(stderr, "Luminance Plane\n");
    fprintf(stderr, "%d blocks wide by %d blocks high\n", lum_->width,
	    lum_->height);
    if ((x < lum_->width) && (y < lum_->height)) {
      ScBlock *scb = &(lum_->firstBlock[y*lum_->width+x]);
      fprintf(stderr, "\t%d,%d block\n", x, y);
      fprintf(stderr, "\tDC value: %d\n", scb->dc);
      fprintf(stderr, "\tAC Index\t AC Value\n");
      for(int idx=0; idx<scb->numOfAC; idx++) {
	fprintf(stderr, "\t%d\t%d\n", scb->index[idx], scb->value[idx]);
      }
    } else {
      fprintf(stderr, "%d,%d out of range\n", x, y);
    }
  }

  x = x / h_subsample_;
  y = y / v_subsample_;

  if (cr_ != 0) {
    fprintf(stderr, "Cr Plane\n");
    fprintf(stderr, "%d blocks wide by %d blocks high\n", cr_->width,
	    cr_->height);

    if ((x < cr_->width) && (y < cr_->height)) {
      ScBlock *scb = &(cr_->firstBlock[y*cr_->width+x]);
      fprintf(stderr, "\t%d,%d block\n", x, y);
      fprintf(stderr, "\tDC value: %d\n", scb->dc);
      fprintf(stderr, "\tAC Index\t AC Value\n");
      for(int idx=0; idx<scb->numOfAC; idx++) {
	fprintf(stderr, "\t%d\t%d\n", scb->index[idx], scb->value[idx]);
      }
    } else {
      fprintf(stderr, "%d,%d out of range\n", x, y);
    }
  }

  if (cb_ != 0) {
    fprintf(stderr, "Cb Plane\n");
    fprintf(stderr, "%d blocks wide by %d blocks high\n", cb_->width,
	    cb_->height);
    if ((x < cb_->width) && (y < cb_->height)) {
      ScBlock *scb = &(cb_->firstBlock[y*cb_->width+x]);
      fprintf(stderr, "\t%d,%d block\n", x, y);
      fprintf(stderr, "\tDC value: %d\n", scb->dc);
      fprintf(stderr, "\tAC Index\t AC Value\n");
      for(int idx=0; idx<scb->numOfAC; idx++) {
	fprintf(stderr, "\t%d\t%d\n", scb->index[idx], scb->value[idx]);
      }
    } else {
      fprintf(stderr, "%d,%d out of range\n", x, y);
    }
  }
}

/*
 * Print all blocks of all planes.
 */
void
Semicompressed::print()
{
  if (lum_ != 0) {
    fprintf(stderr, "Luminance Plane\n");
    fprintf(stderr, "%d blocks wide by %d blocks high\n", lum_->width,
	    lum_->height);
    for (int row=0; row < lum_->height; row++) {
      for (int col=0; col < lum_->width; col++) {
	ScBlock *scb = &(lum_->firstBlock[row*lum_->width+col]);
	fprintf(stderr, "\t%d,%d block\n", col, row);
	fprintf(stderr, "\tDC value: %d\n", scb->dc);
	fprintf(stderr, "\tAC Index\t AC Value\n");
	for(int idx=0; idx<scb->numOfAC; idx++) {
	  fprintf(stderr, "\t%d\t%d\n", scb->index[idx], scb->value[idx]);
	}
      }
    }
  }

  if (cr_ != 0) {
    fprintf(stderr, "Cr Plane\n");
    fprintf(stderr, "%d blocks wide by %d blocks high\n", cr_->width,
	    cr_->height);
    for (int row=0; row < cr_->height; row++) {
      for (int col=0; col < cr_->width; col++) {
	ScBlock *scb = &(cr_->firstBlock[row*cr_->width+col]);
	fprintf(stderr, "\t%d,%d block\n", col, row);
	fprintf(stderr, "\tDC value: %d\n", scb->dc);
	fprintf(stderr, "\tAC Index\t AC Value\n");
	for(int idx=0; idx<scb->numOfAC; idx++) {
	  fprintf(stderr, "\t%d\t%d\n", scb->index[idx], scb->value[idx]);
	}
      }
    }
  }

  if (cb_ != 0) {
    fprintf(stderr, "Cb Plane\n");
    fprintf(stderr, "%d blocks wide by %d blocks high\n", cb_->width,
	    cb_->height);
    for (int row=0; row < cb_->height; row++) {
      for (int col=0; col < cb_->width; col++) {
	ScBlock *scb = &(cb_->firstBlock[row*cb_->width+col]);
	fprintf(stderr, "\t%d,%d block\n", col, row);
	fprintf(stderr, "\tDC value: %d\n", scb->dc);
	fprintf(stderr, "\tAC Index\t AC Value\n");
	for(int idx=0; idx<scb->numOfAC; idx++) {
	  fprintf(stderr, "\t%d\t%d\n", scb->index[idx], scb->value[idx]);
	}
      }
    }
  }
}
