(*
 * Command utilities.
 *
 * ----------------------------------------------------------------
 *
 * @begin[license]
 * Copyright (C) 2003 Jason Hickey, Caltech
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Jason Hickey
 * @email{jyh@cs.caltech.edu}
 * @end[license]
 *)
open Lm_printf

open Fmarshal

open Omake_node
open Omake_marshal
open Omake_ir_print
open Omake_command_type

(************************************************************************
 * Command utilities
 *)

(*
 * Create a command line from a command string.
 *   Flags:
 *      '@' : QuietFlag
 *      '-' : AllowFailureFlag
 *)
let command_flags line =
   let len = String.length line in
   let rec process flags i =
      if i = len then
         flags, ""
      else
         match line.[i] with
            '@' ->
               process (QuietFlag :: flags) (succ i)
          | '-' ->
               process (AllowFailureFlag :: flags) (succ i)
          | '+' ->
               process flags (succ i)
          | _ ->
               let line =
                  if i = 0 then
                     line
                  else
                     String.sub line i (len - i)
               in
                  flags, line
   in
      process [] 0

(*
 * Parse the command lines from the strings.
 *)
let rec parse_commands venv dir target loc lines =
   match lines with
      line :: lines ->
         (match line with
             CommandArgv [] ->
                parse_commands venv dir target loc lines
           | CommandArgv (exe :: args) ->
                let flags, exe = command_flags exe in
                   if exe = "" then
                      match args with
                         _ :: _ ->
                            let command =
                               { command_loc    = loc;
                                 command_dir    = dir;
                                 command_target = target;
                                 command_flags  = flags;
                                 command_venv   = venv;
                                 command_inst   = CommandArgv args
                               }
                            in
                               command :: parse_commands venv dir target loc lines
                       | [] ->
                            parse_commands venv dir target loc lines
                   else
                      let command =
                         { command_loc    = loc;
                           command_dir    = dir;
                           command_target = target;
                           command_flags  = flags;
                           command_venv   = venv;
                           command_inst   = CommandArgv (exe :: args)
                         }
                      in
                         command :: parse_commands venv dir target loc lines
           | CommandEval _
           | CommandValues _ ->
                let command =
                   { command_loc    = loc;
                     command_dir    = dir;
                     command_target = target;
                     command_flags  = [];
                     command_venv   = venv;
                     command_inst   = line
                   }
                in
                   command :: parse_commands venv dir target loc lines)
    | [] ->
         []

(*
 * Allow output in the command.
 *)
let command_allow_output command =
   { command with command_flags = AllowOutputFlag :: command.command_flags }

(*
 * Print a command.
 *)
let pp_print_argv buf argv =
   pp_print_string buf (Lm_string_util.concat_argv argv)

let pp_print_command_inst buf inst =
   match inst with
      CommandArgv argv ->
         pp_print_argv buf argv
    | CommandEval _ ->
         pp_print_string buf "<exp>"
    | CommandValues _ ->
         pp_print_string buf "<values>"

let pp_print_command_line buf line =
   pp_print_command_inst buf line.command_inst

let pp_print_command_lines buf lines =
   List.iter (fun line -> fprintf buf "@ %a" pp_print_command_line line) lines

(*!
 * @docoff
 *
 * -*-
 * Local Variables:
 * Caml-master: "compile"
 * End:
 * -*-
 *)
