/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef OSCSENDER_H
#define OSCSENDER_H

#include <string>
#include <iostream>
#include <lo/lo.h>
#include <pthread.h>
#include "types.h"
#include "Request.h"
#include "config.h"

using std::string;

namespace Om {

class Node;
class Port;
class PortInfo;
class Plugin;
class Patch;
class Connection;
class OSCReceiver;


/** Sends OSC messages (for two-way communication with client)
 *
 * \ingroup engine
 */
class OSCSender
{
public:
	OSCSender(const OSCReceiver* osc_receiver);

	// Control band:
	
	void send_response_ok(lo_address addr, int request_id);
	void send_response_error(lo_address addr, int request_id, const string& msg);
	
	// Notification band:
	
	//void send_client_registration(const string& url, int client_id);
	
	// Error that isn't the direct result of a request
	void send_error(const string& msg);

	void send_plugins(lo_address addr);
	
	void send_node_creation_messages(const Node* const node);
	void send_node_deletion_messages(const Node* const node);
	
	// These versions send to all clients, using single client versions below
	void send_node_removal(const string& path);
	void send_patch(const Patch* const p);
	void send_new_port(const Port* port);
	void send_port_removal(const Port* port);
	void send_node(const Node* const node);
	void send_patch_destruction(const string& patch_path);
	void send_connection(const Connection* const connection);
	void send_disconnection(const string& src_port_path, const string& dst_port_path);
	void send_rename(const string& old_path, const string& new_path);
	void send_all_objects();
	void send_engine_enable();
	void send_engine_disable();
	void send_patch_enable(const string& patch_path);
	void send_patch_disable(const string& patch_path);
	void send_metadata_update(const string& node_path, const string& key, const string& value);
	void send_control_change(const string& port_path, float value);

	
	// These versions send to passed address
	void send_node_removal_to(lo_address addr, const string& path);
	void send_patch_to(lo_address addr, const Patch* const p);
	void send_new_port_to(lo_address addr, const Port* port);
	void send_port_removal_to(lo_address addr, const Port* port);
	void send_node_to(lo_address addr, const Node* const node);
	void send_patch_destruction_to(lo_address addr, const string& patch_path);
	void send_connection_to(lo_address addr, const Connection* const connection);
	void send_disconnection_to(lo_address addr, const string& src_port_path, const string& dst_port_path);
	void send_rename_to(lo_address addr, const string& old_path, const string& new_path);
	void send_all_objects_to(lo_address addr);
	void send_engine_enable_to(lo_address addr);
	void send_engine_disable_to(lo_address addr);
	void send_patch_enable_to(lo_address addr, const string& patch_path);
	void send_patch_disable_to(lo_address addr, const string& patch_path);
	void send_metadata_update_to(lo_address addr, const string& node_path, const string& key, const string& value);
	void send_control_change_to(lo_address addr, const string& port_path, float value);

private:
	const OSCReceiver* m_osc_receiver;
};


} // namespace Om

#endif // OSCSENDER_H
