/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "NodeBase.h"
#include <cassert>
#include <iostream>
#include <stdint.h>
#include "util.h"
#include "Array.h"
#include "Plugin.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "OSCSender.h"
#include "Patch.h"
#include "Om.h"
using std::cout; using std::cerr; using std::endl;

namespace Om {


NodeBase::NodeBase(const string& name, uint poly, Patch* parent, samplerate srate, size_t buffer_size)
: Node(parent, name),
  m_poly(poly),
  m_srate(srate),
  m_buffer_size(buffer_size),
  m_activated(false),
  m_num_ports(0),
  m_traversed(false),
  m_providers(new List<Node*>()),
  m_dependants(new List<Node*>())
{
	assert(m_parent == NULL || (m_poly == parent->internal_poly() || m_poly == 1));
}


NodeBase::~NodeBase()
{
	assert(!m_activated);

	delete m_providers;
	delete m_dependants;
	
	for (uint i=0; i < m_ports.size(); ++i)
		delete m_ports.at(i);
}


void
NodeBase::activate()
{
	assert(!m_activated);
	m_activated = true;
}


void
NodeBase::deactivate()
{
	assert(m_activated);
	m_activated = false;
}


void
NodeBase::send_creation_messages(lo_address addr) const
{
	om->osc_sender()->send_node_to(addr, this);
}


void
NodeBase::send_deletion_messages(lo_address addr) const
{
	om->osc_sender()->send_node_removal_to(addr, path());
}


/** Runs the Node for the specified number of frames (block size)
 */
void
NodeBase::run(size_t nframes)
{
	assert(m_activated);
	// Mix down any ports with multiple inputs
	Port* p;
	for (uint i=0; i < m_ports.size(); ++i) {
		p = m_ports.at(i);
		p->prepare_buffers();
	}
}


Port* const
NodeBase::port(const string& port_name) const
{
	//assert(voice_num < m_poly);
	Port* p = NULL;
	
	for (uint i=0; i < num_ports(); ++i) {
		p = m_ports.at(i);
		if (p->name() == port_name)
			return p;
	}

	cerr << "[NodeBase::port] Failed to find port " << port_name << "!" << endl;

	return NULL;
}

} // namespace Om

