/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "MidiNoteNode.h"
#include <cmath>
#include <iostream>
#include "InputPort.h"
#include "Patch.h"
#include "OutputPort.h"
#include "Plugin.h"
#include "Array.h"
#include "util.h"

using std::cerr; using std::cout; using std::endl;


namespace Om {


MidiNoteNode::MidiNoteNode(const string& path, uint poly, Patch* parent, samplerate srate, size_t buffer_size)
: MidiInNode(path, poly, parent, srate, buffer_size),
  m_voice_notes(new Array<int>(poly, -1)),
  m_voice_ages(new Array<uint>(poly, (uint)0)),
  m_voice_states(new Array<VoiceState>(poly, VOICE_FREE))
{
	m_num_ports = 4;
	m_ports.alloc(m_num_ports);

	Port* p = NULL;
	p = new OutputPort(this, "Frequency", 0, poly,
		new PortInfo("Frequency", AUDIO, OUTPUT, 440, 0, 99999), m_buffer_size);
	m_ports.at(0) = p;
	
	p = new OutputPort(this, "Velocity", 1, poly,
		new PortInfo("Velocity", AUDIO, OUTPUT, 60, 0, 127), m_buffer_size);
	m_ports.at(1) = p;
	
	p = new OutputPort(this, "Gate", 2, poly,
		new PortInfo("Gate", AUDIO, OUTPUT, 0, 0, 1), m_buffer_size);
	m_ports.at(2) = p;
	
	p = new OutputPort(this, "Trigger", 2, poly,
		new PortInfo("Trigger", AUDIO, OUTPUT, 0, 0, 1), m_buffer_size);
	m_ports.at(3) = p;
	
	m_plugin.type(Plugin::Internal);
	m_plugin.lib_path("");
	m_plugin.lib_name("");
	m_plugin.plug_label("note_in");
	m_plugin.name("Om patch note input node (MIDI/OSC)");
}


MidiNoteNode::~MidiNoteNode()
{
	delete m_voice_notes;
	delete m_voice_states;
	delete m_voice_ages;
}


float
MidiNoteNode::note_to_freq(int num)
{
	const float A4 = 440.0f;
	if (num >= 0 && num <= 119)
		return A4 * powf(2.0f, (float)(num - 57.0f) / 12.0f);
			return 1.0f;  // Some LADSPA plugins don't like freq=0
}


void
MidiNoteNode::note_on(int note_num, int velocity, int start_sample)
{
	//cerr << "Note on starting at sample " << start_sample << endl;
	//cerr << "[MidiNoteNode] Note on." << endl;

	assert(start_sample >= 0 && start_sample < static_cast<int>(m_buffer_size));

	uint voice_to_trigger = 0;
	bool found = false;
	
	// Look for free voices
	for (uint i=0; i < m_poly; ++i) {
		if (m_voice_states->at(i) == VOICE_FREE) {
			found = true;
			voice_to_trigger = i;
			break;
		}
	}

	// If we didn't find a free one, steal the oldest
	if (! found) {
		uint oldest_age = 0;
		for (uint i=0; i < m_poly; ++i) {
			if (m_voice_ages->at(i) > oldest_age) {
				found = true;
				voice_to_trigger = i;
				oldest_age = m_voice_ages->at(i);
			}
		}
	}		
	assert(found);

	//cerr << "[MidiNoteNode] Note on.  Voice = " << voice_to_trigger << endl;
	
	// Trigger voice
	m_voice_notes->at(voice_to_trigger) = note_num;
	m_voice_states->at(voice_to_trigger) = VOICE_ACTIVE;
	m_voice_ages->at(voice_to_trigger) = 1;
	

	// one-sample jitter hack to avoid having to deal with trigger sample "next time"
	if (start_sample == (int)(m_buffer_size-1))
		--start_sample;
	
	m_ports.at(0)->set_value(voice_to_trigger, start_sample, note_to_freq(note_num));
	m_ports.at(1)->set_value(voice_to_trigger, start_sample, velocity/127.0);
	m_ports.at(2)->set_value(voice_to_trigger, start_sample, 1.0f);
	
	// trigger (one sample)
	//m_ports.at(3)->buffer(voice_to_trigger)[start_sample] = 1.0f;
	m_ports.at(3)->set_value(voice_to_trigger, start_sample, 1.0f);
	m_ports.at(3)->set_value(voice_to_trigger, start_sample+1, 0.0f);

	//cerr << "[MidiNoteNode] Note on, voice = " << voice_to_trigger << endl;
}


void
MidiNoteNode::note_off(int note_num, int start_sample)
{
	//cerr << "Note off, note = " << note_num << " - ";
	//cerr << "Note off starting at sample " << start_sample << endl;
	assert(start_sample >= 0 && start_sample < static_cast<int>(m_buffer_size));

	for (uint i=0; i < m_poly; ++i) {
		if (m_voice_notes->at(i) == note_num) {
			//cerr << "[MidiNoteNode] Note off, voice = " << i << endl;
			m_ports.at(2)->set_value(i, start_sample, 0.0f);
			m_voice_states->at(i) = VOICE_FREE;
			//break; // ommitted because of thorwil's crazy multi-note-on keyboard
		}
	}
}


void
MidiNoteNode::all_notes_off(int start_sample)
{
	//cerr << "Note off starting at sample " << start_sample << endl;
	assert(start_sample >= 0 && start_sample < static_cast<int>(m_buffer_size));

	for (uint i=0; i < m_poly; ++i) {
		m_ports.at(2)->set_value(i, start_sample, 0.0f);
		m_voice_states->at(i) = VOICE_FREE;
	}
}


void
MidiNoteNode::run(size_t nframes)
{
	NodeBase::run(nframes);

	// Update ages
	// FIXME: wrap around?
	for (uint i=0; i < m_poly; ++i) {
		if (m_voice_states->at(i) == VOICE_ACTIVE)
			++(m_voice_ages->at(i));
	}
}


} // namespace Om

