/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* These are the events that are pushed into a queue by the OSC thread and
 * read by the Gtk thread to do things in the gtk client in response to
 * engine events.
 */

#ifndef GTKCLIENTHOOKSEVENTS_H
#define GTKCLIENTHOOKSEVENTS_H

#include "GtkClientHooks.h"
#include <iostream>
#include <string>
#include <cassert>
#include "OmGtk.h"
#include "Controller.h"
#include "OmGtkApp.h"
#include "OmModule.h"
#include "OmPort.h"
#include "PatchController.h"
#include "PatchModel.h"
#include "NodeModel.h"
#include "ConnectionModel.h"
#include "MetadataModel.h"
#include "ControlModel.h"
#include "PluginModel.h"
#include "OmPath.h"
#include "canvas/PatchBayArea.h"

using std::cout; using std::cerr; using std::endl;
using std::string;

namespace OmGtk {


/**	Pure virtual base class for all gtk client events.
 */
class Event
{
public:
	virtual ~Event() {}
	virtual void execute() = 0;
};



class ErrorEvent : public Event
{
public:
	ErrorEvent(const string& msg) : m_msg(msg) {}
	virtual ~ErrorEvent() {}

	void execute()
	{
		app->error_message(m_msg);
	}
	
private:
	string m_msg;
};


class EngineEnabledEvent : public Event
{
public:
	EngineEnabledEvent() {}
	virtual ~EngineEnabledEvent() {}
	
	void execute()
	{
		app->engine_enabled(true);
	}
};


class EngineDisabledEvent : public Event
{
public:
	EngineDisabledEvent() {}
	virtual ~EngineDisabledEvent() {}
	
	void execute()
	{
		app->engine_enabled(false);
	}
};


class NewPatchEvent : public Event
{
public:
	NewPatchEvent(PatchModel* const pm) : m_patch_model(pm) {}
	virtual ~NewPatchEvent() {}
	
	void execute()
	{
		//cout << "[GtkClientHooks] New patch." << endl;
		if (app->patch(m_patch_model->path()) != NULL) {
			delete m_patch_model;
		} else {
			bool show = false;
			// See if we cached this patch model to store its location (to avoid the
			// module "jumping") and filename (which isn't sent to engine)
			PatchModel* pm = controller->yank_added_patch(m_patch_model->path());
			if (pm != NULL) {
				m_patch_model->x(pm->x());
				m_patch_model->y(pm->y());
				m_patch_model->filename(pm->filename());
				m_patch_model->author(pm->author());
				m_patch_model->description(pm->description());
				//cerr << "Yanking patch, filename = " << pm->filename() << endl;
				// Show only new patch windows that are explicitly added
				if (pm->filename() == "")
					show = true;
			} else {
				//cerr << "Not yanking patch!" << endl;
			}

			// deleting pm here causes a segfault for some reason.. (?)

			//cerr << "[NewPatchEvent] Creating patch " << m_patch_model->path() << endl;
			
			app->add_patch(m_patch_model, show);
		}
	}
	
private:
	PatchModel* m_patch_model;
};



class NewPortEvent : public Event
{
public:
	NewPortEvent(PortModel* const pm)
		: m_port_model(pm) {}
	virtual ~NewPortEvent() {}
	
	void execute()
	{
		const string parent_path = OmPath::parent(m_port_model->path());
	
		// Port on a patch?
		PatchController* pc = app->patch(parent_path);
		if (pc != NULL) {
			pc->new_port(m_port_model);
			return;
		}
		
		// Port on a (normal) module?
		OmModule* module = app->module(parent_path);
		if (module != NULL) {
			module->add_om_port(m_port_model, true);
			return;
		}
		
		cerr << "[NewPortEvent] Could not find parent to add port \'"
			<< m_port_model->path() << endl;
	}

private:
	PortModel* m_port_model;
};


class PortRemovalEvent : public Event
{
public:
	PortRemovalEvent(const string& path)
	: m_path(path) {}
	virtual ~PortRemovalEvent() {}

	void execute()
	{
		const string parent_path = OmPath::parent(m_path);
	
		// Port on a patch?
		PatchController* pc = app->patch(parent_path);
		if (pc != NULL) {
			pc->remove_port(m_path);
			return;
		}
		
		// Port on a (normal) module?
		OmModule* module = app->module(parent_path);
		if (module != NULL) {
			module->remove_port(m_path, true);
			return;
		}
		
		cerr << "[NewPortEvent] Could not find parent to remove port \'"
			<< m_path << endl;
	}
private:
	string m_path;
};
		

class PatchDestructionEvent : public Event
{
public:
	PatchDestructionEvent(const string& path) : m_path(path) {}
	virtual ~PatchDestructionEvent() {}
	
	void execute()
	{
		//cerr << "[GtkClientHooks] Patch destruction" << endl;
	
		app->destroy_patch(m_path);
	}

private:
	string m_path;
};
	

class PatchEnabledEvent : public Event
{
public:
	PatchEnabledEvent(const string& patch_path) : m_patch_path(patch_path) {}
	virtual ~PatchEnabledEvent() {}
	
	void execute()
	{
		app->patch_enabled(m_patch_path);
	}
	
private:
	string m_patch_path;
};


class PatchDisabledEvent : public Event
{
public:
	PatchDisabledEvent(const string& patch_path) : m_patch_path(patch_path) {}
	virtual ~PatchDisabledEvent() {}
	
	void execute()
	{
		app->patch_disabled(m_patch_path);
	}
	
private:
	string m_patch_path;
};


class NewNodeEvent : public Event
{
public:
	NewNodeEvent(NodeModel* const nm) : m_node_model(nm) {}
	virtual ~NewNodeEvent() {}
	
	void execute()
	{
		//cerr << "[GtkClientHooks] New node: " << m_node_model->name() << endl;
		
		string parent_path = OmPath::parent(m_node_model->path());
		
		PatchController* pc = app->patch(parent_path);
		
		if (pc != NULL) {

			// See if we cached this node model to store it's location (to avoid the
			// module "jumping")
			/*NodeModel* nm = controller->yank_added_node(m_node_model->path());
			if (nm != NULL) {
				m_node_model->x(nm->x());
				m_node_model->y(nm->y());
			} else { // pick a default
			*/
				int x, y;
				pc->get_new_module_location(x, y);
				m_node_model->x(x);
				m_node_model->y(y);
			//}*/
			//cerr << "[NewNodeEvent] Creating node " << m_node_model->path() << endl;
			m_node_model->parent(pc->model());
			pc->new_node(m_node_model);
		} else {
			cerr << "[NewNodeEvent] Can not find window for patch " << parent_path
				<< ".  Module will not appear." << endl;
		}
	}
	
private:
	NodeModel* m_node_model;
};



class NodeRemovalEvent : public Event
{
public:
	NodeRemovalEvent(const string& path) : m_path(path) {}
	virtual ~NodeRemovalEvent() {}

	void execute()
	{
		//cerr << "[GtkClientHooks] Node removal" << endl;
	
		PatchController* pc = app->patch(OmPath::parent(m_path));
		
		if (pc != NULL) {
			pc->remove_node(OmPath::name(m_path));
		} else {
			cerr << "[NodeRemoveEvent] Can not find window for patch " << m_path
				<< ".  Module will not be destroyed." << endl;
		}
	}

private:
	string m_path;
};


class ObjectRenamedEvent : public Event
{
public:
	ObjectRenamedEvent(const string& old_path, const string& new_path)
	: m_old_path(old_path), m_new_path(new_path) {}
	virtual ~ObjectRenamedEvent() {}
	
	void execute()
	{
		// Port?
		OmPort* port = app->port(m_old_path);
		if (port != NULL) {
			port->name(OmPath::name(m_new_path));
			// (return deliberately omitted)
		}
		
		// Patch?
		PatchController* pc = app->patch(m_old_path);
		if (pc != NULL) {
			// FIXME: slow, double lookup
			app->patch_renamed(m_old_path, m_new_path);
			assert(pc->model()->path() == m_new_path);
			return;
		}
		
		// Module?
		OmModule* module = app->module(m_old_path);
		if (module != NULL) {
			module->path(m_new_path);
			return;
		}

		// Not found
		if (pc == NULL && module == NULL && port == NULL) 
			cerr << "[ObjectRenamedEvent] Can not find object " << m_old_path
				<< " to rename." << endl;
	}

private:
	string m_old_path;
	string m_new_path;
};


class ConnectionEvent : public Event
{
public:
	ConnectionEvent(ConnectionModel* const cm) : m_connection_model(cm) {}
	virtual ~ConnectionEvent() {}
	
	void execute()
	{
		//cerr << "[GtkClientHooks] Connection" << endl;
		
		assert(OmPath::parent(OmPath::parent(m_connection_model->src_port_path()))
			== OmPath::parent(OmPath::parent(m_connection_model->dst_port_path())));
		
		PatchController* pc = app->patch(m_connection_model->patch_path());
		
		if (pc != NULL) {
			pc->connection(m_connection_model);
		} else {
			cerr << "[ConnectionEvent] Can not find window for patch " << m_connection_model->patch_path()
				<< ".  Connection will not be made." << endl;
		}
	}

private:
	ConnectionModel* m_connection_model;
};



class DisconnectionEvent : public Event
{
public:
	DisconnectionEvent(const string& src_port_path, const string& dst_port_path)
	: m_src_port_path(src_port_path), m_dst_port_path(dst_port_path) {}
	virtual ~DisconnectionEvent() {}
	
	void execute()
	{
		//cerr << "[GtkClientHooks] Disconnection" << endl;
		string patch_path = m_src_port_path;
		patch_path = patch_path.substr(0, patch_path.find_last_of("/"));
		patch_path = patch_path.substr(0, patch_path.find_last_of("/"));
		
		PatchController* pc = app->patch(patch_path);
		
		if (pc != NULL) {
			pc->disconnection(
			m_src_port_path, m_dst_port_path);
		} else {
			cerr << "[DisconnectionEvent] Can not find window for patch " << patch_path
				<< ".  Connection will not be removed." << endl;
		}
	}

private:
	string m_src_port_path;
	string m_dst_port_path;
};
	


class MetadataUpdateEvent : public Event
{
public:
	MetadataUpdateEvent(MetadataModel* const mm) : m_metadata_model(mm) {}
	virtual ~MetadataUpdateEvent() { delete m_metadata_model; }
	
	void execute()
	{
		/* Need to try all different objects consecutively because one path
		 * could be destined for multiple objects (ie control input node on
		 * a patch and the corresponding module) */

		PatchController* patch = app->patch(m_metadata_model->path());
		if (patch != NULL)
			patch->metadata_update(m_metadata_model);
		
		OmModule* module = app->module(m_metadata_model->path());
		if (module != NULL)
			module->metadata_update(m_metadata_model);

		OmPort* port = app->port(m_metadata_model->path());
		if (port != NULL)
			port->metadata_update(m_metadata_model);
		
		if (patch == NULL && module == NULL && port == NULL)
			cerr << "[MetadataUpdateEvent] Could not find object " << m_metadata_model->path() << endl;
	}

private:
	MetadataModel* m_metadata_model;
};



class ControlChangeEvent : public Event
{
public:
	ControlChangeEvent(ControlModel* const cm) : m_control_model(cm) {}
	virtual ~ControlChangeEvent() { delete m_control_model; }
	
	void execute()
	{
		OmPort* port = app->port(m_control_model->port_path());

		//cerr << "[ControlChangeEvent] Control change: " << m_control_model->port_path()
		//	<< " = " << m_control_model->value() << endl;

		if (port != NULL)
			port->control_change(m_control_model);
		else
			cerr << "[ControlChangeEvent] Could not find port " << m_control_model->port_path() << endl;
	}

private:
	ControlModel* m_control_model;
};
	


class NewPluginEvent : public Event
{
public:
	NewPluginEvent(PluginModel* pi) : m_plugin_model(pi) {}
	virtual ~NewPluginEvent() {}

	void execute()
	{
		app->new_plugin(m_plugin_model);
	}

private:
	PluginModel* m_plugin_model;
};

	
} // namespace OmGtk

#endif // GTKCLIENTHOOKSEVENTS_H
