/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "DSSIModule.h"
#include <dirent.h>
#include <unistd.h>
#include <sys/stat.h>
#include "NodeModel.h"
#include "PluginModel.h"
#include "Controller.h"
#include "OmGtk.h"

namespace OmGtk {


DSSIModule::DSSIModule(OmPatchBayArea* patch_bay, NodeModel* nm, PatchController* patch_controller)
: OmModule(patch_bay, nm, patch_controller)
{
	Gtk::Menu::MenuList& items = m_menu.items();
	items[0].property_sensitive() = true; // "Show Control Window" item
	
	items.push_front(Gtk::Menu_Helpers::MenuElem("Show Plugin GUI",
		sigc::mem_fun(this, &DSSIModule::show_gui)));
}


void
DSSIModule::on_double_click()
{
	if (!attempt_to_show_gui())
		show_control_window();
}


/** Show DSSI GUI for this plugin.
 * Exists solely for sigc++ to be pleased with it's damned return type.
 */
void
DSSIModule::show_gui()
{
	attempt_to_show_gui();
}


/** Attempt to show the DSSI GUI for this plugin.
 *
 * Returns whether or not GUI was successfully loaded/shown.
 */
bool
DSSIModule::attempt_to_show_gui()
{
	// Shamelessley "inspired by" jack-dssi-host
	// Copyright 2004 Chris Cannam, Steve Harris and Sean Bolton.

	const bool verbose = false;

	const char*   dllName   = m_node_model->plugin_model()->lib_name().c_str();
	const char*   label     = m_node_model->plugin_model()->plug_label().c_str();
	const char*   myName    = "om_gtk";
	const string& oscUrl    = controller->engine_url() + "/dssi" + m_node_model->path();

	struct dirent* entry;
	char*          dllBase = strdup(dllName);
	char*          subpath;
	DIR*           subdir;
	char*          filename;
	struct stat    buf;
	int            fuzzy;

	char* env_dssi_path = getenv("DSSI_PATH");
	string dssi_path;
	if (!env_dssi_path) {
	 	cerr << "DSSI_PATH is empty.  Assuming /usr/lib/dssi:/usr/local/lib/dssi." << endl;
		dssi_path = "/usr/lib/dssi:/usr/local/lib/dssi";
	} else {
		dssi_path = env_dssi_path;
	}

	if (strlen(dllBase) > 3 && !strcasecmp(dllBase + strlen(dllBase) - 3, ".so")) {
		dllBase[strlen(dllBase) - 3] = '\0';
	}


	// This is pretty nasty, it loops through the path, even if the dllBase is absolute
	while (dssi_path != "") {
		string directory = dssi_path.substr(0, dssi_path.find(':'));
		if (dssi_path.find(':') != string::npos)
			dssi_path = dssi_path.substr(dssi_path.find(':')+1);
		else
			dssi_path = "";
		
		if (*dllBase == '/') {
			subpath = strdup(dllBase);
		} else {
			subpath = (char*)malloc(strlen(directory.c_str()) + strlen(dllBase) + 2);
			sprintf(subpath, "%s/%s", directory.c_str(), dllBase);
		}
	
		for (fuzzy = 0; fuzzy <= 1; ++fuzzy) {
	
			if (!(subdir = opendir(subpath))) {
				if (verbose) {
					fprintf(stderr, "%s: can't open plugin GUI directory \"%s\"\n", myName, subpath);
				}
				break;
			}
	
			while ((entry = readdir(subdir))) {
	
				if (entry->d_name[0] == '.')
					continue;
				if (!strchr(entry->d_name, '_'))
					continue;
	
				if (fuzzy) {
					if (verbose) {
						fprintf(stderr, "checking %s against %s\n", entry->d_name, dllBase);
					}
					if (strncmp(entry->d_name, dllBase, strlen(dllBase)))
						continue;
				} else {
					if (verbose) {
						fprintf(stderr, "checking %s against %s\n", entry->d_name, label);
					}
					if (strncmp(entry->d_name, label, strlen(label)))
						continue;
				}
	
				filename = (char*)malloc(strlen(subpath) + strlen(entry->d_name) + 2);
				sprintf(filename, "%s/%s", subpath, entry->d_name);
	
				if (stat(filename, &buf)) {
					perror("stat failed");
					free(filename);
					continue;
				}
	
				if ((S_ISREG(buf.st_mode) || S_ISLNK(buf.st_mode)) &&
				        (buf.st_mode & (S_IXUSR | S_IXGRP | S_IXOTH))) {
	
					if (verbose) {
						fprintf(stderr, "%s: trying to execute GUI at \"%s\"\n",
						        myName, filename);
					}
	
					if (fork() == 0) {
						execlp(filename, filename, oscUrl.c_str(), dllName, label, m_node_model->name().c_str(), 0);
						perror("exec failed");
						exit(1);
					}
	
					free(filename);
					free(subpath);
					free(dllBase);
					return true;
				}
	
				free(filename);
			}
		}
	}
	
	cerr << "Unable to launch DSSI GUI for " << m_node_model->path() << endl;
	
	free(subpath);
	free(dllBase);
	return false;
}


} // namespace OmGtk
