/* This file is part of Om.  Copyright (C) 2005 Dave Robillard.
 * 
 * Om is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 * 
 * Om is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef CONTROLLER_H
#define CONTROLLER_H

#include <cassert>
#include <string>
#include <list>
#include <glibmm.h>
#include "PluginModel.h"

namespace LibOmClient {
class PatchModel;
class NodeModel;
class PatchLibrarian;
class OSCController;
}

using std::string; using std::list;
using namespace LibOmClient;

namespace OmGtk {

class PatchController;
class GtkClientHooks;
class Loader;

	
/** Controller for the entire om_gtk app.
 *
 * This is hardly more than a trivial wrapper for OSCController, suggesting something
 * needs a rethink around here..
 *
 * \ingroup OmGtk
 */
class Controller
{
public:
	Controller();
	~Controller();

	void attach(const string& engine_url, int client_port = 0);
	
	void register_client();
	
	string engine_url();

	void activate_engine();
	void deactivate_engine();
	void enable_engine();
	void disable_engine();

	void quit();

	void set_control(const string& port_path, float val);
	void set_control(const string& port_path, int voice, float val);
	
	void request_control(const string& port_path);

	void add_node(NodeModel* nm);

	void remove_node(const string& node_path);
	
	void connect(const string& port1_path, const string& port2_path);
	void disconnect(const string& port1_path, const string& port2_path);
	void disconnect_all(const string& node_path);
	
	void load_patch(PatchModel* model);
	void save_patch(PatchModel* model, const string& filename, bool recursive);

	void load_patch_blocking(PatchModel* model);
	void save_patch_blocking(PatchModel* model, const string& filename, bool recursive);
	
	void load_session(const string& filename);
	void save_session(const string& filename);

	void create_patch(PatchModel* pm);
	void destroy_patch(const string& path);

	void rename(const string& path, const string& name);
	
	void enable_patch(const string& patch_path);
	void disable_patch(const string& patch_path);
	
	void midi_learn(const string& node_path);

	void set_metadata(const string& object_path, const string& key, const string& value);
	
	void request_plugins();
	void request_all_objects();
	GtkClientHooks* client_hooks() { return m_client_hooks; }
	bool attached() { return m_is_attached; }

	void set_patch_path(const string& path);

	void push_added_node(NodeModel* nm) { m_added_nodes.push_back(nm); }
	NodeModel* yank_added_node(const string& path);

	void push_added_patch(PatchModel* pm) { m_added_patches.push_back(pm); }
	PatchModel* yank_added_patch(const string& path);

private:
	GtkClientHooks* m_client_hooks;
	OSCController*  m_osc_controller;
	PatchLibrarian* m_patch_librarian;
	Loader*         m_loader;	
	
	
	/** Used to cache added nodes so client can remember some values when the
	 * new node notification comes (location, etc).  Used to prevent the node
	 * jumping locations during the delay between new node and the module-x/y
	 * metadata notifications */
	list<NodeModel*> m_added_nodes;

	/** Used to cache added patches in the same was as m_added_nodes.  Used to
	 * rember filename so File->Save can work without prompting (filename can't
	 * be sent to the server as metadata, because a client on another machine
	 * doesn't want that information) */
	list<PatchModel*> m_added_patches;
	
	bool m_is_attached;
};


} // namespace OmGtk

#endif // CONTROLLER_H
