/***************************************************************************
 *
 * olpc-kbdshim-hal.c: -- low-level support for XO:
 *  - mouse-based scrolling use the XO grab keys
 *  - touchpad and dpad rotation (to match screen rotation)
 *  - user activity monitoring
 *
 * Copyright (C) 2009, Paul G Fox, inspired in places
 *  by code from mouseemu, by Colin Leroy and others.
 *
 * This skeleton of this program program is based heavily on:
 *  addon-input.c : Listen to key events and modify hal device objects
 *
 * Copyright (C) 2005 David Zeuthen, <david@fubar.dk>
 * Copyright (C) 2005 Ryan Lortie <desrt@desrt.ca>
 * Copyright (C) 2006 Matthew Garrett <mjg59@srcf.ucam.org>
 * Copyright (C) 2007 Codethink Ltd. Author Rob Taylor <rob.taylor@codethink.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 **************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sched.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <syslog.h>
#include <signal.h>
#include <errno.h>
#include <stdarg.h>

#include <linux/input.h>
#include <linux/uinput.h>

#include <glib/gmain.h>
#include <glib/gprintf.h>
#include <dbus/dbus-glib-lowlevel.h>

#include <hal/libhal.h>


/* log to syslog (instead of stderr) */
static int logtosyslog;

/* higher values for more debug */
static int debug;

/* suppress any actual tranmission or injection of data */
static int noxmit;

/* which keys do grabbing? */
#define NGRABKEYS 4
static struct keypress {
    int code;
    int pass;
    int pressed;
} grabkey[NGRABKEYS];
static int ngrab;

/* command fifo */
static char *command_fifo = "/var/run/olpc-kbdshim_command";
static int fifo_fd = -1;

/* sysactive path -- where we send user activity/idle events */
static char *sysactive_path = "/var/run/powerevents";
#define MAXTIMERS 3
/* last timer should stay zero */
static int idletimers[MAXTIMERS + 1] = { 10 * 60, 15 * 60, 20 * 60, 0 };;

/* are we in user-idle state? */
static int idleness;
static int wasidle;

/* external commands bound to rotate and brightness keys:
 *
 * the rotate command should inform us via our command fifo when
 * a rotate has happened, so that we can adjust the touchpad and
 * d-pad rotation.
 *
 * the brightness command should accept a single argument of
 * "up", "down", "min", or "max" (the last two when the alt
 * modifier is in effect).
 *
 * these bindings are done here for convenience, and could be
 * done anywhere else in the system just as well.  binding
 * them here effectively "steals" them from sugar, which has
 * internal implementations of rotation and brightness control
 * which are no longer correct when this package and olpc-powerd
 * are used.
 */
static char *brightness_cmd; /*  probably "olpc-brightness" */
static char *volume_cmd;     /*  probably "olpc-volume" */
static char *rotate_cmd;     /*  probably "olpc-rotate" */

static int reflect_x, reflect_y;  /* inverted mouse (e.g., for ebook mode) */
static enum rotation {
    ROT_NORMAL,
    ROT_RIGHT,
    ROT_INVERT,
    ROT_LEFT
} rotation;

/* are we scrolling, or not */
static int scrolling;

#define SCROLL_QUANTUM 20
#define SCROLL_FILTER_PERCENT 33
static int cumul_x, cumul_y;

/* "distance traveled" before we emulate a scroll button event */
static int quantum = SCROLL_QUANTUM;
static int ratio = SCROLL_FILTER_PERCENT;

/* if true, finger moves scrollbars, instead of window contents */
static int reverse = 1;

/* is the D-pad in pointer mode? */
static int dpad_pointer;

/* product, vendor id for keyboard and touchpad */
static struct input_id local_kbd_id = {
    bustype: BUS_I8042,
    product: 0xffff,
    vendor: 0xffff
};
static struct input_id local_tpad_id = {
    bustype: BUS_I8042,
    product: 0xffff,
    vendor: 0xffff
};

extern char *optarg;
extern int optind, opterr, optopt;

/* output events device */
static int uinp_fd = -1;

#define MAX(a, b) (((a) > (b)) ? (a) : (b))

/* how many /dev/input/eventX devices to check */
#define EVENTDEVICES 20

typedef struct input_id input_id_t;

static int idle_timer;

static char me[40];

__attribute__((noreturn)) static void
usage(void)
{
    fprintf(stderr,
        "usage: %s [options]\n"
        " Grab-scroll configuration:\n"
        "   '-g N','-G N' Specify which key(s) are used for grabbing.\n"
        "        A maximum of %d keys can be specified.  (Use \"showkey\"\n"
        "        at the console to find the keycode.)  With '-g', the\n"
        "        key will be processed normally (in addition to instituting\n"
        "        scrolling).  '-G' will cause scrolling but suppress the\n"
        "        key's normal action.  Default is '-g 125 -g 126'.  (The\n"
        "        \"meta\" keys, labeled as \"grab\" keys on the XO laptop.\n"
        "        Use '-g 0' to suppress grab-scoll support.\n"
        "   '-v' to reverse the relative motion of mouse and window\n"
        "        By default the mouse/finger slides the scrollbars, but\n"
        "        with -v it effectively slides the window instead.\n"
        "   '-q N' to set how far the mouse/finger must move before a\n"
        "        scrolling event is generated (default is %d).\n"
        "   '-n N' If the x or y motion is less than N percent of the\n"
        "        other (default N is %d) then the smaller of the two is\n"
        "        dropped. This reduces sideways jitter when scrolling\n"
        "        vertically,and vice versa.\n"
        "\n"
        " Touchpad/D-pad rotation:\n"
        "   '-R <fifoname>'  If set, this gives the name of a fifo which\n"
        "        will be created and monitored for the purpose of rotating\n"
        "        and reflecting the touchpad and the D-pad.  This can be\n"
        "        used to make their function match the screen's rotation.\n"
        "        This fifo will also receive setup commands related to\n"
        "        user activity monitoring (see below).\n"
        "\n"
        " Keyboard/touchpad identification:\n"
        "   '-K BB:VV:PP' Specifies the bustype, vendor, and product id\n"
        "        for the \"local\" keyboard.  BB, VV, and PP are hex\n"
        "        numbers.  ffff can be used as a wildcard for any of them.\n"
        "        Default is '11:ffff:ffff' for the i8042 bus, which will find\n"
        "        the local keyboard.  '03:ffff:ffff' finds any USB keyboard.\n"
        "   '-T BB:VV:PP' Same as -K, but for the pointer device.\n"
        "        Only the \"local\" keyboard and touchpad will be affected by\n"
        "        rotation commands (i.e., rotate with the screen)\n"
        "\n"
        " User activity monitor:\n"
        "   '-A PATH' Specifies path where user idle/activity events will\n"
        "        be written (default is /var/run/powerevents).\n"
        "\n"
        " Key bindings:\n"
        "   '-r rotate-cmd'  If set, the command to be run when the XO\n"
        "        'rotate' button is detected.  (Will be passed no arguments.)\n"
        "   '-b brightness-cmd'  If set, the command to be run when the XO\n"
        "        brightness keys are used.  (Should accept 'min', 'max', 'up'\n"
        "        or 'down' as the sole argument.)\n"
        "   '-V volume-cmd'  If set, the command to be run when the XO\n"
        "        volume keys are used.  (Should accept 'min', 'max', 'up'\n"
        "        or 'down' as the sole argument.)\n"
        " Daemon options:\n"
        "   '-f' to keep program in foreground.\n"
        "   '-l' use syslog, rather than stderr, for messages.\n"
        "   '-s' to run with elevated (sched_fifo) scheduling priority.\n"
        "   '-d' for debugging (repeat for more verbosity).\n"
        "   '-X' don't actually pass on received keystrokes (for debug).\n"
        "(olpc-kbdshim version %d)\n"
        , me, NGRABKEYS, SCROLL_QUANTUM, SCROLL_FILTER_PERCENT, VERSION);
    exit(1);
}

static void
report(const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    if (logtosyslog && debug <= 1) {
        vsyslog(LOG_NOTICE, fmt, ap);
    } else {
        fprintf(stderr, "%s: ", me);
        vfprintf(stderr, fmt, ap);
        fputc('\n', stderr);
    }
}

static void
dbg(int level, const char *fmt, ...)
{
    va_list ap;

    if (debug < level) return;

    va_start(ap, fmt);
    if (logtosyslog)
        vsyslog(LOG_NOTICE, fmt, ap);

    fputc(' ', stderr);
    vfprintf(stderr, fmt, ap);
    fputc('\n', stderr);
}

__attribute__((noreturn)) static void
die(const char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    if (logtosyslog && debug <= 1) {
        vsyslog(LOG_ERR, fmt, ap);
        syslog(LOG_ERR, "exiting -- %m");
    } else {
        fprintf(stderr, "%s: ", me);
        vfprintf(stderr, fmt, ap);
        fprintf(stderr, " - %s", strerror(errno));
        fputc('\n', stderr);
    }
    exit(1);
}

static void
inject_uinput_event(unsigned int type, unsigned int code, int value)
{
    struct input_event event;

    gettimeofday(&event.time, NULL);
    event.type = type;
    event.code = code;
    event.value = value;
    if (!noxmit && write(uinp_fd, &event, sizeof(event)) < 0) {
        report("warning: inject event failed, t/c/v 0x%x/0x%x/0x%x",
            type, code, value);
    }

}

static void
send_a_scroll(int x, int y)
{
    if (reverse) {
        x = -x;
        y = -y;
    }

    if (x)  {
        dbg(1, "scroll %s", x > 0 ? "left" : "right");
        inject_uinput_event(EV_REL, REL_HWHEEL, x < 0 ? 1 : -1);
    }
    if (y) {
        dbg(1, "scroll %s", y > 0 ? "up" : "down");
        inject_uinput_event(EV_REL, REL_WHEEL, y < 0 ? -1 : 1);
    }
    inject_uinput_event(EV_SYN, SYN_REPORT, 0);
}

static const unsigned char dpad[] = { KEY_KP8, KEY_KP6, KEY_KP2, KEY_KP4,
                         KEY_KP8, KEY_KP6, KEY_KP2, KEY_KP4};


void
handle_dpad_pointer(struct input_event *ev)
{
    int xmot, ymot, numdirs;
    static int dpad_up_pressed; // D-pad directions currently pushed?
    static int dpad_down_pressed;
    static int dpad_left_pressed;
    static int dpad_right_pressed;


    switch(ev->code) {
    case KEY_KP8: dpad_up_pressed = ev->value;          break;
    case KEY_KP2: dpad_down_pressed = ev->value;        break;
    case KEY_KP4: dpad_left_pressed = ev->value;        break;
    case KEY_KP6: dpad_right_pressed = ev->value;       break;
    }

    xmot = ymot = numdirs = 0;

    if (dpad_up_pressed) {
        dbg(3, "dpad up pressed");
        ymot -= 1;
        numdirs++;
    }
    if (dpad_down_pressed) {
        dbg(3, "dpad down pressed");
        ymot += 1;
        numdirs++;
    }
    if (dpad_left_pressed) {
        dbg(3, "dpad left pressed");
        xmot -= 1;
        numdirs++;
    }
    if (dpad_right_pressed) {
        dbg(3, "dpad right pressed");
        xmot += 1;
        numdirs++;
    }

    ev->type = EV_REL;
    if (xmot) {
        ev->code = REL_X;
        ev->value = xmot * (ymot ? 2 : 4);
        dbg(3, "dpad X %d", ev->value);
        if (!noxmit && write(uinp_fd, ev, sizeof(ev)) < 0) {
            report("uinput write failed: %s", strerror(errno));
        }
    }
    if (ymot) {
        ev->code = REL_Y;
        ev->value = ymot * (xmot ? 2 : 4);
        dbg(3, "dpad Y %d", ev->value);
        if (!noxmit && write(uinp_fd, ev, sizeof(ev)) < 0) {
            report("uinput write failed: %s", strerror(errno));
        }
    }
    if (xmot || ymot)
        inject_uinput_event(EV_SYN, SYN_REPORT, 0);
}

static int
keyboard_event (GIOChannel *channel, int is_local)
{
    struct input_event ev[1];
    int pass = 1;
    int i;
    gsize read_bytes;
    GError *gerror = NULL;
    static int altdown;

    if (g_io_channel_read_chars (channel,
                    (gchar*)ev, sizeof(ev),
                    &read_bytes, &gerror) != G_IO_STATUS_NORMAL) {
        die("bad read from keyboard");
    }

    if (read_bytes != sizeof(ev)) {
        dbg(1, "short read of ev (%d)", read_bytes);
        return 0;
    }

    dbg(3, "evtype %d code %d value %d", ev->type, ev->code, ev->value);

    for (i = 0; i < ngrab; i++) {
        if (ev->code == grabkey[i].code) {
            /* keep track of how many grab keys are down */
            if (ev->value) {
                if (grabkey[i].pressed++ == 0)
                    scrolling++;
            } else {
                grabkey[i].pressed = 0;
                if (--scrolling == 0)
                    cumul_x = cumul_y = 0;
            }

            pass = grabkey[i].pass;

            dbg(2, "scrolling %d", scrolling);
            break;
        }
    }

    /* implement arrow key grab-scrolling */
    if (scrolling && ev->value) {
        switch(ev->code) {
        case KEY_UP:    send_a_scroll( 0,-1);  pass = 0; break;
        case KEY_DOWN:  send_a_scroll( 0, 1);  pass = 0; break;
        case KEY_LEFT:  send_a_scroll(-1, 0);  pass = 0; break;
        case KEY_RIGHT: send_a_scroll( 1, 0);  pass = 0; break;
        }
    }

    if (is_local) {
        i = 0;
        switch(ev->code) {
        case KEY_KP4: i++; /* rotate the dpad on the screen bezel */
        case KEY_KP2: i++;
        case KEY_KP6: i++;
        case KEY_KP8:
            ev->code = dpad[rotation + i];

            if (dpad_pointer) {
                handle_dpad_pointer(ev);
                pass = 0;
            }
            break;

        case KEY_SWITCHVIDEOMODE:  // rotate button
	    dbg(3, "might rotate");
            if (rotate_cmd && ev->value) {

                // command given, so don't pass the keystroke through.
                pass = 0;

                // if absolute, we can check for presence cheaply
                if (*rotate_cmd != '/' || access(rotate_cmd, X_OK) == 0) {
		    dbg(1, "invoking %s", rotate_cmd);
                    if (system(rotate_cmd) == 0)
                        ;
                }
            }
            break;

        case KEY_F9:        // brightness down key (or minimize, with alt)
        case KEY_F10:       // brightness up key (or maximize, with alt)
	    dbg(3, "might brighten");
            if (brightness_cmd && ev->value) {
                char cmd[256];
                char *arg;

                // command given, so don't pass the keystroke through.
                pass = 0;

                // if absolute, we can check for presence cheaply
                if (*brightness_cmd != '/' ||
                            access(brightness_cmd, X_OK) == 0) {
                    if (altdown) {
                        arg = (ev->code == KEY_F9) ? "min" : "max";
                    } else {
                        arg = (ev->code == KEY_F9) ? "down" : "up";
                    }
                    if (snprintf(cmd, sizeof(cmd), "%s %s",
                                    brightness_cmd, arg) <= sizeof(cmd)) {
			dbg(1, "invoking %s", cmd);
                        if (system(cmd) == 0)
                            ;
                    }
                }
            }
            break;

        case KEY_F11:       // volume down key (or minimize, with alt)
        case KEY_F12:       // volume up key (or maximize, with alt)
	    dbg(3, "might adjust volume");
            if (volume_cmd && ev->value) {
                char cmd[256];
                char *arg;

                // command given, so don't pass the keystroke through.
                pass = 0;

                // if absolute, we can check for presence cheaply
                if (*volume_cmd != '/' ||
                            access(volume_cmd, X_OK) == 0) {
                    if (altdown) {
                        arg = (ev->code == KEY_F11) ? "min" : "max";
                    } else {
                        arg = (ev->code == KEY_F11) ? "down" : "up";
                    }
                    if (snprintf(cmd, sizeof(cmd), "%s %s",
                                    volume_cmd, arg) <= sizeof(cmd)) {
			dbg(1, "invoking %s", cmd);
                        if (system(cmd) == 0)
                            ;
                    }
                }
            }
            break;

        case KEY_LEFTALT:
        case KEY_RIGHTALT:
            if (ev->value == 0)
                altdown--;
            else if (ev->value == 1)
                altdown++;
            else
                /* ignore repeats with ev->value == 2 */ ;
            if (altdown > 2) altdown = 2;       // safety
            else if (altdown < 0) altdown = 0;  // safety
            break;
        }
    }

    if (pass && !noxmit && write(uinp_fd, ev, sizeof(ev)) < 0) {
        report("uinput write failed: %s", strerror(errno));
    }

    return !!(ev->value);  /* indicates press */
}

static void touchpad_event(GIOChannel *channel, int is_local)
{
    struct input_event ev[1];
    gsize read_bytes;
    GError *gerror = NULL;

    if (g_io_channel_read_chars (channel,
                    (gchar*)ev, sizeof(ev),
                    &read_bytes, &gerror) != G_IO_STATUS_NORMAL) {
        die("bad read from pointer");
    }

    if (read_bytes != sizeof(ev)) {
        report("short read of ev (%d)", read_bytes);
        return;
    }

    dbg(3, "evtype %d code %d", ev->type, ev->code);


    if (is_local) {    // only rotate/flip the local touchpad
        if ((reflect_x && ev->code == REL_X) ||
            (reflect_y && ev->code == REL_Y)) {
            ev->value = -ev->value;
        }

        switch(rotation) {
        case ROT_NORMAL:
            break;
        case ROT_RIGHT:
            if (ev->code == REL_Y) {
                ev->code = REL_X;
            } else if (ev->code == REL_X) {
                ev->code = REL_Y;
                ev->value = -ev->value;
            }
            break;
        case ROT_LEFT:
            if (ev->code == REL_Y) {
                ev->code = REL_X;
                ev->value = -ev->value;
            } else if (ev->code == REL_X) {
                ev->code = REL_Y;
            }
            break;
        case ROT_INVERT:
            ev->value = -ev->value;
            break;
        }
    }

    if (scrolling && ev->type == EV_REL && ev->code == REL_X) {
            cumul_x += ev->value;

    } else if (scrolling && ev->type == EV_REL && ev->code == REL_Y) {
            cumul_y += ev->value;

    } else if (scrolling && ev->type == EV_SYN) {
        /* emit our current scroll input */
        int x = 0, y = 0;

        if (abs(cumul_y) > quantum) {
            if (abs(cumul_x) < ratio * abs(cumul_y) / 100)
                cumul_x = 0;
            y = cumul_y;
            cumul_y = 0;
        }
        if (abs(cumul_x) > quantum) {
            if (abs(cumul_y) < ratio * abs(cumul_x) / 100)
                cumul_y = 0;
            x = cumul_x;
            cumul_x = 0;
        }
        send_a_scroll(x, y);

    } else {
        /* passthrough */
        if (!noxmit && write(uinp_fd, ev, sizeof(ev)) < 0) {
            report("uinput write failed: %s", strerror(errno));
        }
    }
}


static void
send_event(char *e)
{
    int fd, n;
    char evtbuf[128];

    fd = open(sysactive_path, O_RDWR|O_NONBLOCK);
    if (fd >= 0) {
        n = snprintf(evtbuf, 128, "%s %d\n", e, (int)time(0));
        n = write(fd, evtbuf, n);
        close(fd);
    }
}

static gboolean
indicate_idleness(gpointer data)
{

    static char useridle[] = "useridleX";

    useridle[8] = idleness + '1';
    if (idleness < MAXTIMERS) idleness++;

    send_event(useridle);
    wasidle = 1;
    if (idletimers[idleness])
        idle_timer = g_timeout_add_seconds(
                idletimers[idleness], indicate_idleness, 0);
    else
        idle_timer = 0;
    return FALSE;
}

static void
reinit_activity(void)
{
    wasidle = 1;
    idleness = 0;
    if (idle_timer)
        g_source_remove(idle_timer);
    idle_timer = g_timeout_add_seconds(
            idletimers[idleness], indicate_idleness, 0);
}

static void
indicate_activity(void)
{

    if (wasidle)
        send_event("useractive");

    reinit_activity();
    wasidle = 0;
}

static void
get_command(GIOChannel *channel)
{
    int n;
    unsigned int a, b, c;
    GError *gerror = NULL;
    GIOStatus s;
    static GString *cmd;

    dbg(1, "reading from command fifo");

    if (!cmd)
        cmd = g_string_new("");

    s = g_io_channel_read_line_string (channel, cmd, 0, &gerror);
    if (s != G_IO_STATUS_NORMAL) {
        if (s == G_IO_STATUS_AGAIN) {
            return;
        }
        die("read from command fifo");
    }

    dbg(1, "got '%c' from command fifo", cmd->str[0]);

    switch (cmd->str[0]) {
        case 'x': reflect_x = 0; break;
        case 'y': reflect_y = 0; break;
        case 'z': reflect_x = 0;
                  reflect_y = 0; break;

        case 'X': reflect_x = 1; break;
        case 'Y': reflect_y = 1; break;
        case 'Z': reflect_x = 1;
                  reflect_y = 1; break;

        case 'n': rotation = ROT_NORMAL; break;
        case 'r': rotation = ROT_RIGHT; break;
        case 'i': rotation = ROT_INVERT; break;
        case 'l': rotation = ROT_LEFT; break;

        case 'I':
                n = sscanf(cmd->str, "I %u %u %u", &a, &b, &c);
                if (n < 1) {
                    a = b = c = 0;
                } else {
                    if (n < 2 || b < a + 1)
                        b = a + 1;
                    if (n < 3 || c < b + 1)
                        c = b + 1;

                }

		{ 
		    static unsigned int old_a, old_b, old_c;
		    if (a != old_a || b != old_b || c != old_c) {
                	report("idle timers changed to %d %d %d", a, b, c);
		    } else {
                	dbg(1,"idle timers set to %d %d %d", a, b, c);
		    }
		    old_a = a; old_b = b; old_c = c;
		}

                idletimers[3] = 0; /* always, so our last sleep is untimed */
                idletimers[2] = c - b;
                idletimers[1] = b - a;
                idletimers[0] = a;

                reinit_activity();

                break;

        case 'd': dpad_pointer = 0; break;
        case 'D': dpad_pointer = 1; break;

        default:
                rotation = ROT_NORMAL;
                reflect_x = 0; reflect_y = 0;
                break;
    }

    dbg(1, "rot/refl_x/refl_y is %d/%d/%d", rotation, reflect_x, reflect_y);

}

/* we must use this kernel-compatible implementation */
#define BITS_PER_LONG (sizeof(long) * 8)
#define NBITS(x) ((((x)-1)/BITS_PER_LONG)+1)
#define OFF(x)  ((x)%BITS_PER_LONG)
#define BIT(x)  (1UL<<OFF(x))
#define LONG(x) ((x)/BITS_PER_LONG)
#define test_bit(bit, array)    ((array[LONG(bit)] >> OFF(bit)) & 1)

#define IS_KBD      1
#define IS_POINTER  2
#define IS_LOCAL    4
#define IS_CMDFIFO  8


static LibHalContext *ctx = NULL;
static GMainLoop *gmain = NULL;
static GHashTable *inputs = NULL;
static GList *devices = NULL;

static gboolean
event_io (GIOChannel *channel, GIOCondition condition, gpointer gdata)
{
        unsigned int info = (unsigned int) gdata;
        int pressed = 0;
        dbg(2, "event_io called, info is 0x%x, (%d)",
                info, !!(info & IS_LOCAL));

        if (condition & (G_IO_HUP | G_IO_ERR | G_IO_NVAL))
                return FALSE;

        if (info & IS_KBD) {
            pressed |= keyboard_event(channel, !!(info & IS_LOCAL));
            dbg(3, "kbd activity");
            dbg(3, "kbd activity (%d)", !!(info & IS_LOCAL));
        }

        if (info & IS_POINTER) {
            touchpad_event(channel, !!(info & IS_LOCAL));
            pressed = 1;
            dbg(3, "pointer activity (%d)", !!(info & IS_LOCAL));
        }

        if (info & IS_CMDFIFO) {
            dbg(3, "got command");
            get_command(channel);
        }

        if (pressed)
            indicate_activity();

        return TRUE;
}


static int
init_fifo(char *fifo_node)
{
    int fd;
    struct stat sbuf;
    GIOChannel *channel;
    unsigned int info;

#define fifomode 0622  /* allow anyone to adjust the rotation */
    dbg(1, "initializing fifo %s", fifo_node);

    if (mkfifo(fifo_node, fifomode)) {
        if (errno != EEXIST) {
            report("mkfifo of %s failed", fifo_node);
            return -1;
        }

        /* the path exists.  is it a fifo? */
        if (stat(fifo_node, &sbuf) < 0) {
            report("stat of %s failed", fifo_node);
            return -1;
        }

        /* if not, remove and recreate */
        if (!S_ISFIFO(sbuf.st_mode)) {
            unlink(fifo_node);
            if (mkfifo(fifo_node, fifomode)) {
                report("recreate of %s failed", fifo_node);
                return -1;
            }
        }
    }

    /* mkfifo was affected by umask */
    if (chmod(fifo_node, fifomode)) {
        report("chmod of %s to 0%o failed", fifo_node, fifomode);
        return -1;
    }

    /* open for read/write, since writers are itinerant */
    fd = open(fifo_node, O_RDWR);
    if (fd < 0) {
        report("open %s failed", fifo_node);
        return -1;
    }
    dbg(1, "done with unix fifo init", fifo_node);

    info = IS_CMDFIFO;

    channel = g_io_channel_unix_new (fd);
    g_io_channel_set_encoding (channel, NULL, NULL);
    g_io_channel_set_flags(channel, G_IO_FLAG_NONBLOCK, NULL);
    g_io_add_watch_full (channel, G_PRIORITY_DEFAULT,
             G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
             event_io, (gpointer)info, NULL);

    dbg(1, "done with glib fifo init", fifo_node);
    return fd;
}

static void
deinit_fifo(void)
{
    if (fifo_fd >= 0) {
        close(fifo_fd);
        fifo_fd = -1;
    }
    unlink(command_fifo);
}

/* Manage the uinput device */
static void
deinit_uinput_device(void)
{
    if (uinp_fd < 0) return;

    if (ioctl(uinp_fd, UI_DEV_DESTROY) < 0)
        die("destroy of uinput dev failed");

    close(uinp_fd);
    uinp_fd = -1;
}

static int
setup_uinput(void)
{
    struct uinput_user_dev uinp;
    int e, i;

    uinp_fd = open("/dev/input/uinput", O_WRONLY | O_NDELAY);
    if (uinp_fd < 0) {
        uinp_fd = open("/dev/uinput", O_WRONLY | O_NDELAY);
        if (uinp_fd < 0) {
            report(
                "Unable to open either /dev/input/uinput or /dev/uinput");
            return -1;
        }
    }

    atexit(deinit_uinput_device);

    memset(&uinp, 0, sizeof(uinp));

    strncpy(uinp.name, "olpc-kbdshim virtual input", UINPUT_MAX_NAME_SIZE);
    uinp.id.bustype = BUS_VIRTUAL;
    uinp.id.vendor  = 'p';
    uinp.id.product = 'g';
    uinp.id.version = 'f';

    for (i = 0; i <= KEY_UNKNOWN; i++) {
        if (ioctl(uinp_fd, UI_SET_KEYBIT, i) != 0) {
            report("uinput setup failed, code %d", i);
            return -1;
        }
    }

    e = 0;
    if (!++e || ioctl(uinp_fd, UI_SET_EVBIT, EV_KEY) < 0 || // keys
        !++e || ioctl(uinp_fd, UI_SET_EVBIT, EV_REP) < 0 ||
        !++e || ioctl(uinp_fd, UI_SET_EVBIT, EV_REL) < 0 || // mouse
        !++e || ioctl(uinp_fd, UI_SET_RELBIT, REL_X) < 0 ||
        !++e || ioctl(uinp_fd, UI_SET_RELBIT, REL_Y) < 0 ||
        !++e || ioctl(uinp_fd, UI_SET_RELBIT, REL_WHEEL) < 0 ||
        !++e || ioctl(uinp_fd, UI_SET_RELBIT, REL_HWHEEL) < 0 ||
        !++e || ioctl(uinp_fd, UI_SET_KEYBIT, BTN_LEFT) < 0 ||
        !++e || ioctl(uinp_fd, UI_SET_KEYBIT, BTN_RIGHT) < 0 ||
        !++e || ioctl(uinp_fd, UI_SET_KEYBIT, BTN_MIDDLE) < 0 ||
        !++e || write(uinp_fd, &uinp, sizeof(uinp)) < 0 ||  // device
        !++e || ioctl(uinp_fd, UI_DEV_CREATE) < 0) {
            report("uinput setup failed, step %d", e);
            return -1;
    }

    /* disable timer-based autorepeat, see http://dev.laptop.org/ticket/9690 */
    inject_uinput_event(EV_REP, REP_DELAY, 0);
    inject_uinput_event(EV_REP, REP_PERIOD, 0);

    dbg(1, "uinput device established");

    return 0;
}

static int
match_input_id(input_id_t *id, input_id_t *tmpl)
{
    return id->bustype == tmpl->bustype &&
        (id->vendor == tmpl->vendor || tmpl->vendor == 0xffff) &&
        (id->product == tmpl->product || tmpl->product == 0xffff);
}


static void
add_device (LibHalContext *ctx,
            const char *udi,
            const LibHalPropertySet *properties)
{
        int eventfp;
        GIOChannel *channel;
        unsigned int info;
        const char* device_file;
        struct input_id id;
        unsigned long bit[NBITS(EV_MAX)];

        dbg(1, "add_device called");

        device_file = libhal_ps_get_string (properties, "input.device");
        if (device_file == NULL) {
            report("%s has no property input.device", udi);
            return;
        }

        dbg(1, "adding %s", device_file);

        eventfp = open(device_file, O_RDONLY | O_NONBLOCK);
        if (!eventfp) {
            report("Unable to open %s for reading", device_file);
            return;
        }


        if (ioctl(eventfp, EVIOCGID, &id) < 0) {
            report("failed ioctl EVIOCGID");
            close(eventfp);
            return;
        }

        if (id.bustype == BUS_VIRTUAL &&
                    id.vendor  == 'p' &&
                    id.product == 'g' &&
                    id.version == 'f') {
            dbg(1, "declining to monitor our own output");
            close(eventfp);
            return;
        }

        if (ioctl(eventfp, EVIOCGBIT(0, EV_MAX), bit) < 0) {
            report("failed ioctl EVIOCGBIT");
            close(eventfp);
            return;
        }

        info = 0;

        dbg(1, "testing EV_KEY bit");
        if (test_bit(EV_KEY, bit) && test_bit(EV_REP, bit)) {

            info |= IS_KBD;
            if (match_input_id(&id, &local_kbd_id))
                info |= IS_LOCAL;

            report("%s keyboard %s (%02x:%02x:%02x)",
                (info & IS_LOCAL) ? "matched local" : "found",
                device_file, id.bustype, id.vendor, id.product);
        }

        dbg(1, "testing EV_REL bit");
        if ( test_bit(EV_REL, bit) ) {

            info |= IS_POINTER;
            if (match_input_id(&id, &local_tpad_id))
                info |= IS_LOCAL;

            report("%s pointer %s (%02x:%02x:%02x)",
                (info & IS_LOCAL) ? "matched local" : "found",
                device_file, id.bustype, id.vendor, id.product);
        }

        if (!(info & IS_KBD) && !(info & IS_POINTER))  {
            dbg(1, "not kbd or pointer");
            close(eventfp);
            return;
        }

        if (ioctl(eventfp, EVIOCGRAB, 1) < 0) {
            report("add_device: couldn't GRAB %s", device_file);
        }

        dbg(1, "Listening on %s", device_file);

        devices = g_list_prepend (devices, g_strdup (device_file));

        channel = g_io_channel_unix_new (eventfp);
        g_io_channel_set_encoding (channel, NULL, NULL);
        g_io_channel_set_buffered (channel, 0);

        g_hash_table_insert (inputs, g_strdup(udi), channel);
        dbg(1, "adding to watch");
        int i = g_io_add_watch_full (channel, G_PRIORITY_DEFAULT,
                             G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
                             event_io, (gpointer)info, NULL);
        dbg(1, "watch full returned %d", i);
}


static void
remove_device (LibHalContext *ctx,
            const char *udi,
            const LibHalPropertySet *properties)
{

        GIOChannel *channel, **p_channel = &channel;
        const gchar *device_file;
        GList *lp;
        gboolean handling_udi;

        dbg(1, "remove_device called");
        dbg(1, "Removing channel for '%s'", udi);

        handling_udi = g_hash_table_lookup_extended (inputs, udi, NULL, (gpointer *)p_channel);

        if (!handling_udi) {
                report ("DeviceRemove called for unknown device: '%s'.", udi);
                return;
        }

        if (channel) {
                g_io_channel_shutdown(channel, FALSE, NULL);
                g_io_channel_unref (channel);
        }

        g_hash_table_remove (inputs, udi);

        if ((device_file = libhal_ps_get_string (properties, "input.device")) == NULL) {
                report ("%s has no property input.device", udi);
                return;
        }

        lp = g_list_find_custom (devices, device_file, (GCompareFunc) strcmp);
        if (lp) {
                devices = g_list_remove_link (devices, lp);
                g_free (lp->data);
                g_list_free_1 (lp);
        }

        if (g_hash_table_size (inputs) == 0) {
                report("no more devices, exiting");
                g_main_loop_quit (gmain);
        }
}

static void
sighandler(int sig)
{
    deinit_uinput_device();
    deinit_fifo();
    die("got signal %d", sig);
}

int
main (int argc, char **argv)
{
        DBusConnection *dbus_connection;
        DBusError error;
        const char *commandline;
        int sched_realtime = 0;
        unsigned short b, v, p;
        char *eargp;
        char *cp;
        int c;

        cp = strrchr(argv[0], '/');
        if (cp)
            cp = cp + 1;
        else
            cp = argv[0];
        strncpy(me, cp, sizeof(me));
        me[sizeof(me)-1] = '\0';

        while ((c = getopt(argc, argv, "lsdXvq:n:K:T:g:G:R:A:r:b:V:")) != -1) {
            switch (c) {

            /* daemon options */
            case 'l':
                logtosyslog = 1;
                openlog(me, LOG_PID, LOG_NOTICE);
                break;
            case 's':
                sched_realtime = 1;
                break;
            case 'd':
                debug++;
                break;
            case 'X':
                noxmit = 1;
                break;

            /* algorithmic tuning */
            case 'v':
                reverse = 0;
                break;
            case 'q':
                quantum = strtol(optarg, &eargp, 10);
                if (*eargp != '\0' || quantum <= 0)
                    usage();
                break;
            case 'n':
                ratio = strtol(optarg, &eargp, 10);
                if (*eargp != '\0' || ratio <= 0)
                    usage();
                break;
            case 'K':
                if (sscanf(optarg, "%hx:%hx:%hx", &b, &v, &p) != 3)
                    usage();
                local_kbd_id.bustype = b;
                local_kbd_id.vendor = v;
                local_kbd_id.product = p;
                break;
            case 'T':
                if (sscanf(optarg, "%hx:%hx:%hx", &b, &v, &p) != 3)
                    usage();
                local_tpad_id.bustype = b;
                local_tpad_id.vendor = v;
                local_tpad_id.product = p;
                break;
            case 'g':
            case 'G':
                if (ngrab >= NGRABKEYS) {
                    fprintf(stderr, "%s: too many grab keys specified, %d max\n",
                        me, NGRABKEYS);
                    exit(1);
                }
                grabkey[ngrab].code = strtol(optarg, &eargp, 10);
                if (*eargp != '\0')
                    usage();
                grabkey[ngrab].pass = (c == 'g');
                ngrab++;
                break;

            case 'R':
                command_fifo = optarg;
                break;

            case 'A':
                sysactive_path = optarg;
                break;

            case 'r':
                rotate_cmd = optarg;
                break;

            case 'b':
                brightness_cmd = optarg;
                break;

            case 'V':
                volume_cmd = optarg;
                break;

            default:
                usage();
                break;
            }
        }

        if (optind < argc) {
            report("found non-option argument(s)");
            usage();
        }

        dbus_error_init (&error);
        if ((ctx = libhal_ctx_init_direct (&error)) == NULL) {
                report ("Unable to init libhal context");
                goto out;
        }

        if ((dbus_connection = libhal_ctx_get_dbus_connection(ctx)) == NULL) {
                report ("Cannot get DBus connection");
                goto out;
        }

        if ((commandline = getenv ("SINGLETON_COMMAND_LINE")) == NULL) {
                report ("SINGLETON_COMMAND_LINE not set");
                goto out;
        }
        dbg(1, "initial ctx calls succeeded");

        /* default to XO grab keys, with passthrough.  since they're
         * just modifiers, nothing will happen with them unless X
         * wants it to.
         */
        if (ngrab == 0) {
            grabkey[0].code = KEY_LEFTMETA;
            grabkey[0].pass = 1;
            grabkey[1].code = KEY_RIGHTMETA;
            grabkey[1].pass = 1;
            ngrab = 2;
        }

        report("starting %s version %d", me, VERSION);

        if (!noxmit) { /* initialize uinput */
            if (system("/sbin/modprobe uinput") == 0)
                sleep(1);
            if (setup_uinput() < 0)
                die("unable to find uinput device");
        }

        atexit(deinit_fifo);

        if (command_fifo) {
            fifo_fd = init_fifo(command_fifo);
            if (fifo_fd < 0)
                report("no fifo for rotation control");
        }

        signal(SIGTERM, sighandler);
        signal(SIGHUP, sighandler);
        signal(SIGINT, sighandler);
        signal(SIGQUIT, sighandler);
        signal(SIGABRT, sighandler);
        signal(SIGUSR1, sighandler);
        signal(SIGUSR2, sighandler);

        if (sched_realtime) {
            struct sched_param sparam;
            int min, max;

            /* first, lock down all our memory */
            long takespace[1024];
            memset(takespace, 0, sizeof(takespace)); /* force paging */
            if (mlockall(MCL_CURRENT|MCL_FUTURE) < 0)
                die("unable to mlockall");

            /* then, raise our scheduling priority */
            min = sched_get_priority_min(SCHED_FIFO);
            max = sched_get_priority_max(SCHED_FIFO);

            sparam.sched_priority = (min + max)/2; /* probably always 50 */
            if (sched_setscheduler(0, SCHED_FIFO, &sparam))
                die("unable to set SCHED_FIFO");

            report("memory locked, scheduler priority set");

        }

        libhal_ctx_set_singleton_device_added (ctx, add_device);
        libhal_ctx_set_singleton_device_removed (ctx, remove_device);
        dbg(1, "set_singleton calls done");

        dbus_connection_setup_with_g_main (dbus_connection, NULL);
        dbus_connection_set_exit_on_disconnect (dbus_connection, 0);
        dbg(1, "dbus connection setup calls done");

        dbus_error_init (&error);

        if (!libhal_device_singleton_addon_is_ready (ctx, commandline, &error)) {
                goto out;
        }
        dbg(1, "dbus singleton ready call done");

        inputs = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);

        gmain = g_main_loop_new (NULL, FALSE);

        reinit_activity();
        dbg(1, "starting main loop");
        g_main_loop_run (gmain);

        return 0;

out:
        dbg(1, "An error occured, exiting cleanly");
        if (ctx != NULL) {
                dbus_error_init (&error);
                libhal_ctx_shutdown (ctx, &error);
                libhal_ctx_free (ctx);
        }

        return 0;
}
