/*
** linux.c - Linux user lookup facility.
** Copyright (C) 1998-2001 Ryan McCabe <odin@numb.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License, version 2,
** as published by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
**
** $Id: linux.c,v 1.1 2001/12/29 16:30:25 odin Exp $
*/

#define _GNU_SOURCE

#include <config.h>

#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <syslog.h>
#include <pwd.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <oidentd.h>
#include <oidentd_util.h>
#include <oidentd_inet_util.h>
#include <oidentd_masq.h>
#include <oidentd_options.h>

#ifdef HAVE_LIBUDB
#	include <udb.h>
#endif

#define CFILE		"/proc/net/tcp"
#define CFILE6		"/proc/net/tcp6"
#define MASQFILE	"/proc/net/ip_masquerade"
#define CONNTRACK	"/proc/net/ip_conntrack"

#ifdef WANT_IPV6

/*
** Determine the owner of an IPv6 connection.
** Returns 0 on success, -1 on failure.
*/

int get_user6(	in_port_t lport,
				in_port_t fport,
				struct sockaddr_storage *laddr,
				struct sockaddr_storage *faddr)
{
	FILE *fp;
	char buf[1024];

	lport = ntohs(lport);
	fport = ntohs(fport);

	fp = fopen(CFILE6, "r");
	if (fp == NULL) {
		debug("fopen: %s: %s", CFILE6, strerror(errno));
		return (-1);
	}

	/* Eat the header line. */
	fgets(buf, sizeof(buf), fp);

	while (fgets(buf, sizeof(buf), fp)) {
		struct in6_addr remote6;
		struct in6_addr local6;
		u_int32_t portl_temp;
		u_int32_t portf_temp;
		in_port_t portl;
		in_port_t portf;
		int uid;
		int ret;

		ret = sscanf(buf,
			"%*d: %8x%8x%8x%8x:%x %8x%8x%8x%8x:%x %*x %*X:%*X %*x:%*X %*x %d %*d %*d",
			&local6.s6_addr32[0], &local6.s6_addr32[1], &local6.s6_addr32[2],
			&local6.s6_addr32[3], &portl_temp,
			&remote6.s6_addr32[0], &remote6.s6_addr32[1], &remote6.s6_addr32[2],
			&remote6.s6_addr32[3], &portf_temp, &uid);

		if (ret != 11)
			continue;

		portl = (in_port_t) portl_temp;
		portf = (in_port_t) portf_temp;

		if (!memcmp(&local6, sin_addr(laddr), sizeof(local6)) &&
			!memcmp(&remote6, sin_addr(faddr), sizeof(remote6)) &&
			portl == lport &&
			portf == fport)
		{
			fclose(fp);
			return (uid);
		}
	}

	fclose(fp);
	return (-1);
}

#endif

/*
** Determine the owner of an IPv4 connection.
** Returns 0 on success, -1 on failure.
*/

int get_user4(	in_port_t lport,
				in_port_t fport,
				struct sockaddr_storage *laddr,
				struct sockaddr_storage *faddr)
{
	int uid;
	FILE *fp;
	char buf[1024];

	in_addr_t laddr4 = SIN4(laddr)->sin_addr.s_addr;
	in_addr_t faddr4 = SIN4(faddr)->sin_addr.s_addr;

	lport = ntohs(lport);
	fport = ntohs(fport);

	fp = fopen(CFILE, "r");
	if (fp == NULL) {
		debug("fopen: %s: %s", CFILE, strerror(errno));
		return (-1);
	}

	/* Eat the header line. */
	fgets(buf, sizeof(buf), fp);

	/*
	** The line should never be longer than 1024 chars, so fgets should be OK.
	*/

	while (fgets(buf, sizeof(buf), fp)) {
		int ret;
		u_int32_t portl_temp;
		u_int32_t portf_temp;
		in_port_t portl;
		in_port_t portf;
		in_addr_t local;
		in_addr_t remote;

		ret = sscanf(buf,
			"%*d: %x:%x %X:%x %*x %*X:%*X %*x:%*X %*x %d %*d %*d",
			&local, &portl_temp, &remote, &portf_temp, &uid);

		if (ret != 5)
			continue;

		portl = (in_port_t) portl_temp;
		portf = (in_port_t) portf_temp;

#ifdef MASQ_SUPPORT
		if (opt_enabled(PROXY)) {
			extern struct sockaddr_storage proxy;

			if (faddr4 == SIN4(&proxy)->sin_addr.s_addr &&
				remote != SIN4(&proxy)->sin_addr.s_addr &&
				lport == portl &&
				fport == portf)
			{
				goto out_success;
			}

			if (local == laddr4 &&
				portl == lport &&
				remote == faddr4 &&
				portf == fport)
			{
				goto out_success;
			}
		}
#endif
		if (local == laddr4 &&
			remote == faddr4 &&
			portl == lport &&
			portf == fport)
		{
			goto out_success;
		}
	}

	fclose(fp);
	return (-1);

out_success:
	fclose(fp);
	return (uid);
}

#ifdef MASQ_SUPPORT

/*
** Handle a request to a host that's IP masquerading through us.
** Returns 0 on success, -1 on failure.
*/

int masq(	int sock,
			in_port_t lport,
			in_port_t fport,
			struct sockaddr_storage *laddr,
			struct sockaddr_storage *faddr)
{
	FILE *fp;
	char buf[2048];
	bool netfilter;

	/* laddr is unneeded on Linux */
	(void) laddr;

	/*
	** There's no masq support for IPv6 yet.
	*/

	if (faddr->ss_family != AF_INET)
		return (-1);

	lport = ntohs(lport);
	fport = ntohs(fport);

	fp = fopen(MASQFILE, "r");
	if (fp == NULL) {
		if (errno != ENOENT)
			debug("fopen: %s: %s", MASQFILE, strerror(errno));

		fp = fopen(CONNTRACK, "r");
		if (fp == NULL) {
			if (errno != ENOENT)
				debug("fopen: %s: %s", CONNTRACK, strerror(errno));
			return (-1);
		}

		netfilter = true;
	} else {
		netfilter = false;

		/* Eat the header line. */
		fgets(buf, sizeof(buf), fp);
	}

	while (fgets(buf, sizeof(buf), fp)) {
		char os[24];
		char proto[16];
		in_port_t mport;
		in_port_t lportm;
		in_port_t fportm;
		char user[MAX_ULEN];
		in_addr_t localm;
		in_addr_t remotem;
		extern struct sockaddr_storage proxy;
		struct sockaddr_storage ss;
		int ret;

		if (netfilter == false) {
			u_int32_t mport_temp;
			u_int32_t lportm_temp;
			u_int32_t fportm_temp;

			ret = sscanf(buf, "%15s %X:%X %X:%X %X %*X %*d %*d %*u",
					proto, &localm, &lportm_temp,
					&remotem, &fportm_temp, &mport_temp);

			if (ret != 6)
				continue;

			mport = (in_port_t) mport_temp;
			lportm = (in_port_t) lportm_temp;
			fportm = (in_port_t) fportm_temp;
		} else {
			int l1, l2, l3, l4, r1, r2, r3, r4;
			u_int32_t mport_temp;
			u_int32_t lportm_temp;
			u_int32_t fportm_temp;

			ret = sscanf(buf,
				"%15s %*d %*d ESTABLISHED src=%d.%d.%d.%d dst=%d.%d.%d.%d sport=%d dport=%d %*s %*s %*s dport=%d",
					proto, &l1, &l2, &l3, &l4, &r1, &r2, &r3, &r4,
					&lportm_temp, &fportm_temp, &mport_temp);

			if (ret != 12)
				continue;

			mport = (in_port_t) mport_temp;
			lportm = (in_port_t) lportm_temp;
			fportm = (in_port_t) fportm_temp;

			localm = l1 << 24 | l2 << 16 | l3 << 8 | l4;
			remotem = r1 << 24 | r2 << 16 | r3 << 8 | r4;
		}

		if (strcasecmp(proto, "tcp"))
			continue;

		if (mport != lport)
			continue;

		if (fportm != fport)
			continue;

		if (remotem != ntohl(SIN4(faddr)->sin_addr.s_addr)) {
			if (!opt_enabled(PROXY))
				continue;

			if (SIN4(faddr)->sin_addr.s_addr != SIN4(&proxy)->sin_addr.s_addr)
				continue;

			if (remotem == SIN4(&proxy)->sin_addr.s_addr)
				continue;
		}

		sin_setv4(htonl(localm), &ss);

		if (opt_enabled(FORWARD)) {
			char ipbuf[MAX_IPLEN];

			if (fwd_request(sock, lport, lportm, fportm, &ss) == 0)
				goto out_success;

			get_ip(&ss, ipbuf, sizeof(ipbuf));

			debug("Forward to %s (%d %d) failed.", ipbuf, lportm, fportm);
		}

		ret = find_masq_entry(&ss, user, sizeof(user), os, sizeof(os));
		if (ret == 0) {
			char ipbuf[MAX_IPLEN];

			sockprintf(sock, "%d , %d : USERID : %s : %s\r\n",
				lport, fport, os, user);

			get_ip(faddr, ipbuf, sizeof(ipbuf));

			o_log(NORMAL,
				"[%s] (Masqueraded) Successful lookup: %d , %d : %s",
				ipbuf, lport, fport, user);

			goto out_success;
		}
	}

	fclose(fp);
	return (-1);

out_success:
	fclose(fp);
	return (0);
}

#endif
